/*! \file logging_web.c
 *  Logging support code using a JS callback. This module sends log
 *  messages to a JavaScript callback named `on_log`.
 *  */
/*
 * (C) 2026 by Timur Davydov <dtv.comp@gmail.com>
 * All Rights Reserved
 *
 * SPDX-License-Identifier: GPL-2.0+
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*! \addtogroup logging
 *  @{
 * \file logging_web.c */

#include "config.h"

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#include <osmocom/core/talloc.h>
#include <osmocom/core/utils.h>
#include <osmocom/core/logging.h>
#include <osmocom/core/byteswap.h>

#if defined(__EMSCRIPTEN__)

#include <emscripten.h>

EM_JS(void, on_log_wrapper, (const char *subsys, int level, const char *msg), {
	return on_log(subsys, level, msg);
});
#else
void on_log_wrapper(const char *subsys, int level, const char *msg)
{
}
#endif

static void _web_raw_output(struct log_target *target, int subsys,
			       unsigned int level, const char *file,
			       int line, int cont, const char *format,
			       va_list ap)
{
	const int msgLen = 4096;
	char msg[msgLen + 1];
	const char *subsys_name = log_category_name(subsys);
	char subsys_buf[16];
	int rc;

	if (subsys_name)
		OSMO_STRLCPY_ARRAY(subsys_buf, subsys_name + 1);
	else
		subsys_buf[0] = '\0';

	rc = vsnprintf(msg, msgLen, format, ap);
	if (rc > 0) {
		if (msg[rc - 1] == '\n') msg[rc - 1] = '\0';
		on_log_wrapper(subsys_buf, level, msg);
	}
}

/*! Create a new logging target for JS callback logging (uses `on_log`)
 *  \returns Log target in case of success, NULL in case of error
 */
struct log_target *log_target_create_web(void)
{
	struct log_target *target;

	target = log_target_create();
	if (!target)
		return NULL;

	target->type = LOG_TGT_TYPE_WEB;
	target->raw_output = _web_raw_output;

	return target;
}

/* @} */
