#pragma once

#include <cstdint>

/*!
 * \class rijndael
 * \brief  Implement rijndael as defined in ETSI TS 135 206 V16.0.0 (2020-08)
 */ 
class rijndael {
  const uint8_t S[256]{ //<! Rijndael S box table
    99,124,119,123,242,107,111,197, 48,  1,103, 43,254,215,171,118,
    202,130,201,125,250, 89, 71,240,173,212,162,175,156,164,114,192,
    183,253,147, 38, 54, 63,247,204, 52,165,229,241,113,216, 49, 21,
    4,199, 35,195, 24,150,  5,154,  7, 18,128,226,235, 39,178,117,
    9,131, 44, 26, 27,110, 90,160, 82, 59,214,179, 41,227, 47,132,
    83,209,  0,237, 32,252,177, 91,106,203,190, 57, 74, 76, 88,207,
    208,239,170,251, 67, 77, 51,133, 69,249,  2,127, 80, 60,159,168,
    81,163, 64,143,146,157, 56,245,188,182,218, 33, 16,255,243,210,
    205, 12, 19,236, 95,151, 68, 23,196,167,126, 61,100, 93, 25,115,
    96,129, 79,220, 34, 42,144,136, 70,238,184, 20,222, 94, 11,219,
    224, 50, 58, 10, 73,  6, 36, 92,194,211,172, 98,145,149,228,121,
    231,200, 55,109,141,213, 78,169,108, 86,244,234,101,122,174,  8,
    186,120, 37, 46, 28,166,180,198,232,221,116, 31, 75,189,139,138,
    112, 62,181,102, 72,  3,246, 14, 97, 53, 87,185,134,193, 29,158,
    225,248,152, 17,105,217,142,148,155, 30,135,233,206, 85, 40,223,
    140,161,137, 13,191,230, 66,104, 65,153, 45, 15,176, 84,187, 22,
   };
  const uint8_t XTIME[256]{ //<! Multiplication by x in GF(2^8)
    0,  2,  4,  6,  8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30,
    32, 34, 36, 38, 40, 42, 44, 46, 48, 50, 52, 54, 56, 58, 60, 62,
    64, 66, 68, 70, 72, 74, 76, 78, 80, 82, 84, 86, 88, 90, 92, 94,
    96, 98,100,102,104,106,108,110,112,114,116,118,120,122,124,126,
    128,130,132,134,136,138,140,142,144,146,148,150,152,154,156,158,
    160,162,164,166,168,170,172,174,176,178,180,182,184,186,188,190,
    192,194,196,198,200,202,204,206,208,210,212,214,216,218,220,222,
    224,226,228,230,232,234,236,238,240,242,244,246,248,250,252,254,
    27, 25, 31, 29, 19, 17, 23, 21, 11,  9, 15, 13,  3,  1,  7,  5,
    59, 57, 63, 61, 51, 49, 55, 53, 43, 41, 47, 45, 35, 33, 39, 37,
    91, 89, 95, 93, 83, 81, 87, 85, 75, 73, 79, 77, 67, 65, 71, 69,
    123,121,127,125,115,113,119,117,107,105,111,109, 99, 97,103,101,
    155,153,159,157,147,145,151,149,139,137,143,141,131,129,135,133,
    187,185,191,189,179,177,183,181,171,169,175,173,163,161,167,165,
    219,217,223,221,211,209,215,213,203,201,207,205,195,193,199,197,
    251,249,255,253,243,241,247,245,235,233,239,237,227,225,231,229
   };

  uint8_t _round_keys[11][4][4]; //<! Rijndael round subkeys

public: //! \publicsection
  /*!
   * \fn codec();
   * \brief  Default constructor
   */
  rijndael();
  virtual ~rijndael();

  /*!
   * \fn int rijndael_key_schedule(const uint8_t p_key[16]);
   * \brief  Rijndael key schedule function. Takes 16-byte key and creates all Rijndael's internal subkeys ready for encryption
   * \param[in] p_key The data to encrypt
   * \param[out] p_output The encrypted result
   * \return 0 on success, -1 otherwise
   */
  int rijndael_key_schedule(const uint8_t p_key[16]);

  /*!
   * \fn void rijndael_encrypt(const uint8_t p_input[16], uint8_t p_output[16]);
   * \brief  Rijndael encryption function. Takes 16-byte input and creates 16-byte output (using round keys already derived from 16-byte key)
   * \param[in] p_input The data to encrypt
   * \param[out] p_output The encrypted result
   */
  void rijndael_encrypt(const uint8_t p_input[16], uint8_t p_output[16]);

private:
  void key_add(uint8_t p_state[4][4], const uint8_t p_round_keys[11][4][4], const int p_round);

  /*!
   * \fn void byte_sub(uint8_t p_state[4][4]);
   * \brief  Byte substitution transformation
   * \param[in] p_state
   */
  void byte_sub(uint8_t p_state[4][4]);

  /*!
   * \fn void shift_row(uint8_t p_state[4][4]);
   * \brief  Row shift transformation
   * \param[in] p_state
   */
  void shift_row(uint8_t p_state[4][4]);

  /*!
   * \fn void mix_column(uint8_t p_state[4][4]);
   * \brief  MixColumn transformation
   * \param[in] p_state 
   */
  void mix_column(uint8_t p_state[4][4]);
}; // End of class rijndael
