#!/bin/bash -xe

SCRIPT_DIR=$( cd -- "$( dirname -- "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )

NUM_ENB=200

NUM_SEGMENTS=2
NUM_ENB_SEGMENT=$((NUM_ENB / NUM_SEGMENTS))

# Interface towards UPF:
ADDR_GW_CN="172.16.31.200"
ADDR_GW_CN2="172.16.31.201"

case $(hostname) in
epyc1)
        IFACE_RAN="enp129s0f0np0"
        IFACE_CN="enp129s0f1np1"
        # TRex peer is c240-1 Mellanox NIC:
        MAC_RAN_PEER="88:e9:a4:3b:f0:48"
        MAC_CN_PEER="88:e9:a4:3b:f0:49"
        # TRex peer is c240-2 Mellanox NIC:
        MAC_RAN_PEER2="88:e9:a4:3b:21:38"
        MAC_CN_PEER2="88:e9:a4:3b:21:39"
        ;;
*)
        echo "UNEXPECTED HOSTNAME: $(hostname)"
        exit 1
        ;;
esac

setup_arp_enb() {
        local enb_offset="$1"
        local iface_ran="$2"
        local mac_ran_peer="$3"
        local enb_start="$((2 + enb_offset))"
        local enb_end="$((enb_start + NUM_ENB_SEGMENT - 1))"
        # TRex doesn't answer ARPs, so we need to set up the peers:
        for i in $(seq "$enb_start" "$enb_end"); do
                local ip_addr
                ip_addr="$(printf "172.16.32.%u" "$i")"
                # Delete needed to potentially drop incomplete entries created when trying to Tx traffic:
                sudo ip neigh del "$ip_addr" lladdr "$mac_ran_peer" nud permanent dev "$iface_ran" || true
                sudo ip neigh add "$ip_addr" lladdr "$mac_ran_peer" nud permanent dev "$iface_ran" || true
        done
}

setup_arp_cn() {
        local cn_ip_addr="$1"
        local iface_cn="$2"
        local mac_cn_peer="$3"
        sudo ip neigh del "$cn_ip_addr" lladdr "$mac_cn_peer" nud permanent dev "$iface_cn" || true
        sudo ip neigh add "$cn_ip_addr" lladdr "$mac_cn_peer" nud permanent dev "$iface_cn" || true
}

# Disable ethernet flow control:
sudo ethtool -A $IFACE_RAN autoneg off rx off tx off
sudo ethtool -A $IFACE_CN autoneg off rx off tx off
# Disable GRO / LRO:
sudo ethtool -K $IFACE_RAN gro off lro off
sudo ethtool -K $IFACE_CN gro off lro off
# Increase NIC buffers:
sudo ethtool -G $IFACE_RAN rx 8192 tx 8192
sudo ethtool -G $IFACE_CN rx 8192 tx 8192

# Enable IP forwarding:
sudo sysctl -w net.ipv4.ip_forward=1

# Increase UDP buffer memory:
sudo sysctl -w net.ipv4.udp_mem="763563 900000000 1000000000"
sudo sysctl -w net.core.optmem_max=16000000
sudo sysctl -w net.core.rmem_max=2000000000
sudo sysctl -w net.core.rmem_default=2000000000
sudo sysctl -w net.core.wmem_max=2000000000
sudo sysctl -w net.core.wmem_default=2000000000
sudo sysctl -w net.core.netdev_max_backlog=2000
sudo sysctl -w net.core.netdev_budget=600

# Set up UPF address:
sudo ip addr add 172.16.32.1/24 dev $IFACE_RAN || true
sudo ip link set up dev $IFACE_RAN || true
sudo ip addr add 172.16.31.2/24 dev $IFACE_CN || true
sudo ip link set up dev $IFACE_CN || true

# TRex doesn't answer ARPs, so we need to set up the peers:
setup_arp_enb 0 "$IFACE_RAN" "$MAC_RAN_PEER"
setup_arp_enb "$NUM_ENB_SEGMENT" "$IFACE_RAN" "$MAC_RAN_PEER2"

setup_arp_cn "$ADDR_GW_CN" "$IFACE_CN" "$MAC_CN_PEER"
setup_arp_cn "$ADDR_GW_CN2" "$IFACE_CN" "$MAC_CN_PEER2"

# Add route towards HTTP server on the CN side, aka "the Internet"
sudo ip route del 48.0.0.0/16 via $ADDR_GW_CN dev $IFACE_CN || true
sudo ip route add 48.0.0.0/16 via $ADDR_GW_CN dev $IFACE_CN || true
sudo ip route del 48.$NUM_ENB_SEGMENT.0.0/16 via $ADDR_GW_CN2 dev $IFACE_CN || true
sudo ip route add 48.$NUM_ENB_SEGMENT.0.0/16 via $ADDR_GW_CN2 dev $IFACE_CN || true

# Apply host specific iface names:
sed -i "s/TEMPLATE_IFACE_RAN/$IFACE_RAN/g" "${SCRIPT_DIR}/eupf.yaml"
sed -i "s/TEMPLATE_IFACE_CN/$IFACE_CN/g" "${SCRIPT_DIR}/eupf.yaml"
