#!/usr/bin/env python3
# SPDX-License-Identifier: GPL-2.0-or-later
# Copyright 2022 sysmocom - s.f.m.c. GmbH <info@sysmocom.de>
import glob
import os
import shutil
import lib
import lib.git


def get_spec_in_path(project):
    repo_path = lib.git.get_repo_path(project)
    ret = glob.glob(f"{repo_path}/**/contrib/*.spec.in", recursive=True)
    assert len(ret) < 2
    return ret[0] if ret else None


def get_source_name(project):
    """ Get the name to the .tar.xz file generated by dpkg-buildpackage, that
        we also use in the rpm spec file. This requires
        lib.debian.move_files_to_output() to run first."""
    path_output = lib.get_output_path(project)
    ret = glob.glob(f"{path_output}/*.tar.*")
    assert len(ret) == 1
    return os.path.basename(ret[0])


def add_depend(project, pkgname, version):
    """ Add a 'Requires: ...' line below the 'Name:' and '%package' lines of
        the .spec.in file.
        :param pkgname: of the meta-package to depend on (e.g. osmocom-nightly)
        :param version: of the meta-pkgname to depend on or None """
    path = get_spec_in_path(project)

    req_line = f"Requires: {pkgname}"
    if version:
        req_line += f" = {version}"

    # Main package
    lib.run_cmd(["sed", "-e", "/^Name:/a\\", "-e", req_line, "-i", path])

    # Subpackages
    lib.run_cmd(["sed", "-e", "/^%package/a\\", "-e", req_line, "-i", path])


def generate(project, version, epoch):
    """ Update the version and source in the .spec.in file.
        :param version: from get-version-gen script """
    print(f"{project}: generating rpm spec file")
    path = get_spec_in_path(project)
    source = get_source_name(project)

    # rpmbuild doesn't allow '-' in version
    version = version.replace("-", ".")

    # Version
    lib.run_cmd(["sed", f"s/^Version:.*/Version: {version}/", "-i", path])
    if epoch:
        epoch_line = f"Epoch: {epoch}"
        lib.run_cmd(["sed", "-e", "/^Version:/a\\", "-e", epoch_line, "-i",
                     path])

    # Source: add the one generated by dpkg-buildpackage and another line with
    # rpmlintrc
    lib.run_cmd(["sed", f"s/^Source:.*/Source: {source}/", "-i", path])
    rpmlint_line = "Source1: rpmlintrc"
    lib.run_cmd(["sed", "-e", "/^Source:/a\\", "-e", rpmlint_line, "-i", path])

    # rpmbuild assumes the directory inside the source tarball has the version
    # at the end, but dpkg-buildpackage doesn't generate it that way. Set the
    # directory with "%setup -n dirname".
    lib.run_cmd(["sed", f"s/^%setup/%setup -n {os.path.basename(project)}/",
                 "-i", path])


def copy_to_output(project):
    # rpm spec
    path_spec_in = get_spec_in_path(project)
    path_output = lib.get_output_path(project)
    name_spec = os.path.basename(path_spec_in)[:-3]  # remove '.in'
    shutil.copy(path_spec_in, f"{path_output}/{name_spec}")

    # rpmlintrc
    shutil.copy(f"{lib.config.path_top}/data/rpmlintrc", path_output)
