#!/bin/sh -e
DIR="$(realpath "$(dirname "$0")/..")"
TEMP="$DIR/.temp.apk"
KEYSTORE="$DIR/.apk-sign-keystore.p12"
PROJECT="$1"
VERSION="$2"

check_usage() {
	if [ "$#" != 2 ]; then
		echo "usage:"
		echo "  contrib/add-apk.sh PROJECT VERSION"
		echo
		echo "examples:"
		echo "  contrib/add-apk.sh android-apdu-proxy 0.1.0"
		echo "  contrib/add-apk.sh easyuicc 1.0.0"
		echo "  contrib/add-apk.sh omnt 0.4"
		exit 1
	fi
}

check_fdroid_init() {
	if ! [ -f "$DIR/public/fdroid/config.yml" ]; then
		echo "ERROR: follow instructions in INITIALIZE.md first"
		exit 1
	fi
}

check_keystore() {
	if ! [ -f "$KEYSTORE" ] || ! [ -f "$(realpath "$KEYSTORE")" ]; then
		echo "ERROR: keystore not found!"
		echo
		echo "Put a symlink to the real apk sign keystore here:"
		echo "  $KEYSTORE"
		echo
		echo "Or use contrib/generate-test-apk-sign-keystore.sh for development."
		exit 1
	fi
}

download_apk() {
	local url

	case "$PROJECT" in
		android-apdu-proxy)
			url="https://downloads.osmocom.org/binaries/android/.apks_testsig/android-apdu-proxy/android-apdu-proxy-$VERSION.apk"
			;;
		easyuicc)
			url="https://gitea.angry.im/PeterCxy/OpenEUICC/releases/download/unpriv-v$VERSION/app-unpriv-release.apk"
			;;
		omnt)
			url="https://github.com/omnt/OpenMobileNetworkToolkit/releases/download/$VERSION/omnt-release.apk"
			;;
		*)
			echo "ERROR: unknown project: $PROJECT"
			exit 1
	esac

	if ! wget -O "$TEMP" "$url"; then
		set +x
		case "$url" in
			"https://downloads.osmocom.org"*)
				echo "Maybe you need to trigger this job?"
				echo "https://jenkins.osmocom.org/jenkins/job/Osmocom-release-android-apks/"
				echo
				;;
		esac

		echo "ERROR: download failed"
		exit 1
	fi
}

sign_apk() {
	apksigner verify --print-certs "$TEMP"
	apksigner sign --ks "$(realpath "$KEYSTORE")" "$TEMP"
	apksigner verify --print-certs "$TEMP"
}

update_fdroid() {
	mkdir -p "$DIR/public/fdroid/repo"
	cp "$TEMP" "$DIR/public/fdroid/repo/$PROJECT-$VERSION.apk"
	cd "$DIR/public/fdroid"
	fdroid update -c
}

create_symlink() {
	mkdir -p "$DIR/public/apks/$PROJECT"
	ln -sf "../../fdroid/repo/$PROJECT-$VERSION.apk" "$DIR/public/apks/$PROJECT/$PROJECT-$VERSION.apk"
	ln -sf "../../fdroid/repo/$PROJECT-$VERSION.apk" "$DIR/public/apks/$PROJECT/$PROJECT-latest.apk"
}

success() {
	set +x
	echo "Success!"
	echo
	echo "Next steps:"
	echo "* 'git add -A'"
	echo "* 'git commit'"
	echo "* 'git push'"
	echo "* Run https://jenkins.osmocom.org/jenkins/job/master-osmocom-android-binaries/"
}

check_usage "$@"
check_fdroid_init
check_keystore
set -x
download_apk
sign_apk
update_fdroid
create_symlink
success
