%%%-------------------------------------------------------------------
%%% @author Andreas Schultz <aschultz@tpip.net>
%%% @copyright (C) 2011, Andreas Schultz
%%% @doc
%%%
%%% @end
%%% Created : 29 Jun 2011 by Andreas Schultz <aschultz@tpip.net>
%%%-------------------------------------------------------------------
-module(netlink_SUITE).

-compile(export_all).

-include_lib("common_test/include/ct.hrl").
-include("../include/netlink.hrl").

-define(equal(Expected, Actual),
    (fun (Expected@@@, Expected@@@) -> true;
         (Expected@@@, Actual@@@) ->
             ct:pal("MISMATCH(~s:~b, ~s)~nExpected: ~p~nActual:   ~p~n",
                    [?FILE, ?LINE, ??Actual, Expected@@@, Actual@@@]),
             false
     end)(Expected, Actual) orelse error(badmatch)).

% hexstr2bin
hexstr2bin(S) ->
    list_to_binary(hexstr2list(S)).

hexstr2list([X,Y|T]) ->
    [mkint(X)*16 + mkint(Y) | hexstr2list(T)];
hexstr2list([]) ->
    [].

mkint(C) when $0 =< C, C =< $9 ->
    C - $0;
mkint(C) when $A =< C, C =< $F ->
    C - $A + 10;
mkint(C) when $a =< C, C =< $f ->
    C - $a + 10.

%%--------------------------------------------------------------------

conntrack_new() ->
    hexstr2bin("c400000000010000000000000000000002000000340001801400018008000100"
	       "c0a8021808000200c0a802091c000280050001000600000006000200d7640000"
	       "0600030000190000340002801400018008000100c0a8020908000200c0a80218"
	       "1c0002800500010006000000060002000019000006000300d764000008000c00"
	       "35e69900080003000000018e080007000000003c300004802c00018005000100"
	       "0500000005000200070000000500030007000000060004003700000006000500"
	       "23000000").

rt_newneigh_1() ->
    hexstr2bin("4c0000001c000000000000000000000002000000050000000200000108000100"
	       "c0a802030a000200000e0cba03a2000008000400040000001400030000000000"
	       "000000000000000004000000").

rt_newneigh_2() ->
    hexstr2bin("580000001c00000000000000000000000a000000050000000200000114000100"
	       "200106f812d9000000000000000000090a000200005056ae29ac000008000400"
	       "010000001400030021010000210100000000000003000000").

rt_delroute() ->
    hexstr2bin("9c0000001900000000000000000000000a800000ff0000010002000008000f00"
	       "ff00000014000100ff0200000000000000000000000000fb1c00080008000200"
	       "dc05000008000800a005000008000a00ffffffff14000500ff02000000000000"
	       "00000000000000fb0800040005000000080006000000000024000c0000000000"
	       "00120000000000000000000002000000000000000000000000000000").

rt_newprefix() ->
    hexstr2bin("3c0000003400000000000000000000000a000000050000000340030014000100"
	       "200106f812d9000000000000000000000c000200803a0900008d2700").

rt_newlink_1() ->
    hexstr2bin("080200001000000000000000000000000a000100060000000310010000000000"
	       "0a000300766c616e340000000a0001000019b947fa13000008000400dc050000"
	       "0800050005000000c0010c00080001000000008014000500ffff00006c354a04"
	       "c010000064000000780002000000000040000000dc0500000100000001000000"
	       "010000000100000003000000900100006400000000000000803a090080510100"
	       "0500000058020000100000000000000001000000010000000000000000000000"
	       "0000000000000000000000000000000000000000000000000100000000000000"
	       "fc0003001f000000000000005001000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000005001000000000000000000000000000057010000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000050010000000000005a0100000000000000000000"
	       "0000000000000000000000006f910000000000004f930000000000006f910000"
	       "000000004794000000000000000000000000000000000000000000002c000600"
	       "0500000000000000000000000000000000000000000000000700000000000000"
	       "0000000000000000").

rt_newlink_2() ->
    hexstr2bin("bc01000010000000000000000000000000000100020000000310000001000000"
	       "09000300657468310000000008000d00e8030000050010000200000005001100"
	       "0000000008000400dc0500000f000600706669666f5f66617374000024000e00"
	       "0000000000000000000000000000000080ef330000c9ffff1300000800000000"
	       "0a0001000050044d739e00000a000200ffffffffffff00006000070000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "000000000000000000000000000000000000000000000000bc00170000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000000000000000000000800150000000000").


rt_linkinfo_1() ->
    hexstr2bin("e001000010000000000000000000000000000100060000000210000001000000"
	       "0a000300766c616e3400000008000d0000000000050010000200000005001100"
	       "0000000008000400dc05000008000500050000000c0006006e6f717565756500"
	       "24000e0000000000000000000000000000000000000000000000000000000000"
	       "000000000a0001000019b947fa1300000a000200ffffffffffff000060000700"
	       "040f000089120000da170600a261070000000000000000000000000000000000"
	       "040f000000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000000000000000000000000000000000000bc001700"
	       "040f0000000000008912000000000000da17060000000000a261070000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "040f000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000002800120009000100"
	       "766c616e000000001800020006000100040000000c00020001000000ffffffff"
	       "").

rt_linkinfo_complex() ->
    hexstr2bin("ac01000010000200000000000000ffff00000403010000004900010000000000"
	       "070003006c6f000008000d000000000005001000000000000500110000000000"
	       "08000400344000000c0006006e6f71756575650024000e000000000000000000"
	       "0000000000000000000000000000000000000000000000000a00010000000000"
	       "000000000a0002000000000000000000600007009a5c01009a5c01006811c300"
	       "6811c30000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000000000000bc0017009a5c0100000000009a5c0100"
	       "000000006811c300000000006811c30000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "000000000000000000000000b401000010000200000000000000ffff00000100"
	       "02000000431001000000000009000300657468300000000008000d00e8030000"
	       "0500100006000000050011000000000008000400dc050000070006006d710000"
	       "24000e0000000000000000000000000000000000000000000000000010000000"
	       "000000000a0001000019b947fa1300000a000200ffffffffffff000060000700"
	       "66798003d41ad3026bc1df4e99521ce500000000000000000000000000000000"
	       "e06e0f0000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000000000000000000000000000000000000bc001700"
	       "6679800300000000d41ad302000000006bc1df4e0c00000099521ce506000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "e06e0f0000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000800150000000000"
	       "bc01000010000200000000000000ffff00000100030000000310000000000000"
	       "09000300657468310000000008000d00e8030000050010000200000005001100"
	       "0000000008000400dc0500000f000600706669666f5f66617374000024000e00"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0a000100000e0cc3b91b00000a000200ffffffffffff00006000070000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "000000000000000000000000000000000000000000000000bc00170000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000000000000000000000000000000000000800150000000000c0010000"
	       "10000200000000000000ffff0000010004000000431001000000000009000300"
	       "746170300000000008000d00f401000005001000000000000500110000000000"
	       "08000400dc0500000f000600706669666f5f66617374000024000e0000000000"
	       "000000000000000000000000000000000000000000000000000000000a000100"
	       "d2f88f65f2ef00000a000200ffffffffffff000060000700b4070000cba31400"
	       "b13205007cdc6f0b000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000bc001700b407000000000000"
	       "cba3140000000000b1320500000000007cdc6f0b000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "000000000000000000000000000000000c0012000800010074756e00ac010000"
	       "10000200000000000000ffff0000010005000000431001000000000008000300"
	       "6272300008000d00000000000500100000000000050011000000000008000400"
	       "dc0500000c0006006e6f71756575650024000e00000000000000000000000000"
	       "00000000000000000000000000000000000000000a0001000019b947fa130000"
	       "0a000200ffffffffffff00006000070082295602eebf0602bcc06bce5d4907a5"
	       "000000000000000000000000000000000fc31300000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "000000000000000000000000bc0017008229560200000000eebf060200000000"
	       "bcc06bce0b0000005d4907a50600000000000000000000000000000000000000"
	       "000000000000000000000000000000000fc31300000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000e001000010000200000000000000ffff0000010006000000"
	       "02100000000000000a000300766c616e3400000008000d000000000005001000"
	       "02000000050011000000000008000400dc05000008000500050000000c000600"
	       "6e6f71756575650024000e000000000000000000000000000000000000000000"
	       "0000000000000000000000000a0001000019b947fa1300000a000200ffffffff"
	       "ffff000060000700040f000089120000da170600a26107000000000000000000"
	       "0000000000000000040f00000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000bc001700040f0000000000008912000000000000da17060000000000"
	       "a261070000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000040f00000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "2800120009000100766c616e000000001800020006000100040000000c000200"
	       "01000000ffffffffe001000010000200000000000000ffff0000010007000000"
	       "43100100000000000c000300766c616e3130310008000d000000000005001000"
	       "06000000050011000000000008000400dc05000008000500050000000c000600"
	       "6e6f71756575650024000e000000000000000000000000000000000000000000"
	       "0000000000000000000000000a0001000019b947fa1300000a000200ffffffff"
	       "ffff000060000700000000008103000000000000a29b01000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000bc001700000000000000000081030000000000000000000000000000"
	       "a29b010000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "2800120009000100766c616e000000001800020006000100650000000c000200"
	       "01000000ffffffffe001000010000200000000000000ffff0000010008000000"
	       "43100100000000000c000300766c616e3130330008000d000000000005001000"
	       "06000000050011000000000008000400dc05000008000500050000000c000600"
	       "6e6f71756575650024000e000000000000000000000000000000000000000000"
	       "0000000000000000000000000a0001000019b947fa1300000a000200ffffffff"
	       "ffff000060000700000000008003000000000000579b01000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "00000000bc001700000000000000000080030000000000000000000000000000"
	       "579b010000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "0000000000000000000000000000000000000000000000000000000000000000"
	       "2800120009000100766c616e000000001800020006000100670000000c000200"
	       "01000000ffffffff").

nfq_unbind() ->
    hexstr2bin("1c000000020305000000000000000000000000000800010004000002").

nfq_unbind_answer() ->
    hexstr2bin("240000000200000000000000ae040000000000001c0000000203050000000000"
	       "00000000").

nfq_bind_queue() ->
    hexstr2bin("1c000000020305000000000000000000000000000800010003000002").

nfq_bind_queue_answer() ->
    hexstr2bin("240000000200000000000000bd040000000000001c0000000203050000000000"
	       "00000000").

nfq_bind_socket() ->
    hexstr2bin("1c000000020305000000000000000000000000000800010001000000").

nfq_bind_socket_answer() ->
    hexstr2bin("240000000200000000000000bd040000000000001c0000000203050000000000"
	       "00000000").

nfq_set_copy_mode() ->
    hexstr2bin("2000000002030500000000000000000000000000090002000000ffff02000000").

nfq_set_copy_mode_answer() ->
    hexstr2bin("240000000200000000000000bd04000000000000200000000203050000000000"
"00000000").

nfq_set_verdict() ->
    hexstr2bin("20000000010301000000000000000000000000000c0002000000000100000001").

nfq_packet() ->
    hexstr2bin("b0000000000300000000000000000000020000000b0001000000000108000000"
	       "080005000000000b08000300080c0000100009000006000000505696c4030000"
	       "140004000000000055a7c58900000000000ad18a5c000a0045c0005857220000"
	       "0159d538ac1c0011e000000502010034ac1c0011000000000000000200000110"
	       "00075e9cffffff00000a020100000028ac1c0011ac1c00200a000001ac1c0010"
	       "e7944f54d33f540c642e23c7b99d3f09").

nft_requests() ->
    [hexstr2bin("1400000010000100000000000000000002000a0014000000090a050001000000"
		"00000000020000001400000011000100020000000000000002000a00"),
     hexstr2bin("14000000100a0100030000000000000000000000"),
     hexstr2bin("14000000010a0103050000000000000000000000"),
     hexstr2bin("1c000000010a05000500000000000000020000000800010072617700"),
     hexstr2bin("1c0000000a0a05030500000000000000020000000800010072617700"),
     hexstr2bin("14000000040a0103050000000000000002000000"),
     hexstr2bin("14000000070a0103050000000000000002000000"),
     hexstr2bin("20000000010a05000500000000000000020000000b0001006d616e676c650000"),
     hexstr2bin("200000000a0a05030500000000000000020000000b0001006d616e676c650000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b0001006d616e676c650000"
		"090002006d61703000000000"),
     hexstr2bin("14000000040a0103050000000000000002000000"),
     hexstr2bin("14000000070a0103050000000000000002000000"),
     hexstr2bin("30000000000b0500f5c0c15500000000020000000b000100736f636b65740000"
		"08000200000000020800030000000000"),
     hexstr2bin("30000000000b0500f5c0c15500000000020000000b000100736f636b65740000"
		"08000200000000020800030000000000"),
     hexstr2bin("2c000000000b0500f5c0c1550000000002000000080001007365740008000200"
		"000000040800030000000000"),
     hexstr2bin("2c000000000b0500f5c0c1550000000002000000080001007365740008000200"
		"000000040800030000000000"),
     hexstr2bin("2c000000000b0500f5c0c1550000000002000000080001007365740008000200"
		"000000040800030000000000"),
     hexstr2bin("2c000000000b0500f5c0c1550000000002000000080001007365740008000200"
		"000000040800030000000000"),
     %% RTM_GETLINK
     %% hexstr2bin("1400000012000103f5c0c1550000000011000000"),
     hexstr2bin("1c000000010a0500050000000000000002000000080001006e617400"),
     hexstr2bin("1c0000000a0a0503050000000000000002000000080001006e617400"),
     hexstr2bin("14000000040a0103050000000000000002000000"),
     hexstr2bin("14000000070a0103050000000000000002000000"),
     hexstr2bin("20000000010a05000500000000000000020000000b00010066696c7465720000"),
     hexstr2bin("200000000a0a05030500000000000000020000000b00010066696c7465720000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703000000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703100000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703200000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703300000000"),

     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703400000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703500000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703600000000"),
     hexstr2bin("2c0000000d0a05030500000000000000020000000b00010066696c7465720000"
		"090002006d61703700000000"),
     hexstr2bin("14000000040a0103050000000000000002000000"),
     hexstr2bin("14000000070a0103050000000000000002000000"),
     hexstr2bin("20000000130a05030000000000000000020000000b0001006d616e676c650000"),
     hexstr2bin("20000000130a05030000000000000000020000000b00010066696c7465720000"),
     hexstr2bin("20000000130a050300000000000000000a0000000b0001006d616e676c650000"),
     hexstr2bin("20000000130a050300000000000000000a0000000b00010066696c7465720000"),
     hexstr2bin("58000000120A0208000000003F7B0000020000560B00010066696C7465720000"
		"0A00020043496E5F35000000080003000000000108000500000000001C000400"
		"0C00010000000000000000000C000200000000000000000058000000120A0208"
		"000000003F7B0000020000560B00010066696C74657200000B000200434F7574"
		"5F350000080003000000000108000500000000001C0004000C00010000000000"
		"000000000C0002000000000000000000")
    ].

genl_request() ->
    hexstr2bin("240000001000050077ce08560000000003010000060001001000000008000200"
	       "67747000").

vrf_enslave_link() ->
    [hexstr2bin("2800000010000500a35b355800000000000000006f0000000000000000000000"
		"08000a0005000000")].

vrf_add_route() ->
    [hexstr2bin("3000000012000100c5b030580000000000000000000000000000000000000000"
		"0d0003007672662d626c756500000000"),
     hexstr2bin("f004000010000000c5b03058f02f00000000010005000000c104010000000000"
		"0d0003007672662d626c75650000000008000d00e80300000500100000000000"
		"050011000000000008000400dc05000008001b000000000008001e0000000000"
		"08001f000100000008002800ffff000008002900000001000800200001000000"
		"05002100010000000f000600706669666f5f6661737400000800230000000000"
		"050027000000000024000e000000000000000000000000000000000000000000"
		"0000000000000000000000000a0001004e5fb1609a2b00000a000200ffffffff"
		"ffff000064000700000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000c40017000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000001800120008000100767266000c00020008000100"
		"0a000000cc021a00840002008000010000000000000000000000000001000000"
		"0100000001000000010000000100000001000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"00000000000000000000000000000000000000000000000010270000e8030000"
		"00000000000000000000000044020a00080001000000008014000500ffff0000"
		"de843500bc530000e8030000b00002000000000040000000dc05000001000000"
		"01000000010000000100000003000000a00f0000e803000002000000803a0900"
		"8051010003000000580200001000000000000000010000000100000001000000"
		"60ea0000000000000000000000000000000000000000000000000000ffffffff"
		"000000000000000010270000e803000001000000000000000000000001000000"
		"0000000000000000010000000000000000000000000000000000000024010300"
		"2400000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000000000000000000000000000"
		"3400060006000000000000000000000000000000000000000000000000000000"
		"0000000000000000000000000000000000000000140007000000000000000000"
		"00000000000000000500080000000000"),
     hexstr2bin("1c00000018000506c6b0305800000000020000000a03000700000000")
    ].

%%--------------------------------------------------------------------
%% @spec suite() -> Info
%% Info = [tuple()]
%% @end
%%--------------------------------------------------------------------
suite() ->
	[{timetrap,{seconds,30}}].

test_conntrack_new(_Config) ->
	Msg = conntrack_new(),
	Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_rt_newneigh_1(_Config) ->
	Msg = rt_newneigh_1(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_newneigh_2(_Config) ->
	Msg = rt_newneigh_2(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_delroute(_Config) ->
	Msg = rt_delroute(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_newprefix(_Config) ->
	Msg = rt_newprefix(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_newlink_1(_Config) ->
	Msg = rt_newlink_1(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_newlink_2(_Config) ->
	Msg = rt_newlink_2(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_linkinfo_1(_Config) ->
	Msg = rt_linkinfo_1(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_rt_linkinfo_complex(_Config) ->
	Msg = rt_linkinfo_complex(),
	Msg = netlink:nl_rt_enc(netlink:nl_rt_dec(Msg)),
    ok.

test_nfq_unbind(_Config) ->
    Msg = nfq_unbind(),
    Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_nfq_bind_queue(_Config) ->
    Msg = nfq_bind_queue(),
    Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_nfq_bind_socket(_Config) ->
    Msg = nfq_bind_socket(),
    Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_nfq_set_copy_mode(_Config) ->
    Msg = nfq_set_copy_mode(),
    Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_nfq_set_verdict(_Config) ->
    Msg = nfq_set_verdict(),
    Msg = netlink:nl_ct_enc(netlink:nl_ct_dec(Msg)),
    ok.

test_nft_requests(_Config) ->
    lists:foreach(fun(Msg) ->
			  D = netlink:nl_ct_dec(Msg),
			  ct:pal("D: ~p", [D]),
			  ?equal(Msg, netlink:nl_ct_enc(D))
		  end, nft_requests()),
    ok.

test_genl(_Config) ->
    Msg = genl_request(),
    Msg = netlink:nl_enc(?NETLINK_GENERIC, netlink:nl_dec(?NETLINK_GENERIC, Msg)),
    ok.

test_vrf_enslave_link(_Config) ->
    lists:foreach(fun(Msg) ->
			  D = netlink:nl_rt_dec(Msg),
			  ct:pal("D: ~p", [D]),
			  ?equal(Msg, netlink:nl_rt_enc(D))
		  end, vrf_enslave_link()),
    ok.

test_vrf_add_route(_Config) ->
    lists:foreach(fun(Msg) ->
			  D = netlink:nl_rt_dec(Msg),
			  ct:pal("D: ~p", [D]),
			  ?equal(Msg, netlink:nl_rt_enc(D))
		  end, vrf_add_route()),
    ok.

all() ->
	[test_conntrack_new,
	 test_rt_newneigh_1, test_rt_newneigh_2, test_rt_delroute,
	 test_rt_newprefix,
	 test_rt_newlink_1, test_rt_newlink_2,
	 test_rt_linkinfo_1, test_rt_linkinfo_complex,
	 test_nfq_unbind, test_nfq_bind_queue,
	 test_nfq_bind_socket, test_nfq_set_copy_mode,
	 test_nfq_set_verdict,
	 test_nft_requests,
	 test_genl,
	 test_vrf_enslave_link, test_vrf_add_route
	].

init_per_suite(Config) ->
	R = netlink:start(),
	Config.

end_per_suite(_Config) ->
	ok.

