/* TETRA TEA3 keystream generator implementation */
/*
 * Copyright (C) 2023 Midnight Blue B.V.
 *
 * Author: Wouter Bokslag <w.bokslag [ ] midnightblue [ ] nl>
 *
 * SPDX-License-Identifier: AGPL-3.0+
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * See the COPYING file in the main directory for details.
 */

#include <stdio.h>
#include <inttypes.h>
#include <string.h>

#include "tea3.h"


const uint16_t g_awTea3LutA[8] = { 0x92A7, 0xA761, 0x974C, 0x6B8C, 0x29CE, 0x176C, 0x39D4, 0x7463 };
const uint16_t g_awTea3LutB[8] = { 0x9D58, 0xA46D, 0x176C, 0x79C4, 0xC62B, 0xB2C9, 0x4D93, 0x2E93 };
const uint8_t g_abTea3Sbox[256] = {
	0x7D, 0xBF, 0x7B, 0x92, 0xAE, 0x7C, 0xF2, 0x10, 0x5A, 0x0F, 0x61, 0x7A, 0x98, 0x76, 0x07, 0x64,
	0xEE, 0x89, 0xF7, 0xBA, 0xC2, 0x02, 0x0D, 0xE8, 0x56, 0x2E, 0xCA, 0x58, 0xC0, 0xFA, 0x2A, 0x01,
	0x57, 0x6E, 0x3F, 0x4B, 0x9C, 0xDA, 0xA6, 0x5B, 0x41, 0x26, 0x50, 0x24, 0x3E, 0xF8, 0x0A, 0x86,
	0xB6, 0x5C, 0x34, 0xE9, 0x06, 0x88, 0x1F, 0x39, 0x33, 0xDF, 0xD9, 0x78, 0xD8, 0xA8, 0x51, 0xB2,
	0x09, 0xCD, 0xA1, 0xDD, 0x8E, 0x62, 0x69, 0x4D, 0x23, 0x2B, 0xA9, 0xE1, 0x53, 0x94, 0x90, 0x1E,
	0xB4, 0x3B, 0xF9, 0x4E, 0x36, 0xFE, 0xB5, 0xD1, 0xA2, 0x8D, 0x66, 0xCE, 0xB7, 0xC4, 0x60, 0xED,
	0x96, 0x4F, 0x31, 0x79, 0x35, 0xEB, 0x8F, 0xBB, 0x54, 0x14, 0xCB, 0xDE, 0x6B, 0x2D, 0x19, 0x82,
	0x80, 0xAC, 0x17, 0x05, 0xFF, 0xA4, 0xCF, 0xC6, 0x6F, 0x65, 0xE6, 0x74, 0xC8, 0x93, 0xF4, 0x7E,
	0xF3, 0x43, 0x9F, 0x71, 0xAB, 0x9A, 0x0B, 0x87, 0x55, 0x70, 0x0C, 0xAD, 0xCC, 0xA5, 0x44, 0xE7,
	0x46, 0x45, 0x03, 0x30, 0x1A, 0xEA, 0x67, 0x99, 0xDB, 0x4A, 0x42, 0xD7, 0xAA, 0xE4, 0xC2, 0xD5,
	0xF0, 0x77, 0x20, 0xC3, 0x3C, 0x16, 0xB9, 0xE2, 0xEF, 0x6C, 0x3D, 0x1B, 0x22, 0x84, 0x2F, 0x81,
	0x1D, 0xB1, 0x3A, 0xE5, 0x73, 0x40, 0xD0, 0x18, 0xC7, 0x6A, 0x9E, 0x91, 0x48, 0x27, 0x95, 0x72,
	0x68, 0x0E, 0x00, 0xFC, 0xC5, 0x5F, 0xF1, 0xF5, 0x38, 0x11, 0x7F, 0xE3, 0x5E, 0x13, 0xAF, 0x37,
	0xE0, 0x8A, 0x49, 0x1C, 0x21, 0x47, 0xD4, 0xDC, 0xB0, 0xEC, 0x83, 0x28, 0xB8, 0xF6, 0xA7, 0xC9,
	0x63, 0x59, 0xBD, 0x32, 0x85, 0x08, 0xBE, 0xD3, 0xFD, 0x4C, 0x2C, 0xFB, 0xA0, 0xC1, 0x9D, 0xB3,
	0x52, 0x8C, 0x5D, 0x29, 0x6D, 0x04, 0xBC, 0x25, 0x15, 0x8B, 0x12, 0x9B, 0xD6, 0x75, 0xA3, 0x97
};


static uint64_t tea3_compute_iv(uint32_t dwFrameNumbers)
{
	uint32_t dwXorred = dwFrameNumbers ^ 0xC43A7D51;
	dwXorred = (dwXorred << 8) | (dwXorred >> 24);
	uint64_t qwIv = ((uint64_t)dwFrameNumbers << 32) | dwXorred;
	return (qwIv >> 8) | (qwIv << 56);
}

static uint8_t tea3_state_word_to_newbyte(uint16_t wSt, const uint16_t *awLut)
{
	uint8_t bSt0 = wSt;
	uint8_t bSt1 = wSt >> 8;

	uint8_t bDist;
	uint8_t bOut = 0;

	for (int i = 0; i < 8; i++) {
		/* taps on bit 5,6 for bSt0 and bit 5,6 for bSt1 */
		bDist = ((bSt0 >> 5) & 3) | ((bSt1 >> 3) & 12);
		if (awLut[i] & (1 << bDist))
			bOut |= 1 << i;

		/* rotate one position */
		bSt0 = ((bSt0 >> 1) | (bSt0 << 7));
		bSt1 = ((bSt1 >> 1) | (bSt1 << 7));
	}

	return bOut;
}

static uint8_t tea3_reorder_state_byte(uint8_t bStByte)
{
	/* simple re-ordering of bits */
	uint8_t bOut = 0;
	bOut |= ((bStByte << 6) & 0x40);
	bOut |= ((bStByte << 1) & 0x20);
	bOut |= ((bStByte << 2) & 0x98);
	bOut |= ((bStByte >> 4) & 0x04);
	bOut |= ((bStByte >> 3) & 0x01);
	bOut |= ((bStByte >> 6) & 0x02);
	return bOut;
}

void tea3(uint32_t dwFrameNumbers, uint8_t *lpKey, uint32_t dwNumKsBytes, uint8_t *lpKsOut)
{
	uint8_t abKeyReg[10];
	uint32_t dwNumSkipRounds = 51;

	/* init registers */
	uint64_t qwIvReg = tea3_compute_iv(dwFrameNumbers);
	memcpy(abKeyReg, lpKey, 10);

	for (int i = 0; i < dwNumKsBytes; i++) {
		for (int j = 0; j < dwNumSkipRounds; j++) {
			/* Step 1: Derive a non-linear feedback byte through sbox and feed back into key register */
			uint8_t bSboxOut = g_abTea3Sbox[abKeyReg[7] ^ abKeyReg[2]] ^ abKeyReg[0];
			memmove(abKeyReg, abKeyReg + 1, 9);
			abKeyReg[9] = bSboxOut;

			/* Step 2: Compute 3 bytes derived from current state */
			uint8_t bDerivByte12 = tea3_state_word_to_newbyte((qwIvReg >>  8) & 0xffff, g_awTea3LutA);
			uint8_t bDerivByte56 = tea3_state_word_to_newbyte((qwIvReg >> 40) & 0xffff, g_awTea3LutB);
			uint8_t bReordByte4  = tea3_reorder_state_byte((qwIvReg >> 32) & 0xff);

			/* Step 3: Combine current state with state derived values, and xor in key derived sbox output */
			uint8_t bNewByte = ((qwIvReg >> 56) ^ bReordByte4 ^ bDerivByte12 ^ bSboxOut) & 0xff;
			uint8_t bMixByte = bDerivByte56;

			/* Step 4: Update lfsr: leftshift 8, feed/mix in previously generated bytes */
			qwIvReg = ((qwIvReg << 8) ^ ((uint64_t)bMixByte << 40)) | bNewByte;
		}
		lpKsOut[i] = (qwIvReg >> 56);
		dwNumSkipRounds = 19;
	}
}
