# Copyright 2017-2019 Harald Welte
# Copyright 2018-2025 sysmocom - s.f.m.c. GmbH
# SPDX-License-Identifier: Apache-2.0
# Wrapper around the TITAN make file generator

if [ -z "$NAME" ]; then
	echo "ERROR: NAME is not set!"
	exit 1
fi

if [ -z "$FILES" ]; then
	echo "ERROR: FILES is not set!"
	exit 1
fi

if ! command -v ttcn3_makefilegen >/dev/null; then
	echo "ERROR: ttcn3_makefilegen not in PATH"
	exit 1
fi

TOPDIR="$(realpath "$(dirname "$0")/..")"
BUILDDIR="${BUILDDIR:-$TOPDIR/_build}"
PROJECTDIR=$(realpath . --relative-to "$TOPDIR")  # e.g. "msc", "library/rua"

cd "$BUILDDIR/$PROJECTDIR"

ttcn3_makefilegen -g -p -l -U 8 -f -e "$NAME" $FILES

sed -i -e 's/# TTCN3_DIR = /TTCN3_DIR = \/usr/' Makefile
sed -i -e 's/LDFLAGS = /LDFLAGS = -L \/usr\/lib\/titan/' Makefile
sed -i -e 's/LINUX_LIBS = -lxml2/LINUX_LIBS = -lxml2 -lsctp -lssl/' Makefile

# The -DMAKEDEPEND_RUN is a workaround for Debian packaging issue,
# see https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=879816 for details
sed -i -e 's/CPPFLAGS = -D$(PLATFORM)/CPPFLAGS = -D$(PLATFORM) -DMAKEDEPEND_RUN -DUSE_SCTP -DLKSCTP_MULTIHOMING_ENABLED -DAS_USE_SSL/' Makefile

# Remove -Wall from CXXFLAGS: we're not interested in generic warnings for
# autogenerated code cluttering the logs
sed -i -e 's/-Wall//' Makefile

if [ -n "$CPPFLAGS_TTCN3" ]; then
	CPPFLAGS_TTCN3="$(echo "$CPPFLAGS_TTCN3" | tr -d '\n' | tr '\t' ' ')"
	sed -i -e "s/CPPFLAGS_TTCN3 =/CPPFLAGS_TTCN3 = $CPPFLAGS_TTCN3/" Makefile
fi

# For TITAN 6.3.0
if cat /etc/issue | grep "Arch Linux" >/dev/null 2>&1; then
        sed -i -e 's/TTCN3_DIR = $/TTCN3_DIR = \/usr\/ttcn3/' Makefile
else
        sed -i -e 's/TTCN3_DIR = $/TTCN3_DIR = \/usr/' Makefile
fi

# Enable ccache if it can be found in path. This speeds up repeated builds of
# the TTCN3 tests by an order of magnitude since most of the generated C++
# source files don't change very often. Roughly, for an initial build which
# takes N minutes, a complete rebuild after 'make clean' will only take N
# seconds with ccache. Note that ccache cannot speed up compilation of .o files
# to .so files.
if [ "$USE_CCACHE" != 0 ] && command -v ccache >/dev/null; then
	sed -i -e 's/^CXX = g++ $/CXX = env CCACHE_SLOPPINESS=time_macros ccache g++/' Makefile
	# Append the -D option to compiler flags. This option disables
	# timestamps inside comments in the generated C++ code which interfere
	# with ccache.
	sed -i -e 's/^COMPILER_FLAGS = \(.*\)/& -D/' Makefile
fi

#
# Make output more readable (skip with make V=1)
#

make_pretty_rule() {
	local rule_name="$1"
	local cmd_start="$2"
	local cmd_msg="@echo '  $3'"

	sed -i -z "s/$rule_name\\n\\t$cmd_start/$rule_name\\n\\t$cmd_msg\n\t@$cmd_start/" Makefile
}

make_pretty_rules() {
	make_pretty_rule \
		'$(EXECUTABLE): $(SHARED_OBJECTS)' \
		'if' \
		'CCLD     $@'

	make_pretty_rule \
		'$(LIBRARY): $(OBJECTS)' \
		'$(CXX)' \
		'CCLD     $@'

	make_pretty_rule \
		'.cc.o .c.o:' \
		'$(CXX)' \
		'CC       $@'

	make_pretty_rule \
		'%.so: %.o' \
		'$(CXX)' \
		'CCLD     $@'

	make_pretty_rule \
		'%.ttcn: %.ttcnpp $(TTCN3_INCLUDES)' \
		'$(CPP)' \
		'PP       $@'

	make_pretty_rule \
		'compile: $(TTCN3_MODULES)  $(PREPROCESSED_TTCN3_MODULES) $(ASN1_MODULES)' \
		'$(TTCN3_DIR)' \
		'TTCN     *.ttcn *.asn'

	sed -i "s/@echo Creating dependency file for/@echo '  DEP     '/" Makefile

	# TTCN3 compiler: Silence "Notify: Parsing..." output with -q, it still
	# prints warnings and errors
	sed -i -e 's/^COMPILER_FLAGS = \(.*\)/& -q/' Makefile
}

if [ "$V" != 1 ]; then
	make_pretty_rules
fi
