/* SS7 TCAP Loadsharing VTY Interface */

/* (C) 2025 by sysmocom s.f.m.c. GmbH <info@sysmocom.de>
 * Author: Alexander Couzens <lynxis@fe80.eu>
 * Author: Daniel Willmann <dwillmann@sysmocom.de>
 * All Rights Reserved
 *
 * SPDX-License-Identifier: GPL-2.0+
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdlib.h>
#include <unistd.h>

#include <osmocom/core/hashtable.h>

#include <osmocom/sigtran/osmo_ss7.h>

#include <osmocom/vty/vty.h>
#include <osmocom/vty/command.h>
#include <osmocom/vty/logging.h>
#include <osmocom/vty/misc.h>
#include <osmocom/vty/tdef_vty.h>
#include <osmocom/vty/telnet_interface.h>

#include "ss7_as.h"
#include "ss7_asp.h"
#include "ss7_vty.h"
#include "ss7_as_loadshare_tcap.h"

#ifdef WITH_TCAP_LOADSHARING

DEFUN(tcap_show_timer, tcap_show_timer_cmd,
      "show cs7 instance <0-15> tcap timer " OSMO_TDEF_VTY_ARG_T_OPTIONAL,
      SHOW_STR CS7_STR INST_STR INST_STR
      "Show tcap timers\n"
      OSMO_TDEF_VTY_DOC_T)
{
	int id = atoi(argv[0]);
	struct osmo_ss7_instance *ss7;

	ss7 = osmo_ss7_instance_find(id);
	if (!ss7) {
		vty_out(vty, "No SS7 instance %d found%s", id, VTY_NEWLINE);
		return CMD_WARNING;
	}

	const char *T_arg = argc > 1 ? argv[1] : NULL;
	return osmo_tdef_vty_show_cmd(vty, ss7->cfg.tcap.T_defs, T_arg, NULL);
}

DEFUN(cfg_tcap_timer, cfg_tcap_timer_cmd,
      "tcap timer " OSMO_TDEF_VTY_ARG_SET_OPTIONAL,
      "Configure or show timers\n"
      OSMO_TDEF_VTY_DOC_SET)
{
	struct osmo_ss7_instance *ss7 = vty->index;

	/* If any arguments are missing, redirect to 'show' */
	if (argc < 2)
		return tcap_show_timer(self, vty, argc, argv);
	return osmo_tdef_vty_set_cmd(vty, ss7->cfg.tcap.T_defs, argv);
}

static int show_one_tcap_range(struct vty *vty, const struct tcap_range *tcrng, bool print_pc, bool print_ssn)
{
	vty_out(vty, "%-7u %-7u %-12s %3u %3u%s", tcrng->tid_start, tcrng->tid_end, tcrng->asp->cfg.name, tcrng->ssn, tcrng->pc, VTY_NEWLINE);
	return CMD_SUCCESS;
}

DEFUN(show_cs7_as_tcapranges_name, show_cs7_as_tcapranges_name_cmd,
      "show cs7 instance <0-15> as tcap-ranges name AS_NAME",
      SHOW_STR CS7_STR INST_STR INST_STR "Application Server (AS)\n"
					 "Display tcap ranges\n"
					 "Look up AS with a given name\n"
					 "Name of the Application Server (AS)\n")
{
	int id = atoi(argv[0]);
	const char *as_name = argv[1];
	struct osmo_ss7_instance *inst;
	struct osmo_ss7_as *as = NULL;
	int i;
	struct tcap_range *tcrng;

	inst = osmo_ss7_instance_find(id);
	if (!inst) {
		vty_out(vty, "No SS7 instance %d found%s", id, VTY_NEWLINE);
		return CMD_WARNING;
	}

	if (as_name) {
		as = osmo_ss7_as_find_by_name(inst, as_name);
		if (!as) {
			vty_out(vty, "No AS %s found%s", as_name, VTY_NEWLINE);
			return CMD_WARNING;
		}
	}

	vty_out(vty, "Tid Min      Tid Max      ASP Name      SSN       PC       %s", VTY_NEWLINE);
	vty_out(vty, "------------ ------------ ------------- --------- ---------%s", VTY_NEWLINE);

	hash_for_each(as->tcap.tid_ranges, i, tcrng, list) {
		show_one_tcap_range(vty, tcrng, as->tcap.contains_pc, as->tcap.contains_ssn);
	}


	return CMD_SUCCESS;
}

DEFUN(show_cs7_as_tcaproute_name, show_cs7_as_tcapranges_name_tid_cmd,
      "show cs7 instance <0-15> as tcap-ranges name AS_NAME tid TID",
      SHOW_STR CS7_STR INST_STR INST_STR "Application Server (AS)\n"
					 "Display tcap range\n"
					 "Look up AS with a given name\n"
					 "Name of the Application Server (AS)\n"
					 "Show tcap range for a given TID\n"
					 "TID\n")
{
	int id = atoi(argv[0]);
	const char *as_name = argv[1];
	int tid = atoi(argv[2]);
	struct osmo_ss7_instance *inst;
	struct osmo_ss7_as *as = NULL;
	int i;
	struct tcap_range *tcrng = NULL;

	inst = osmo_ss7_instance_find(id);
	if (!inst) {
		vty_out(vty, "No SS7 instance %d found%s", id, VTY_NEWLINE);
		return CMD_WARNING;
	}

	if (as_name) {
		as = osmo_ss7_as_find_by_name(inst, as_name);
		if (!as) {
			vty_out(vty, "No AS %s found%s", as_name, VTY_NEWLINE);
			return CMD_WARNING;
		}
	}

	vty_out(vty, "Tid Min      Tid Max      ASP Name      SSN       PC       %s", VTY_NEWLINE);
	vty_out(vty, "------------ ------------ ------------- --------- ---------%s", VTY_NEWLINE);

	hash_for_each(as->tcap.tid_ranges, i, tcrng, list) {
		if (tid < tcrng->tid_start || tid > tcrng->tid_end)
			continue;
		show_one_tcap_range(vty, tcrng, as->tcap.contains_pc, as->tcap.contains_ssn);
	}

	return CMD_SUCCESS;
}

void ss7_tcap_vty_init(void)
{
	install_lib_element(L_CS7_NODE, &cfg_tcap_timer_cmd);

	install_lib_element_ve(&tcap_show_timer_cmd);
	install_lib_element_ve(&show_cs7_as_tcapranges_name_cmd);
	install_lib_element_ve(&show_cs7_as_tcapranges_name_tid_cmd);
}

void ss7_tcap_vty_write_cs7(struct vty *vty, struct osmo_ss7_instance *ss7)
{
	osmo_tdef_vty_write(vty, ss7->cfg.tcap.T_defs, " tcap timer ");
}

#else
void ss7_tcap_vty_init(void)
{
}

void ss7_tcap_vty_write_cs7(struct vty *vty, struct osmo_ss7_instance *ss7)
{
}
#endif /* WITH_TCAP_LOADSHARING */
