/* Iridium AMBE vocoder - Tables */

/* (C) 2015 by Sylvain Munaut <tnt@246tNt.com>
 * All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*! \addtogroup codec_private
 *  @{
 */

/*! \file codec/tables.c
 *  \brief Iridium AMBE vocoder tables
 */

#include <stdint.h>

/*! \brief Bit prioritization table */
const struct {
	uint8_t pos;
	uint8_t len;
} ir77_ambe_prio_tbl[] = {
	{   0, 6 },
	{  10, 3 },
	{  15, 3 },
	{  28, 8 },
	{  36, 6 },
	{  49, 8 },
	{  63, 7 },
	{  73, 7 },
	{  20, 2 },
	{  24, 2 },
	{  42, 7 },
	{  13, 1 },
	{  83, 4 },
	{  93, 4 },
	{   6, 4 },
	{  14, 1 },
	{  18, 2 },
	{  87, 3 },
	{  97, 3 },
	{  57, 6 },
	{  70, 3 },
	{  80, 3 },
	{  90, 3 },
	{ 100, 3 },
	{  22, 2 },
	{  26, 2 },
	{   0, 0 },	/* Termination */
};

/*! \brief f0log/pitch difference vector quantization table */
const float ir77_ambe_pitch_diff_vq[64][2] = {
	{ -0.9111f,  0.8867f },
	{ -0.7461f,  0.8027f },
	{ -0.7188f,  0.6191f },
	{ -0.5527f,  0.6084f },
	{ -0.5635f,  0.4619f },
	{ -0.3809f,  0.4971f },
	{ -0.4189f,  0.4209f },
	{ -0.3809f,  0.2744f },
	{ -0.2324f,  0.3340f },
	{ -0.2490f,  0.2051f },
	{ -0.1357f,  0.2393f },
	{ -0.2217f,  0.1006f },
	{ -0.0566f,  0.1670f },
	{ -0.1289f,  0.0898f },
	{ -0.1357f,  0.0029f },
	{ -0.0498f,  0.0898f },
	{  0.0088f,  0.1211f },
	{ -0.0576f,  0.0322f },
	{ -0.0723f, -0.0117f },
	{  0.0146f,  0.0664f },
	{  0.0498f,  0.0996f },
	{ -0.0898f, -0.0469f },
	{ -0.0107f,  0.0342f },
	{ -0.0283f, -0.0088f },
	{ -0.0400f, -0.0420f },
	{  0.0000f,  0.0000f },
	{  0.0576f,  0.0557f },
	{ -0.0732f, -0.0762f },
	{  0.0313f,  0.0254f },
	{  0.0840f,  0.0801f },
	{  0.0166f, -0.0137f },
	{ -0.0381f, -0.0762f },
	{ -0.0020f, -0.0508f },
	{  0.0986f,  0.0449f },
	{  0.0674f,  0.0098f },
	{ -0.0498f, -0.1094f },
	{  0.0420f, -0.0254f },
	{ -0.0010f, -0.0869f },
	{  0.1211f,  0.0107f },
	{  0.0371f, -0.0771f },
	{ -0.0098f, -0.1299f },
	{  0.0977f, -0.0381f },
	{  0.0615f, -0.1328f },
	{  0.1748f, -0.0410f },
	{  0.1260f, -0.1182f },
	{  0.0596f, -0.2080f },
	{  0.2305f, -0.1152f },
	{  0.1357f, -0.2100f },
	{  0.2256f, -0.2383f },
	{  0.2949f, -0.2021f },
	{  0.3711f, -0.2314f },
	{  0.2480f, -0.3701f },
	{  0.3652f, -0.3994f },
	{  0.4443f, -0.3740f },
	{  0.4316f, -0.5420f },
	{  0.5410f, -0.4580f },
	{  0.5410f, -0.5791f },
	{  0.7012f, -0.5859f },
	{  0.5977f, -0.7402f },
	{  0.7139f, -0.7588f },
	{  0.8369f, -0.7471f },
	{  0.8369f, -0.9160f },
	{  0.9873f, -0.9453f },
	{  1.0732f, -1.1279f },
};

/*! \brief Number of harmonics per group for a given L (starts at L=9) */
const uint8_t ir77_ambe_hpg_tbl[48][4] = {
	{  2,  2,  2,  3 },	/* L = 9 */
	{  2,  2,  3,  3 },
	{  2,  3,  3,  3 },
	{  2,  3,  3,  4 },
	{  3,  3,  3,  4 },
	{  3,  3,  4,  4 },
	{  3,  3,  4,  5 },
	{  3,  4,  4,  5 },
	{  3,  4,  5,  5 },
	{  4,  4,  5,  5 },
	{  4,  4,  5,  6 },
	{  4,  4,  6,  6 },
	{  4,  5,  6,  6 },
	{  4,  5,  6,  7 },
	{  5,  5,  6,  7 },
	{  5,  5,  7,  7 },
	{  5,  6,  7,  7 },
	{  5,  6,  7,  8 },
	{  5,  6,  8,  8 },
	{  6,  6,  8,  8 },
	{  6,  6,  8,  9 },
	{  6,  7,  8,  9 },
	{  6,  7,  9,  9 },
	{  6,  7,  9, 10 },
	{  7,  7,  9, 10 },
	{  7,  8,  9, 10 },
	{  7,  8, 10, 10 },
	{  7,  8, 10, 11 },
	{  8,  8, 10, 11 },
	{  8,  9, 10, 11 },
	{  8,  9, 11, 11 },
	{  8,  9, 11, 12 },
	{  8,  9, 11, 13 },
	{  8,  9, 12, 13 },
	{  8, 10, 12, 13 },
	{  9, 10, 12, 13 },
	{  9, 10, 12, 14 },
	{  9, 10, 13, 14 },
	{  9, 11, 13, 14 },
	{ 10, 11, 13, 14 },
	{ 10, 11, 13, 15 },
	{ 10, 11, 14, 15 },
	{ 10, 12, 14, 15 },
	{ 10, 12, 14, 16 },
	{ 11, 12, 14, 16 },
	{ 11, 12, 15, 16 },
	{ 11, 12, 15, 17 },
	{ 11, 13, 15, 17 },	/* L = 56 */
};

/*! \brief Gain difference (in dual-subframe quant) vector quantization table */
const float ir77_ambe_gain_diff_vq[32][2] = {
	{ -3.2695f,  3.2710f },
	{ -2.7949f,  2.7544f },
	{ -2.3730f,  2.3701f },
	{ -1.8853f,  1.8672f },
	{ -1.5293f,  1.5093f },
	{ -1.2393f,  1.2842f },
	{ -1.0020f,  1.0195f },
	{ -0.9229f,  0.7280f },
	{ -0.6196f,  0.7944f },
	{ -0.6592f,  0.4897f },
	{ -0.3691f,  0.5425f },
	{ -0.4219f,  0.2510f },
	{ -0.1582f,  0.3042f },
	{ -0.2373f,  0.0791f },
	{ -0.1450f, -0.0532f },
	{  0.0264f,  0.1851f },
	{  0.0103f, -0.0239f },
	{  0.1592f,  0.0596f },
	{  0.0103f, -0.2153f },
	{  0.2549f, -0.0957f },
	{  0.1699f, -0.3350f },
	{  0.4033f, -0.2275f },
	{  0.3076f, -0.4907f },
	{  0.4883f, -0.6460f },
	{  0.5732f, -0.3950f },
	{  0.7964f, -0.6221f },
	{  0.7222f, -0.8735f },
	{  1.0195f, -0.9570f },
	{  1.2529f, -1.2324f },
	{  1.5293f, -1.5552f },
	{  1.9326f, -1.9502f },
	{  2.7080f, -2.9189f },
};

/*! \brief V/UV decisions (MSBs = low freq) */
const uint8_t ir77_ambe_v_uv_tbl[16] = {
	0x00, 0x80, 0xc0, 0xc1, 0xe0, 0xe1, 0xf0, 0xf1,
	0xf9, 0xf8, 0xfb, 0xdf, 0xfc, 0xfe, 0xfd, 0xff,
};

/*! \brief PRBA[1:2] Sum vector quantization table */
const float ir77_ambe_prba_sum12_vq[256][2] = {
	{  0.1758f, -0.1240f },
	{ -0.2070f, -0.1523f },
	{  0.5034f, -0.0322f },
	{  0.1270f,  0.0137f },
	{ -0.1704f, -0.2300f },
	{ -0.7412f, -0.3330f },
	{  0.1211f, -0.0293f },
	{ -0.2129f, -0.0063f },
	{ -0.0186f,  0.0195f },
	{ -0.5044f,  0.1113f },
	{  0.2544f,  0.1719f },
	{ -0.0552f,  0.2490f },
	{ -0.5107f,  0.0024f },
	{ -1.0205f,  0.1919f },
	{ -0.0796f,  0.1401f },
	{ -0.5410f,  0.3867f },
	{  0.2827f,  0.0679f },
	{ -0.0210f,  0.1226f },
	{  0.6826f,  0.3813f },
	{  0.2207f,  0.3008f },
	{ -0.0483f,  0.0762f },
	{ -0.4487f,  0.2202f },
	{  0.2134f,  0.2373f },
	{ -0.1177f,  0.4199f },
	{  0.0825f,  0.2544f },
	{ -0.2207f,  0.3560f },
	{  0.4067f,  0.5469f },
	{  0.1172f,  0.5298f },
	{ -0.4004f,  0.4961f },
	{ -1.0483f,  0.5596f },
	{ -0.0210f,  0.6274f },
	{ -0.2207f,  1.1494f },
	{  0.4546f, -0.4072f },
	{  0.1714f, -0.1772f },
	{  1.0742f, -0.2739f },
	{  0.4678f, -0.0869f },
	{  0.1055f, -0.3828f },
	{ -0.2705f, -0.4189f },
	{  0.4087f, -0.1372f },
	{  0.0791f, -0.1011f },
	{  0.2173f, -0.0908f },
	{ -0.0464f, -0.0303f },
	{  0.5601f,  0.0742f },
	{  0.1978f,  0.0864f },
	{ -0.0659f, -0.0889f },
	{ -0.5801f, -0.1108f },
	{  0.1909f,  0.0381f },
	{ -0.1055f,  0.0845f },
	{  0.6553f, -0.0361f },
	{  0.2363f,  0.0181f },
	{  1.0098f,  0.1489f },
	{  0.6123f,  0.1577f },
	{  0.2417f, -0.0249f },
	{ -0.1021f, -0.0361f },
	{  0.4727f,  0.1035f },
	{  0.1665f,  0.1128f },
	{  0.3706f,  0.1938f },
	{  0.0806f,  0.1577f },
	{  0.6768f,  0.5283f },
	{  0.3599f,  0.4219f },
	{  0.0322f,  0.2168f },
	{ -0.3438f,  0.3452f },
	{  0.3115f,  0.3115f },
	{ -0.0054f,  0.4561f },
	{  0.4946f, -0.1411f },
	{  0.2036f, -0.0498f },
	{  0.8926f, -0.0313f },
	{  0.4219f,  0.0767f },
	{  0.1646f, -0.0830f },
	{ -0.1411f, -0.0977f },
	{  0.4121f,  0.0186f },
	{  0.1211f,  0.0913f },
	{  0.2954f,  0.1079f },
	{  0.0239f,  0.1553f },
	{  0.5430f,  0.2241f },
	{  0.2568f,  0.3794f },
	{  0.0044f,  0.0664f },
	{ -0.3301f,  0.1909f },
	{  0.2666f,  0.2324f },
	{ -0.0439f,  0.3486f },
	{  0.7178f,  0.2471f },
	{  0.3057f,  0.2007f },
	{  1.1772f,  0.3525f },
	{  0.5415f,  0.4644f },
	{  0.2524f,  0.1187f },
	{ -0.0239f,  0.1909f },
	{  0.5532f,  0.3457f },
	{  0.2524f,  0.4941f },
	{  0.4590f,  0.3853f },
	{  0.1113f,  0.4219f },
	{  1.1128f,  0.9194f },
	{  0.4648f,  0.9521f },
	{  0.0581f,  0.3560f },
	{ -0.3066f,  0.7578f },
	{  0.5234f,  0.6426f },
	{  0.0171f,  0.8335f },
	{  0.9063f, -0.4927f },
	{  0.4883f, -0.2866f },
	{  1.7275f, -0.2300f },
	{  1.0479f, -0.0308f },
	{  0.4458f, -0.2134f },
	{  0.1274f, -0.1304f },
	{  0.8921f, -0.2202f },
	{  0.4248f, -0.0273f },
	{  0.5732f, -0.0073f },
	{  0.2832f,  0.0259f },
	{  0.9771f,  0.3452f },
	{  0.4741f,  0.1924f },
	{  0.1982f, -0.0073f },
	{ -0.0566f,  0.0225f },
	{  0.5308f,  0.1426f },
	{  0.1841f,  0.1792f },
	{  1.4692f,  0.1084f },
	{  0.7202f,  0.1387f },
	{  1.8579f,  0.2144f },
	{  1.4692f,  0.4258f },
	{  0.6553f,  0.0542f },
	{  0.2373f,  0.0630f },
	{  1.2026f,  0.1235f },
	{  0.6118f,  0.2656f },
	{  0.8711f,  0.5254f },
	{  0.3887f,  0.3262f },
	{  1.5845f,  0.8184f },
	{  0.7993f,  0.7397f },
	{  0.3599f,  0.2563f },
	{  0.0288f,  0.2900f },
	{  1.1089f,  0.5947f },
	{  0.2661f,  0.6797f },
	{  0.0000f, -0.4746f },
	{ -0.6055f, -0.4746f },
	{  0.2271f, -0.2246f },
	{ -0.0908f, -0.2759f },
	{ -0.3105f, -0.7300f },
	{ -1.0898f, -0.7383f },
	{ -0.1665f, -0.3325f },
	{ -0.6963f, -0.6392f },
	{ -0.3711f, -0.2617f },
	{ -0.9312f, -0.2646f },
	{  0.0527f, -0.0264f },
	{ -0.3784f, -0.0347f },
	{ -1.2490f, -0.3608f },
	{ -1.6650f, -0.2192f },
	{ -0.4771f, -0.0972f },
	{ -1.2339f, -0.0688f },
	{  0.0791f, -0.1548f },
	{ -0.2896f, -0.1143f },
	{  0.3613f, -0.0171f },
	{  0.0645f,  0.0635f },
	{ -0.2827f, -0.1880f },
	{ -0.7744f, -0.1377f },
	{  0.0068f, -0.0239f },
	{ -0.3115f,  0.0737f },
	{ -0.1597f,  0.1206f },
	{ -0.6079f,  0.2114f },
	{  0.1514f,  0.2349f },
	{ -0.1377f,  0.2886f },
	{ -0.7959f,  0.1172f },
	{ -1.4468f,  0.2446f },
	{ -0.2681f,  0.2515f },
	{ -0.6802f,  0.6611f },
	{  0.2607f, -0.7041f },
	{  0.0327f, -0.6387f },
	{  0.5161f, -0.5391f },
	{  0.2495f, -0.2993f },
	{ -0.0396f, -0.8564f },
	{ -0.5337f, -0.9341f },
	{  0.1938f, -0.3677f },
	{ -0.1504f, -0.4458f },
	{  0.0996f, -0.2900f },
	{ -0.3613f, -0.3833f },
	{  0.3105f, -0.1499f },
	{ -0.0337f, -0.1528f },
	{ -0.3950f, -0.5513f },
	{ -0.8667f, -0.4736f },
	{ -0.0728f, -0.2061f },
	{ -0.6001f, -0.2588f },
	{  0.3052f, -0.2334f },
	{  0.0190f, -0.1802f },
	{  0.5674f, -0.1064f },
	{  0.2954f, -0.0176f },
	{  0.0190f, -0.3257f },
	{ -0.2622f, -0.2900f },
	{  0.2715f, -0.1104f },
	{  0.0386f, -0.0684f },
	{  0.1646f, -0.0327f },
	{ -0.1509f,  0.0356f },
	{  0.3687f,  0.1226f },
	{  0.1357f,  0.1450f },
	{ -0.1606f, -0.0366f },
	{ -0.6406f,  0.0317f },
	{  0.1064f,  0.0508f },
	{ -0.2524f,  0.1470f },
	{  0.3076f, -0.3652f },
	{ -0.0127f, -0.2559f },
	{  0.5674f, -0.2183f },
	{  0.2593f, -0.1772f },
	{ -0.0728f, -0.3652f },
	{ -0.5020f, -0.3599f },
	{  0.2173f, -0.1421f },
	{ -0.1211f, -0.1494f },
	{  0.1206f, -0.0752f },
	{ -0.2178f, -0.0825f },
	{  0.3682f,  0.0386f },
	{  0.0903f,  0.1055f },
	{ -0.3750f, -0.1440f },
	{ -0.9429f, -0.0474f },
	{  0.0420f,  0.0200f },
	{ -0.3994f,  0.0850f },
	{  0.3906f, -0.0649f },
	{  0.0894f,  0.0044f },
	{  0.8457f,  0.1577f },
	{  0.3140f,  0.1484f },
	{  0.0830f, -0.0513f },
	{ -0.2886f, -0.0234f },
	{  0.3394f,  0.0811f },
	{  0.0381f,  0.1064f },
	{  0.2114f,  0.1411f },
	{ -0.1025f,  0.1948f },
	{  0.4805f,  0.2852f },
	{  0.1602f,  0.3687f },
	{ -0.1733f,  0.1948f },
	{ -0.7690f,  0.3433f },
	{  0.1343f,  0.3013f },
	{ -0.2310f,  0.5215f },
	{  0.6050f, -0.7075f },
	{  0.3267f, -0.4951f },
	{  1.3008f, -0.4463f },
	{  0.6958f, -0.4771f },
	{  0.1597f, -0.5156f },
	{ -0.1743f, -0.5640f },
	{  0.5913f, -0.3438f },
	{  0.1597f, -0.2539f },
	{  0.3896f, -0.2974f },
	{  0.1113f, -0.2002f },
	{  0.6606f, -0.1616f },
	{  0.3198f, -0.0669f },
	{  0.0483f, -0.2310f },
	{ -0.4453f, -0.2080f },
	{  0.2642f, -0.0645f },
	{ -0.0073f, -0.0747f },
	{  0.7197f, -0.2764f },
	{  0.3511f, -0.1099f },
	{  1.2910f, -0.1123f },
	{  0.7676f,  0.0303f },
	{  0.3687f, -0.1958f },
	{  0.0234f, -0.1172f },
	{  0.7617f, -0.1240f },
	{  0.3213f,  0.0303f },
	{  0.4761f,  0.0352f },
	{  0.1543f,  0.0615f },
	{  0.8208f,  0.3232f },
	{  0.4224f,  0.2378f },
	{  0.1606f,  0.0112f },
	{ -0.1904f,  0.0664f },
	{  0.4165f,  0.1494f },
	{  0.1187f,  0.1948f },
};

/*! \brief PRBA[3:4] Sum vector quantization table */
const float ir77_ambe_prba_sum34_vq[64][2] = {
	{  0.7690f,  0.1548f },
	{  0.3657f,  0.2974f },
	{  0.4819f, -0.1167f },
	{  0.2837f,  0.0547f },
	{  0.5605f,  0.3843f },
	{  0.2563f,  0.4556f },
	{  0.3315f,  0.1514f },
	{  0.0171f,  0.1743f },
	{  0.6904f, -0.1558f },
	{  0.3452f, -0.0488f },
	{  0.4956f, -0.3496f },
	{  0.2017f, -0.2012f },
	{  0.4170f,  0.0449f },
	{  0.2085f,  0.1289f },
	{  0.2598f, -0.1104f },
	{ -0.0957f, -0.0425f },
	{  0.4995f,  0.1812f },
	{  0.1172f,  0.2080f },
	{  0.2231f, -0.0186f },
	{ -0.0493f,  0.0181f },
	{  0.1245f,  0.3232f },
	{ -0.0117f,  0.4590f },
	{  0.0073f,  0.0815f },
	{ -0.1665f,  0.2837f },
	{  0.3633f, -0.1968f },
	{  0.0640f, -0.0786f },
	{  0.1094f, -0.2900f },
	{ -0.1592f, -0.2900f },
	{  0.0791f,  0.0513f },
	{ -0.0723f,  0.0996f },
	{ -0.0078f, -0.1392f },
	{ -0.4541f, -0.0605f },
	{  0.5718f,  0.0234f },
	{  0.0967f,  0.1221f },
	{  0.1611f, -0.0962f },
	{ -0.0205f, -0.0601f },
	{  0.2334f,  0.2417f },
	{ -0.0122f,  0.2778f },
	{  0.0322f, -0.0034f },
	{ -0.3062f,  0.1792f },
	{  0.2817f, -0.3203f },
	{  0.0967f, -0.1621f },
	{  0.1450f, -0.4668f },
	{ -0.0322f, -0.3652f },
	{  0.1206f, -0.0156f },
	{ -0.1450f,  0.0322f },
	{ -0.0806f, -0.2246f },
	{ -0.4351f, -0.2217f },
	{  0.1660f,  0.0557f },
	{ -0.1880f,  0.1260f },
	{ -0.0991f, -0.1270f },
	{ -0.3154f, -0.1235f },
	{ -0.0889f,  0.1855f },
	{ -0.3115f,  0.4067f },
	{ -0.2407f, -0.0005f },
	{ -0.5278f,  0.1504f },
	{  0.0244f, -0.2319f },
	{ -0.2021f, -0.1865f },
	{ -0.2124f, -0.4775f },
	{ -0.5098f, -0.4111f },
	{ -0.1880f, -0.0845f },
	{ -0.3506f,  0.0313f },
	{ -0.3081f, -0.2881f },
	{ -0.7222f, -0.1060f },
};

/*! \brief PRBA[5:7] Sum vector quantization table */
const float ir77_ambe_prba_sum57_vq[128][3] = {
	{ -0.0352f, -0.0244f, -0.1870f },
	{ -0.1187f, -0.0645f, -0.2529f },
	{  0.0903f, -0.0566f, -0.0991f },
	{  0.0205f, -0.0483f, -0.1104f },
	{ -0.2021f, -0.0806f, -0.1431f },
	{ -0.2993f, -0.1128f, -0.1104f },
	{ -0.0562f, -0.0083f,  0.0215f },
	{ -0.0840f, -0.0645f, -0.0005f },
	{ -0.0142f,  0.1045f, -0.1543f },
	{  0.0137f,  0.0400f, -0.1211f },
	{  0.1387f,  0.0806f, -0.0332f },
	{  0.1182f,  0.0322f, -0.0884f },
	{ -0.1396f,  0.1289f, -0.0552f },
	{ -0.1675f,  0.0322f, -0.0444f },
	{  0.0205f,  0.1128f,  0.0767f },
	{ -0.0073f,  0.0483f,  0.0435f },
	{ -0.1235f,  0.1519f, -0.1816f },
	{ -0.1157f,  0.0474f, -0.1377f },
	{  0.0610f,  0.1235f, -0.0825f },
	{  0.0303f,  0.0664f, -0.0386f },
	{ -0.2539f,  0.0757f, -0.1484f },
	{ -0.3154f,  0.0947f, -0.0278f },
	{ -0.0850f,  0.1328f,  0.0601f },
	{ -0.1001f,  0.0474f,  0.0601f },
	{  0.0381f,  0.2568f, -0.1924f },
	{ -0.0313f,  0.1899f, -0.0605f },
	{  0.1611f,  0.3423f, -0.0278f },
	{  0.1074f,  0.2188f, -0.0386f },
	{ -0.1157f,  0.3042f, -0.0605f },
	{ -0.1924f,  0.2378f,  0.0493f },
	{  0.0229f,  0.2949f,  0.1479f },
	{ -0.0156f,  0.2188f,  0.0601f },
	{  0.0723f, -0.1597f, -0.2695f },
	{  0.0562f, -0.1313f, -0.1738f },
	{  0.2573f, -0.1807f, -0.1860f },
	{  0.2495f, -0.1240f, -0.0542f },
	{ -0.0649f, -0.0957f, -0.1260f },
	{ -0.1211f, -0.1665f, -0.0903f },
	{  0.0400f, -0.0674f, -0.0186f },
	{  0.0400f, -0.1382f, -0.0063f },
	{  0.2334f, -0.0039f, -0.2339f },
	{  0.1768f, -0.0674f, -0.1382f },
	{  0.3784f, -0.0107f, -0.1260f },
	{  0.3140f,  0.0034f,  0.0054f },
	{  0.0479f,  0.0034f, -0.0542f },
	{ -0.0005f,  0.0103f, -0.0186f },
	{  0.2012f,  0.0317f,  0.0415f },
	{  0.1123f, -0.0107f,  0.1011f },
	{  0.0703f,  0.0977f, -0.2432f },
	{  0.0845f,  0.0039f, -0.1934f },
	{  0.2476f,  0.0635f, -0.0688f },
	{  0.2051f, -0.0132f, -0.0439f },
	{ -0.0498f,  0.0039f, -0.0815f },
	{ -0.0781f,  0.0039f, -0.0317f },
	{  0.0776f,  0.0552f,  0.0181f },
	{  0.0493f,  0.0039f,  0.0308f },
	{  0.2402f,  0.2090f, -0.1685f },
	{  0.1694f,  0.1064f, -0.1313f },
	{  0.3608f,  0.2002f, -0.0068f },
	{  0.2192f,  0.1489f,  0.0181f },
	{  0.0845f,  0.1489f,  0.0181f },
	{ -0.0005f,  0.1318f,  0.0059f },
	{  0.1978f,  0.2432f,  0.1426f },
	{  0.0986f,  0.1577f,  0.1426f },
	{ -0.0684f, -0.1797f, -0.1621f },
	{ -0.1973f, -0.2529f, -0.1621f },
	{  0.0298f, -0.2310f,  0.0122f },
	{ -0.0459f, -0.1870f,  0.0122f },
	{ -0.2349f, -0.2017f, -0.0127f },
	{ -0.3711f, -0.2456f, -0.0376f },
	{ -0.1367f, -0.2017f,  0.0991f },
	{ -0.2651f, -0.2603f,  0.1489f },
	{ -0.0308f, -0.0479f, -0.0503f },
	{ -0.0459f, -0.1138f, -0.0503f },
	{  0.0981f, -0.0698f,  0.0371f },
	{  0.0449f, -0.0845f,  0.0620f },
	{ -0.1440f, -0.0332f,  0.0244f },
	{ -0.2046f, -0.1064f,  0.0742f },
	{ -0.0762f, -0.0405f,  0.1616f },
	{ -0.1138f, -0.1357f,  0.2114f },
	{ -0.1118f, -0.0513f, -0.0820f },
	{ -0.1714f, -0.0854f, -0.0410f },
	{ -0.0005f, -0.0376f,  0.0308f },
	{ -0.0522f, -0.0444f,  0.0718f },
	{ -0.2534f, -0.0376f,  0.0000f },
	{ -0.3203f, -0.0786f,  0.1128f },
	{ -0.1567f, -0.0239f,  0.1128f },
	{ -0.2012f, -0.0171f,  0.2256f },
	{ -0.0522f,  0.0854f, -0.0718f },
	{ -0.0674f,  0.0649f, -0.0103f },
	{  0.1411f,  0.0854f,  0.0923f },
	{  0.0591f,  0.0513f,  0.1025f },
	{ -0.1865f,  0.0649f,  0.0513f },
	{ -0.2759f,  0.0991f,  0.1436f },
	{ -0.0078f,  0.1196f,  0.1948f },
	{ -0.1196f,  0.0786f,  0.1743f },
	{  0.0732f, -0.2876f, -0.1118f },
	{ -0.0332f, -0.3340f, -0.2012f },
	{  0.2451f, -0.2876f, -0.0225f },
	{  0.1060f, -0.3525f,  0.0669f },
	{ -0.0903f, -0.2783f, -0.0225f },
	{ -0.1719f, -0.3804f,  0.0220f },
	{  0.0649f, -0.2134f,  0.1265f },
	{ -0.0410f, -0.2876f,  0.1563f },
	{  0.1226f, -0.1021f, -0.0376f },
	{  0.1226f, -0.1855f, -0.0674f },
	{  0.3350f, -0.1299f,  0.0815f },
	{  0.2124f, -0.1855f,  0.1113f },
	{ -0.0249f, -0.1021f,  0.0371f },
	{  0.0078f, -0.1577f,  0.0669f },
	{  0.1226f, -0.1021f,  0.1411f },
	{  0.0898f, -0.1577f,  0.2603f },
	{  0.0273f, -0.1265f, -0.0791f },
	{ -0.0137f, -0.1987f, -0.0791f },
	{  0.1846f, -0.0723f,  0.0469f },
	{  0.1299f, -0.1626f,  0.0361f },
	{ -0.0957f, -0.1177f,  0.0679f },
	{ -0.1230f, -0.1538f,  0.0049f },
	{  0.0205f, -0.0903f,  0.1411f },
	{ -0.0410f, -0.1265f,  0.1309f },
	{  0.1299f,  0.0088f,  0.0151f },
	{  0.1025f, -0.0273f, -0.0269f },
	{  0.3076f,  0.0811f,  0.1621f },
	{  0.2188f, -0.0273f,  0.1411f },
	{  0.0205f, -0.0093f,  0.1099f },
	{ -0.0410f,  0.0361f,  0.1099f },
	{  0.1299f,  0.0449f,  0.2358f },
	{  0.0205f, -0.0093f,  0.2358f },
};

/*! \brief PRBA[1:3] Difference vector quantization table */
const float ir77_ambe_prba_dif13_vq[256][3] = {
	{ -0.0952f,  0.0181f,  0.0547f },
	{ -0.2441f, -0.0371f,  0.0547f },
	{ -0.2983f, -0.1104f,  0.1401f },
	{ -0.6504f, -0.2017f,  0.0303f },
	{ -0.2036f,  0.0181f, -0.1162f },
	{ -0.4746f,  0.0361f, -0.1650f },
	{ -0.5962f, -0.0552f, -0.0918f },
	{ -0.8130f, -0.2017f, -0.1650f },
	{  0.0269f, -0.0552f,  0.0059f },
	{ -0.1357f, -0.0918f, -0.0430f },
	{ -0.0815f, -0.2202f,  0.0791f },
	{ -0.3252f, -0.2568f, -0.0430f },
	{ -0.0679f, -0.0552f, -0.0796f },
	{ -0.2983f, -0.1470f, -0.2383f },
	{ -0.3389f, -0.2935f, -0.1772f },
	{ -0.5830f, -0.5313f, -0.2139f },
	{ -0.0640f,  0.1646f,  0.0581f },
	{ -0.1914f,  0.1523f,  0.0400f },
	{ -0.1489f,  0.0791f,  0.0674f },
	{ -0.4995f,  0.0791f,  0.0220f },
	{ -0.1382f,  0.1768f, -0.0591f },
	{ -0.3823f,  0.2622f, -0.1494f },
	{ -0.3506f,  0.1279f, -0.0591f },
	{ -0.5947f,  0.1890f, -0.1313f },
	{  0.0635f,  0.0303f,  0.0132f },
	{ -0.0215f,  0.0059f, -0.0229f },
	{ -0.0107f, -0.0308f,  0.0400f },
	{ -0.1807f, -0.0552f, -0.0049f },
	{  0.0210f,  0.0547f, -0.0771f },
	{ -0.1489f,  0.0791f, -0.1855f },
	{ -0.1274f, -0.0063f, -0.0684f },
	{ -0.4146f, -0.1040f, -0.2036f },
	{  0.0640f,  0.0166f, -0.0391f },
	{ -0.0771f,  0.0166f, -0.0498f },
	{ -0.0771f, -0.0845f,  0.0049f },
	{ -0.2949f, -0.1743f, -0.1157f },
	{ -0.0259f,  0.0391f, -0.0718f },
	{ -0.3081f,  0.0391f, -0.1709f },
	{ -0.2695f, -0.0845f, -0.1489f },
	{ -0.6025f, -0.1294f, -0.3354f },
	{  0.1919f, -0.0620f, -0.0718f },
	{ -0.0005f, -0.0732f, -0.1050f },
	{  0.1152f, -0.1631f, -0.0278f },
	{ -0.1924f, -0.2305f, -0.0938f },
	{  0.0640f, -0.0508f, -0.1270f },
	{ -0.0898f, -0.1743f, -0.2915f },
	{ -0.0645f, -0.1069f, -0.1377f },
	{ -0.4106f, -0.2979f, -0.3354f },
	{  0.1538f,  0.1621f, -0.0771f },
	{ -0.0122f,  0.1157f, -0.0659f },
	{ -0.0239f,  0.0415f,  0.0107f },
	{ -0.1304f,  0.0693f, -0.0771f },
	{ -0.0356f,  0.2085f, -0.1318f },
	{ -0.2251f,  0.1714f, -0.1978f },
	{ -0.0596f,  0.1250f, -0.1099f },
	{ -0.2251f, -0.0049f, -0.3296f },
	{  0.5088f,  0.1343f, -0.0991f },
	{  0.0942f,  0.0415f, -0.1099f },
	{  0.1416f, -0.0234f, -0.0332f },
	{ -0.0239f, -0.0234f, -0.0659f },
	{  0.2363f,  0.0415f, -0.2856f },
	{ -0.0239f,  0.0322f, -0.2856f },
	{  0.1299f, -0.0049f, -0.1978f },
	{ -0.1777f, -0.0698f, -0.2090f },
	{  0.0327f, -0.0645f,  0.0557f },
	{ -0.0991f, -0.1289f,  0.0557f },
	{ -0.0552f, -0.1450f,  0.1528f },
	{ -0.3955f, -0.2900f,  0.1670f },
	{ -0.0659f, -0.0483f, -0.0278f },
	{ -0.2310f, -0.0806f, -0.0698f },
	{ -0.3076f, -0.1450f,  0.0000f },
	{ -0.5054f, -0.2739f, -0.1113f },
	{  0.1758f, -0.2095f,  0.0557f },
	{  0.0439f, -0.2095f,  0.0000f },
	{  0.0547f, -0.3545f,  0.0835f },
	{ -0.2090f, -0.4028f, -0.0420f },
	{  0.0767f, -0.0967f, -0.0835f },
	{ -0.0771f, -0.2256f, -0.1533f },
	{ -0.0659f, -0.2900f, -0.0142f },
	{ -0.2417f, -0.5317f, -0.2505f },
	{  0.1348f,  0.0659f,  0.1279f },
	{ -0.0391f,  0.0132f,  0.0337f },
	{  0.0190f, -0.0132f,  0.0767f },
	{ -0.1353f, -0.0483f,  0.0679f },
	{  0.0288f,  0.0659f, -0.0259f },
	{ -0.2222f,  0.1011f, -0.0688f },
	{ -0.0581f,  0.0044f, -0.0088f },
	{ -0.3184f, -0.0220f, -0.0771f },
	{  0.2891f, -0.0308f, -0.0347f },
	{  0.0767f, -0.0659f, -0.0005f },
	{  0.0576f, -0.1099f,  0.0425f },
	{ -0.0391f, -0.1626f, -0.0176f },
	{  0.2217f,  0.0044f, -0.1113f },
	{  0.0288f, -0.0044f, -0.0859f },
	{ -0.0098f, -0.1099f, -0.0518f },
	{ -0.1548f, -0.1450f, -0.1284f },
	{  0.2119f, -0.0366f,  0.0127f },
	{  0.0381f, -0.0610f, -0.0522f },
	{  0.0576f, -0.1221f, -0.0132f },
	{ -0.1162f, -0.1831f, -0.0264f },
	{  0.0864f, -0.0244f, -0.0522f },
	{ -0.0581f, -0.0366f, -0.1816f },
	{ -0.0776f, -0.1221f, -0.0649f },
	{ -0.2026f, -0.2441f, -0.2334f },
	{  0.3950f, -0.2686f, -0.1040f },
	{  0.1445f, -0.1587f, -0.1167f },
	{  0.1924f, -0.2686f, -0.0522f },
	{ -0.0005f, -0.4028f, -0.1558f },
	{  0.2407f, -0.2319f, -0.2202f },
	{  0.0962f, -0.1343f, -0.3110f },
	{  0.0767f, -0.2686f, -0.1816f },
	{  0.0288f, -0.3784f, -0.3755f },
	{  0.4521f,  0.1436f,  0.0801f },
	{  0.0879f,  0.0601f, -0.0103f },
	{  0.1504f,  0.0137f,  0.0200f },
	{ -0.0005f, -0.0142f,  0.0000f },
	{  0.1758f,  0.0972f, -0.1304f },
	{  0.0122f,  0.1064f, -0.1304f },
	{  0.0122f,  0.0229f, -0.0400f },
	{ -0.0630f,  0.0322f, -0.1201f },
	{  0.7163f, -0.0513f, -0.0400f },
	{  0.3140f,  0.0601f, -0.1001f },
	{  0.2510f, -0.1348f, -0.0703f },
	{  0.1255f, -0.0884f, -0.0400f },
	{  0.5151f, -0.0327f, -0.2202f },
	{  0.3140f, -0.0698f, -0.1904f },
	{  0.1504f, -0.0791f, -0.1504f },
	{  0.0249f, -0.1162f, -0.1802f },
	{  0.0000f,  0.0762f,  0.1948f },
	{ -0.1685f,  0.0923f,  0.1831f },
	{ -0.0986f, -0.0044f,  0.2637f },
	{ -0.4917f,  0.0278f,  0.2061f },
	{ -0.1123f,  0.1162f,  0.0225f },
	{ -0.3369f,  0.1084f,  0.0571f },
	{ -0.3511f,  0.0117f,  0.1372f },
	{ -0.7300f,  0.0679f,  0.0913f },
	{  0.1401f, -0.0288f,  0.0913f },
	{ -0.0562f, -0.0127f,  0.0801f },
	{  0.0000f, -0.0366f,  0.1719f },
	{ -0.1826f, -0.1172f,  0.1260f },
	{  0.0137f,  0.0195f, -0.0005f },
	{ -0.1548f,  0.0278f, -0.0234f },
	{ -0.1968f, -0.1333f, -0.0005f },
	{ -0.4355f, -0.1333f, -0.0806f },
	{  0.0513f,  0.3936f,  0.2974f },
	{ -0.0308f,  0.2158f,  0.1641f },
	{ -0.0410f,  0.1650f,  0.2974f },
	{ -0.2871f,  0.1143f,  0.2256f },
	{  0.0103f,  0.3301f,  0.0308f },
	{ -0.2256f,  0.3174f,  0.1333f },
	{ -0.2051f,  0.1777f,  0.1333f },
	{ -0.4204f,  0.2285f,  0.1128f },
	{  0.2256f,  0.1523f,  0.1538f },
	{  0.0308f,  0.1396f,  0.0615f },
	{  0.0718f,  0.0000f,  0.1641f },
	{ -0.1128f,  0.0381f,  0.1128f },
	{  0.0718f,  0.1396f, -0.0103f },
	{ -0.0718f,  0.1396f, -0.0205f },
	{ -0.0718f,  0.0762f,  0.0615f },
	{ -0.2563f,  0.0254f,  0.0000f },
	{  0.1553f,  0.1582f,  0.0601f },
	{ -0.0132f,  0.0791f,  0.0415f },
	{ -0.0259f,  0.0352f,  0.0786f },
	{ -0.1943f, -0.0088f,  0.1436f },
	{  0.0127f,  0.1143f,  0.0044f },
	{ -0.1426f,  0.1055f, -0.0234f },
	{ -0.0776f,  0.0527f,  0.0044f },
	{ -0.3882f, -0.0352f,  0.0044f },
	{  0.4141f,  0.0264f,  0.0137f },
	{  0.0903f,  0.0000f,  0.0044f },
	{  0.1294f, -0.1143f,  0.0508f },
	{ -0.0259f, -0.0879f,  0.0229f },
	{  0.1294f,  0.0439f, -0.0605f },
	{  0.0127f,  0.0176f, -0.1440f },
	{  0.0127f, -0.0176f, -0.0420f },
	{ -0.1426f, -0.0527f, -0.1348f },
	{  0.3652f,  0.4199f,  0.2012f },
	{  0.0752f,  0.2261f,  0.0620f },
	{  0.0645f,  0.1484f,  0.1318f },
	{ -0.0859f,  0.1353f,  0.1318f },
	{  0.1074f,  0.2778f, -0.0908f },
	{ -0.1611f,  0.3164f, -0.1465f },
	{  0.0000f,  0.1743f,  0.0063f },
	{ -0.2363f,  0.2261f, -0.0073f },
	{  0.4297f,  0.3555f, -0.0073f },
	{  0.2686f,  0.1484f,  0.0063f },
	{  0.2148f,  0.0576f,  0.0762f },
	{  0.0000f,  0.0190f,  0.0342f },
	{  0.2578f,  0.2905f, -0.2300f },
	{  0.0752f,  0.1353f, -0.2163f },
	{  0.0752f,  0.0967f, -0.0630f },
	{ -0.0645f,  0.0835f, -0.0493f },
	{  0.1206f, -0.0054f,  0.2803f },
	{ -0.0005f,  0.0244f,  0.1187f },
	{  0.0117f, -0.1255f,  0.2520f },
	{ -0.2422f, -0.0654f,  0.2710f },
	{  0.0479f,  0.0347f,  0.0615f },
	{ -0.1694f,  0.0244f,  0.0327f },
	{ -0.0972f, -0.0454f,  0.1470f },
	{ -0.4473f, -0.1455f,  0.0708f },
	{  0.3018f, -0.1157f,  0.2329f },
	{  0.1084f, -0.1055f,  0.1567f },
	{  0.1206f, -0.2456f,  0.1851f },
	{ -0.1572f, -0.2656f,  0.2041f },
	{  0.1328f, -0.0557f,  0.0234f },
	{ -0.0122f, -0.0654f, -0.0146f },
	{ -0.0122f, -0.1357f,  0.0708f },
	{ -0.2300f, -0.2358f,  0.0518f },
	{  0.2588f,  0.2451f,  0.3184f },
	{  0.1294f,  0.1494f,  0.2231f },
	{  0.2588f,  0.1016f,  0.2900f },
	{ -0.0581f,  0.0415f,  0.1567f },
	{  0.1724f,  0.2930f,  0.1377f },
	{ -0.1157f,  0.2451f,  0.0234f },
	{ -0.0005f,  0.1016f,  0.0996f },
	{ -0.2163f,  0.0654f,  0.0996f },
	{  0.6768f,  0.0654f,  0.2422f },
	{  0.2158f,  0.0059f,  0.1470f },
	{  0.3022f,  0.0537f,  0.1758f },
	{  0.0571f, -0.0659f,  0.1089f },
	{  0.3164f,  0.0776f,  0.0518f },
	{  0.0430f,  0.0654f,  0.0234f },
	{  0.0859f, -0.0181f,  0.0615f },
	{ -0.0581f, -0.0420f,  0.0327f },
	{  0.4556f,  0.0659f,  0.2202f },
	{  0.1211f,  0.0396f,  0.0562f },
	{  0.2124f, -0.1187f,  0.1279f },
	{ -0.0308f, -0.0659f,  0.0869f },
	{  0.1514f,  0.1055f, -0.0054f },
	{ -0.0205f,  0.0527f, -0.0259f },
	{  0.0400f, -0.0132f,  0.0356f },
	{ -0.1016f, -0.0132f, -0.0054f },
	{  0.5264f, -0.2637f,  0.0562f },
	{  0.3340f, -0.1582f,  0.0459f },
	{  0.3135f, -0.3032f,  0.1074f },
	{  0.0605f, -0.1714f,  0.0972f },
	{  0.4253f, -0.0791f, -0.0771f },
	{  0.1211f, -0.0264f, -0.0977f },
	{  0.1919f, -0.1187f,  0.0049f },
	{  0.0200f, -0.1714f, -0.0977f },
	{  0.6699f,  0.3623f,  0.1587f },
	{  0.4766f,  0.2549f,  0.2202f },
	{  0.3276f,  0.2280f,  0.1382f },
	{  0.0596f,  0.0532f,  0.1074f },
	{  0.2231f,  0.2549f,  0.0049f },
	{  0.0298f,  0.1875f, -0.0669f },
	{  0.0894f,  0.0938f,  0.0459f },
	{ -0.0298f,  0.0938f,  0.0049f },
	{  0.8936f,  0.1206f, -0.0156f },
	{  0.6255f,  0.1475f,  0.0767f },
	{  0.4912f, -0.0405f,  0.0869f },
	{  0.2979f, -0.0405f,  0.0767f },
	{  0.6553f,  0.2817f, -0.0874f },
	{  0.3276f,  0.1606f, -0.0977f },
	{  0.2085f,  0.0400f, -0.0259f },
	{  0.0444f, -0.0273f, -0.0054f },
};

/*! \brief PRBA[4:7] Difference vector quantization table */
const float ir77_ambe_prba_dif47_vq[64][4] = {
	{  0.0044f,  0.0229f,  0.0151f, -0.0024f },
	{  0.0044f, -0.0298f,  0.0239f,  0.0142f },
	{ -0.0308f,  0.0757f, -0.0815f, -0.0356f },
	{ -0.0483f, -0.0298f, -0.0464f, -0.0107f },
	{  0.0747f,  0.0098f,  0.0151f, -0.0439f },
	{  0.1099f, -0.0825f,  0.0327f, -0.0605f },
	{  0.0747f,  0.0361f, -0.0640f, -0.0938f },
	{  0.0396f, -0.0562f, -0.0376f, -0.0605f },
	{ -0.0659f, -0.0166f,  0.0327f,  0.0059f },
	{ -0.0835f, -0.1089f,  0.0151f,  0.0225f },
	{ -0.1362f, -0.0034f, -0.0903f, -0.0356f },
	{ -0.1538f, -0.1221f, -0.0991f, -0.0356f },
	{ -0.0132f, -0.0298f,  0.0151f, -0.0356f },
	{  0.0044f, -0.1089f,  0.0151f, -0.0273f },
	{ -0.0308f, -0.0298f, -0.0991f, -0.0938f },
	{ -0.0483f, -0.1221f, -0.0728f, -0.0522f },
	{ -0.0127f,  0.0991f,  0.0933f, -0.0137f },
	{ -0.0293f,  0.0142f,  0.0767f,  0.0000f },
	{ -0.0791f,  0.1416f, -0.0063f, -0.0137f },
	{ -0.0625f,  0.0425f,  0.0020f,  0.0000f },
	{  0.0703f,  0.0566f,  0.0850f, -0.0820f },
	{  0.0371f, -0.0142f,  0.0767f, -0.0615f },
	{  0.0039f,  0.1133f,  0.0103f, -0.0889f },
	{  0.0039f,  0.0142f, -0.0063f, -0.0615f },
	{ -0.1289f,  0.0708f,  0.1016f,  0.0137f },
	{ -0.1289f, -0.0425f,  0.1099f,  0.0000f },
	{ -0.1787f,  0.0708f,  0.0186f, -0.0410f },
	{ -0.1787f, -0.0425f,  0.0020f,  0.0068f },
	{ -0.0625f,  0.0283f,  0.0684f, -0.0752f },
	{ -0.0459f, -0.0708f,  0.1016f, -0.0615f },
	{ -0.0791f,  0.0425f, -0.0146f, -0.0820f },
	{ -0.0791f, -0.0566f,  0.0020f, -0.0684f },
	{  0.0981f,  0.0474f, -0.0029f,  0.0791f },
	{  0.0640f, -0.0288f,  0.0186f,  0.0688f },
	{  0.0640f,  0.0601f, -0.0996f,  0.0483f },
	{  0.0298f, -0.0288f, -0.0674f,  0.0688f },
	{  0.1836f,  0.0093f, -0.0029f, -0.0234f },
	{  0.1665f, -0.0796f,  0.0186f,  0.0483f },
	{  0.1494f,  0.0347f, -0.1104f, -0.0439f },
	{  0.1152f, -0.0542f, -0.0781f,  0.0278f },
	{ -0.0044f, -0.0161f,  0.0078f,  0.0791f },
	{ -0.0044f, -0.1050f, -0.0029f,  0.1099f },
	{ -0.0386f,  0.0220f, -0.0996f,  0.0483f },
	{ -0.0728f, -0.0923f, -0.1211f,  0.0483f },
	{  0.0640f, -0.0415f,  0.0078f, -0.0029f },
	{  0.0640f, -0.1304f,  0.0400f,  0.0278f },
	{  0.0298f, -0.0161f, -0.0996f, -0.0132f },
	{  0.0469f, -0.1304f, -0.0996f,  0.0073f },
	{  0.0684f,  0.1128f,  0.0811f,  0.0586f },
	{  0.0171f,  0.0151f,  0.0996f,  0.0732f },
	{  0.0000f,  0.1128f, -0.0396f,  0.0586f },
	{  0.0000f,  0.0273f, -0.0117f,  0.0513f },
	{  0.1538f,  0.1250f,  0.0625f,  0.0000f },
	{  0.1196f,  0.0029f,  0.0903f,  0.0146f },
	{  0.0684f,  0.1128f, -0.0303f, -0.0146f },
	{  0.0684f,  0.0151f, -0.0303f,  0.0073f },
	{ -0.0513f,  0.0640f,  0.0439f,  0.0732f },
	{ -0.0854f, -0.0337f,  0.0811f,  0.0952f },
	{ -0.1025f,  0.0640f, -0.0303f,  0.0586f },
	{ -0.0854f, -0.0215f, -0.0210f,  0.0659f },
	{  0.0342f,  0.0518f,  0.0439f,  0.0073f },
	{  0.0342f, -0.0581f,  0.0811f,  0.0220f },
	{  0.0000f,  0.0396f, -0.0396f, -0.0073f },
	{  0.0000f, -0.0337f, -0.0210f,  0.0146f },
};

/*! \brief HOC 1st block Sum vector quantization table */
const float ir77_ambe_hoc0_sum_vq[128][4] = {
	{ -0.1294f,  0.0190f,  0.0112f, -0.0073f },
	{ -0.2329f,  0.0698f,  0.0562f, -0.0630f },
	{ -0.1812f,  0.0952f,  0.0562f,  0.1597f },
	{ -0.3105f,  0.2476f,  0.1909f,  0.0762f },
	{ -0.0259f,  0.0190f, -0.0337f, -0.0073f },
	{ -0.1294f,  0.1206f, -0.0337f, -0.0630f },
	{ -0.0518f,  0.0698f, -0.1011f,  0.0205f },
	{ -0.1035f,  0.2983f,  0.0337f,  0.1040f },
	{ -0.0518f,  0.0698f,  0.0337f, -0.1187f },
	{ -0.1553f,  0.1460f,  0.1011f, -0.1743f },
	{ -0.1035f,  0.1460f,  0.1011f, -0.0073f },
	{ -0.1035f,  0.3491f,  0.2358f, -0.0352f },
	{  0.0776f, -0.0317f, -0.1011f, -0.1187f },
	{  0.0518f,  0.1206f, -0.0562f, -0.2578f },
	{  0.0000f,  0.1460f, -0.0562f, -0.0630f },
	{  0.0518f,  0.2983f,  0.0112f, -0.1187f },
	{ -0.1416f,  0.0151f, -0.0747f,  0.0493f },
	{ -0.2905f,  0.1665f,  0.0000f, -0.0356f },
	{ -0.2012f,  0.1060f, -0.1245f,  0.1060f },
	{ -0.3799f,  0.2573f, -0.0747f,  0.1060f },
	{ -0.1118f, -0.0757f, -0.2241f,  0.0210f },
	{ -0.2310f,  0.0454f, -0.1245f, -0.0640f },
	{ -0.0225f,  0.0454f, -0.1992f,  0.1626f },
	{ -0.2012f,  0.1968f, -0.2490f,  0.0493f },
	{ -0.0820f, -0.0151f, -0.0996f, -0.0923f },
	{ -0.1714f,  0.1968f, -0.0996f, -0.1772f },
	{ -0.1118f,  0.2271f, -0.0996f, -0.0356f },
	{ -0.2607f,  0.3784f,  0.0000f, -0.1206f },
	{  0.0669f, -0.0757f, -0.3735f, -0.0640f },
	{ -0.1416f, -0.0151f, -0.2739f, -0.2622f },
	{  0.0073f,  0.1060f, -0.1992f, -0.0356f },
	{  0.0073f,  0.3179f, -0.1992f, -0.0356f },
	{  0.1543f, -0.0957f,  0.1250f,  0.0732f },
	{  0.0771f,  0.0479f,  0.1250f,  0.0000f },
	{  0.1929f,  0.0479f,  0.2549f,  0.1709f },
	{  0.0000f,  0.1914f,  0.2363f,  0.1221f },
	{  0.2700f, -0.2871f,  0.0137f,  0.1221f },
	{  0.1929f,  0.0000f, -0.0234f,  0.0244f },
	{  0.4629f,  0.1436f,  0.0508f,  0.1465f },
	{  0.1543f,  0.2393f,  0.0508f,  0.2441f },
	{  0.3086f,  0.0957f,  0.1807f, -0.0244f },
	{  0.1157f,  0.1436f,  0.1992f, -0.1221f },
	{  0.3472f,  0.2393f,  0.1992f,  0.1221f },
	{  0.1929f,  0.3350f,  0.2178f,  0.0244f },
	{  0.3857f, -0.0957f,  0.0322f,  0.0488f },
	{  0.3086f,  0.2393f,  0.0322f, -0.1221f },
	{  0.5786f,  0.2871f,  0.1621f,  0.0488f },
	{  0.2314f,  0.4307f, -0.0049f,  0.0732f },
	{  0.1245f, -0.1230f, -0.0464f,  0.0410f },
	{  0.0532f, -0.0137f, -0.0669f,  0.0410f },
	{  0.1958f,  0.0137f, -0.0259f,  0.2051f },
	{  0.0176f,  0.0957f,  0.0356f,  0.1348f },
	{  0.2671f, -0.1504f, -0.1284f,  0.0176f },
	{  0.1245f, -0.0137f, -0.1489f,  0.0410f },
	{  0.3027f,  0.0137f, -0.2104f,  0.1113f },
	{  0.0889f,  0.1504f, -0.1694f,  0.0879f },
	{  0.1958f, -0.0410f,  0.0151f, -0.1230f },
	{  0.1245f,  0.0957f, -0.0464f, -0.0762f },
	{  0.3027f,  0.1230f, -0.0259f,  0.0176f },
	{  0.1245f,  0.2051f,  0.0767f,  0.0176f },
	{  0.4810f, -0.1777f, -0.1899f, -0.0293f },
	{  0.3027f, -0.0137f, -0.1694f, -0.1465f },
	{  0.5522f,  0.1230f, -0.0669f, -0.0762f },
	{  0.2671f,  0.2324f, -0.2310f, -0.0059f },
	{ -0.5029f, -0.1719f,  0.1777f, -0.0068f },
	{ -0.7456f, -0.0996f,  0.0986f,  0.0195f },
	{ -0.4336f,  0.0088f,  0.2041f,  0.1514f },
	{ -0.6763f,  0.2256f,  0.2305f, -0.0068f },
	{ -0.3643f, -0.3164f, -0.0068f, -0.0859f },
	{ -0.5723f, -0.1357f, -0.0596f, -0.0859f },
	{ -0.3643f,  0.0088f, -0.0332f,  0.1250f },
	{ -0.5029f,  0.0811f,  0.0459f,  0.0195f },
	{ -0.3643f, -0.0996f,  0.0723f, -0.1650f },
	{ -0.5723f,  0.0449f,  0.0459f, -0.1914f },
	{ -0.3643f,  0.0088f,  0.1250f, -0.0332f },
	{ -0.3989f,  0.1533f,  0.1514f, -0.1650f },
	{ -0.2603f, -0.2080f, -0.1650f, -0.1650f },
	{ -0.2603f, -0.0273f, -0.0859f, -0.2441f },
	{ -0.2256f, -0.1357f, -0.1387f, -0.0068f },
	{ -0.3643f,  0.0811f, -0.0332f, -0.1650f },
	{ -0.6401f, -0.4102f, -0.0518f,  0.0444f },
	{ -0.9185f, -0.2461f, -0.1035f, -0.1079f },
	{ -0.4731f, -0.1914f,  0.0000f,  0.1460f },
	{ -0.6958f,  0.0273f, -0.1294f,  0.0190f },
	{ -0.3618f, -0.5195f, -0.2847f, -0.1333f },
	{ -0.6401f, -0.3008f, -0.3364f, -0.0825f },
	{ -0.4175f, -0.2461f, -0.1812f,  0.0190f },
	{ -0.4175f, -0.0273f, -0.2070f,  0.0444f },
	{ -0.5288f, -0.3008f, -0.1035f, -0.2349f },
	{ -0.7515f, -0.0273f, -0.1812f, -0.2095f },
	{ -0.3618f, -0.0820f, -0.0518f, -0.0571f },
	{ -0.5288f,  0.3008f, -0.0776f, -0.0825f },
	{ -0.0835f, -0.4102f, -0.3882f, -0.2349f },
	{ -0.4175f, -0.1914f, -0.2847f, -0.2095f },
	{ -0.2505f, -0.1367f, -0.3623f, -0.0825f },
	{ -0.4175f,  0.0820f, -0.2329f, -0.1333f },
	{ -0.1489f, -0.2637f,  0.1646f,  0.0000f },
	{ -0.2681f, -0.1230f,  0.2231f, -0.0864f },
	{ -0.1489f, -0.0527f,  0.2378f,  0.2305f },
	{ -0.2085f,  0.0645f,  0.2378f,  0.0576f },
	{  0.0298f, -0.2871f,  0.0620f,  0.0576f },
	{ -0.0298f, -0.1465f,  0.0913f, -0.0576f },
	{  0.0298f, -0.0996f,  0.0474f,  0.1729f },
	{ -0.0894f, -0.0293f,  0.0474f,  0.0864f },
	{  0.0596f, -0.2637f,  0.2085f, -0.1152f },
	{ -0.1191f, -0.0527f,  0.1792f, -0.2017f },
	{ -0.0298f, -0.0762f,  0.1938f,  0.0288f },
	{ -0.1489f, -0.0059f,  0.0913f, -0.0864f },
	{  0.1787f, -0.1934f,  0.0474f, -0.0864f },
	{  0.0000f, -0.0996f,  0.0327f, -0.2017f },
	{  0.0596f, -0.0762f,  0.0181f,  0.0000f },
	{ -0.0298f, -0.0293f,  0.0474f, -0.0576f },
	{ -0.2231f, -0.4468f,  0.0298f,  0.1191f },
	{ -0.2847f, -0.1587f,  0.0898f,  0.0894f },
	{ -0.2539f, -0.1299f, -0.0103f,  0.2383f },
	{ -0.2231f, -0.0146f,  0.0098f,  0.0596f },
	{  0.1460f, -0.4468f, -0.1904f,  0.0298f },
	{ -0.1924f, -0.3027f, -0.1704f,  0.0298f },
	{  0.0229f, -0.1875f, -0.1304f,  0.2085f },
	{ -0.1001f, -0.1011f, -0.0503f,  0.1191f },
	{ -0.0693f, -0.3315f, -0.0503f, -0.1191f },
	{ -0.1924f, -0.1587f,  0.0098f, -0.0894f },
	{ -0.1001f, -0.1875f,  0.0498f,  0.0894f },
	{ -0.1924f, -0.0723f,  0.1099f,  0.0000f },
	{  0.1768f, -0.2451f, -0.1904f, -0.2085f },
	{ -0.0693f, -0.1875f, -0.1904f, -0.2085f },
	{ -0.0078f, -0.1587f, -0.1304f, -0.0298f },
	{ -0.0693f, -0.0723f, -0.0103f,  0.0000f },
};

/*! \brief HOC 1st block Difference vector quantization table */
const float ir77_ambe_hoc0_dif_vq[8][4] = {
	{ -0.0303f, -0.1313f,  0.0117f,  0.0107f },
	{ -0.2422f, -0.0771f, -0.0508f, -0.0166f },
	{  0.1211f, -0.0771f, -0.0352f,  0.0107f },
	{  0.0000f,  0.0132f, -0.0742f,  0.0264f },
	{  0.0000f, -0.0049f,  0.0352f, -0.0166f },
	{ -0.1211f,  0.0493f,  0.0430f, -0.0049f },
	{  0.2119f,  0.0854f,  0.0195f,  0.0146f },
	{  0.0303f,  0.1396f,  0.0039f, -0.0283f },
};

/*! \brief HOC 2nd block Sum vector quantization table */
const float ir77_ambe_hoc1_sum_vq[128][4] = {
	{  0.4424f, -0.0303f,  0.0767f,  0.1050f },
	{  0.2607f,  0.1816f,  0.1997f,  0.0649f },
	{  0.1699f,  0.0000f,  0.0767f, -0.0151f },
	{  0.0791f,  0.1211f,  0.1997f, -0.0552f },
	{  0.2153f, -0.0605f, -0.1079f,  0.0249f },
	{  0.0791f,  0.1816f, -0.0259f, -0.0752f },
	{ -0.0117f, -0.0303f, -0.0054f, -0.0151f },
	{ -0.1025f,  0.1514f,  0.0972f, -0.1152f },
	{  0.1245f,  0.1211f,  0.0767f,  0.1851f },
	{  0.0791f,  0.2725f,  0.1587f,  0.1050f },
	{ -0.0117f,  0.1211f,  0.0767f,  0.0249f },
	{ -0.1025f,  0.3027f,  0.1997f, -0.0752f },
	{  0.0337f,  0.0605f, -0.0464f,  0.0850f },
	{ -0.0117f,  0.3936f, -0.0259f,  0.0449f },
	{ -0.1479f,  0.0908f, -0.0669f, -0.0552f },
	{ -0.2388f,  0.3027f, -0.0054f, -0.0952f },
	{  0.3013f, -0.2007f,  0.1489f,  0.0947f },
	{  0.1621f, -0.0195f,  0.2212f,  0.0547f },
	{  0.1157f, -0.2266f,  0.2031f, -0.0854f },
	{ -0.0234f, -0.0195f,  0.2754f, -0.0854f },
	{  0.1621f, -0.2007f,  0.0405f,  0.0146f },
	{  0.0229f, -0.0195f,  0.1309f, -0.0054f },
	{ -0.1162f, -0.2266f,  0.1128f, -0.1255f },
	{ -0.2090f, -0.0195f,  0.1670f, -0.1255f },
	{  0.1157f, -0.0713f,  0.1489f,  0.1748f },
	{ -0.0234f,  0.0840f,  0.2031f,  0.1147f },
	{ -0.1162f, -0.0972f,  0.1489f,  0.0347f },
	{ -0.2090f,  0.0840f,  0.2031f, -0.0054f },
	{ -0.0698f, -0.0713f,  0.0044f,  0.0547f },
	{ -0.1626f,  0.1099f,  0.0405f,  0.0146f },
	{ -0.3018f, -0.1230f,  0.0225f, -0.0054f },
	{ -0.3945f,  0.1616f,  0.1309f, -0.1255f },
	{  0.1816f, -0.0063f,  0.0088f,  0.1206f },
	{  0.0908f,  0.0654f,  0.1143f,  0.0576f },
	{  0.0000f, -0.1260f,  0.0439f,  0.0366f },
	{ -0.0908f,  0.0176f,  0.0967f, -0.0054f },
	{  0.0000f, -0.1021f, -0.0615f,  0.0996f },
	{ -0.0454f,  0.0415f, -0.0088f,  0.0366f },
	{ -0.1362f, -0.1260f, -0.0439f, -0.0474f },
	{ -0.3179f,  0.0894f, -0.0264f, -0.0474f },
	{  0.0000f,  0.0415f, -0.0615f,  0.2676f },
	{ -0.0908f,  0.1851f,  0.0264f,  0.1626f },
	{ -0.1362f,  0.0176f,  0.0264f,  0.0996f },
	{ -0.2725f,  0.2329f,  0.1143f,  0.0786f },
	{ -0.1362f,  0.0415f, -0.1143f,  0.1416f },
	{ -0.2725f,  0.1851f, -0.0967f,  0.0996f },
	{ -0.3179f, -0.0303f, -0.1494f,  0.0786f },
	{ -0.4995f,  0.2329f, -0.0439f,  0.0156f },
	{  0.0820f, -0.2373f,  0.0322f,  0.1699f },
	{ -0.0117f, -0.1582f,  0.1826f,  0.0820f },
	{ -0.0586f, -0.3428f,  0.1182f,  0.0820f },
	{ -0.1992f, -0.1846f,  0.2686f, -0.0498f },
	{ -0.1055f, -0.2637f, -0.0537f,  0.1699f },
	{ -0.1523f, -0.1582f,  0.0322f,  0.0381f },
	{ -0.2930f, -0.3691f, -0.0322f, -0.0059f },
	{ -0.3867f, -0.1582f,  0.1182f, -0.0718f },
	{ -0.1055f, -0.1318f,  0.0752f,  0.2578f },
	{ -0.1992f,  0.0264f,  0.1396f,  0.1699f },
	{ -0.2930f, -0.2637f,  0.1396f,  0.1040f },
	{ -0.3398f,  0.0000f,  0.2256f,  0.1040f },
	{ -0.2461f, -0.1318f, -0.0322f,  0.1479f },
	{ -0.3398f,  0.0000f,  0.0537f,  0.0601f },
	{ -0.4805f, -0.2109f, -0.0322f,  0.0820f },
	{ -0.6211f, -0.0527f,  0.0967f, -0.0059f },
	{  0.7163f,  0.0000f, -0.0718f,  0.0054f },
	{  0.4648f,  0.1777f,  0.0161f, -0.0161f },
	{  0.2134f, -0.0508f, -0.0278f, -0.0806f },
	{  0.2637f,  0.1523f,  0.0820f, -0.1235f },
	{  0.4146f, -0.0254f, -0.1816f, -0.0806f },
	{  0.2637f,  0.1270f, -0.1377f, -0.1450f },
	{  0.1631f, -0.0508f, -0.1816f, -0.1880f },
	{  0.0122f,  0.1523f, -0.0498f, -0.2310f },
	{  0.3643f,  0.1270f, -0.1157f,  0.0913f },
	{  0.3140f,  0.3301f,  0.0381f,  0.0269f },
	{  0.1631f,  0.0762f, -0.0718f, -0.0161f },
	{  0.1128f,  0.2793f,  0.0601f, -0.0806f },
	{  0.2134f,  0.1270f, -0.2476f,  0.0269f },
	{  0.1631f,  0.3047f, -0.1597f, -0.0806f },
	{  0.0625f,  0.1016f, -0.2256f, -0.1021f },
	{ -0.0381f,  0.2539f, -0.1157f, -0.0806f },
	{  0.5156f, -0.2051f,  0.0176f, -0.0054f },
	{  0.3281f, -0.0371f,  0.1230f, -0.0698f },
	{  0.2813f, -0.2051f,  0.0352f, -0.1558f },
	{  0.1406f, -0.0161f,  0.1582f, -0.1558f },
	{  0.2344f, -0.2051f, -0.1055f, -0.0698f },
	{  0.0938f,  0.0049f, -0.0176f, -0.1343f },
	{  0.0938f, -0.1841f, -0.0703f, -0.1558f },
	{ -0.0469f, -0.0791f,  0.0703f, -0.2632f },
	{  0.2813f, -0.1001f,  0.0176f,  0.0591f },
	{  0.1406f,  0.1099f,  0.0352f,  0.0161f },
	{  0.0938f, -0.1001f,  0.0352f, -0.0698f },
	{  0.0000f,  0.0259f,  0.0527f, -0.1128f },
	{  0.0938f, -0.0371f, -0.0527f,  0.0161f },
	{  0.0000f,  0.0889f, -0.0352f, -0.0483f },
	{ -0.0938f, -0.0581f, -0.0527f, -0.1128f },
	{ -0.1875f,  0.0889f,  0.0176f, -0.1772f },
	{  0.3848f, -0.0942f, -0.1001f,  0.1318f },
	{  0.3037f,  0.0938f,  0.0259f,  0.0220f },
	{  0.1011f, -0.1479f, -0.1001f,  0.0000f },
	{  0.0605f,  0.0132f,  0.0259f,  0.0000f },
	{  0.2227f, -0.1479f, -0.2891f,  0.0000f },
	{  0.0605f, -0.0137f, -0.1211f, -0.0659f },
	{  0.0200f, -0.1211f, -0.2051f, -0.0659f },
	{ -0.1016f, -0.0137f, -0.1631f, -0.1978f },
	{  0.1416f,  0.0132f, -0.1421f,  0.1099f },
	{  0.1416f,  0.2012f, -0.0791f,  0.0879f },
	{ -0.0205f, -0.0137f, -0.1001f,  0.0220f },
	{ -0.0610f,  0.2012f, -0.0161f,  0.0220f },
	{  0.0200f, -0.0405f, -0.2681f,  0.0659f },
	{ -0.0205f,  0.1475f, -0.1841f,  0.0879f },
	{ -0.1421f,  0.0400f, -0.2051f, -0.0220f },
	{ -0.2231f,  0.2549f, -0.2051f, -0.0220f },
	{  0.3267f, -0.3203f, -0.0557f,  0.0479f },
	{  0.1255f, -0.0957f,  0.0557f,  0.0479f },
	{  0.0752f, -0.3428f,  0.0186f, -0.0718f },
	{ -0.0254f, -0.1182f,  0.0742f, -0.0957f },
	{  0.0752f, -0.3428f, -0.1484f,  0.0479f },
	{ -0.0254f, -0.1406f, -0.0742f, -0.0239f },
	{ -0.0757f, -0.2979f, -0.2041f, -0.0718f },
	{ -0.2266f, -0.1631f, -0.0186f, -0.1675f },
	{  0.1758f, -0.1855f, -0.0928f,  0.1914f },
	{ -0.0254f, -0.0059f,  0.0557f,  0.1196f },
	{ -0.0757f, -0.2529f, -0.0371f, -0.0239f },
	{ -0.1763f, -0.0059f,  0.0186f, -0.0479f },
	{ -0.0757f, -0.1631f, -0.1484f,  0.0718f },
	{ -0.1763f, -0.0283f, -0.0742f,  0.0239f },
	{ -0.2769f, -0.1855f, -0.1484f, -0.0479f },
	{ -0.4277f, -0.0508f, -0.0928f, -0.0957f },
};

/*! \brief HOC 2nd block Difference vector quantization table */
const float ir77_ambe_hoc1_dif_vq[8][4] = {
	{ -0.0122f,  0.1094f,  0.0498f, -0.0337f },
	{  0.1313f,  0.0957f, -0.0415f, -0.0122f },
	{  0.0356f, -0.0137f,  0.0581f,  0.0200f },
	{  0.1792f, -0.0684f,  0.0000f,  0.0093f },
	{ -0.1797f,  0.0547f,  0.0083f, -0.0068f },
	{ -0.0361f,  0.0273f, -0.0664f,  0.0308f },
	{ -0.1079f, -0.0957f,  0.0332f,  0.0146f },
	{  0.0117f, -0.0820f, -0.0581f, -0.0444f },
};

/*! \brief HOC 3rd block Sum vector quantization table */
const float ir77_ambe_hoc2_sum_vq[128][4] = {
	{ -0.0093f,  0.1221f,  0.0742f, -0.0044f },
	{  0.1021f,  0.3101f,  0.1025f, -0.0044f },
	{  0.1021f, -0.0283f,  0.0317f, -0.0210f },
	{  0.1763f,  0.0845f,  0.0601f,  0.0039f },
	{  0.1392f,  0.0093f,  0.1733f, -0.0542f },
	{  0.2505f,  0.1973f,  0.1592f, -0.0210f },
	{  0.2876f, -0.1787f,  0.1450f, -0.0542f },
	{  0.3989f,  0.0093f,  0.1309f, -0.0376f },
	{ -0.1577f,  0.1221f, -0.0107f,  0.0288f },
	{ -0.0835f,  0.3853f, -0.0107f,  0.0039f },
	{ -0.0464f, -0.0659f, -0.0391f,  0.0371f },
	{  0.0278f,  0.1221f, -0.0391f,  0.0620f },
	{ -0.0093f, -0.0283f,  0.0601f,  0.0288f },
	{  0.1021f,  0.1597f,  0.0317f,  0.0039f },
	{  0.1021f, -0.1787f, -0.0107f, -0.0127f },
	{  0.1763f, -0.0283f,  0.0034f,  0.0205f },
	{  0.1479f,  0.0952f, -0.0601f, -0.0195f },
	{  0.2271f,  0.3379f, -0.0771f,  0.0195f },
	{  0.2271f, -0.0435f, -0.0771f, -0.0039f },
	{  0.3457f,  0.1299f, -0.0942f, -0.0273f },
	{  0.2666f,  0.0259f,  0.0083f,  0.0039f },
	{  0.4248f,  0.2686f,  0.0596f,  0.0273f },
	{  0.4644f, -0.1821f, -0.0088f, -0.0039f },
	{  0.5830f,  0.0605f, -0.0430f, -0.0195f },
	{ -0.0103f,  0.0952f, -0.1455f,  0.0430f },
	{  0.0688f,  0.2339f, -0.1284f,  0.0508f },
	{  0.0688f, -0.0781f, -0.1968f,  0.0742f },
	{  0.1875f,  0.0605f, -0.1968f,  0.0898f },
	{  0.1084f, -0.0088f, -0.0942f,  0.0586f },
	{  0.2271f,  0.1646f, -0.0601f,  0.0742f },
	{  0.2271f, -0.1821f, -0.1797f,  0.0352f },
	{  0.3853f, -0.0088f, -0.1626f,  0.0820f },
	{ -0.1245f,  0.0571f,  0.0742f, -0.1616f },
	{ -0.0581f,  0.2534f,  0.0742f, -0.1206f },
	{ -0.0581f, -0.0737f, -0.0107f, -0.1821f },
	{  0.0083f,  0.0571f,  0.0034f, -0.1001f },
	{  0.0083f, -0.0410f,  0.1167f, -0.1309f },
	{  0.0747f,  0.1226f,  0.1025f, -0.1411f },
	{  0.1079f, -0.1719f,  0.0742f, -0.1616f },
	{  0.2075f, -0.0083f,  0.0459f, -0.1309f },
	{ -0.2905f,  0.0571f,  0.0176f, -0.1001f },
	{ -0.2241f,  0.2207f, -0.0391f, -0.0796f },
	{ -0.2241f, -0.1064f, -0.0957f, -0.1104f },
	{ -0.0913f,  0.0898f, -0.0815f, -0.0386f },
	{ -0.1245f, -0.0410f,  0.0459f, -0.0591f },
	{ -0.0581f,  0.1226f,  0.0034f, -0.0693f },
	{ -0.0581f, -0.2373f, -0.0532f, -0.1104f },
	{  0.0083f, -0.0410f, -0.0107f, -0.0283f },
	{ -0.0435f,  0.0869f, -0.0903f, -0.1709f },
	{  0.0259f,  0.2456f, -0.0723f, -0.1343f },
	{  0.0952f, -0.0718f, -0.1084f, -0.1709f },
	{  0.1646f,  0.0869f, -0.1445f, -0.1221f },
	{  0.1299f, -0.0083f, -0.0361f, -0.0732f },
	{  0.2339f,  0.1504f,  0.0181f, -0.1099f },
	{  0.2339f, -0.1987f, -0.0542f, -0.1343f },
	{  0.3379f, -0.0083f, -0.1265f, -0.1343f },
	{ -0.1821f,  0.0552f, -0.1626f, -0.0732f },
	{ -0.1128f,  0.2139f, -0.2168f,  0.0000f },
	{ -0.0781f, -0.1035f, -0.2168f, -0.0854f },
	{  0.0259f,  0.0552f, -0.2529f, -0.0610f },
	{ -0.0088f, -0.0083f, -0.1084f, -0.0366f },
	{  0.0605f,  0.0869f, -0.1084f, -0.0244f },
	{  0.0952f, -0.2305f, -0.1987f, -0.0854f },
	{  0.1299f, -0.0718f, -0.1445f, -0.0366f },
	{ -0.1445f,  0.0757f,  0.1685f,  0.0718f },
	{ -0.0361f,  0.2441f,  0.1538f,  0.0718f },
	{ -0.0361f, -0.1265f,  0.0806f,  0.0308f },
	{  0.0361f,  0.0757f,  0.0952f,  0.0615f },
	{  0.0000f, -0.0254f,  0.1978f,  0.0410f },
	{  0.1084f,  0.1431f,  0.2271f,  0.0718f },
	{  0.1084f, -0.1938f,  0.1392f,  0.0103f },
	{  0.2168f, -0.0254f,  0.1538f,  0.0718f },
	{ -0.3252f,  0.0083f,  0.1099f,  0.1230f },
	{ -0.2168f,  0.2104f,  0.0513f,  0.1128f },
	{ -0.2529f, -0.1938f,  0.0073f,  0.1538f },
	{ -0.1084f,  0.0083f,  0.0220f,  0.1128f },
	{ -0.1084f, -0.0928f,  0.1245f,  0.1333f },
	{ -0.0361f,  0.0757f,  0.0952f,  0.1641f },
	{ -0.0361f, -0.2612f,  0.0513f,  0.1128f },
	{  0.0723f, -0.1265f,  0.0806f,  0.1436f },
	{  0.0171f,  0.0078f, -0.0039f,  0.0767f },
	{  0.0864f,  0.2300f,  0.0107f,  0.1484f },
	{  0.0864f, -0.0874f, -0.0186f,  0.0459f },
	{  0.1558f,  0.0713f, -0.0332f,  0.0767f },
	{  0.1211f, -0.0239f,  0.0547f,  0.0869f },
	{  0.2251f,  0.1348f,  0.0986f,  0.1382f },
	{  0.2251f, -0.2144f, -0.0039f,  0.0664f },
	{  0.3638f, -0.0239f,  0.0400f,  0.1177f },
	{ -0.1563f,  0.0396f, -0.0918f,  0.1587f },
	{ -0.0869f,  0.1665f, -0.1064f,  0.1382f },
	{ -0.0869f, -0.1509f, -0.1211f,  0.1279f },
	{  0.0171f,  0.0396f, -0.1211f,  0.1587f },
	{  0.0171f, -0.0874f, -0.0186f,  0.1484f },
	{  0.0864f,  0.0713f, -0.0039f,  0.1997f },
	{  0.0864f, -0.2461f, -0.1064f,  0.1484f },
	{  0.1904f, -0.0557f, -0.0625f,  0.1792f },
	{ -0.3115f,  0.0615f,  0.1914f, -0.0586f },
	{ -0.2256f,  0.2256f,  0.1445f, -0.0352f },
	{ -0.2256f, -0.1436f,  0.0508f, -0.0820f },
	{ -0.0537f,  0.0205f,  0.0508f, -0.0273f },
	{ -0.1396f, -0.1025f,  0.2070f, -0.0664f },
	{ -0.0537f,  0.0615f,  0.1758f, -0.0586f },
	{ -0.0537f, -0.2256f,  0.1602f, -0.0508f },
	{  0.0752f, -0.0615f,  0.0977f, -0.0273f },
	{ -0.5693f, -0.0205f,  0.1133f,  0.0273f },
	{ -0.4404f,  0.1846f,  0.0195f, -0.0117f },
	{ -0.4404f, -0.2666f, -0.0273f,  0.0117f },
	{ -0.2686f, -0.0205f, -0.0117f,  0.0352f },
	{ -0.3115f, -0.1436f,  0.1445f,  0.0352f },
	{ -0.1826f,  0.0205f,  0.0820f, -0.0039f },
	{ -0.2256f, -0.3896f,  0.0352f,  0.0117f },
	{ -0.1396f, -0.1436f,  0.0352f,  0.0195f },
	{ -0.1353f,  0.0176f, -0.0405f, -0.0552f },
	{ -0.0522f,  0.1958f, -0.0244f,  0.0063f },
	{ -0.0522f, -0.1250f, -0.0728f, -0.0464f },
	{  0.0308f,  0.0176f, -0.0405f, -0.0024f },
	{  0.0308f, -0.1250f,  0.0239f, -0.0640f },
	{  0.0723f,  0.0532f,  0.0562f, -0.0288f },
	{  0.1138f, -0.3389f, -0.0083f, -0.0376f },
	{  0.2383f, -0.0894f,  0.0400f, -0.0112f },
	{ -0.3843f, -0.0181f, -0.1050f,  0.0063f },
	{ -0.3013f,  0.1958f, -0.1372f,  0.0679f },
	{ -0.2598f, -0.2319f, -0.1855f, -0.0200f },
	{ -0.1353f, -0.0181f, -0.1694f,  0.0415f },
	{ -0.1768f, -0.1250f, -0.0566f,  0.0327f },
	{ -0.0522f,  0.0532f, -0.0244f,  0.0503f },
	{ -0.0522f, -0.3389f, -0.1372f,  0.0239f },
	{  0.0308f, -0.1250f, -0.0728f,  0.0327f },
};

/*! \brief HOC 3rd block Difference vector quantization table */
const float ir77_ambe_hoc2_dif_vq[8][4] = {
	{  0.0112f,  0.0884f,  0.0020f,  0.0527f },
	{ -0.1265f,  0.0884f,  0.0283f, -0.0020f },
	{  0.1719f,  0.0762f, -0.0156f, -0.0156f },
	{  0.0112f,  0.0029f,  0.0547f, -0.0430f },
	{ -0.0117f, -0.0093f, -0.0771f, -0.0156f },
	{ -0.1724f, -0.0581f, -0.0068f, -0.0156f },
	{  0.1260f, -0.0947f, -0.0156f,  0.0049f },
	{ -0.0347f, -0.0947f,  0.0459f,  0.0391f },
};

/*! \brief HOC 4th block Sum vector quantization table */
const float ir77_ambe_hoc3_sum_vq[128][4] = {
	{ -0.0532f,  0.0049f, -0.1143f, -0.0688f },
	{  0.0317f,  0.1650f, -0.1396f, -0.1147f },
	{  0.0742f, -0.0151f, -0.1016f,  0.0688f },
	{  0.2441f,  0.1450f, -0.1904f, -0.0688f },
	{ -0.2231f,  0.0649f, -0.2412f,  0.0918f },
	{ -0.0957f,  0.2051f, -0.2412f, -0.0229f },
	{ -0.0107f,  0.0049f, -0.1904f,  0.1836f },
	{  0.0317f,  0.1851f, -0.2285f,  0.0918f },
	{ -0.2231f, -0.0552f, -0.0762f, -0.1606f },
	{ -0.1382f,  0.0850f, -0.0889f, -0.1377f },
	{ -0.0532f, -0.0952f, -0.0508f,  0.0229f },
	{  0.0317f,  0.0850f, -0.0762f, -0.0229f },
	{ -0.3931f, -0.0552f, -0.1777f,  0.0229f },
	{ -0.3081f,  0.1050f, -0.1523f, -0.0688f },
	{ -0.2231f, -0.0952f, -0.1523f,  0.1377f },
	{ -0.0957f,  0.1050f, -0.1523f,  0.0229f },
	{  0.0698f, -0.1812f, -0.1880f, -0.0859f },
	{  0.1499f, -0.0259f, -0.1470f, -0.1289f },
	{  0.2300f, -0.2588f, -0.1196f,  0.0645f },
	{  0.3501f, -0.0518f, -0.1743f, -0.0215f },
	{ -0.0903f, -0.1553f, -0.2427f, -0.0215f },
	{ -0.0103f,  0.0259f, -0.2427f, -0.0430f },
	{  0.0698f, -0.1553f, -0.1743f,  0.1289f },
	{  0.1499f, -0.0259f, -0.2427f,  0.0430f },
	{ -0.0903f, -0.2329f, -0.0513f, -0.1289f },
	{ -0.0503f, -0.0518f, -0.0923f, -0.1934f },
	{ -0.0103f, -0.3623f, -0.0513f,  0.0000f },
	{  0.0698f, -0.1553f, -0.0376f, -0.0645f },
	{ -0.2505f, -0.2329f, -0.1606f, -0.0430f },
	{ -0.1704f,  0.0000f, -0.1606f, -0.0430f },
	{ -0.0903f, -0.2070f, -0.0923f,  0.0859f },
	{ -0.0503f, -0.0518f, -0.1333f,  0.0430f },
	{  0.1011f,  0.0479f, -0.0063f, -0.0718f },
	{  0.1416f,  0.2632f, -0.0190f, -0.1333f },
	{  0.2227f,  0.0239f,  0.0063f,  0.0513f },
	{  0.3848f,  0.2632f,  0.0317f,  0.0103f },
	{ -0.0610f,  0.1196f, -0.0571f,  0.0513f },
	{  0.0200f,  0.3589f, -0.1206f, -0.0103f },
	{  0.1011f,  0.0957f, -0.0825f,  0.1743f },
	{  0.1821f,  0.2632f, -0.0825f,  0.0923f },
	{ -0.1016f,  0.0000f,  0.0698f, -0.0923f },
	{ -0.0205f,  0.1675f,  0.0317f, -0.1333f },
	{  0.0605f,  0.0000f,  0.0571f,  0.0103f },
	{  0.1821f,  0.1436f,  0.0190f, -0.0103f },
	{ -0.2231f,  0.0479f, -0.0063f, -0.0103f },
	{ -0.1826f,  0.2393f, -0.0444f, -0.0923f },
	{ -0.1016f,  0.0718f,  0.0063f,  0.1128f },
	{  0.0200f,  0.1914f, -0.0698f,  0.0308f },
	{  0.1885f, -0.1538f, -0.0352f, -0.1313f },
	{  0.3218f,  0.0718f, -0.0195f, -0.1313f },
	{  0.3662f, -0.1743f,  0.0742f, -0.0054f },
	{  0.5439f,  0.0103f, -0.0039f, -0.0054f },
	{  0.0552f, -0.0513f, -0.0508f, -0.0054f },
	{  0.0996f,  0.0923f, -0.1133f, -0.0054f },
	{  0.2329f, -0.0923f, -0.0664f,  0.1206f },
	{  0.2773f,  0.0923f, -0.1133f,  0.0996f },
	{  0.0107f, -0.1743f,  0.1211f, -0.1313f },
	{  0.0996f, -0.0103f,  0.0430f, -0.1943f },
	{  0.1440f, -0.2153f,  0.1211f, -0.0474f },
	{  0.2329f, -0.0513f,  0.0586f, -0.0474f },
	{ -0.1226f, -0.1128f,  0.0273f,  0.0156f },
	{ -0.0781f,  0.0513f, -0.0195f, -0.0474f },
	{  0.0107f, -0.0718f,  0.0273f,  0.0786f },
	{  0.0552f,  0.0513f, -0.0039f,  0.0576f },
	{ -0.1504f,  0.0000f, -0.0293f, -0.0054f },
	{ -0.0645f,  0.2021f, -0.0293f, -0.0264f },
	{ -0.0215f,  0.0000f, -0.0146f,  0.0366f },
	{  0.0645f,  0.1348f,  0.0146f,  0.0366f },
	{ -0.3223f,  0.0898f, -0.1172f,  0.1206f },
	{ -0.2363f,  0.2920f, -0.1611f,  0.0576f },
	{ -0.1504f,  0.0449f, -0.0439f,  0.2256f },
	{ -0.1074f,  0.2021f, -0.1025f,  0.1416f },
	{ -0.3652f, -0.0449f,  0.0146f, -0.0684f },
	{ -0.3223f,  0.1348f,  0.0146f, -0.0894f },
	{ -0.2363f, -0.0449f,  0.0586f,  0.0576f },
	{ -0.1074f,  0.0898f,  0.0439f, -0.0264f },
	{ -0.5801f, -0.0225f,  0.0000f,  0.0576f },
	{ -0.4941f,  0.2246f, -0.0879f,  0.0156f },
	{ -0.3652f,  0.0000f,  0.0146f,  0.1626f },
	{ -0.2793f,  0.1797f, -0.0293f,  0.0996f },
	{ -0.1074f, -0.1187f, -0.0552f, -0.0703f },
	{  0.0098f,  0.0059f, -0.0186f, -0.0879f },
	{  0.0879f, -0.1934f,  0.0059f,  0.0703f },
	{  0.1660f, -0.0439f, -0.0430f,  0.0000f },
	{ -0.2246f, -0.0938f, -0.0674f,  0.0352f },
	{ -0.1855f,  0.1304f, -0.0918f,  0.0352f },
	{ -0.0684f, -0.0938f, -0.0186f,  0.1758f },
	{  0.0098f,  0.0557f, -0.0796f,  0.1055f },
	{ -0.2637f, -0.2432f,  0.0547f, -0.0352f },
	{ -0.1855f, -0.0190f,  0.0303f, -0.0879f },
	{ -0.0684f, -0.2432f,  0.0913f,  0.0527f },
	{ -0.0293f, -0.0688f,  0.0303f, -0.0176f },
	{ -0.4199f, -0.2432f, -0.0063f,  0.0703f },
	{ -0.3418f,  0.0308f, -0.0430f,  0.0176f },
	{ -0.2246f, -0.1934f,  0.0181f,  0.1758f },
	{ -0.1465f,  0.0059f, -0.0674f,  0.0879f },
	{ -0.0210f,  0.0483f,  0.0996f, -0.0254f },
	{  0.0210f,  0.3276f,  0.0830f, -0.0454f },
	{  0.0630f,  0.0913f,  0.1328f,  0.0947f },
	{  0.1890f,  0.2417f,  0.1494f,  0.0747f },
	{ -0.1890f,  0.1558f,  0.0332f,  0.0547f },
	{ -0.1470f,  0.3491f, -0.0166f,  0.0747f },
	{ -0.0210f,  0.1128f,  0.0664f,  0.2148f },
	{  0.0630f,  0.2632f,  0.0166f,  0.1348f },
	{ -0.2729f,  0.0269f,  0.1992f, -0.0654f },
	{ -0.1470f,  0.1987f,  0.1328f, -0.0854f },
	{ -0.1050f,  0.0483f,  0.2324f,  0.0547f },
	{  0.0210f,  0.1772f,  0.1826f,  0.0146f },
	{ -0.4409f,  0.0913f,  0.1494f, -0.0054f },
	{ -0.3569f,  0.3062f,  0.0996f,  0.0146f },
	{ -0.2729f,  0.1128f,  0.1328f,  0.1348f },
	{ -0.1050f,  0.2202f,  0.0996f,  0.0947f },
	{  0.0767f, -0.0728f,  0.0762f, -0.0498f },
	{  0.1206f,  0.1206f,  0.1230f, -0.0996f },
	{  0.2085f, -0.1050f,  0.1113f,  0.0996f },
	{  0.3403f,  0.0400f,  0.1230f,  0.0498f },
	{ -0.0991f, -0.0083f,  0.0645f,  0.0498f },
	{ -0.0552f,  0.1206f,  0.0293f,  0.0249f },
	{  0.0767f, -0.0405f,  0.0410f,  0.1992f },
	{  0.1646f,  0.1045f,  0.0410f,  0.1245f },
	{ -0.1431f, -0.1211f,  0.1699f, -0.0498f },
	{ -0.0991f,  0.0400f,  0.1582f, -0.1743f },
	{  0.0327f, -0.1050f,  0.2051f,  0.0498f },
	{  0.1646f, -0.0083f,  0.1934f, -0.0249f },
	{ -0.3188f, -0.1050f,  0.1699f,  0.0747f },
	{ -0.1870f,  0.0562f,  0.0996f,  0.0000f },
	{ -0.0991f, -0.0566f,  0.1465f,  0.1743f },
	{ -0.0112f,  0.0562f,  0.0762f,  0.0996f },
};

/*! \brief HOC 4th block Difference vector quantization table */
const float ir77_ambe_hoc3_dif_vq[8][4] = {
	{  0.1914f, -0.0171f,  0.0103f,  0.0044f },
	{  0.0479f, -0.0718f, -0.0645f,  0.0220f },
	{  0.0239f, -0.0444f,  0.0601f, -0.0366f },
	{ -0.0957f, -0.0991f, -0.0063f,  0.0044f },
	{  0.0718f,  0.1060f, -0.0063f, -0.0073f },
	{ -0.0479f,  0.0376f, -0.0645f, -0.0190f },
	{ -0.0239f,  0.0513f,  0.0601f,  0.0513f },
	{ -0.1675f,  0.0376f,  0.0103f, -0.0015f },
};

/*! @} */
