/*
 * Copyright 2013-2020 Software Radio Systems Limited
 *
 * This file is part of srsLTE.
 *
 * srsLTE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsLTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

/*******************************************************************************
 *
 *                      3GPP TS ASN1 RRC v15.4.0 (2018-12)
 *
 ******************************************************************************/

#ifndef SRSASN1_RRC_H
#define SRSASN1_RRC_H

#include "asn1_utils.h"
#include <cstdio>
#include <stdarg.h>

namespace asn1 {
namespace rrc {

/*******************************************************************************
 *                        Functions for external logging
 ******************************************************************************/

void log_invalid_access_choice_id(uint32_t val, uint32_t choice_id);

void assert_choice_type(uint32_t val, uint32_t choice_id);

void assert_choice_type(const std::string& access_type,
                        const std::string& current_type,
                        const std::string& choice_type);

const char* convert_enum_idx(const char* array[], uint32_t nof_types, uint32_t enum_val, const char* enum_type);

template <class ItemType>
ItemType map_enum_number(ItemType* array, uint32_t nof_types, uint32_t enum_val, const char* enum_type);

/*******************************************************************************
 *                             Constant Definitions
 ******************************************************************************/

#define ASN1_RRC_MAX_ACCESS_CAT_MINUS1_R15 63
#define ASN1_RRC_MAX_ACDC_CAT_R13 16
#define ASN1_RRC_MAX_AVAIL_NARROW_BANDS_R13 16
#define ASN1_RRC_MAX_BAND_COMB_R10 128
#define ASN1_RRC_MAX_BAND_COMB_R11 256
#define ASN1_RRC_MAX_BAND_COMB_R13 384
#define ASN1_RRC_MAX_BANDS 64
#define ASN1_RRC_MAX_BANDS_NR_R15 1024
#define ASN1_RRC_MAX_BW_CLASS_R10 16
#define ASN1_RRC_MAX_BW_COMB_SET_R10 32
#define ASN1_RRC_MAX_BARR_INFO_SET_R15 8
#define ASN1_RRC_MAX_BT_ID_REPORT_R15 32
#define ASN1_RRC_MAX_BT_NAME_R15 4
#define ASN1_RRC_MAX_CBR_LEVEL_R14 16
#define ASN1_RRC_MAX_CBR_LEVEL_MINUS1_R14 15
#define ASN1_RRC_MAX_CBR_REPORT_R14 72
#define ASN1_RRC_MAX_CDMA_BAND_CLASS 32
#define ASN1_RRC_MAX_CE_LEVEL_R13 4
#define ASN1_RRC_MAX_CELL_BLACK 16
#define ASN1_RRC_MAX_CELL_HISTORY_R12 16
#define ASN1_RRC_MAX_CELL_INFO_GERAN_R9 32
#define ASN1_RRC_MAX_CELL_INFO_UTRA_R9 16
#define ASN1_RRC_MAX_CELL_MEAS_IDLE_R15 8
#define ASN1_RRC_MAX_COMB_IDC_R11 128
#define ASN1_RRC_MAX_CSI_IM_R11 3
#define ASN1_RRC_MAX_CSI_IM_R12 4
#define ASN1_RRC_MIN_CSI_IM_R13 5
#define ASN1_RRC_MAX_CSI_IM_R13 24
#define ASN1_RRC_MAX_CSI_IM_V1310 20
#define ASN1_RRC_MAX_CSI_PROC_R11 4
#define ASN1_RRC_MAX_CSI_RS_NZP_R11 3
#define ASN1_RRC_MIN_CSI_RS_NZP_R13 4
#define ASN1_RRC_MAX_CSI_RS_NZP_R13 24
#define ASN1_RRC_MAX_CSI_RS_NZP_V1310 21
#define ASN1_RRC_MAX_CSI_RS_ZP_R11 4
#define ASN1_RRC_MAX_CQI_PROC_EXT_R11 3
#define ASN1_RRC_MAX_FREQ_UTRA_TDD_R10 6
#define ASN1_RRC_MAX_CELL_INTER 16
#define ASN1_RRC_MAX_CELL_INTRA 16
#define ASN1_RRC_MAX_CELL_LIST_GERAN 3
#define ASN1_RRC_MAX_CELL_MEAS 32
#define ASN1_RRC_MAX_CELL_REPORT 8
#define ASN1_RRC_MAX_CELL_SFTD 3
#define ASN1_RRC_MAX_CFG_SPS_R14 8
#define ASN1_RRC_MAX_CFG_SPS_R15 6
#define ASN1_RRC_MAX_CSI_RS_MEAS_R12 96
#define ASN1_RRC_MAX_DRB 11
#define ASN1_RRC_MAX_DRB_EXT_R15 4
#define ASN1_RRC_MAX_DRB_R15 15
#define ASN1_RRC_MAX_DS_DUR_R12 5
#define ASN1_RRC_MAX_DS_ZTP_CSI_RS_R12 5
#define ASN1_RRC_MAX_EARFCN 65535
#define ASN1_RRC_MAX_EARFCN_PLUS1 65536
#define ASN1_RRC_MAX_EARFCN2 262143
#define ASN1_RRC_MAX_EPDCCH_SET_R11 2
#define ASN1_RRC_MAX_FBI 64
#define ASN1_RRC_MAX_FBI_NR_R15 1024
#define ASN1_RRC_MAX_FBI_PLUS1 65
#define ASN1_RRC_MAX_FBI2 256
#define ASN1_RRC_MAX_FEATURE_SETS_R15 256
#define ASN1_RRC_MAX_PER_CC_FEATURE_SETS_R15 32
#define ASN1_RRC_MAX_FREQ 8
#define ASN1_RRC_MAX_FREQ_IDC_R11 32
#define ASN1_RRC_MAX_FREQ_IDLE_R15 8
#define ASN1_RRC_MAX_FREQ_MBMS_R11 5
#define ASN1_RRC_MAX_FREQ_NR_R15 5
#define ASN1_RRC_MAX_FREQ_V2X_R14 8
#define ASN1_RRC_MAX_FREQ_V2X_MINUS1_R14 7
#define ASN1_RRC_MAX_GERAN_SI 10
#define ASN1_RRC_MAX_GNFG 16
#define ASN1_RRC_MAX_IDLE_MEAS_CARRIERS_R15 3
#define ASN1_RRC_MAX_LCG_R13 4
#define ASN1_RRC_MAX_LOG_MEAS_REPORT_R10 520
#define ASN1_RRC_MAX_MBSFN_ALLOCS 8
#define ASN1_RRC_MAX_MBSFN_AREA 8
#define ASN1_RRC_MAX_MBSFN_AREA_MINUS1 7
#define ASN1_RRC_MAX_MBMS_SERVICE_LIST_PER_UE_R13 15
#define ASN1_RRC_MAX_MEAS_ID 32
#define ASN1_RRC_MAX_MEAS_ID_PLUS1 33
#define ASN1_RRC_MAX_MEAS_ID_R12 64
#define ASN1_RRC_MAX_MULTI_BANDS 8
#define ASN1_RRC_MAX_MULTI_BANDS_NR_R15 32
#define ASN1_RRC_MAX_NS_PMAX_R10 8
#define ASN1_RRC_MAX_NAICS_ENTRIES_R12 8
#define ASN1_RRC_MAX_NEIGH_CELL_R12 8
#define ASN1_RRC_MAX_NEIGH_CELL_SCPTM_R13 8
#define ASN1_RRC_MAX_NROF_S_NSSAI_R15 8
#define ASN1_RRC_MAX_OBJ_ID 32
#define ASN1_RRC_MAX_OBJ_ID_PLUS1_R13 33
#define ASN1_RRC_MAX_OBJ_ID_R13 64
#define ASN1_RRC_MAX_P_A_PER_NEIGH_CELL_R12 3
#define ASN1_RRC_MAX_PAGE_REC 16
#define ASN1_RRC_MAX_PCI_RANGE_R9 4
#define ASN1_RRC_MAX_PLMN_R11 6
#define ASN1_RRC_MAX_PLMN_MINUS1_R14 5
#define ASN1_RRC_MAX_PLMN_R15 8
#define ASN1_RRC_MAX_PLMN_NR_R15 12
#define ASN1_RRC_MAX_PN_OFFSET 511
#define ASN1_RRC_MAX_PMCH_PER_MBSFN 15
#define ASN1_RRC_MAX_PSSCH_TX_CFG_R14 16
#define ASN1_RRC_MAX_QUANT_SETS_NR_R15 2
#define ASN1_RRC_MAX_QCI_R13 6
#define ASN1_RRC_MAX_RAT_CAP 8
#define ASN1_RRC_MAX_RE_MAP_QCL_R11 4
#define ASN1_RRC_MAX_REPORT_CFG_ID 32
#define ASN1_RRC_MAX_RESERV_PERIOD_R14 16
#define ASN1_RRC_MAX_RS_IDX_R15 64
#define ASN1_RRC_MAX_RS_IDX_MINUS1_R15 63
#define ASN1_RRC_MAX_RS_IDX_CELL_QUAL_R15 16
#define ASN1_RRC_MAX_RS_IDX_REPORT_R15 32
#define ASN1_RRC_MAX_RSTD_FREQ_R10 3
#define ASN1_RRC_MAX_SAI_MBMS_R11 64
#define ASN1_RRC_MAX_SCELL_R10 4
#define ASN1_RRC_MAX_SCELL_R13 31
#define ASN1_RRC_MAX_SCELL_GROUPS_R15 4
#define ASN1_RRC_MAX_SC_MTCH_R13 1023
#define ASN1_RRC_MAX_SC_MTCH_BR_R14 128
#define ASN1_RRC_MAX_SL_COMM_RX_POOL_NFREQ_R13 32
#define ASN1_RRC_MAX_SL_COMM_RX_POOL_PRECONF_V1310 12
#define ASN1_RRC_MAX_SL_TX_POOL_R12_PLUS1_R13 5
#define ASN1_RRC_MAX_SL_TX_POOL_V1310 4
#define ASN1_RRC_MAX_SL_TX_POOL_R13 8
#define ASN1_RRC_MAX_SL_COMM_TX_POOL_PRECONF_V1310 7
#define ASN1_RRC_MAX_SL_DEST_R12 16
#define ASN1_RRC_MAX_SL_DISC_CELLS_R13 16
#define ASN1_RRC_MAX_SL_DISC_PWR_CLASS_R12 3
#define ASN1_RRC_MAX_SL_DISC_RX_POOL_PRECONF_R13 16
#define ASN1_RRC_MAX_SL_DISC_SYS_INFO_REPORT_FREQ_R13 8
#define ASN1_RRC_MAX_SL_DISC_TX_POOL_PRECONF_R13 4
#define ASN1_RRC_MAX_SL_GP_R13 8
#define ASN1_RRC_MAX_SL_POOL_TO_MEASURE_R14 72
#define ASN1_RRC_MAX_SL_PRIO_R13 8
#define ASN1_RRC_MAX_SL_RX_POOL_R12 16
#define ASN1_RRC_MAX_SL_RELIABILITY_R15 8
#define ASN1_RRC_MAX_SL_SYNC_CFG_R12 16
#define ASN1_RRC_MAX_SL_TF_IDX_PAIR_R12 64
#define ASN1_RRC_MAX_SL_TX_POOL_R12 4
#define ASN1_RRC_MAX_SL_V2X_RX_POOL_R14 16
#define ASN1_RRC_MAX_SL_V2X_RX_POOL_PRECONF_R14 16
#define ASN1_RRC_MAX_SL_V2X_TX_POOL_R14 8
#define ASN1_RRC_MAX_SL_V2X_TX_POOL_PRECONF_R14 8
#define ASN1_RRC_MAX_SL_V2X_SYNC_CFG_R14 16
#define ASN1_RRC_MAX_SL_V2X_CBR_CFG_R14 4
#define ASN1_RRC_MAX_SL_V2X_CBR_CFG_MINUS1_R14 3
#define ASN1_RRC_MAX_SL_V2X_TX_CFG_R14 64
#define ASN1_RRC_MAX_SL_V2X_TX_CFG_MINUS1_R14 63
#define ASN1_RRC_MAX_SL_V2X_CBR_CFG2_R14 8
#define ASN1_RRC_MAX_SL_V2X_CBR_CFG2_MINUS1_R14 7
#define ASN1_RRC_MAX_SL_V2X_TX_CFG2_R14 128
#define ASN1_RRC_MAX_SL_V2X_TX_CFG2_MINUS1_R14 127
#define ASN1_RRC_MAX_STAG_R11 3
#define ASN1_RRC_MAX_SERV_CELL_R10 5
#define ASN1_RRC_MAX_SERV_CELL_R13 32
#define ASN1_RRC_MAX_SERV_CELL_NR_R15 16
#define ASN1_RRC_MAX_SERVICE_COUNT 16
#define ASN1_RRC_MAX_SERVICE_COUNT_MINUS1 15
#define ASN1_RRC_MAX_SESSION_PER_PMCH 29
#define ASN1_RRC_MAX_SESSION_PER_PMCH_MINUS1 28
#define ASN1_RRC_MAX_SIB 32
#define ASN1_RRC_MAX_SIB_MINUS1 31
#define ASN1_RRC_MAX_SI_MSG 32
#define ASN1_RRC_MAX_SIMUL_BANDS_R10 64
#define ASN1_RRC_MAX_SF_PATTERN_IDC_R11 8
#define ASN1_RRC_MAX_TRAFFIC_PATTERN_R14 8
#define ASN1_RRC_MAX_UTRA_FDD_CARRIER 16
#define ASN1_RRC_MAX_UTRA_TDD_CARRIER 16
#define ASN1_RRC_MAX_WAY_POINT_R15 20
#define ASN1_RRC_MAX_WLAN_ID_R12 16
#define ASN1_RRC_MAX_WLAN_BANDS_R13 8
#define ASN1_RRC_MAX_WLAN_ID_R13 32
#define ASN1_RRC_MAX_WLAN_CHS_R13 16
#define ASN1_RRC_MAX_WLAN_CARRIER_INFO_R13 8
#define ASN1_RRC_MAX_WLAN_ID_REPORT_R14 32
#define ASN1_RRC_MAX_WLAN_NAME_R15 4
#define ASN1_RRC_MAX_LOG_MEAS_R10 4060
#define ASN1_RRC_MAX_REESTAB_INFO 32
#define ASN1_RRC_MAX_NPRACH_RES_NB_R13 3
#define ASN1_RRC_MAX_NON_ANCHOR_CARRIERS_NB_R14 15
#define ASN1_RRC_MAX_DRB_NB_R13 2
#define ASN1_RRC_MAX_NEIGH_CELL_SCPTM_NB_R14 8
#define ASN1_RRC_MAX_NS_PMAX_NB_R13 4
#define ASN1_RRC_MAX_SC_MTCH_NB_R14 64
#define ASN1_RRC_MAX_SI_MSG_NB_R13 8

/*******************************************************************************
 *                              Struct Definitions
 ******************************************************************************/

// PHICH-Config ::= SEQUENCE
struct phich_cfg_s {
  struct phich_dur_opts {
    enum options { normal, extended, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<phich_dur_opts> phich_dur_e_;
  struct phich_res_opts {
    enum options { one_sixth, half, one, two, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<phich_res_opts> phich_res_e_;

  // member variables
  phich_dur_e_ phich_dur;
  phich_res_e_ phich_res;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MasterInformationBlock ::= SEQUENCE
struct mib_s {
  struct dl_bw_opts {
    enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<dl_bw_opts> dl_bw_e_;

  // member variables
  dl_bw_e_           dl_bw;
  phich_cfg_s        phich_cfg;
  fixed_bitstring<8> sys_frame_num;
  uint8_t            sched_info_sib1_br_r13    = 0;
  bool               sys_info_unchanged_br_r15 = false;
  fixed_bitstring<4> spare;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-BCH-MessageType ::= MasterInformationBlock
typedef mib_s bcch_bch_msg_type_s;

// BCCH-BCH-Message ::= SEQUENCE
struct bcch_bch_msg_s {
  bcch_bch_msg_type_s msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MasterInformationBlock-MBMS-r14 ::= SEQUENCE
struct mib_mbms_r14_s {
  struct dl_bw_mbms_r14_opts {
    enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<dl_bw_mbms_r14_opts> dl_bw_mbms_r14_e_;

  // member variables
  dl_bw_mbms_r14_e_   dl_bw_mbms_r14;
  fixed_bitstring<6>  sys_frame_num_r14;
  uint8_t             add_non_mbsfn_sfs_r14 = 0;
  fixed_bitstring<13> spare;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-BCH-MessageType-MBMS-r14 ::= MasterInformationBlock-MBMS-r14
typedef mib_mbms_r14_s bcch_bch_msg_type_mbms_r14_s;

// BCCH-BCH-Message-MBMS ::= SEQUENCE
struct bcch_bch_msg_mbms_s {
  bcch_bch_msg_type_mbms_r14_s msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// GNSS-ID-r15 ::= SEQUENCE
struct gnss_id_r15_s {
  struct gnss_id_r15_opts {
    enum options { gps, sbas, qzss, galileo, glonass, bds, /*...*/ nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<gnss_id_r15_opts, true> gnss_id_r15_e_;

  // member variables
  bool           ext = false;
  gnss_id_r15_e_ gnss_id_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MCC ::= SEQUENCE (SIZE (3)) OF INTEGER (0..9)
using mcc_l = std::array<uint8_t, 3>;

// MNC ::= SEQUENCE (SIZE (2..3)) OF INTEGER (0..9)
using mnc_l = bounded_array<uint8_t, 3>;

// SBAS-ID-r15 ::= SEQUENCE
struct sbas_id_r15_s {
  struct sbas_id_r15_opts {
    enum options { waas, egnos, msas, gagan, /*...*/ nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<sbas_id_r15_opts, true> sbas_id_r15_e_;

  // member variables
  bool           ext = false;
  sbas_id_r15_e_ sbas_id_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-Identity ::= SEQUENCE
struct plmn_id_s {
  bool  mcc_present = false;
  mcc_l mcc;
  mnc_l mnc;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PosSIB-Type-r15 ::= SEQUENCE
struct pos_sib_type_r15_s {
  struct pos_sib_type_r15_opts {
    enum options {
      pos_sib_type1_minus1,
      pos_sib_type1_minus2,
      pos_sib_type1_minus3,
      pos_sib_type1_minus4,
      pos_sib_type1_minus5,
      pos_sib_type1_minus6,
      pos_sib_type1_minus7,
      pos_sib_type2_minus1,
      pos_sib_type2_minus2,
      pos_sib_type2_minus3,
      pos_sib_type2_minus4,
      pos_sib_type2_minus5,
      pos_sib_type2_minus6,
      pos_sib_type2_minus7,
      pos_sib_type2_minus8,
      pos_sib_type2_minus9,
      pos_sib_type2_minus10,
      pos_sib_type2_minus11,
      pos_sib_type2_minus12,
      pos_sib_type2_minus13,
      pos_sib_type2_minus14,
      pos_sib_type2_minus15,
      pos_sib_type2_minus16,
      pos_sib_type2_minus17,
      pos_sib_type2_minus18,
      pos_sib_type2_minus19,
      pos_sib_type3_minus1,
      // ...
      nulltype
    } value;

    std::string to_string() const;
  };
  typedef enumerated<pos_sib_type_r15_opts, true> pos_sib_type_r15_e_;

  // member variables
  bool                ext                   = false;
  bool                encrypted_r15_present = false;
  bool                gnss_id_r15_present   = false;
  bool                sbas_id_r15_present   = false;
  gnss_id_r15_s       gnss_id_r15;
  sbas_id_r15_s       sbas_id_r15;
  pos_sib_type_r15_e_ pos_sib_type_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityInfo-r15 ::= SEQUENCE
struct plmn_id_info_r15_s {
  struct plmn_id_minus5_gc_r15_c_ {
    struct types_opts {
      enum options { plmn_id_r15, plmn_idx_r15, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    plmn_id_minus5_gc_r15_c_() = default;
    plmn_id_minus5_gc_r15_c_(const plmn_id_minus5_gc_r15_c_& other);
    plmn_id_minus5_gc_r15_c_& operator=(const plmn_id_minus5_gc_r15_c_& other);
    ~plmn_id_minus5_gc_r15_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    plmn_id_s& plmn_id_r15()
    {
      assert_choice_type("plmn-Identity-r15", type_.to_string(), "plmn-Identity-5GC-r15");
      return c.get<plmn_id_s>();
    }
    uint8_t& plmn_idx_r15()
    {
      assert_choice_type("plmn-Index-r15", type_.to_string(), "plmn-Identity-5GC-r15");
      return c.get<uint8_t>();
    }
    const plmn_id_s& plmn_id_r15() const
    {
      assert_choice_type("plmn-Identity-r15", type_.to_string(), "plmn-Identity-5GC-r15");
      return c.get<plmn_id_s>();
    }
    const uint8_t& plmn_idx_r15() const
    {
      assert_choice_type("plmn-Index-r15", type_.to_string(), "plmn-Identity-5GC-r15");
      return c.get<uint8_t>();
    }
    plmn_id_s& set_plmn_id_r15()
    {
      set(types::plmn_id_r15);
      return c.get<plmn_id_s>();
    }
    uint8_t& set_plmn_idx_r15()
    {
      set(types::plmn_idx_r15);
      return c.get<uint8_t>();
    }

  private:
    types                      type_;
    choice_buffer_t<plmn_id_s> c;

    void destroy_();
  };
  struct cell_reserved_for_oper_r15_opts {
    enum options { reserved, not_reserved, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<cell_reserved_for_oper_r15_opts> cell_reserved_for_oper_r15_e_;
  struct cell_reserved_for_oper_crs_r15_opts {
    enum options { reserved, not_reserved, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<cell_reserved_for_oper_crs_r15_opts> cell_reserved_for_oper_crs_r15_e_;

  // member variables
  plmn_id_minus5_gc_r15_c_          plmn_id_minus5_gc_r15;
  cell_reserved_for_oper_r15_e_     cell_reserved_for_oper_r15;
  cell_reserved_for_oper_crs_r15_e_ cell_reserved_for_oper_crs_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PosSIB-MappingInfo-r15 ::= SEQUENCE (SIZE (1..32)) OF PosSIB-Type-r15
using pos_sib_map_info_r15_l = dyn_array<pos_sib_type_r15_s>;

// CellIdentity-5GC-r15 ::= CHOICE
struct cell_id_minus5_gc_r15_c {
  struct types_opts {
    enum options { cell_id_r15, cell_id_idx_r15, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  cell_id_minus5_gc_r15_c() = default;
  cell_id_minus5_gc_r15_c(const cell_id_minus5_gc_r15_c& other);
  cell_id_minus5_gc_r15_c& operator=(const cell_id_minus5_gc_r15_c& other);
  ~cell_id_minus5_gc_r15_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  fixed_bitstring<28>& cell_id_r15()
  {
    assert_choice_type("cellIdentity-r15", type_.to_string(), "CellIdentity-5GC-r15");
    return c.get<fixed_bitstring<28> >();
  }
  uint8_t& cell_id_idx_r15()
  {
    assert_choice_type("cellId-Index-r15", type_.to_string(), "CellIdentity-5GC-r15");
    return c.get<uint8_t>();
  }
  const fixed_bitstring<28>& cell_id_r15() const
  {
    assert_choice_type("cellIdentity-r15", type_.to_string(), "CellIdentity-5GC-r15");
    return c.get<fixed_bitstring<28> >();
  }
  const uint8_t& cell_id_idx_r15() const
  {
    assert_choice_type("cellId-Index-r15", type_.to_string(), "CellIdentity-5GC-r15");
    return c.get<uint8_t>();
  }
  fixed_bitstring<28>& set_cell_id_r15()
  {
    set(types::cell_id_r15);
    return c.get<fixed_bitstring<28> >();
  }
  uint8_t& set_cell_id_idx_r15()
  {
    set(types::cell_id_idx_r15);
    return c.get<uint8_t>();
  }

private:
  types                                 type_;
  choice_buffer_t<fixed_bitstring<28> > c;

  void destroy_();
};

// PLMN-IdentityInfo-v1530 ::= SEQUENCE
struct plmn_id_info_v1530_s {
  struct cell_reserved_for_oper_crs_r15_opts {
    enum options { reserved, not_reserved, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<cell_reserved_for_oper_crs_r15_opts> cell_reserved_for_oper_crs_r15_e_;

  // member variables
  cell_reserved_for_oper_crs_r15_e_ cell_reserved_for_oper_crs_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityList-r15 ::= SEQUENCE (SIZE (1..6)) OF PLMN-IdentityInfo-r15
using plmn_id_list_r15_l = dyn_array<plmn_id_info_r15_s>;

// PosSchedulingInfo-r15 ::= SEQUENCE
struct pos_sched_info_r15_s {
  struct pos_si_periodicity_r15_opts {
    enum options { rf8, rf16, rf32, rf64, rf128, rf256, rf512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<pos_si_periodicity_r15_opts> pos_si_periodicity_r15_e_;

  // member variables
  pos_si_periodicity_r15_e_ pos_si_periodicity_r15;
  pos_sib_map_info_r15_l    pos_sib_map_info_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellAccessRelatedInfo-5GC-r15 ::= SEQUENCE
struct cell_access_related_info_minus5_gc_r15_s {
  bool                    ran_area_code_r15_present = false;
  plmn_id_list_r15_l      plmn_id_list_r15;
  uint16_t                ran_area_code_r15 = 0;
  fixed_bitstring<24>     tac_minus5_gc_r15;
  cell_id_minus5_gc_r15_c cell_id_minus5_gc_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfoCE-v1530 ::= SEQUENCE
struct cell_sel_info_ce_v1530_s {
  struct pwr_class14dbm_offset_r15_opts {
    enum options { db_minus6, db_minus3, db3, db6, db9, db12, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<pwr_class14dbm_offset_r15_opts> pwr_class14dbm_offset_r15_e_;

  // member variables
  pwr_class14dbm_offset_r15_e_ pwr_class14dbm_offset_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MCS-PSSCH-Range-r15 ::= SEQUENCE
struct mcs_pssch_range_r15_s {
  uint8_t min_mcs_pssch_r15 = 0;
  uint8_t max_mcs_pssch_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityInfo ::= SEQUENCE
struct plmn_id_info_s {
  struct cell_reserved_for_oper_opts {
    enum options { reserved, not_reserved, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<cell_reserved_for_oper_opts> cell_reserved_for_oper_e_;

  // member variables
  plmn_id_s                 plmn_id;
  cell_reserved_for_oper_e_ cell_reserved_for_oper;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityList-v1530 ::= SEQUENCE (SIZE (1..6)) OF PLMN-IdentityInfo-v1530
using plmn_id_list_v1530_l = dyn_array<plmn_id_info_v1530_s>;

// PosSchedulingInfoList-r15 ::= SEQUENCE (SIZE (1..32)) OF PosSchedulingInfo-r15
using pos_sched_info_list_r15_l = dyn_array<pos_sched_info_r15_s>;

// SL-PriorityList-r13 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..8)
using sl_prio_list_r13_l = bounded_array<uint8_t, 8>;

// SL-TxPower-r14 ::= CHOICE
struct sl_tx_pwr_r14_c {
  struct types_opts {
    enum options { minusinfinity_r14, tx_pwr_r14, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sl_tx_pwr_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  int8_t& tx_pwr_r14()
  {
    assert_choice_type("txPower-r14", type_.to_string(), "SL-TxPower-r14");
    return c;
  }
  const int8_t& tx_pwr_r14() const
  {
    assert_choice_type("txPower-r14", type_.to_string(), "SL-TxPower-r14");
    return c;
  }
  int8_t& set_tx_pwr_r14()
  {
    set(types::tx_pwr_r14);
    return c;
  }

private:
  types  type_;
  int8_t c;
};

// Alpha-r12 ::= ENUMERATED
struct alpha_r12_opts {
  enum options { al0, al04, al05, al06, al07, al08, al09, al1, nulltype } value;
  typedef float number_type;

  std::string to_string() const;
  float       to_number() const;
  std::string to_number_string() const;
};
typedef enumerated<alpha_r12_opts> alpha_r12_e;

// PLMN-IdentityList ::= SEQUENCE (SIZE (1..6)) OF PLMN-IdentityInfo
using plmn_id_list_l = dyn_array<plmn_id_info_s>;

// SL-MinT2Value-r15 ::= SEQUENCE
struct sl_min_t2_value_r15_s {
  sl_prio_list_r13_l prio_list_r15;
  uint8_t            min_t2_value_r15 = 10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PPPP-TxConfigIndex-r14 ::= SEQUENCE
struct sl_pppp_tx_cfg_idx_r14_s {
  using tx_cfg_idx_list_r14_l_ = bounded_array<uint8_t, 16>;

  // member variables
  uint8_t                prio_thres_r14         = 1;
  uint8_t                default_tx_cfg_idx_r14 = 0;
  uint8_t                cbr_cfg_idx_r14        = 0;
  tx_cfg_idx_list_r14_l_ tx_cfg_idx_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PPPP-TxConfigIndex-v1530 ::= SEQUENCE
struct sl_pppp_tx_cfg_idx_v1530_s {
  using mcs_pssch_range_list_r15_l_ = dyn_array<mcs_pssch_range_r15_s>;

  // member variables
  bool                        mcs_pssch_range_list_r15_present = false;
  mcs_pssch_range_list_r15_l_ mcs_pssch_range_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PSSCH-TxParameters-r14 ::= SEQUENCE
struct sl_pssch_tx_params_r14_s {
  struct allowed_retx_num_pssch_r14_opts {
    enum options { n0, n1, both, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<allowed_retx_num_pssch_r14_opts> allowed_retx_num_pssch_r14_e_;

  // member variables
  bool                          max_tx_pwr_r14_present   = false;
  uint8_t                       min_mcs_pssch_r14        = 0;
  uint8_t                       max_mcs_pssch_r14        = 0;
  uint8_t                       min_sub_ch_num_pssch_r14 = 1;
  uint8_t                       max_subch_num_pssch_r14  = 1;
  allowed_retx_num_pssch_r14_e_ allowed_retx_num_pssch_r14;
  sl_tx_pwr_r14_c               max_tx_pwr_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PSSCH-TxParameters-v1530 ::= SEQUENCE
struct sl_pssch_tx_params_v1530_s {
  uint8_t min_mcs_pssch_r15 = 0;
  uint8_t max_mcs_pssch_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-RestrictResourceReservationPeriod-r14 ::= ENUMERATED
struct sl_restrict_res_reserv_period_r14_opts {
  enum options {
    v0dot2,
    v0dot5,
    v1,
    v2,
    v3,
    v4,
    v5,
    v6,
    v7,
    v8,
    v9,
    v10,
    spare4,
    spare3,
    spare2,
    spare1,
    nulltype
  } value;
  typedef float number_type;

  std::string to_string() const;
  float       to_number() const;
  std::string to_number_string() const;
};
typedef enumerated<sl_restrict_res_reserv_period_r14_opts> sl_restrict_res_reserv_period_r14_e;

// SL-TypeTxSync-r14 ::= ENUMERATED
struct sl_type_tx_sync_r14_opts {
  enum options { gnss, enb, ue, nulltype } value;

  std::string to_string() const;
};
typedef enumerated<sl_type_tx_sync_r14_opts> sl_type_tx_sync_r14_e;

// SystemInformationBlockType1-v1530-IEs ::= SEQUENCE
struct sib_type1_v1530_ies_s {
  struct crs_intf_mitig_cfg_r15_c_ {
    struct crs_intf_mitig_num_prbs_r15_opts {
      enum options { n6, n24, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<crs_intf_mitig_num_prbs_r15_opts> crs_intf_mitig_num_prbs_r15_e_;
    struct types_opts {
      enum options { crs_intf_mitig_enabled_minus15, crs_intf_mitig_num_prbs_r15, nulltype } value;
      typedef int8_t number_type;

      std::string to_string() const;
      int8_t      to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    crs_intf_mitig_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    crs_intf_mitig_num_prbs_r15_e_& crs_intf_mitig_num_prbs_r15()
    {
      assert_choice_type("crs-IntfMitigNumPRBs-r15", type_.to_string(), "crs-IntfMitigConfig-r15");
      return c;
    }
    const crs_intf_mitig_num_prbs_r15_e_& crs_intf_mitig_num_prbs_r15() const
    {
      assert_choice_type("crs-IntfMitigNumPRBs-r15", type_.to_string(), "crs-IntfMitigConfig-r15");
      return c;
    }
    crs_intf_mitig_num_prbs_r15_e_& set_crs_intf_mitig_num_prbs_r15()
    {
      set(types::crs_intf_mitig_num_prbs_r15);
      return c;
    }

  private:
    types                          type_;
    crs_intf_mitig_num_prbs_r15_e_ c;
  };
  struct cell_barred_crs_r15_opts {
    enum options { barred, not_barred, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<cell_barred_crs_r15_opts> cell_barred_crs_r15_e_;
  struct cell_access_related_info_minus5_gc_r15_s_ {
    struct cell_barred_minus5_gc_r15_opts {
      enum options { barred, not_barred, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<cell_barred_minus5_gc_r15_opts> cell_barred_minus5_gc_r15_e_;
    struct cell_barred_minus5_gc_crs_r15_opts {
      enum options { barred, not_barred, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<cell_barred_minus5_gc_crs_r15_opts> cell_barred_minus5_gc_crs_r15_e_;
    using cell_access_related_info_list_minus5_gc_r15_l_ = dyn_array<cell_access_related_info_minus5_gc_r15_s>;

    // member variables
    cell_barred_minus5_gc_r15_e_                   cell_barred_minus5_gc_r15;
    cell_barred_minus5_gc_crs_r15_e_               cell_barred_minus5_gc_crs_r15;
    cell_access_related_info_list_minus5_gc_r15_l_ cell_access_related_info_list_minus5_gc_r15;
  };

  // member variables
  bool                                      hsdn_cell_r15_present                          = false;
  bool                                      cell_sel_info_ce_v1530_present                 = false;
  bool                                      crs_intf_mitig_cfg_r15_present                 = false;
  bool                                      plmn_id_list_v1530_present                     = false;
  bool                                      pos_sched_info_list_r15_present                = false;
  bool                                      cell_access_related_info_minus5_gc_r15_present = false;
  bool                                      ims_emergency_support5_gc_r15_present          = false;
  bool                                      ecall_over_ims_support5_gc_r15_present         = false;
  bool                                      non_crit_ext_present                           = false;
  cell_sel_info_ce_v1530_s                  cell_sel_info_ce_v1530;
  crs_intf_mitig_cfg_r15_c_                 crs_intf_mitig_cfg_r15;
  cell_barred_crs_r15_e_                    cell_barred_crs_r15;
  plmn_id_list_v1530_l                      plmn_id_list_v1530;
  pos_sched_info_list_r15_l                 pos_sched_info_list_r15;
  cell_access_related_info_minus5_gc_r15_s_ cell_access_related_info_minus5_gc_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TDD-Config-v1450 ::= SEQUENCE
struct tdd_cfg_v1450_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellAccessRelatedInfo-r14 ::= SEQUENCE
struct cell_access_related_info_r14_s {
  plmn_id_list_l      plmn_id_list_r14;
  fixed_bitstring<16> tac_r14;
  fixed_bitstring<28> cell_id_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfoCE1-v1360 ::= SEQUENCE
struct cell_sel_info_ce1_v1360_s {
  int8_t delta_rx_lev_min_ce1_v1360 = -8;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NS-PmaxValue-v10l0 ::= SEQUENCE
struct ns_pmax_value_v10l0_s {
  bool     add_spec_emission_v10l0_present = false;
  uint16_t add_spec_emission_v10l0         = 33;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CBR-PPPP-TxConfigList-r14 ::= SEQUENCE (SIZE (1..8)) OF SL-PPPP-TxConfigIndex-r14
using sl_cbr_pppp_tx_cfg_list_r14_l = dyn_array<sl_pppp_tx_cfg_idx_r14_s>;

// SL-CBR-PPPP-TxConfigList-v1530 ::= SEQUENCE (SIZE (1..8)) OF SL-PPPP-TxConfigIndex-v1530
using sl_cbr_pppp_tx_cfg_list_v1530_l = dyn_array<sl_pppp_tx_cfg_idx_v1530_s>;

// SL-CP-Len-r12 ::= ENUMERATED
struct sl_cp_len_r12_opts {
  enum options { normal, extended, nulltype } value;

  std::string to_string() const;
};
typedef enumerated<sl_cp_len_r12_opts> sl_cp_len_r12_e;

// SL-MinT2ValueList-r15 ::= SEQUENCE (SIZE (1..8)) OF SL-MinT2Value-r15
using sl_min_t2_value_list_r15_l = dyn_array<sl_min_t2_value_r15_s>;

// SL-OffsetIndicator-r12 ::= CHOICE
struct sl_offset_ind_r12_c {
  struct types_opts {
    enum options { small_r12, large_r12, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sl_offset_ind_r12_c() = default;
  sl_offset_ind_r12_c(const sl_offset_ind_r12_c& other);
  sl_offset_ind_r12_c& operator=(const sl_offset_ind_r12_c& other);
  ~sl_offset_ind_r12_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  uint16_t& small_r12()
  {
    assert_choice_type("small-r12", type_.to_string(), "SL-OffsetIndicator-r12");
    return c.get<uint16_t>();
  }
  uint16_t& large_r12()
  {
    assert_choice_type("large-r12", type_.to_string(), "SL-OffsetIndicator-r12");
    return c.get<uint16_t>();
  }
  const uint16_t& small_r12() const
  {
    assert_choice_type("small-r12", type_.to_string(), "SL-OffsetIndicator-r12");
    return c.get<uint16_t>();
  }
  const uint16_t& large_r12() const
  {
    assert_choice_type("large-r12", type_.to_string(), "SL-OffsetIndicator-r12");
    return c.get<uint16_t>();
  }
  uint16_t& set_small_r12()
  {
    set(types::small_r12);
    return c.get<uint16_t>();
  }
  uint16_t& set_large_r12()
  {
    set(types::large_r12);
    return c.get<uint16_t>();
  }

private:
  types               type_;
  pod_choice_buffer_t c;

  void destroy_();
};

// SL-P2X-ResourceSelectionConfig-r14 ::= SEQUENCE
struct sl_p2_x_res_sel_cfg_r14_s {
  bool partial_sensing_r14_present = false;
  bool random_sel_r14_present      = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PSSCH-TxConfig-r14 ::= SEQUENCE
struct sl_pssch_tx_cfg_r14_s {
  struct thres_ue_speed_r14_opts {
    enum options { kmph60, kmph80, kmph100, kmph120, kmph140, kmph160, kmph180, kmph200, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<thres_ue_speed_r14_opts> thres_ue_speed_r14_e_;

  // member variables
  bool                     ext                      = false;
  bool                     type_tx_sync_r14_present = false;
  sl_type_tx_sync_r14_e    type_tx_sync_r14;
  thres_ue_speed_r14_e_    thres_ue_speed_r14;
  sl_pssch_tx_params_r14_s params_above_thres_r14;
  sl_pssch_tx_params_r14_s params_below_thres_r14;
  // ...
  // group 0
  copy_ptr<sl_pssch_tx_params_v1530_s> params_above_thres_v1530;
  copy_ptr<sl_pssch_tx_params_v1530_s> params_below_thres_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-RestrictResourceReservationPeriodList-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-RestrictResourceReservationPeriod-r14
using sl_restrict_res_reserv_period_list_r14_l = bounded_array<sl_restrict_res_reserv_period_r14_e, 16>;

// SL-SyncAllowed-r14 ::= SEQUENCE
struct sl_sync_allowed_r14_s {
  bool gnss_sync_r14_present = false;
  bool enb_sync_r14_present  = false;
  bool ue_sync_r14_present   = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TxParameters-r12 ::= SEQUENCE
struct sl_tx_params_r12_s {
  alpha_r12_e alpha_r12;
  int8_t      p0_r12 = -126;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SubframeBitmapSL-r12 ::= CHOICE
struct sf_bitmap_sl_r12_c {
  struct types_opts {
    enum options { bs4_r12, bs8_r12, bs12_r12, bs16_r12, bs30_r12, bs40_r12, bs42_r12, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sf_bitmap_sl_r12_c() = default;
  sf_bitmap_sl_r12_c(const sf_bitmap_sl_r12_c& other);
  sf_bitmap_sl_r12_c& operator=(const sf_bitmap_sl_r12_c& other);
  ~sf_bitmap_sl_r12_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  fixed_bitstring<4>& bs4_r12()
  {
    assert_choice_type("bs4-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<4> >();
  }
  fixed_bitstring<8>& bs8_r12()
  {
    assert_choice_type("bs8-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<8> >();
  }
  fixed_bitstring<12>& bs12_r12()
  {
    assert_choice_type("bs12-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<12> >();
  }
  fixed_bitstring<16>& bs16_r12()
  {
    assert_choice_type("bs16-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<16> >();
  }
  fixed_bitstring<30>& bs30_r12()
  {
    assert_choice_type("bs30-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<30> >();
  }
  fixed_bitstring<40>& bs40_r12()
  {
    assert_choice_type("bs40-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<40> >();
  }
  fixed_bitstring<42>& bs42_r12()
  {
    assert_choice_type("bs42-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<42> >();
  }
  const fixed_bitstring<4>& bs4_r12() const
  {
    assert_choice_type("bs4-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<4> >();
  }
  const fixed_bitstring<8>& bs8_r12() const
  {
    assert_choice_type("bs8-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<8> >();
  }
  const fixed_bitstring<12>& bs12_r12() const
  {
    assert_choice_type("bs12-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<12> >();
  }
  const fixed_bitstring<16>& bs16_r12() const
  {
    assert_choice_type("bs16-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<16> >();
  }
  const fixed_bitstring<30>& bs30_r12() const
  {
    assert_choice_type("bs30-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<30> >();
  }
  const fixed_bitstring<40>& bs40_r12() const
  {
    assert_choice_type("bs40-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<40> >();
  }
  const fixed_bitstring<42>& bs42_r12() const
  {
    assert_choice_type("bs42-r12", type_.to_string(), "SubframeBitmapSL-r12");
    return c.get<fixed_bitstring<42> >();
  }
  fixed_bitstring<4>& set_bs4_r12()
  {
    set(types::bs4_r12);
    return c.get<fixed_bitstring<4> >();
  }
  fixed_bitstring<8>& set_bs8_r12()
  {
    set(types::bs8_r12);
    return c.get<fixed_bitstring<8> >();
  }
  fixed_bitstring<12>& set_bs12_r12()
  {
    set(types::bs12_r12);
    return c.get<fixed_bitstring<12> >();
  }
  fixed_bitstring<16>& set_bs16_r12()
  {
    set(types::bs16_r12);
    return c.get<fixed_bitstring<16> >();
  }
  fixed_bitstring<30>& set_bs30_r12()
  {
    set(types::bs30_r12);
    return c.get<fixed_bitstring<30> >();
  }
  fixed_bitstring<40>& set_bs40_r12()
  {
    set(types::bs40_r12);
    return c.get<fixed_bitstring<40> >();
  }
  fixed_bitstring<42>& set_bs42_r12()
  {
    set(types::bs42_r12);
    return c.get<fixed_bitstring<42> >();
  }

private:
  types                                 type_;
  choice_buffer_t<fixed_bitstring<42> > c;

  void destroy_();
};

// SubframeBitmapSL-r14 ::= CHOICE
struct sf_bitmap_sl_r14_c {
  struct types_opts {
    enum options { bs10_r14, bs16_r14, bs20_r14, bs30_r14, bs40_r14, bs50_r14, bs60_r14, bs100_r14, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sf_bitmap_sl_r14_c() = default;
  sf_bitmap_sl_r14_c(const sf_bitmap_sl_r14_c& other);
  sf_bitmap_sl_r14_c& operator=(const sf_bitmap_sl_r14_c& other);
  ~sf_bitmap_sl_r14_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  fixed_bitstring<10>& bs10_r14()
  {
    assert_choice_type("bs10-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<10> >();
  }
  fixed_bitstring<16>& bs16_r14()
  {
    assert_choice_type("bs16-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<16> >();
  }
  fixed_bitstring<20>& bs20_r14()
  {
    assert_choice_type("bs20-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<20> >();
  }
  fixed_bitstring<30>& bs30_r14()
  {
    assert_choice_type("bs30-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<30> >();
  }
  fixed_bitstring<40>& bs40_r14()
  {
    assert_choice_type("bs40-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<40> >();
  }
  fixed_bitstring<50>& bs50_r14()
  {
    assert_choice_type("bs50-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<50> >();
  }
  fixed_bitstring<60>& bs60_r14()
  {
    assert_choice_type("bs60-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<60> >();
  }
  fixed_bitstring<100>& bs100_r14()
  {
    assert_choice_type("bs100-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<100> >();
  }
  const fixed_bitstring<10>& bs10_r14() const
  {
    assert_choice_type("bs10-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<10> >();
  }
  const fixed_bitstring<16>& bs16_r14() const
  {
    assert_choice_type("bs16-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<16> >();
  }
  const fixed_bitstring<20>& bs20_r14() const
  {
    assert_choice_type("bs20-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<20> >();
  }
  const fixed_bitstring<30>& bs30_r14() const
  {
    assert_choice_type("bs30-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<30> >();
  }
  const fixed_bitstring<40>& bs40_r14() const
  {
    assert_choice_type("bs40-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<40> >();
  }
  const fixed_bitstring<50>& bs50_r14() const
  {
    assert_choice_type("bs50-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<50> >();
  }
  const fixed_bitstring<60>& bs60_r14() const
  {
    assert_choice_type("bs60-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<60> >();
  }
  const fixed_bitstring<100>& bs100_r14() const
  {
    assert_choice_type("bs100-r14", type_.to_string(), "SubframeBitmapSL-r14");
    return c.get<fixed_bitstring<100> >();
  }
  fixed_bitstring<10>& set_bs10_r14()
  {
    set(types::bs10_r14);
    return c.get<fixed_bitstring<10> >();
  }
  fixed_bitstring<16>& set_bs16_r14()
  {
    set(types::bs16_r14);
    return c.get<fixed_bitstring<16> >();
  }
  fixed_bitstring<20>& set_bs20_r14()
  {
    set(types::bs20_r14);
    return c.get<fixed_bitstring<20> >();
  }
  fixed_bitstring<30>& set_bs30_r14()
  {
    set(types::bs30_r14);
    return c.get<fixed_bitstring<30> >();
  }
  fixed_bitstring<40>& set_bs40_r14()
  {
    set(types::bs40_r14);
    return c.get<fixed_bitstring<40> >();
  }
  fixed_bitstring<50>& set_bs50_r14()
  {
    set(types::bs50_r14);
    return c.get<fixed_bitstring<50> >();
  }
  fixed_bitstring<60>& set_bs60_r14()
  {
    set(types::bs60_r14);
    return c.get<fixed_bitstring<60> >();
  }
  fixed_bitstring<100>& set_bs100_r14()
  {
    set(types::bs100_r14);
    return c.get<fixed_bitstring<100> >();
  }

private:
  types                                  type_;
  choice_buffer_t<fixed_bitstring<100> > c;

  void destroy_();
};

// SystemInformationBlockType1-v1450-IEs ::= SEQUENCE
struct sib_type1_v1450_ies_s {
  bool                  tdd_cfg_v1450_present = false;
  bool                  non_crit_ext_present  = false;
  tdd_cfg_v1450_s       tdd_cfg_v1450;
  sib_type1_v1530_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TDD-Config ::= SEQUENCE
struct tdd_cfg_s {
  struct sf_assign_opts {
    enum options { sa0, sa1, sa2, sa3, sa4, sa5, sa6, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<sf_assign_opts> sf_assign_e_;
  struct special_sf_patterns_opts {
    enum options { ssp0, ssp1, ssp2, ssp3, ssp4, ssp5, ssp6, ssp7, ssp8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<special_sf_patterns_opts> special_sf_patterns_e_;

  // member variables
  sf_assign_e_           sf_assign;
  special_sf_patterns_e_ special_sf_patterns;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TDD-Config-v1430 ::= SEQUENCE
struct tdd_cfg_v1430_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandclassCDMA2000 ::= ENUMERATED
struct bandclass_cdma2000_opts {
  enum options {
    bc0,
    bc1,
    bc2,
    bc3,
    bc4,
    bc5,
    bc6,
    bc7,
    bc8,
    bc9,
    bc10,
    bc11,
    bc12,
    bc13,
    bc14,
    bc15,
    bc16,
    bc17,
    bc18_v9a0,
    bc19_v9a0,
    bc20_v9a0,
    bc21_v9a0,
    spare10,
    spare9,
    spare8,
    spare7,
    spare6,
    spare5,
    spare4,
    spare3,
    spare2,
    spare1,
    // ...
    nulltype
  } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<bandclass_cdma2000_opts, true> bandclass_cdma2000_e;

// InterFreqCarrierFreqInfo-v1360 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v1360_s {
  bool                      cell_sel_info_ce1_v1360_present = false;
  cell_sel_info_ce1_v1360_s cell_sel_info_ce1_v1360;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NS-PmaxList-v10l0 ::= SEQUENCE (SIZE (1..8)) OF NS-PmaxValue-v10l0
using ns_pmax_list_v10l0_l = dyn_array<ns_pmax_value_v10l0_s>;

// NS-PmaxValue-r10 ::= SEQUENCE
struct ns_pmax_value_r10_s {
  bool    add_pmax_r10_present = false;
  int8_t  add_pmax_r10         = -30;
  uint8_t add_spec_emission    = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysCellIdList-r13 ::= SEQUENCE (SIZE (1..16)) OF INTEGER (0..503)
using pci_list_r13_l = bounded_array<uint16_t, 16>;

// SL-CommResourcePoolV2X-r14 ::= SEQUENCE
struct sl_comm_res_pool_v2x_r14_s {
  struct size_subch_r14_opts {
    enum options {
      n4,
      n5,
      n6,
      n8,
      n9,
      n10,
      n12,
      n15,
      n16,
      n18,
      n20,
      n25,
      n30,
      n48,
      n50,
      n72,
      n75,
      n96,
      n100,
      spare13,
      spare12,
      spare11,
      spare10,
      spare9,
      spare8,
      spare7,
      spare6,
      spare5,
      spare4,
      spare3,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<size_subch_r14_opts> size_subch_r14_e_;
  struct num_subch_r14_opts {
    enum options { n1, n3, n5, n8, n10, n15, n20, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<num_subch_r14_opts> num_subch_r14_e_;
  struct rx_params_ncell_r14_s_ {
    bool      tdd_cfg_r14_present = false;
    tdd_cfg_s tdd_cfg_r14;
    uint8_t   sync_cfg_idx_r14 = 0;
  };

  // member variables
  bool                                     ext                                    = false;
  bool                                     sl_offset_ind_r14_present              = false;
  bool                                     start_rb_pscch_pool_r14_present        = false;
  bool                                     rx_params_ncell_r14_present            = false;
  bool                                     data_tx_params_r14_present             = false;
  bool                                     zone_id_r14_present                    = false;
  bool                                     thresh_s_rssi_cbr_r14_present          = false;
  bool                                     pool_report_id_r14_present             = false;
  bool                                     cbr_pssch_tx_cfg_list_r14_present      = false;
  bool                                     res_sel_cfg_p2_x_r14_present           = false;
  bool                                     sync_allowed_r14_present               = false;
  bool                                     restrict_res_reserv_period_r14_present = false;
  sl_offset_ind_r12_c                      sl_offset_ind_r14;
  sf_bitmap_sl_r14_c                       sl_sf_r14;
  bool                                     adjacency_pscch_pssch_r14 = false;
  size_subch_r14_e_                        size_subch_r14;
  num_subch_r14_e_                         num_subch_r14;
  uint8_t                                  start_rb_subch_r14      = 0;
  uint8_t                                  start_rb_pscch_pool_r14 = 0;
  rx_params_ncell_r14_s_                   rx_params_ncell_r14;
  sl_tx_params_r12_s                       data_tx_params_r14;
  uint8_t                                  zone_id_r14           = 0;
  uint8_t                                  thresh_s_rssi_cbr_r14 = 0;
  uint8_t                                  pool_report_id_r14    = 1;
  sl_cbr_pppp_tx_cfg_list_r14_l            cbr_pssch_tx_cfg_list_r14;
  sl_p2_x_res_sel_cfg_r14_s                res_sel_cfg_p2_x_r14;
  sl_sync_allowed_r14_s                    sync_allowed_r14;
  sl_restrict_res_reserv_period_list_r14_l restrict_res_reserv_period_r14;
  // ...
  // group 0
  copy_ptr<sl_min_t2_value_list_r15_l>      sl_min_t2_value_list_r15;
  copy_ptr<sl_cbr_pppp_tx_cfg_list_v1530_l> cbr_pssch_tx_cfg_list_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PSSCH-TxConfigList-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-PSSCH-TxConfig-r14
using sl_pssch_tx_cfg_list_r14_l = dyn_array<sl_pssch_tx_cfg_r14_s>;

// SL-PoolSelectionConfig-r12 ::= SEQUENCE
struct sl_pool_sel_cfg_r12_s {
  uint8_t thresh_low_r12  = 0;
  uint8_t thresh_high_r12 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-SyncConfigNFreq-r13 ::= SEQUENCE
struct sl_sync_cfg_nfreq_r13_s {
  struct async_params_r13_s_ {
    sl_cp_len_r12_e sync_cp_len_r13;
    uint8_t         sync_offset_ind_r13 = 0;
    uint8_t         slssid_r13          = 0;
  };
  struct tx_params_r13_s_ {
    bool                sync_info_reserved_r13_present = false;
    bool                sync_tx_periodic_r13_present   = false;
    sl_tx_params_r12_s  sync_tx_params_r13;
    uint8_t             sync_tx_thresh_ic_r13 = 0;
    fixed_bitstring<19> sync_info_reserved_r13;
  };
  struct rx_params_r13_s_ {
    struct disc_sync_win_r13_opts {
      enum options { w1, w2, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<disc_sync_win_r13_opts> disc_sync_win_r13_e_;

    // member variables
    disc_sync_win_r13_e_ disc_sync_win_r13;
  };

  // member variables
  bool                ext                      = false;
  bool                async_params_r13_present = false;
  bool                tx_params_r13_present    = false;
  bool                rx_params_r13_present    = false;
  async_params_r13_s_ async_params_r13;
  tx_params_r13_s_    tx_params_r13;
  rx_params_r13_s_    rx_params_r13;
  // ...
  // group 0
  bool    sync_offset_ind_v1430_present = false;
  bool    gnss_sync_r14_present         = false;
  uint8_t sync_offset_ind_v1430         = 40;
  // group 1
  bool    sync_offset_ind2_r14_present = false;
  bool    sync_offset_ind3_r14_present = false;
  uint8_t sync_offset_ind2_r14         = 0;
  uint8_t sync_offset_ind3_r14         = 0;
  // group 2
  bool slss_tx_disabled_r15_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TF-ResourceConfig-r12 ::= SEQUENCE
struct sl_tf_res_cfg_r12_s {
  uint8_t             prb_num_r12   = 1;
  uint8_t             prb_start_r12 = 0;
  uint8_t             prb_end_r12   = 0;
  sl_offset_ind_r12_c offset_ind_r12;
  sf_bitmap_sl_r12_c  sf_bitmap_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-ThresPSSCH-RSRP-List-r14 ::= SEQUENCE (SIZE (64)) OF INTEGER (0..66)
using sl_thres_pssch_rsrp_list_r14_l = std::array<uint8_t, 64>;

// SystemInformationBlockType1-v1430-IEs ::= SEQUENCE
struct sib_type1_v1430_ies_s {
  using cell_access_related_info_list_r14_l_ = dyn_array<cell_access_related_info_r14_s>;

  // member variables
  bool                                 ecall_over_ims_support_r14_present        = false;
  bool                                 tdd_cfg_v1430_present                     = false;
  bool                                 cell_access_related_info_list_r14_present = false;
  bool                                 non_crit_ext_present                      = false;
  tdd_cfg_v1430_s                      tdd_cfg_v1430;
  cell_access_related_info_list_r14_l_ cell_access_related_info_list_r14;
  sib_type1_v1450_ies_s                non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TDD-Config-v1130 ::= SEQUENCE
struct tdd_cfg_v1130_s {
  struct special_sf_patterns_v1130_opts {
    enum options { ssp7, ssp9, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<special_sf_patterns_v1130_opts> special_sf_patterns_v1130_e_;

  // member variables
  special_sf_patterns_v1130_e_ special_sf_patterns_v1130;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandClassInfoCDMA2000 ::= SEQUENCE
struct band_class_info_cdma2000_s {
  bool                 ext                     = false;
  bool                 cell_resel_prio_present = false;
  bandclass_cdma2000_e band_class;
  uint8_t              cell_resel_prio = 0;
  uint8_t              thresh_x_high   = 0;
  uint8_t              thresh_x_low    = 0;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfoCE1-r13 ::= SEQUENCE
struct cell_sel_info_ce1_r13_s {
  bool   q_qual_min_rsrq_ce1_r13_present = false;
  int8_t q_rx_lev_min_ce1_r13            = -70;
  int8_t q_qual_min_rsrq_ce1_r13         = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqList-v13a0 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1360
using inter_freq_carrier_freq_list_v13a0_l = dyn_array<inter_freq_carrier_freq_info_v1360_s>;

// MultiBandInfoList-v10l0 ::= SEQUENCE (SIZE (1..8)) OF NS-PmaxList-v10l0
using multi_band_info_list_v10l0_l = dyn_array<ns_pmax_list_v10l0_l>;

// NS-PmaxList-r10 ::= SEQUENCE (SIZE (1..8)) OF NS-PmaxValue-r10
using ns_pmax_list_r10_l = dyn_array<ns_pmax_value_r10_s>;

// NeighCellsPerBandclassCDMA2000-r11 ::= SEQUENCE
struct neigh_cells_per_bandclass_cdma2000_r11_s {
  using pci_list_r11_l_ = dyn_array<uint16_t>;

  // member variables
  uint16_t        arfcn = 0;
  pci_list_r11_l_ pci_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysCellIdListCDMA2000 ::= SEQUENCE (SIZE (1..16)) OF INTEGER (0..511)
using pci_list_cdma2000_l = bounded_array<uint16_t, 16>;

// PhysCellIdListCDMA2000-v920 ::= SEQUENCE (SIZE (0..24)) OF INTEGER (0..511)
using pci_list_cdma2000_v920_l = bounded_array<uint16_t, 24>;

// SL-CommRxPoolListV2X-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-CommResourcePoolV2X-r14
using sl_comm_rx_pool_list_v2x_r14_l = dyn_array<sl_comm_res_pool_v2x_r14_s>;

// SL-CommTxPoolListV2X-r14 ::= SEQUENCE (SIZE (1..8)) OF SL-CommResourcePoolV2X-r14
using sl_comm_tx_pool_list_v2x_r14_l = dyn_array<sl_comm_res_pool_v2x_r14_s>;

// SL-CommTxPoolSensingConfig-r14 ::= SEQUENCE
struct sl_comm_tx_pool_sensing_cfg_r14_s {
  struct prob_res_keep_r14_opts {
    enum options { v0, v0dot2, v0dot4, v0dot6, v0dot8, spare3, spare2, spare1, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<prob_res_keep_r14_opts> prob_res_keep_r14_e_;
  struct p2x_sensing_cfg_r14_s_ {
    uint8_t             min_num_candidate_sf_r14 = 1;
    fixed_bitstring<10> gap_candidate_sensing_r14;
  };
  struct sl_reselect_after_r14_opts {
    enum options {
      n1,
      n2,
      n3,
      n4,
      n5,
      n6,
      n7,
      n8,
      n9,
      spare7,
      spare6,
      spare5,
      spare4,
      spare3,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<sl_reselect_after_r14_opts> sl_reselect_after_r14_e_;

  // member variables
  bool                                     restrict_res_reserv_period_r14_present = false;
  bool                                     p2x_sensing_cfg_r14_present            = false;
  bool                                     sl_reselect_after_r14_present          = false;
  sl_pssch_tx_cfg_list_r14_l               pssch_tx_cfg_list_r14;
  sl_thres_pssch_rsrp_list_r14_l           thres_pssch_rsrp_list_r14;
  sl_restrict_res_reserv_period_list_r14_l restrict_res_reserv_period_r14;
  prob_res_keep_r14_e_                     prob_res_keep_r14;
  p2x_sensing_cfg_r14_s_                   p2x_sensing_cfg_r14;
  sl_reselect_after_r14_e_                 sl_reselect_after_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

struct setup_opts {
  enum options { release, setup, nulltype } value;

  std::string to_string() const;
};
typedef enumerated<setup_opts> setup_e;

// SL-DiscResourcePool-r12 ::= SEQUENCE
struct sl_disc_res_pool_r12_s {
  struct disc_period_r12_opts {
    enum options { rf32, rf64, rf128, rf256, rf512, rf1024, rf16_v1310, spare, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<disc_period_r12_opts> disc_period_r12_e_;
  struct tx_params_r12_s_ {
    struct ue_sel_res_cfg_r12_s_ {
      struct pool_sel_r12_c_ {
        struct types_opts {
          enum options { rsrp_based_r12, random_r12, nulltype } value;

          std::string to_string() const;
        };
        typedef enumerated<types_opts> types;

        // choice methods
        pool_sel_r12_c_() = default;
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        sl_pool_sel_cfg_r12_s& rsrp_based_r12()
        {
          assert_choice_type("rsrpBased-r12", type_.to_string(), "poolSelection-r12");
          return c;
        }
        const sl_pool_sel_cfg_r12_s& rsrp_based_r12() const
        {
          assert_choice_type("rsrpBased-r12", type_.to_string(), "poolSelection-r12");
          return c;
        }
        sl_pool_sel_cfg_r12_s& set_rsrp_based_r12()
        {
          set(types::rsrp_based_r12);
          return c;
        }

      private:
        types                 type_;
        sl_pool_sel_cfg_r12_s c;
      };
      struct tx_probability_r12_opts {
        enum options { p25, p50, p75, p100, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<tx_probability_r12_opts> tx_probability_r12_e_;

      // member variables
      pool_sel_r12_c_       pool_sel_r12;
      tx_probability_r12_e_ tx_probability_r12;
    };

    // member variables
    bool                  ue_sel_res_cfg_r12_present = false;
    sl_tx_params_r12_s    tx_params_general_r12;
    ue_sel_res_cfg_r12_s_ ue_sel_res_cfg_r12;
  };
  struct rx_params_r12_s_ {
    bool      tdd_cfg_r12_present = false;
    tdd_cfg_s tdd_cfg_r12;
    uint8_t   sync_cfg_idx_r12 = 0;
  };
  struct disc_period_v1310_c_ {
    struct setup_opts {
      enum options { rf4, rf6, rf7, rf8, rf12, rf14, rf24, rf28, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    disc_period_v1310_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discPeriod-v1310");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discPeriod-v1310");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };
  struct rx_params_add_neigh_freq_r13_c_ {
    struct setup_s_ {
      pci_list_r13_l pci_r13;
    };
    typedef setup_e types;

    // choice methods
    rx_params_add_neigh_freq_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "rxParamsAddNeighFreq-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "rxParamsAddNeighFreq-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct tx_params_add_neigh_freq_r13_c_ {
    struct setup_s_ {
      struct freq_info_s_ {
        struct ul_bw_opts {
          enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<ul_bw_opts> ul_bw_e_;

        // member variables
        bool     ul_carrier_freq_present = false;
        bool     ul_bw_present           = false;
        uint32_t ul_carrier_freq         = 0;
        ul_bw_e_ ul_bw;
        uint8_t  add_spec_emission = 1;
      };

      // member variables
      bool            p_max_present            = false;
      bool            tdd_cfg_r13_present      = false;
      bool            tdd_cfg_v1130_present    = false;
      bool            sync_cfg_idx_r13_present = false;
      pci_list_r13_l  pci_r13;
      int8_t          p_max = -30;
      tdd_cfg_s       tdd_cfg_r13;
      tdd_cfg_v1130_s tdd_cfg_v1130;
      freq_info_s_    freq_info;
      int8_t          ref_sig_pwr      = -60;
      uint8_t         sync_cfg_idx_r13 = 0;
    };
    typedef setup_e types;

    // choice methods
    tx_params_add_neigh_freq_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "txParamsAddNeighFreq-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "txParamsAddNeighFreq-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct tx_params_add_neigh_freq_v1370_c_ {
    struct setup_s_ {
      struct freq_info_v1370_s_ {
        uint16_t add_spec_emission_v1370 = 33;
      };

      // member variables
      freq_info_v1370_s_ freq_info_v1370;
    };
    typedef setup_e types;

    // choice methods
    tx_params_add_neigh_freq_v1370_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "txParamsAddNeighFreq-v1370");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "txParamsAddNeighFreq-v1370");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                ext                   = false;
  bool                tx_params_r12_present = false;
  bool                rx_params_r12_present = false;
  sl_cp_len_r12_e     cp_len_r12;
  disc_period_r12_e_  disc_period_r12;
  uint8_t             num_retx_r12   = 0;
  uint8_t             num_repeat_r12 = 1;
  sl_tf_res_cfg_r12_s tf_res_cfg_r12;
  tx_params_r12_s_    tx_params_r12;
  rx_params_r12_s_    rx_params_r12;
  // ...
  // group 0
  copy_ptr<disc_period_v1310_c_>            disc_period_v1310;
  copy_ptr<rx_params_add_neigh_freq_r13_c_> rx_params_add_neigh_freq_r13;
  copy_ptr<tx_params_add_neigh_freq_r13_c_> tx_params_add_neigh_freq_r13;
  // group 1
  copy_ptr<tx_params_add_neigh_freq_v1370_c_> tx_params_add_neigh_freq_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-SyncConfigListNFreqV2X-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-SyncConfigNFreq-r13
using sl_sync_cfg_list_nfreq_v2x_r14_l = dyn_array<sl_sync_cfg_nfreq_r13_s>;

// SL-ZoneConfig-r14 ::= SEQUENCE
struct sl_zone_cfg_r14_s {
  struct zone_len_r14_opts {
    enum options { m5, m10, m20, m50, m100, m200, m500, spare1, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<zone_len_r14_opts> zone_len_r14_e_;
  struct zone_width_r14_opts {
    enum options { m5, m10, m20, m50, m100, m200, m500, spare1, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<zone_width_r14_opts> zone_width_r14_e_;

  // member variables
  zone_len_r14_e_   zone_len_r14;
  zone_width_r14_e_ zone_width_r14;
  uint8_t           zone_id_longi_mod_r14 = 1;
  uint8_t           zone_id_lati_mod_r14  = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v1360-IEs ::= SEQUENCE
struct sib_type1_v1360_ies_s {
  bool                      cell_sel_info_ce1_v1360_present = false;
  bool                      non_crit_ext_present            = false;
  cell_sel_info_ce1_v1360_s cell_sel_info_ce1_v1360;
  sib_type1_v1430_ies_s     non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandClassListCDMA2000 ::= SEQUENCE (SIZE (1..32)) OF BandClassInfoCDMA2000
using band_class_list_cdma2000_l = dyn_array<band_class_info_cdma2000_s>;

// InterFreqCarrierFreqInfo-v10l0 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v10l0_s {
  bool                         freq_band_info_v10l0_present       = false;
  bool                         multi_band_info_list_v10l0_present = false;
  ns_pmax_list_v10l0_l         freq_band_info_v10l0;
  multi_band_info_list_v10l0_l multi_band_info_list_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MultiBandInfo-v9e0 ::= SEQUENCE
struct multi_band_info_v9e0_s {
  bool     freq_band_ind_v9e0_present = false;
  uint16_t freq_band_ind_v9e0         = 65;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MultiBandInfoList-v10j0 ::= SEQUENCE (SIZE (1..8)) OF NS-PmaxList-r10
using multi_band_info_list_v10j0_l = dyn_array<ns_pmax_list_r10_l>;

// NeighCellCDMA2000-r11 ::= SEQUENCE
struct neigh_cell_cdma2000_r11_s {
  using neigh_freq_info_list_r11_l_ = dyn_array<neigh_cells_per_bandclass_cdma2000_r11_s>;

  // member variables
  bandclass_cdma2000_e        band_class;
  neigh_freq_info_list_r11_l_ neigh_freq_info_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellsPerBandclassCDMA2000 ::= SEQUENCE
struct neigh_cells_per_bandclass_cdma2000_s {
  uint16_t            arfcn = 0;
  pci_list_cdma2000_l pci_list;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellsPerBandclassCDMA2000-v920 ::= SEQUENCE
struct neigh_cells_per_bandclass_cdma2000_v920_s {
  pci_list_cdma2000_v920_l pci_list_v920;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysCellIdRange ::= SEQUENCE
struct pci_range_s {
  struct range_opts {
    enum options {
      n4,
      n8,
      n12,
      n16,
      n24,
      n32,
      n48,
      n64,
      n84,
      n96,
      n128,
      n168,
      n252,
      n504,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<range_opts> range_e_;

  // member variables
  bool     range_present = false;
  uint16_t start         = 0;
  range_e_ range;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// Q-OffsetRange ::= ENUMERATED
struct q_offset_range_opts {
  enum options {
    db_minus24,
    db_minus22,
    db_minus20,
    db_minus18,
    db_minus16,
    db_minus14,
    db_minus12,
    db_minus10,
    db_minus8,
    db_minus6,
    db_minus5,
    db_minus4,
    db_minus3,
    db_minus2,
    db_minus1,
    db0,
    db1,
    db2,
    db3,
    db4,
    db5,
    db6,
    db8,
    db10,
    db12,
    db14,
    db16,
    db18,
    db20,
    db22,
    db24,
    nulltype
  } value;
  typedef int8_t number_type;

  std::string to_string() const;
  int8_t      to_number() const;
};
typedef enumerated<q_offset_range_opts> q_offset_range_e;

// RedistributionNeighCell-r13 ::= SEQUENCE
struct redist_neigh_cell_r13_s {
  uint16_t pci_r13                = 0;
  uint8_t  redist_factor_cell_r13 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscTxPoolList-r12 ::= SEQUENCE (SIZE (1..4)) OF SL-DiscResourcePool-r12
using sl_disc_tx_pool_list_r12_l = dyn_array<sl_disc_res_pool_r12_s>;

// SL-DiscTxPowerInfo-r12 ::= SEQUENCE
struct sl_disc_tx_pwr_info_r12_s {
  bool   ext                 = false;
  int8_t disc_max_tx_pwr_r12 = -30;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-V2X-FreqSelectionConfig-r15 ::= SEQUENCE
struct sl_v2x_freq_sel_cfg_r15_s {
  bool               thresh_cbr_freq_resel_r15_present   = false;
  bool               thresh_cbr_freq_keeping_r15_present = false;
  sl_prio_list_r13_l prio_list_r15;
  uint8_t            thresh_cbr_freq_resel_r15   = 0;
  uint8_t            thresh_cbr_freq_keeping_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-V2X-InterFreqUE-Config-r14 ::= SEQUENCE
struct sl_v2x_inter_freq_ue_cfg_r14_s {
  bool                              ext                                      = false;
  bool                              pci_list_r14_present                     = false;
  bool                              type_tx_sync_r14_present                 = false;
  bool                              v2x_sync_cfg_r14_present                 = false;
  bool                              v2x_comm_rx_pool_r14_present             = false;
  bool                              v2x_comm_tx_pool_normal_r14_present      = false;
  bool                              p2x_comm_tx_pool_normal_r14_present      = false;
  bool                              v2x_comm_tx_pool_exceptional_r14_present = false;
  bool                              v2x_res_sel_cfg_r14_present              = false;
  bool                              zone_cfg_r14_present                     = false;
  bool                              offset_dfn_r14_present                   = false;
  pci_list_r13_l                    pci_list_r14;
  sl_type_tx_sync_r14_e             type_tx_sync_r14;
  sl_sync_cfg_list_nfreq_v2x_r14_l  v2x_sync_cfg_r14;
  sl_comm_rx_pool_list_v2x_r14_l    v2x_comm_rx_pool_r14;
  sl_comm_tx_pool_list_v2x_r14_l    v2x_comm_tx_pool_normal_r14;
  sl_comm_tx_pool_list_v2x_r14_l    p2x_comm_tx_pool_normal_r14;
  sl_comm_res_pool_v2x_r14_s        v2x_comm_tx_pool_exceptional_r14;
  sl_comm_tx_pool_sensing_cfg_r14_s v2x_res_sel_cfg_r14;
  sl_zone_cfg_r14_s                 zone_cfg_r14;
  uint16_t                          offset_dfn_r14 = 0;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SchedulingInfo-BR-r13 ::= SEQUENCE
struct sched_info_br_r13_s {
  struct si_tbs_r13_opts {
    enum options { b152, b208, b256, b328, b408, b504, b600, b712, b808, b936, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<si_tbs_r13_opts> si_tbs_r13_e_;

  // member variables
  uint8_t       si_nb_r13 = 1;
  si_tbs_r13_e_ si_tbs_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SecondaryPreRegistrationZoneIdListHRPD ::= SEQUENCE (SIZE (1..2)) OF INTEGER (0..255)
using secondary_pre_regist_zone_id_list_hrpd_l = bounded_array<uint16_t, 2>;

// SpeedStateScaleFactors ::= SEQUENCE
struct speed_state_scale_factors_s {
  struct sf_medium_opts {
    enum options { odot25, odot5, odot75, ldot0, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<sf_medium_opts> sf_medium_e_;
  struct sf_high_opts {
    enum options { odot25, odot5, odot75, ldot0, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<sf_high_opts> sf_high_e_;

  // member variables
  sf_medium_e_ sf_medium;
  sf_high_e_   sf_high;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v1350-IEs ::= SEQUENCE
struct sib_type1_v1350_ies_s {
  bool                    cell_sel_info_ce1_r13_present = false;
  bool                    non_crit_ext_present          = false;
  cell_sel_info_ce1_r13_s cell_sel_info_ce1_r13;
  sib_type1_v1360_ies_s   non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5-v13a0-IEs ::= SEQUENCE
struct sib_type5_v13a0_ies_s {
  bool                                 late_non_crit_ext_present                  = false;
  bool                                 inter_freq_carrier_freq_list_v13a0_present = false;
  bool                                 non_crit_ext_present                       = false;
  dyn_octstring                        late_non_crit_ext;
  inter_freq_carrier_freq_list_v13a0_l inter_freq_carrier_freq_list_v13a0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AC-BarringConfig1XRTT-r9 ::= SEQUENCE
struct ac_barr_cfg1_xrtt_r9_s {
  uint8_t ac_barr0to9_r9 = 0;
  uint8_t ac_barr10_r9   = 0;
  uint8_t ac_barr11_r9   = 0;
  uint8_t ac_barr12_r9   = 0;
  uint8_t ac_barr13_r9   = 0;
  uint8_t ac_barr14_r9   = 0;
  uint8_t ac_barr15_r9   = 0;
  uint8_t ac_barr_msg_r9 = 0;
  uint8_t ac_barr_reg_r9 = 0;
  uint8_t ac_barr_emg_r9 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AllowedMeasBandwidth ::= ENUMERATED
struct allowed_meas_bw_opts {
  enum options { mbw6, mbw15, mbw25, mbw50, mbw75, mbw100, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<allowed_meas_bw_opts> allowed_meas_bw_e;

// BandIndicatorGERAN ::= ENUMERATED
struct band_ind_geran_opts {
  enum options { dcs1800, pcs1900, nulltype } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<band_ind_geran_opts> band_ind_geran_e;

// BarringPerACDC-Category-r13 ::= SEQUENCE
struct barr_per_acdc_category_r13_s {
  struct acdc_barr_cfg_r13_s_ {
    struct ac_barr_factor_r13_opts {
      enum options { p00, p05, p10, p15, p20, p25, p30, p40, p50, p60, p70, p75, p80, p85, p90, p95, nulltype } value;
      typedef float number_type;

      std::string to_string() const;
      float       to_number() const;
      std::string to_number_string() const;
    };
    typedef enumerated<ac_barr_factor_r13_opts> ac_barr_factor_r13_e_;
    struct ac_barr_time_r13_opts {
      enum options { s4, s8, s16, s32, s64, s128, s256, s512, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<ac_barr_time_r13_opts> ac_barr_time_r13_e_;

    // member variables
    ac_barr_factor_r13_e_ ac_barr_factor_r13;
    ac_barr_time_r13_e_   ac_barr_time_r13;
  };

  // member variables
  bool                 acdc_barr_cfg_r13_present = false;
  uint8_t              acdc_category_r13         = 1;
  acdc_barr_cfg_r13_s_ acdc_barr_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSFB-RegistrationParam1XRTT ::= SEQUENCE
struct csfb_regist_param1_xrtt_s {
  fixed_bitstring<15> sid;
  fixed_bitstring<16> nid;
  bool                multiple_sid    = false;
  bool                multiple_nid    = false;
  bool                home_reg        = false;
  bool                foreign_sid_reg = false;
  bool                foreign_nid_reg = false;
  bool                param_reg       = false;
  bool                pwr_up_reg      = false;
  fixed_bitstring<7>  regist_period;
  fixed_bitstring<12> regist_zone;
  fixed_bitstring<3>  total_zone;
  fixed_bitstring<3>  zone_timer;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSFB-RegistrationParam1XRTT-v920 ::= SEQUENCE
struct csfb_regist_param1_xrtt_v920_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellList-r15 ::= SEQUENCE (SIZE (1..8)) OF PhysCellIdRange
using cell_list_r15_l = dyn_array<pci_range_s>;

// CellReselectionParametersCDMA2000-r11 ::= SEQUENCE
struct cell_resel_params_cdma2000_r11_s {
  using neigh_cell_list_r11_l_ = dyn_array<neigh_cell_cdma2000_r11_s>;

  // member variables
  bool                        t_resel_cdma2000_sf_present = false;
  band_class_list_cdma2000_l  band_class_list;
  neigh_cell_list_r11_l_      neigh_cell_list_r11;
  uint8_t                     t_resel_cdma2000 = 0;
  speed_state_scale_factors_s t_resel_cdma2000_sf;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfoCE-r13 ::= SEQUENCE
struct cell_sel_info_ce_r13_s {
  bool   q_qual_min_rsrq_ce_r13_present = false;
  int8_t q_rx_lev_min_ce_r13            = -70;
  int8_t q_qual_min_rsrq_ce_r13         = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfoNFreq-r13 ::= SEQUENCE
struct cell_sel_info_nfreq_r13_s {
  struct q_hyst_r13_opts {
    enum options {
      db0,
      db1,
      db2,
      db3,
      db4,
      db5,
      db6,
      db8,
      db10,
      db12,
      db14,
      db16,
      db18,
      db20,
      db22,
      db24,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<q_hyst_r13_opts> q_hyst_r13_e_;

  // member variables
  bool          q_rx_lev_min_offset_present = false;
  int8_t        q_rx_lev_min_r13            = -70;
  uint8_t       q_rx_lev_min_offset         = 1;
  q_hyst_r13_e_ q_hyst_r13;
  int8_t        q_rx_lev_min_resel_r13 = -70;
  uint8_t       t_resel_eutra_r13      = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ExplicitListOfARFCNs ::= SEQUENCE (SIZE (0..31)) OF INTEGER (0..1023)
using explicit_list_of_arfcns_l = bounded_array<uint16_t, 31>;

// InterFreqCarrierFreqInfo-v10j0 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v10j0_s {
  bool                         freq_band_info_r10_present         = false;
  bool                         multi_band_info_list_v10j0_present = false;
  ns_pmax_list_r10_l           freq_band_info_r10;
  multi_band_info_list_v10j0_l multi_band_info_list_v10j0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqNeighCellInfo ::= SEQUENCE
struct inter_freq_neigh_cell_info_s {
  uint16_t         pci = 0;
  q_offset_range_e q_offset_cell;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MultiBandInfoList-v9e0 ::= SEQUENCE (SIZE (1..8)) OF MultiBandInfo-v9e0
using multi_band_info_list_v9e0_l = dyn_array<multi_band_info_v9e0_s>;

// NS-PmaxValueNR-r15 ::= SEQUENCE
struct ns_pmax_value_nr_r15_s {
  bool    add_pmax_nr_r15_present  = false;
  int8_t  add_pmax_nr_r15          = -30;
  uint8_t add_spec_emission_nr_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellsPerBandclassListCDMA2000 ::= SEQUENCE (SIZE (1..16)) OF NeighCellsPerBandclassCDMA2000
using neigh_cells_per_bandclass_list_cdma2000_l = dyn_array<neigh_cells_per_bandclass_cdma2000_s>;

// NeighCellsPerBandclassListCDMA2000-v920 ::= SEQUENCE (SIZE (1..16)) OF NeighCellsPerBandclassCDMA2000-v920
using neigh_cells_per_bandclass_list_cdma2000_v920_l = dyn_array<neigh_cells_per_bandclass_cdma2000_v920_s>;

// PLMN-IdentityInfo2-r12 ::= CHOICE
struct plmn_id_info2_r12_c {
  struct types_opts {
    enum options { plmn_idx_r12, plmn_id_r12, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  plmn_id_info2_r12_c() = default;
  plmn_id_info2_r12_c(const plmn_id_info2_r12_c& other);
  plmn_id_info2_r12_c& operator=(const plmn_id_info2_r12_c& other);
  ~plmn_id_info2_r12_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  uint8_t& plmn_idx_r12()
  {
    assert_choice_type("plmn-Index-r12", type_.to_string(), "PLMN-IdentityInfo2-r12");
    return c.get<uint8_t>();
  }
  plmn_id_s& plmn_id_r12()
  {
    assert_choice_type("plmnIdentity-r12", type_.to_string(), "PLMN-IdentityInfo2-r12");
    return c.get<plmn_id_s>();
  }
  const uint8_t& plmn_idx_r12() const
  {
    assert_choice_type("plmn-Index-r12", type_.to_string(), "PLMN-IdentityInfo2-r12");
    return c.get<uint8_t>();
  }
  const plmn_id_s& plmn_id_r12() const
  {
    assert_choice_type("plmnIdentity-r12", type_.to_string(), "PLMN-IdentityInfo2-r12");
    return c.get<plmn_id_s>();
  }
  uint8_t& set_plmn_idx_r12()
  {
    set(types::plmn_idx_r12);
    return c.get<uint8_t>();
  }
  plmn_id_s& set_plmn_id_r12()
  {
    set(types::plmn_id_r12);
    return c.get<plmn_id_s>();
  }

private:
  types                      type_;
  choice_buffer_t<plmn_id_s> c;

  void destroy_();
};

// PRACH-ParametersCE-r13 ::= SEQUENCE
struct prach_params_ce_r13_s {
  struct prach_start_sf_r13_opts {
    enum options { sf2, sf4, sf8, sf16, sf32, sf64, sf128, sf256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<prach_start_sf_r13_opts> prach_start_sf_r13_e_;
  struct max_num_preamb_attempt_ce_r13_opts {
    enum options { n3, n4, n5, n6, n7, n8, n10, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_num_preamb_attempt_ce_r13_opts> max_num_preamb_attempt_ce_r13_e_;
  struct num_repeat_per_preamb_attempt_r13_opts {
    enum options { n1, n2, n4, n8, n16, n32, n64, n128, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<num_repeat_per_preamb_attempt_r13_opts> num_repeat_per_preamb_attempt_r13_e_;
  using mpdcch_nbs_to_monitor_r13_l_ = bounded_array<uint8_t, 2>;
  struct mpdcch_num_repeat_ra_r13_opts {
    enum options { r1, r2, r4, r8, r16, r32, r64, r128, r256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<mpdcch_num_repeat_ra_r13_opts> mpdcch_num_repeat_ra_r13_e_;
  struct prach_hop_cfg_r13_opts {
    enum options { on, off, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<prach_hop_cfg_r13_opts> prach_hop_cfg_r13_e_;

  // member variables
  bool                                 prach_start_sf_r13_present            = false;
  bool                                 max_num_preamb_attempt_ce_r13_present = false;
  uint8_t                              prach_cfg_idx_r13                     = 0;
  uint8_t                              prach_freq_offset_r13                 = 0;
  prach_start_sf_r13_e_                prach_start_sf_r13;
  max_num_preamb_attempt_ce_r13_e_     max_num_preamb_attempt_ce_r13;
  num_repeat_per_preamb_attempt_r13_e_ num_repeat_per_preamb_attempt_r13;
  mpdcch_nbs_to_monitor_r13_l_         mpdcch_nbs_to_monitor_r13;
  mpdcch_num_repeat_ra_r13_e_          mpdcch_num_repeat_ra_r13;
  prach_hop_cfg_r13_e_                 prach_hop_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PreRegistrationInfoHRPD ::= SEQUENCE
struct pre_regist_info_hrpd_s {
  bool                                     pre_regist_zone_id_present                = false;
  bool                                     secondary_pre_regist_zone_id_list_present = false;
  bool                                     pre_regist_allowed                        = false;
  uint16_t                                 pre_regist_zone_id                        = 0;
  secondary_pre_regist_zone_id_list_hrpd_l secondary_pre_regist_zone_id_list;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-CE-LevelInfo-r13 ::= SEQUENCE
struct rach_ce_level_info_r13_s {
  struct preamb_map_info_r13_s_ {
    uint8_t first_preamb_r13 = 0;
    uint8_t last_preamb_r13  = 0;
  };
  struct ra_resp_win_size_r13_opts {
    enum options { sf20, sf50, sf80, sf120, sf180, sf240, sf320, sf400, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<ra_resp_win_size_r13_opts> ra_resp_win_size_r13_e_;
  struct mac_contention_resolution_timer_r13_opts {
    enum options { sf80, sf100, sf120, sf160, sf200, sf240, sf480, sf960, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<mac_contention_resolution_timer_r13_opts> mac_contention_resolution_timer_r13_e_;
  struct rar_hop_cfg_r13_opts {
    enum options { on, off, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<rar_hop_cfg_r13_opts> rar_hop_cfg_r13_e_;
  struct edt_params_r15_s_ {
    struct edt_tbs_r15_opts {
      enum options { b328, b408, b504, b600, b712, b808, b936, b1000or456, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<edt_tbs_r15_opts> edt_tbs_r15_e_;
    struct mac_contention_resolution_timer_r15_opts {
      enum options { sf240, sf480, sf960, sf1920, sf3840, sf5760, sf7680, sf10240, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<mac_contention_resolution_timer_r15_opts> mac_contention_resolution_timer_r15_e_;

    // member variables
    bool                                   mac_contention_resolution_timer_r15_present = false;
    uint8_t                                edt_last_preamb_r15                         = 0;
    bool                                   edt_small_tbs_enabled_r15                   = false;
    edt_tbs_r15_e_                         edt_tbs_r15;
    mac_contention_resolution_timer_r15_e_ mac_contention_resolution_timer_r15;
  };

  // member variables
  bool                                   ext = false;
  preamb_map_info_r13_s_                 preamb_map_info_r13;
  ra_resp_win_size_r13_e_                ra_resp_win_size_r13;
  mac_contention_resolution_timer_r13_e_ mac_contention_resolution_timer_r13;
  rar_hop_cfg_r13_e_                     rar_hop_cfg_r13;
  // ...
  // group 0
  copy_ptr<edt_params_r15_s_> edt_params_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RedistributionNeighCellList-r13 ::= SEQUENCE (SIZE (1..16)) OF RedistributionNeighCell-r13
using redist_neigh_cell_list_r13_l = dyn_array<redist_neigh_cell_r13_s>;

// SL-AllowedCarrierFreqList-r15 ::= SEQUENCE
struct sl_allowed_carrier_freq_list_r15_s {
  using allowed_carrier_freq_set1_l_ = bounded_array<uint32_t, 8>;
  using allowed_carrier_freq_set2_l_ = bounded_array<uint32_t, 8>;

  // member variables
  allowed_carrier_freq_set1_l_ allowed_carrier_freq_set1;
  allowed_carrier_freq_set2_l_ allowed_carrier_freq_set2;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DestinationInfoList-r12 ::= SEQUENCE (SIZE (1..16)) OF BIT STRING (SIZE (24))
using sl_dest_info_list_r12_l = bounded_array<fixed_bitstring<24>, 16>;

// SL-DiscRxPoolList-r12 ::= SEQUENCE (SIZE (1..16)) OF SL-DiscResourcePool-r12
using sl_disc_rx_pool_list_r12_l = dyn_array<sl_disc_res_pool_r12_s>;

// SL-DiscTxPowerInfoList-r12 ::= SEQUENCE (SIZE (3)) OF SL-DiscTxPowerInfo-r12
using sl_disc_tx_pwr_info_list_r12_l = std::array<sl_disc_tx_pwr_info_r12_s, 3>;

// SL-DiscTxResourcesInterFreq-r13 ::= CHOICE
struct sl_disc_tx_res_inter_freq_r13_c {
  struct types_opts {
    enum options {
      acquire_si_from_carrier_r13,
      disc_tx_pool_common_r13,
      request_ded_r13,
      no_tx_on_carrier_r13,
      nulltype
    } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sl_disc_tx_res_inter_freq_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  sl_disc_tx_pool_list_r12_l& disc_tx_pool_common_r13()
  {
    assert_choice_type("discTxPoolCommon-r13", type_.to_string(), "SL-DiscTxResourcesInterFreq-r13");
    return c;
  }
  const sl_disc_tx_pool_list_r12_l& disc_tx_pool_common_r13() const
  {
    assert_choice_type("discTxPoolCommon-r13", type_.to_string(), "SL-DiscTxResourcesInterFreq-r13");
    return c;
  }
  sl_disc_tx_pool_list_r12_l& set_disc_tx_pool_common_r13()
  {
    set(types::disc_tx_pool_common_r13);
    return c;
  }

private:
  types                      type_;
  sl_disc_tx_pool_list_r12_l c;
};

// SL-SyncConfigListNFreq-r13 ::= SEQUENCE (SIZE (1..16)) OF SL-SyncConfigNFreq-r13
using sl_sync_cfg_list_nfreq_r13_l = dyn_array<sl_sync_cfg_nfreq_r13_s>;

// SL-V2X-FreqSelectionConfigList-r15 ::= SEQUENCE (SIZE (1..8)) OF SL-V2X-FreqSelectionConfig-r15
using sl_v2x_freq_sel_cfg_list_r15_l = dyn_array<sl_v2x_freq_sel_cfg_r15_s>;

// SL-V2X-UE-ConfigList-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-V2X-InterFreqUE-Config-r14
using sl_v2x_ue_cfg_list_r14_l = dyn_array<sl_v2x_inter_freq_ue_cfg_r14_s>;

// SchedulingInfoList-BR-r13 ::= SEQUENCE (SIZE (1..32)) OF SchedulingInfo-BR-r13
using sched_info_list_br_r13_l = dyn_array<sched_info_br_r13_s>;

// SystemInfoValueTagList-r13 ::= SEQUENCE (SIZE (1..32)) OF INTEGER (0..3)
using sys_info_value_tag_list_r13_l = bounded_array<uint8_t, 32>;

// SystemInformationBlockType1-v1320-IEs ::= SEQUENCE
struct sib_type1_v1320_ies_s {
  struct freq_hop_params_dl_r13_s_ {
    struct mpdcch_pdsch_hop_nb_r13_opts {
      enum options { nb2, nb4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<mpdcch_pdsch_hop_nb_r13_opts> mpdcch_pdsch_hop_nb_r13_e_;
    struct interv_dl_hop_cfg_common_mode_a_r13_c_ {
      struct interv_fdd_r13_opts {
        enum options { int1, int2, int4, int8, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
      struct interv_tdd_r13_opts {
        enum options { int1, int5, int10, int20, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
      struct types_opts {
        enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      interv_dl_hop_cfg_common_mode_a_r13_c_() = default;
      interv_dl_hop_cfg_common_mode_a_r13_c_(const interv_dl_hop_cfg_common_mode_a_r13_c_& other);
      interv_dl_hop_cfg_common_mode_a_r13_c_& operator=(const interv_dl_hop_cfg_common_mode_a_r13_c_& other);
      ~interv_dl_hop_cfg_common_mode_a_r13_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      interv_fdd_r13_e_& interv_fdd_r13()
      {
        assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeA-r13");
        return c.get<interv_fdd_r13_e_>();
      }
      interv_tdd_r13_e_& interv_tdd_r13()
      {
        assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeA-r13");
        return c.get<interv_tdd_r13_e_>();
      }
      const interv_fdd_r13_e_& interv_fdd_r13() const
      {
        assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeA-r13");
        return c.get<interv_fdd_r13_e_>();
      }
      const interv_tdd_r13_e_& interv_tdd_r13() const
      {
        assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeA-r13");
        return c.get<interv_tdd_r13_e_>();
      }
      interv_fdd_r13_e_& set_interv_fdd_r13()
      {
        set(types::interv_fdd_r13);
        return c.get<interv_fdd_r13_e_>();
      }
      interv_tdd_r13_e_& set_interv_tdd_r13()
      {
        set(types::interv_tdd_r13);
        return c.get<interv_tdd_r13_e_>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };
    struct interv_dl_hop_cfg_common_mode_b_r13_c_ {
      struct interv_fdd_r13_opts {
        enum options { int2, int4, int8, int16, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
      struct interv_tdd_r13_opts {
        enum options { int5, int10, int20, int40, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
      struct types_opts {
        enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      interv_dl_hop_cfg_common_mode_b_r13_c_() = default;
      interv_dl_hop_cfg_common_mode_b_r13_c_(const interv_dl_hop_cfg_common_mode_b_r13_c_& other);
      interv_dl_hop_cfg_common_mode_b_r13_c_& operator=(const interv_dl_hop_cfg_common_mode_b_r13_c_& other);
      ~interv_dl_hop_cfg_common_mode_b_r13_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      interv_fdd_r13_e_& interv_fdd_r13()
      {
        assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeB-r13");
        return c.get<interv_fdd_r13_e_>();
      }
      interv_tdd_r13_e_& interv_tdd_r13()
      {
        assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeB-r13");
        return c.get<interv_tdd_r13_e_>();
      }
      const interv_fdd_r13_e_& interv_fdd_r13() const
      {
        assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeB-r13");
        return c.get<interv_fdd_r13_e_>();
      }
      const interv_tdd_r13_e_& interv_tdd_r13() const
      {
        assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-DLHoppingConfigCommonModeB-r13");
        return c.get<interv_tdd_r13_e_>();
      }
      interv_fdd_r13_e_& set_interv_fdd_r13()
      {
        set(types::interv_fdd_r13);
        return c.get<interv_fdd_r13_e_>();
      }
      interv_tdd_r13_e_& set_interv_tdd_r13()
      {
        set(types::interv_tdd_r13);
        return c.get<interv_tdd_r13_e_>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };

    // member variables
    bool                                   mpdcch_pdsch_hop_nb_r13_present             = false;
    bool                                   interv_dl_hop_cfg_common_mode_a_r13_present = false;
    bool                                   interv_dl_hop_cfg_common_mode_b_r13_present = false;
    bool                                   mpdcch_pdsch_hop_offset_r13_present         = false;
    mpdcch_pdsch_hop_nb_r13_e_             mpdcch_pdsch_hop_nb_r13;
    interv_dl_hop_cfg_common_mode_a_r13_c_ interv_dl_hop_cfg_common_mode_a_r13;
    interv_dl_hop_cfg_common_mode_b_r13_c_ interv_dl_hop_cfg_common_mode_b_r13;
    uint8_t                                mpdcch_pdsch_hop_offset_r13 = 1;
  };

  // member variables
  bool                      freq_hop_params_dl_r13_present = false;
  bool                      non_crit_ext_present           = false;
  freq_hop_params_dl_r13_s_ freq_hop_params_dl_r13;
  sib_type1_v1350_ies_s     non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType2-v10m0-IEs ::= SEQUENCE
struct sib_type2_v10m0_ies_s {
  struct freq_info_v10l0_s_ {
    uint16_t add_spec_emission_v10l0 = 33;
  };
  using multi_band_info_list_v10l0_l_ = bounded_array<uint16_t, 8>;

  // member variables
  bool                          freq_info_v10l0_present            = false;
  bool                          multi_band_info_list_v10l0_present = false;
  bool                          non_crit_ext_present               = false;
  freq_info_v10l0_s_            freq_info_v10l0;
  multi_band_info_list_v10l0_l_ multi_band_info_list_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5-v10l0-IEs ::= SEQUENCE
struct sib_type5_v10l0_ies_s {
  using inter_freq_carrier_freq_list_v10l0_l_ = dyn_array<inter_freq_carrier_freq_info_v10l0_s>;

  // member variables
  bool                                  inter_freq_carrier_freq_list_v10l0_present = false;
  bool                                  non_crit_ext_present                       = false;
  inter_freq_carrier_freq_list_v10l0_l_ inter_freq_carrier_freq_list_v10l0;
  sib_type5_v13a0_ies_s                 non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemTimeInfoCDMA2000 ::= SEQUENCE
struct sys_time_info_cdma2000_s {
  struct cdma_sys_time_c_ {
    struct types_opts {
      enum options { sync_sys_time, async_sys_time, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    cdma_sys_time_c_() = default;
    cdma_sys_time_c_(const cdma_sys_time_c_& other);
    cdma_sys_time_c_& operator=(const cdma_sys_time_c_& other);
    ~cdma_sys_time_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<39>& sync_sys_time()
    {
      assert_choice_type("synchronousSystemTime", type_.to_string(), "cdma-SystemTime");
      return c.get<fixed_bitstring<39> >();
    }
    fixed_bitstring<49>& async_sys_time()
    {
      assert_choice_type("asynchronousSystemTime", type_.to_string(), "cdma-SystemTime");
      return c.get<fixed_bitstring<49> >();
    }
    const fixed_bitstring<39>& sync_sys_time() const
    {
      assert_choice_type("synchronousSystemTime", type_.to_string(), "cdma-SystemTime");
      return c.get<fixed_bitstring<39> >();
    }
    const fixed_bitstring<49>& async_sys_time() const
    {
      assert_choice_type("asynchronousSystemTime", type_.to_string(), "cdma-SystemTime");
      return c.get<fixed_bitstring<49> >();
    }
    fixed_bitstring<39>& set_sync_sys_time()
    {
      set(types::sync_sys_time);
      return c.get<fixed_bitstring<39> >();
    }
    fixed_bitstring<49>& set_async_sys_time()
    {
      set(types::async_sys_time);
      return c.get<fixed_bitstring<49> >();
    }

  private:
    types                                 type_;
    choice_buffer_t<fixed_bitstring<49> > c;

    void destroy_();
  };

  // member variables
  bool             cdma_eutra_synchronisation = false;
  cdma_sys_time_c_ cdma_sys_time;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UAC-BarringPerCat-r15 ::= SEQUENCE
struct uac_barr_per_cat_r15_s {
  uint8_t access_category_r15       = 1;
  uint8_t uac_barr_info_set_idx_r15 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AC-BarringConfig ::= SEQUENCE
struct ac_barr_cfg_s {
  struct ac_barr_factor_opts {
    enum options { p00, p05, p10, p15, p20, p25, p30, p40, p50, p60, p70, p75, p80, p85, p90, p95, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<ac_barr_factor_opts> ac_barr_factor_e_;
  struct ac_barr_time_opts {
    enum options { s4, s8, s16, s32, s64, s128, s256, s512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<ac_barr_time_opts> ac_barr_time_e_;

  // member variables
  ac_barr_factor_e_  ac_barr_factor;
  ac_barr_time_e_    ac_barr_time;
  fixed_bitstring<5> ac_barr_for_special_ac;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BarringPerACDC-CategoryList-r13 ::= SEQUENCE (SIZE (1..16)) OF BarringPerACDC-Category-r13
using barr_per_acdc_category_list_r13_l = dyn_array<barr_per_acdc_category_r13_s>;

// CarrierFreqsGERAN ::= SEQUENCE
struct carrier_freqs_geran_s {
  struct following_arfcns_c_ {
    struct equally_spaced_arfcns_s_ {
      uint8_t arfcn_spacing        = 1;
      uint8_t nof_following_arfcns = 0;
    };
    struct types_opts {
      enum options { explicit_list_of_arfcns, equally_spaced_arfcns, variable_bit_map_of_arfcns, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    following_arfcns_c_() = default;
    following_arfcns_c_(const following_arfcns_c_& other);
    following_arfcns_c_& operator=(const following_arfcns_c_& other);
    ~following_arfcns_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    explicit_list_of_arfcns_l& explicit_list_of_arfcns()
    {
      assert_choice_type("explicitListOfARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<explicit_list_of_arfcns_l>();
    }
    equally_spaced_arfcns_s_& equally_spaced_arfcns()
    {
      assert_choice_type("equallySpacedARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<equally_spaced_arfcns_s_>();
    }
    dyn_octstring& variable_bit_map_of_arfcns()
    {
      assert_choice_type("variableBitMapOfARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<dyn_octstring>();
    }
    const explicit_list_of_arfcns_l& explicit_list_of_arfcns() const
    {
      assert_choice_type("explicitListOfARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<explicit_list_of_arfcns_l>();
    }
    const equally_spaced_arfcns_s_& equally_spaced_arfcns() const
    {
      assert_choice_type("equallySpacedARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<equally_spaced_arfcns_s_>();
    }
    const dyn_octstring& variable_bit_map_of_arfcns() const
    {
      assert_choice_type("variableBitMapOfARFCNs", type_.to_string(), "followingARFCNs");
      return c.get<dyn_octstring>();
    }
    explicit_list_of_arfcns_l& set_explicit_list_of_arfcns()
    {
      set(types::explicit_list_of_arfcns);
      return c.get<explicit_list_of_arfcns_l>();
    }
    equally_spaced_arfcns_s_& set_equally_spaced_arfcns()
    {
      set(types::equally_spaced_arfcns);
      return c.get<equally_spaced_arfcns_s_>();
    }
    dyn_octstring& set_variable_bit_map_of_arfcns()
    {
      set(types::variable_bit_map_of_arfcns);
      return c.get<dyn_octstring>();
    }

  private:
    types                                                                               type_;
    choice_buffer_t<dyn_octstring, equally_spaced_arfcns_s_, explicit_list_of_arfcns_l> c;

    void destroy_();
  };

  // member variables
  uint16_t            start_arfcn = 0;
  band_ind_geran_e    band_ind;
  following_arfcns_c_ following_arfcns;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellReselectionSubPriority-r13 ::= ENUMERATED
struct cell_resel_sub_prio_r13_opts {
  enum options { odot2, odot4, odot6, odot8, nulltype } value;
  typedef float number_type;

  std::string to_string() const;
  float       to_number() const;
  std::string to_number_string() const;
};
typedef enumerated<cell_resel_sub_prio_r13_opts> cell_resel_sub_prio_r13_e;

// CellSelectionInfo-v1250 ::= SEQUENCE
struct cell_sel_info_v1250_s {
  int8_t q_qual_min_rsrq_on_all_symbols_r12 = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DeltaFList-PUCCH ::= SEQUENCE
struct delta_flist_pucch_s {
  struct delta_f_pucch_format1_opts {
    enum options { delta_f_minus2, delta_f0, delta_f2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format1_opts> delta_f_pucch_format1_e_;
  struct delta_f_pucch_format1b_opts {
    enum options { delta_f1, delta_f3, delta_f5, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format1b_opts> delta_f_pucch_format1b_e_;
  struct delta_f_pucch_format2_opts {
    enum options { delta_f_minus2, delta_f0, delta_f1, delta_f2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format2_opts> delta_f_pucch_format2_e_;
  struct delta_f_pucch_format2a_opts {
    enum options { delta_f_minus2, delta_f0, delta_f2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format2a_opts> delta_f_pucch_format2a_e_;
  struct delta_f_pucch_format2b_opts {
    enum options { delta_f_minus2, delta_f0, delta_f2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format2b_opts> delta_f_pucch_format2b_e_;

  // member variables
  delta_f_pucch_format1_e_  delta_f_pucch_format1;
  delta_f_pucch_format1b_e_ delta_f_pucch_format1b;
  delta_f_pucch_format2_e_  delta_f_pucch_format2;
  delta_f_pucch_format2a_e_ delta_f_pucch_format2a;
  delta_f_pucch_format2b_e_ delta_f_pucch_format2b;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EDT-PRACH-ParametersCE-r15 ::= SEQUENCE
struct edt_prach_params_ce_r15_s {
  struct edt_prach_params_ce_r15_s_ {
    struct prach_start_sf_r15_opts {
      enum options { sf2, sf4, sf8, sf16, sf32, sf64, sf128, sf256, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<prach_start_sf_r15_opts> prach_start_sf_r15_e_;
    using mpdcch_nbs_to_monitor_r15_l_ = bounded_array<uint8_t, 2>;

    // member variables
    bool                         prach_start_sf_r15_present = false;
    uint8_t                      prach_cfg_idx_r15          = 0;
    uint8_t                      prach_freq_offset_r15      = 0;
    prach_start_sf_r15_e_        prach_start_sf_r15;
    mpdcch_nbs_to_monitor_r15_l_ mpdcch_nbs_to_monitor_r15;
  };

  // member variables
  bool                       edt_prach_params_ce_r15_present = false;
  edt_prach_params_ce_r15_s_ edt_prach_params_ce_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FilterCoefficient ::= ENUMERATED
struct filt_coef_opts {
  enum options {
    fc0,
    fc1,
    fc2,
    fc3,
    fc4,
    fc5,
    fc6,
    fc7,
    fc8,
    fc9,
    fc11,
    fc13,
    fc15,
    fc17,
    fc19,
    spare1,
    /*...*/ nulltype
  } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<filt_coef_opts, true> filt_coef_e;

// InterFreqBlackCellList ::= SEQUENCE (SIZE (1..16)) OF PhysCellIdRange
using inter_freq_black_cell_list_l = dyn_array<pci_range_s>;

// InterFreqCarrierFreqInfo-v9e0 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v9e0_s {
  bool                        dl_carrier_freq_v9e0_present      = false;
  bool                        multi_band_info_list_v9e0_present = false;
  uint32_t                    dl_carrier_freq_v9e0              = 65536;
  multi_band_info_list_v9e0_l multi_band_info_list_v9e0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqNeighCellList ::= SEQUENCE (SIZE (1..16)) OF InterFreqNeighCellInfo
using inter_freq_neigh_cell_list_l = dyn_array<inter_freq_neigh_cell_info_s>;

// InterFreqNeighHSDN-CellList-r15 ::= SEQUENCE (SIZE (1..16)) OF PhysCellIdRange
using inter_freq_neigh_hsdn_cell_list_r15_l = dyn_array<pci_range_s>;

// MBMS-SAI-List-r11 ::= SEQUENCE (SIZE (1..64)) OF INTEGER (0..65535)
using mbms_sai_list_r11_l = dyn_array<uint32_t>;

// MTC-SSB-NR-r15 ::= SEQUENCE
struct mtc_ssb_nr_r15_s {
  struct periodicity_and_offset_r15_c_ {
    struct types_opts {
      enum options { sf5_r15, sf10_r15, sf20_r15, sf40_r15, sf80_r15, sf160_r15, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    periodicity_and_offset_r15_c_() = default;
    periodicity_and_offset_r15_c_(const periodicity_and_offset_r15_c_& other);
    periodicity_and_offset_r15_c_& operator=(const periodicity_and_offset_r15_c_& other);
    ~periodicity_and_offset_r15_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& sf5_r15()
    {
      assert_choice_type("sf5-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& sf10_r15()
    {
      assert_choice_type("sf10-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& sf20_r15()
    {
      assert_choice_type("sf20-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& sf40_r15()
    {
      assert_choice_type("sf40-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& sf80_r15()
    {
      assert_choice_type("sf80-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& sf160_r15()
    {
      assert_choice_type("sf160-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf5_r15() const
    {
      assert_choice_type("sf5-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf10_r15() const
    {
      assert_choice_type("sf10-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf20_r15() const
    {
      assert_choice_type("sf20-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf40_r15() const
    {
      assert_choice_type("sf40-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf80_r15() const
    {
      assert_choice_type("sf80-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    const uint8_t& sf160_r15() const
    {
      assert_choice_type("sf160-r15", type_.to_string(), "periodicityAndOffset-r15");
      return c.get<uint8_t>();
    }
    uint8_t& set_sf5_r15()
    {
      set(types::sf5_r15);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf10_r15()
    {
      set(types::sf10_r15);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf20_r15()
    {
      set(types::sf20_r15);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf40_r15()
    {
      set(types::sf40_r15);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf80_r15()
    {
      set(types::sf80_r15);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf160_r15()
    {
      set(types::sf160_r15);
      return c.get<uint8_t>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };
  struct ssb_dur_r15_opts {
    enum options { sf1, sf2, sf3, sf4, sf5, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ssb_dur_r15_opts> ssb_dur_r15_e_;

  // member variables
  periodicity_and_offset_r15_c_ periodicity_and_offset_r15;
  ssb_dur_r15_e_                ssb_dur_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasIdleCarrierEUTRA-r15 ::= SEQUENCE
struct meas_idle_carrier_eutra_r15_s {
  struct report_quantities_opts {
    enum options { rsrp, rsrq, both, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<report_quantities_opts> report_quantities_e_;
  struct quality_thres_r15_s_ {
    bool    idle_rsrp_thres_r15_present = false;
    bool    idle_rsrq_thres_r15_present = false;
    uint8_t idle_rsrp_thres_r15         = 0;
    int8_t  idle_rsrq_thres_r15         = -30;
  };

  // member variables
  bool                 ext                        = false;
  bool                 validity_area_r15_present  = false;
  bool                 meas_cell_list_r15_present = false;
  bool                 quality_thres_r15_present  = false;
  uint32_t             carrier_freq_r15           = 0;
  allowed_meas_bw_e    allowed_meas_bw_r15;
  cell_list_r15_l      validity_area_r15;
  cell_list_r15_l      meas_cell_list_r15;
  report_quantities_e_ report_quantities;
  quality_thres_r15_s_ quality_thres_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MultiBandInfoList ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..64)
using multi_band_info_list_l = bounded_array<uint8_t, 8>;

// MultiBandInfoList-r11 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..256)
using multi_band_info_list_r11_l = bounded_array<uint16_t, 8>;

// MultiFrequencyBandListNR-r15 ::= SEQUENCE (SIZE (1..32)) OF INTEGER (1..1024)
using multi_freq_band_list_nr_r15_l = bounded_array<uint16_t, 32>;

// N1PUCCH-AN-InfoList-r13 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (0..2047)
using n1_pucch_an_info_list_r13_l = bounded_array<uint16_t, 4>;

// NS-PmaxListNR-r15 ::= SEQUENCE (SIZE (1..8)) OF NS-PmaxValueNR-r15
using ns_pmax_list_nr_r15_l = dyn_array<ns_pmax_value_nr_r15_s>;

// NeighCellCDMA2000 ::= SEQUENCE
struct neigh_cell_cdma2000_s {
  bandclass_cdma2000_e                      band_class;
  neigh_cells_per_bandclass_list_cdma2000_l neigh_cells_per_freq_list;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellCDMA2000-v920 ::= SEQUENCE
struct neigh_cell_cdma2000_v920_s {
  neigh_cells_per_bandclass_list_cdma2000_v920_l neigh_cells_per_freq_list_v920;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityList4-r12 ::= SEQUENCE (SIZE (1..6)) OF PLMN-IdentityInfo2-r12
using plmn_id_list4_r12_l = dyn_array<plmn_id_info2_r12_c>;

// PRACH-ConfigInfo ::= SEQUENCE
struct prach_cfg_info_s {
  uint8_t prach_cfg_idx             = 0;
  bool    high_speed_flag           = false;
  uint8_t zero_correlation_zone_cfg = 0;
  uint8_t prach_freq_offset         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-ParametersListCE-r13 ::= SEQUENCE (SIZE (1..4)) OF PRACH-ParametersCE-r13
using prach_params_list_ce_r13_l = dyn_array<prach_params_ce_r13_s>;

// ParametersCDMA2000-r11 ::= SEQUENCE
struct params_cdma2000_r11_s {
  struct sys_time_info_r11_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sys_time_info_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sys_time_info_cdma2000_s& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "systemTimeInfo-r11");
      return c;
    }
    const sys_time_info_cdma2000_s& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "systemTimeInfo-r11");
      return c;
    }
    sys_time_info_cdma2000_s& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types                    type_;
    sys_time_info_cdma2000_s c;
  };
  struct params_hrpd_r11_s_ {
    bool                             cell_resel_params_hrpd_r11_present = false;
    pre_regist_info_hrpd_s           pre_regist_info_hrpd_r11;
    cell_resel_params_cdma2000_r11_s cell_resel_params_hrpd_r11;
  };
  struct params1_xrtt_r11_s_ {
    bool                             csfb_regist_param1_xrtt_r11_present      = false;
    bool                             csfb_regist_param1_xrtt_ext_r11_present  = false;
    bool                             long_code_state1_xrtt_r11_present        = false;
    bool                             cell_resel_params1_xrtt_r11_present      = false;
    bool                             ac_barr_cfg1_xrtt_r11_present            = false;
    bool                             csfb_support_for_dual_rx_ues_r11_present = false;
    bool                             csfb_dual_rx_tx_support_r11_present      = false;
    csfb_regist_param1_xrtt_s        csfb_regist_param1_xrtt_r11;
    csfb_regist_param1_xrtt_v920_s   csfb_regist_param1_xrtt_ext_r11;
    fixed_bitstring<42>              long_code_state1_xrtt_r11;
    cell_resel_params_cdma2000_r11_s cell_resel_params1_xrtt_r11;
    ac_barr_cfg1_xrtt_r9_s           ac_barr_cfg1_xrtt_r11;
    bool                             csfb_support_for_dual_rx_ues_r11 = false;
  };

  // member variables
  bool                 ext                       = false;
  bool                 sys_time_info_r11_present = false;
  bool                 params_hrpd_r11_present   = false;
  bool                 params1_xrtt_r11_present  = false;
  sys_time_info_r11_c_ sys_time_info_r11;
  uint8_t              search_win_size_r11 = 0;
  params_hrpd_r11_s_   params_hrpd_r11;
  params1_xrtt_r11_s_  params1_xrtt_r11;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PowerRampingParameters ::= SEQUENCE
struct pwr_ramp_params_s {
  struct pwr_ramp_step_opts {
    enum options { db0, db2, db4, db6, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pwr_ramp_step_opts> pwr_ramp_step_e_;
  struct preamb_init_rx_target_pwr_opts {
    enum options {
      dbm_minus120,
      dbm_minus118,
      dbm_minus116,
      dbm_minus114,
      dbm_minus112,
      dbm_minus110,
      dbm_minus108,
      dbm_minus106,
      dbm_minus104,
      dbm_minus102,
      dbm_minus100,
      dbm_minus98,
      dbm_minus96,
      dbm_minus94,
      dbm_minus92,
      dbm_minus90,
      nulltype
    } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<preamb_init_rx_target_pwr_opts> preamb_init_rx_target_pwr_e_;

  // member variables
  pwr_ramp_step_e_             pwr_ramp_step;
  preamb_init_rx_target_pwr_e_ preamb_init_rx_target_pwr;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PreambleTransMax ::= ENUMERATED
struct preamb_trans_max_opts {
  enum options { n3, n4, n5, n6, n7, n8, n10, n20, n50, n100, n200, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<preamb_trans_max_opts> preamb_trans_max_e;

// RACH-CE-LevelInfoList-r13 ::= SEQUENCE (SIZE (1..4)) OF RACH-CE-LevelInfo-r13
using rach_ce_level_info_list_r13_l = dyn_array<rach_ce_level_info_r13_s>;

// RSRP-ThresholdsPrachInfoList-r13 ::= SEQUENCE (SIZE (1..3)) OF INTEGER (0..97)
using rsrp_thress_prach_info_list_r13_l = bounded_array<uint8_t, 3>;

// RedistributionInterFreqInfo-r13 ::= SEQUENCE
struct redist_inter_freq_info_r13_s {
  bool                         redist_factor_freq_r13_present     = false;
  bool                         redist_neigh_cell_list_r13_present = false;
  uint8_t                      redist_factor_freq_r13             = 1;
  redist_neigh_cell_list_r13_l redist_neigh_cell_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CBR-Levels-Config-r14 ::= SEQUENCE (SIZE (1..16)) OF INTEGER (0..100)
using sl_cbr_levels_cfg_r14_l = bounded_array<uint8_t, 16>;

// SL-CBR-PSSCH-TxConfig-r14 ::= SEQUENCE
struct sl_cbr_pssch_tx_cfg_r14_s {
  uint16_t                 cr_limit_r14 = 0;
  sl_pssch_tx_params_r14_s tx_params_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscConfigOtherInterFreq-r13 ::= SEQUENCE
struct sl_disc_cfg_other_inter_freq_r13_s {
  bool                           tx_pwr_info_r13_present        = false;
  bool                           ref_carrier_common_r13_present = false;
  bool                           disc_sync_cfg_r13_present      = false;
  bool                           disc_cell_sel_info_r13_present = false;
  sl_disc_tx_pwr_info_list_r12_l tx_pwr_info_r13;
  sl_sync_cfg_list_nfreq_r13_l   disc_sync_cfg_r13;
  cell_sel_info_nfreq_r13_s      disc_cell_sel_info_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-HoppingConfigComm-r12 ::= SEQUENCE
struct sl_hop_cfg_comm_r12_s {
  struct num_subbands_r12_opts {
    enum options { ns1, ns2, ns4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<num_subbands_r12_opts> num_subbands_r12_e_;

  // member variables
  uint16_t            hop_param_r12 = 0;
  num_subbands_r12_e_ num_subbands_r12;
  uint8_t             rb_offset_r12 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-InterFreqInfoV2X-r14 ::= SEQUENCE
struct sl_inter_freq_info_v2x_r14_s {
  struct sl_bw_r14_opts {
    enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<sl_bw_r14_opts> sl_bw_r14_e_;
  struct add_spec_emission_v2x_r14_c_ {
    struct types_opts {
      enum options { add_spec_emission_r14, add_spec_emission_v1440, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    add_spec_emission_v2x_r14_c_() = default;
    add_spec_emission_v2x_r14_c_(const add_spec_emission_v2x_r14_c_& other);
    add_spec_emission_v2x_r14_c_& operator=(const add_spec_emission_v2x_r14_c_& other);
    ~add_spec_emission_v2x_r14_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& add_spec_emission_r14()
    {
      assert_choice_type("additionalSpectrumEmission-r14", type_.to_string(), "additionalSpectrumEmissionV2X-r14");
      return c.get<uint8_t>();
    }
    uint16_t& add_spec_emission_v1440()
    {
      assert_choice_type("additionalSpectrumEmission-v1440", type_.to_string(), "additionalSpectrumEmissionV2X-r14");
      return c.get<uint16_t>();
    }
    const uint8_t& add_spec_emission_r14() const
    {
      assert_choice_type("additionalSpectrumEmission-r14", type_.to_string(), "additionalSpectrumEmissionV2X-r14");
      return c.get<uint8_t>();
    }
    const uint16_t& add_spec_emission_v1440() const
    {
      assert_choice_type("additionalSpectrumEmission-v1440", type_.to_string(), "additionalSpectrumEmissionV2X-r14");
      return c.get<uint16_t>();
    }
    uint8_t& set_add_spec_emission_r14()
    {
      set(types::add_spec_emission_r14);
      return c.get<uint8_t>();
    }
    uint16_t& set_add_spec_emission_v1440()
    {
      set(types::add_spec_emission_v1440);
      return c.get<uint16_t>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };

  // member variables
  bool                       ext                         = false;
  bool                       plmn_id_list_r14_present    = false;
  bool                       sl_max_tx_pwr_r14_present   = false;
  bool                       sl_bw_r14_present           = false;
  bool                       v2x_sched_pool_r14_present  = false;
  bool                       v2x_ue_cfg_list_r14_present = false;
  plmn_id_list_l             plmn_id_list_r14;
  uint32_t                   v2x_comm_carrier_freq_r14 = 0;
  int8_t                     sl_max_tx_pwr_r14         = -30;
  sl_bw_r14_e_               sl_bw_r14;
  sl_comm_res_pool_v2x_r14_s v2x_sched_pool_r14;
  sl_v2x_ue_cfg_list_r14_l   v2x_ue_cfg_list_r14;
  // ...
  // group 0
  copy_ptr<add_spec_emission_v2x_r14_c_> add_spec_emission_v2x_r14;
  // group 1
  copy_ptr<sl_v2x_freq_sel_cfg_list_r15_l> v2x_freq_sel_cfg_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PPPR-Dest-CarrierFreq ::= SEQUENCE
struct sl_pppr_dest_carrier_freq_s {
  bool                               dest_info_list_r15_present            = false;
  bool                               allowed_carrier_freq_list_r15_present = false;
  sl_dest_info_list_r12_l            dest_info_list_r15;
  sl_allowed_carrier_freq_list_r15_s allowed_carrier_freq_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PeriodComm-r12 ::= ENUMERATED
struct sl_period_comm_r12_opts {
  enum options {
    sf40,
    sf60,
    sf70,
    sf80,
    sf120,
    sf140,
    sf160,
    sf240,
    sf280,
    sf320,
    spare6,
    spare5,
    spare4,
    spare3,
    spare2,
    spare,
    nulltype
  } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<sl_period_comm_r12_opts> sl_period_comm_r12_e;

// SL-ResourcesInterFreq-r13 ::= SEQUENCE
struct sl_res_inter_freq_r13_s {
  bool                            disc_rx_res_inter_freq_r13_present = false;
  bool                            disc_tx_res_inter_freq_r13_present = false;
  sl_disc_rx_pool_list_r12_l      disc_rx_res_inter_freq_r13;
  sl_disc_tx_res_inter_freq_r13_c disc_tx_res_inter_freq_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-SyncConfig-r12 ::= SEQUENCE
struct sl_sync_cfg_r12_s {
  struct tx_params_r12_s_ {
    bool                sync_info_reserved_r12_present = false;
    sl_tx_params_r12_s  sync_tx_params_r12;
    uint8_t             sync_tx_thresh_ic_r12 = 0;
    fixed_bitstring<19> sync_info_reserved_r12;
  };
  struct rx_params_ncell_r12_s_ {
    struct disc_sync_win_r12_opts {
      enum options { w1, w2, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<disc_sync_win_r12_opts> disc_sync_win_r12_e_;

    // member variables
    uint16_t             pci_r12 = 0;
    disc_sync_win_r12_e_ disc_sync_win_r12;
  };

  // member variables
  bool                   ext                         = false;
  bool                   tx_params_r12_present       = false;
  bool                   rx_params_ncell_r12_present = false;
  sl_cp_len_r12_e        sync_cp_len_r12;
  uint8_t                sync_offset_ind_r12 = 0;
  uint8_t                slssid_r12          = 0;
  tx_params_r12_s_       tx_params_r12;
  rx_params_ncell_r12_s_ rx_params_ncell_r12;
  // ...
  // group 0
  bool sync_tx_periodic_r13_present = false;
  // group 1
  bool    sync_offset_ind_v1430_present = false;
  bool    gnss_sync_r14_present         = false;
  uint8_t sync_offset_ind_v1430         = 40;
  // group 2
  bool    sync_offset_ind2_r14_present = false;
  bool    sync_offset_ind3_r14_present = false;
  uint8_t sync_offset_ind2_r14         = 0;
  uint8_t sync_offset_ind3_r14         = 0;
  // group 3
  bool slss_tx_disabled_r15_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SS-RSSI-Measurement-r15 ::= SEQUENCE
struct ss_rssi_meas_r15_s {
  bounded_bitstring<1, 80> meas_slots_r15;
  uint8_t                  end_symbol_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v10l0-IEs ::= SEQUENCE
struct sib_type1_v10l0_ies_s {
  bool                         freq_band_info_v10l0_present       = false;
  bool                         multi_band_info_list_v10l0_present = false;
  bool                         non_crit_ext_present               = false;
  ns_pmax_list_v10l0_l         freq_band_info_v10l0;
  multi_band_info_list_v10l0_l multi_band_info_list_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v1310-IEs ::= SEQUENCE
struct sib_type1_v1310_ies_s {
  struct bw_reduced_access_related_info_r13_s_ {
    struct si_win_len_br_r13_opts {
      enum options { ms20, ms40, ms60, ms80, ms120, ms160, ms200, spare, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<si_win_len_br_r13_opts> si_win_len_br_r13_e_;
    struct si_repeat_pattern_r13_opts {
      enum options { every_rf, every2nd_rf, every4th_rf, every8th_rf, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<si_repeat_pattern_r13_opts> si_repeat_pattern_r13_e_;
    struct fdd_dl_or_tdd_sf_bitmap_br_r13_c_ {
      struct types_opts {
        enum options { sf_pattern10_r13, sf_pattern40_r13, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      fdd_dl_or_tdd_sf_bitmap_br_r13_c_() = default;
      fdd_dl_or_tdd_sf_bitmap_br_r13_c_(const fdd_dl_or_tdd_sf_bitmap_br_r13_c_& other);
      fdd_dl_or_tdd_sf_bitmap_br_r13_c_& operator=(const fdd_dl_or_tdd_sf_bitmap_br_r13_c_& other);
      ~fdd_dl_or_tdd_sf_bitmap_br_r13_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      fixed_bitstring<10>& sf_pattern10_r13()
      {
        assert_choice_type("subframePattern10-r13", type_.to_string(), "fdd-DownlinkOrTddSubframeBitmapBR-r13");
        return c.get<fixed_bitstring<10> >();
      }
      fixed_bitstring<40>& sf_pattern40_r13()
      {
        assert_choice_type("subframePattern40-r13", type_.to_string(), "fdd-DownlinkOrTddSubframeBitmapBR-r13");
        return c.get<fixed_bitstring<40> >();
      }
      const fixed_bitstring<10>& sf_pattern10_r13() const
      {
        assert_choice_type("subframePattern10-r13", type_.to_string(), "fdd-DownlinkOrTddSubframeBitmapBR-r13");
        return c.get<fixed_bitstring<10> >();
      }
      const fixed_bitstring<40>& sf_pattern40_r13() const
      {
        assert_choice_type("subframePattern40-r13", type_.to_string(), "fdd-DownlinkOrTddSubframeBitmapBR-r13");
        return c.get<fixed_bitstring<40> >();
      }
      fixed_bitstring<10>& set_sf_pattern10_r13()
      {
        set(types::sf_pattern10_r13);
        return c.get<fixed_bitstring<10> >();
      }
      fixed_bitstring<40>& set_sf_pattern40_r13()
      {
        set(types::sf_pattern40_r13);
        return c.get<fixed_bitstring<40> >();
      }

    private:
      types                                 type_;
      choice_buffer_t<fixed_bitstring<40> > c;

      void destroy_();
    };
    struct si_hop_cfg_common_r13_opts {
      enum options { on, off, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<si_hop_cfg_common_r13_opts> si_hop_cfg_common_r13_e_;

    // member variables
    bool                              sched_info_list_br_r13_present         = false;
    bool                              fdd_dl_or_tdd_sf_bitmap_br_r13_present = false;
    bool                              fdd_ul_sf_bitmap_br_r13_present        = false;
    bool                              si_validity_time_r13_present           = false;
    bool                              sys_info_value_tag_list_r13_present    = false;
    si_win_len_br_r13_e_              si_win_len_br_r13;
    si_repeat_pattern_r13_e_          si_repeat_pattern_r13;
    sched_info_list_br_r13_l          sched_info_list_br_r13;
    fdd_dl_or_tdd_sf_bitmap_br_r13_c_ fdd_dl_or_tdd_sf_bitmap_br_r13;
    fixed_bitstring<10>               fdd_ul_sf_bitmap_br_r13;
    uint8_t                           start_symbol_br_r13 = 1;
    si_hop_cfg_common_r13_e_          si_hop_cfg_common_r13;
    sys_info_value_tag_list_r13_l     sys_info_value_tag_list_r13;
  };

  // member variables
  bool                                  hyper_sfn_r13_present                      = false;
  bool                                  edrx_allowed_r13_present                   = false;
  bool                                  cell_sel_info_ce_r13_present               = false;
  bool                                  bw_reduced_access_related_info_r13_present = false;
  bool                                  non_crit_ext_present                       = false;
  fixed_bitstring<10>                   hyper_sfn_r13;
  cell_sel_info_ce_r13_s                cell_sel_info_ce_r13;
  bw_reduced_access_related_info_r13_s_ bw_reduced_access_related_info_r13;
  sib_type1_v1320_ies_s                 non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType2-v9i0-IEs ::= SEQUENCE
struct sib_type2_v9i0_ies_s {
  bool          non_crit_ext_present = false;
  bool          dummy_present        = false;
  dyn_octstring non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5-v10j0-IEs ::= SEQUENCE
struct sib_type5_v10j0_ies_s {
  using inter_freq_carrier_freq_list_v10j0_l_ = dyn_array<inter_freq_carrier_freq_info_v10j0_s>;

  // member variables
  bool                                  inter_freq_carrier_freq_list_v10j0_present = false;
  bool                                  non_crit_ext_present                       = false;
  inter_freq_carrier_freq_list_v10j0_l_ inter_freq_carrier_freq_list_v10j0;
  sib_type5_v10l0_ies_s                 non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ThresholdListNR-r15 ::= SEQUENCE
struct thres_list_nr_r15_s {
  bool    nr_rsrp_r15_present = false;
  bool    nr_rsrq_r15_present = false;
  bool    nr_sinr_r15_present = false;
  uint8_t nr_rsrp_r15         = 0;
  uint8_t nr_rsrq_r15         = 0;
  uint8_t nr_sinr_r15         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UAC-BarringPerCatList-r15 ::= SEQUENCE (SIZE (1..63)) OF UAC-BarringPerCat-r15
using uac_barr_per_cat_list_r15_l = dyn_array<uac_barr_per_cat_r15_s>;

// UDT-Restricting-r13 ::= SEQUENCE
struct udt_restricting_r13_s {
  struct udt_restricting_time_r13_opts {
    enum options { s4, s8, s16, s32, s64, s128, s256, s512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<udt_restricting_time_r13_opts> udt_restricting_time_r13_e_;

  // member variables
  bool                        udt_restricting_r13_present      = false;
  bool                        udt_restricting_time_r13_present = false;
  udt_restricting_time_r13_e_ udt_restricting_time_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UL-ReferenceSignalsPUSCH ::= SEQUENCE
struct ul_ref_sigs_pusch_s {
  bool    group_hop_enabled  = false;
  uint8_t group_assign_pusch = 0;
  bool    seq_hop_enabled    = false;
  uint8_t cyclic_shift       = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-Identifiers-r12 ::= SEQUENCE
struct wlan_ids_r12_s {
  bool               ext                = false;
  bool               ssid_r12_present   = false;
  bool               bssid_r12_present  = false;
  bool               hessid_r12_present = false;
  dyn_octstring      ssid_r12;
  fixed_octstring<6> bssid_r12;
  fixed_octstring<6> hessid_r12;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-backhaulRate-r12 ::= ENUMERATED
struct wlan_backhaul_rate_r12_opts {
  enum options {
    r0,
    r4,
    r8,
    r16,
    r32,
    r64,
    r128,
    r256,
    r512,
    r1024,
    r2048,
    r4096,
    r8192,
    r16384,
    r32768,
    r65536,
    r131072,
    r262144,
    r524288,
    r1048576,
    r2097152,
    r4194304,
    r8388608,
    r16777216,
    r33554432,
    r67108864,
    r134217728,
    r268435456,
    r536870912,
    r1073741824,
    r2147483648,
    r4294967296,
    nulltype
  } value;
  typedef uint64_t number_type;

  std::string to_string() const;
  uint64_t    to_number() const;
};
typedef enumerated<wlan_backhaul_rate_r12_opts> wlan_backhaul_rate_r12_e;

// AC-BarringPerPLMN-r12 ::= SEQUENCE
struct ac_barr_per_plmn_r12_s {
  struct ac_barr_info_r12_s_ {
    bool          ac_barr_for_mo_sig_r12_present  = false;
    bool          ac_barr_for_mo_data_r12_present = false;
    bool          ac_barr_for_emergency_r12       = false;
    ac_barr_cfg_s ac_barr_for_mo_sig_r12;
    ac_barr_cfg_s ac_barr_for_mo_data_r12;
  };

  // member variables
  bool                ac_barr_info_r12_present                 = false;
  bool                ac_barr_skip_for_mmtel_voice_r12_present = false;
  bool                ac_barr_skip_for_mmtel_video_r12_present = false;
  bool                ac_barr_skip_for_sms_r12_present         = false;
  bool                ac_barr_for_csfb_r12_present             = false;
  bool                ssac_barr_for_mmtel_voice_r12_present    = false;
  bool                ssac_barr_for_mmtel_video_r12_present    = false;
  uint8_t             plmn_id_idx_r12                          = 1;
  ac_barr_info_r12_s_ ac_barr_info_r12;
  ac_barr_cfg_s       ac_barr_for_csfb_r12;
  ac_barr_cfg_s       ssac_barr_for_mmtel_voice_r12;
  ac_barr_cfg_s       ssac_barr_for_mmtel_video_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ACDC-BarringPerPLMN-r13 ::= SEQUENCE
struct acdc_barr_per_plmn_r13_s {
  uint8_t                           plmn_id_idx_r13         = 1;
  bool                              acdc_only_for_hplmn_r13 = false;
  barr_per_acdc_category_list_r13_l barr_per_acdc_category_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-Config ::= SEQUENCE
struct bcch_cfg_s {
  struct mod_period_coeff_opts {
    enum options { n2, n4, n8, n16, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<mod_period_coeff_opts> mod_period_coeff_e_;

  // member variables
  mod_period_coeff_e_ mod_period_coeff;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-Config-v1310 ::= SEQUENCE
struct bcch_cfg_v1310_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CIOT-OptimisationPLMN-r13 ::= SEQUENCE
struct ciot_optim_plmn_r13_s {
  bool up_cio_t_eps_optim_r13_present         = false;
  bool cp_cio_t_eps_optim_r13_present         = false;
  bool attach_without_pdn_connect_r13_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqInfoUTRA-FDD-v8h0 ::= SEQUENCE
struct carrier_freq_info_utra_fdd_v8h0_s {
  using multi_band_info_list_l_ = bounded_array<uint8_t, 8>;

  // member variables
  bool                    multi_band_info_list_present = false;
  multi_band_info_list_l_ multi_band_info_list;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqNR-r15 ::= SEQUENCE
struct carrier_freq_nr_r15_s {
  struct subcarrier_spacing_ssb_r15_opts {
    enum options { khz15, khz30, khz120, khz240, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<subcarrier_spacing_ssb_r15_opts> subcarrier_spacing_ssb_r15_e_;
  struct thresh_x_q_r15_s_ {
    uint8_t thresh_x_high_q_r15 = 0;
    uint8_t thresh_x_low_q_r15  = 0;
  };

  // member variables
  bool                          ext                                  = false;
  bool                          multi_band_info_list_r15_present     = false;
  bool                          multi_band_info_list_sul_r15_present = false;
  bool                          meas_timing_cfg_r15_present          = false;
  bool                          ss_rssi_meas_r15_present             = false;
  bool                          cell_resel_prio_r15_present          = false;
  bool                          cell_resel_sub_prio_r15_present      = false;
  bool                          thresh_x_q_r15_present               = false;
  bool                          q_rx_lev_min_sul_r15_present         = false;
  bool                          ns_pmax_list_nr_r15_present          = false;
  bool                          q_qual_min_r15_present               = false;
  bool                          max_rs_idx_cell_qual_r15_present     = false;
  bool                          thresh_rs_idx_r15_present            = false;
  uint32_t                      carrier_freq_r15                     = 0;
  multi_freq_band_list_nr_r15_l multi_band_info_list_r15;
  multi_freq_band_list_nr_r15_l multi_band_info_list_sul_r15;
  mtc_ssb_nr_r15_s              meas_timing_cfg_r15;
  subcarrier_spacing_ssb_r15_e_ subcarrier_spacing_ssb_r15;
  ss_rssi_meas_r15_s            ss_rssi_meas_r15;
  uint8_t                       cell_resel_prio_r15 = 0;
  cell_resel_sub_prio_r13_e     cell_resel_sub_prio_r15;
  uint8_t                       thresh_x_high_r15 = 0;
  uint8_t                       thresh_x_low_r15  = 0;
  thresh_x_q_r15_s_             thresh_x_q_r15;
  int8_t                        q_rx_lev_min_r15     = -70;
  int8_t                        q_rx_lev_min_sul_r15 = -70;
  int8_t                        p_max_nr_r15         = -30;
  ns_pmax_list_nr_r15_l         ns_pmax_list_nr_r15;
  int8_t                        q_qual_min_r15               = -34;
  bool                          derive_ssb_idx_from_cell_r15 = false;
  uint8_t                       max_rs_idx_cell_qual_r15     = 1;
  thres_list_nr_r15_s           thresh_rs_idx_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqUTRA-FDD ::= SEQUENCE
struct carrier_freq_utra_fdd_s {
  struct thresh_x_q_r9_s_ {
    uint8_t thresh_x_high_q_r9 = 0;
    uint8_t thresh_x_low_q_r9  = 0;
  };

  // member variables
  bool     ext                     = false;
  bool     cell_resel_prio_present = false;
  uint16_t carrier_freq            = 0;
  uint8_t  cell_resel_prio         = 0;
  uint8_t  thresh_x_high           = 0;
  uint8_t  thresh_x_low            = 0;
  int8_t   q_rx_lev_min            = -60;
  int8_t   p_max_utra              = -50;
  int8_t   q_qual_min              = -24;
  // ...
  // group 0
  copy_ptr<thresh_x_q_r9_s_> thresh_x_q_r9;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqUTRA-FDD-Ext-r12 ::= SEQUENCE
struct carrier_freq_utra_fdd_ext_r12_s {
  struct thresh_x_q_r12_s_ {
    uint8_t thresh_x_high_q_r12 = 0;
    uint8_t thresh_x_low_q_r12  = 0;
  };
  using multi_band_info_list_r12_l_ = bounded_array<uint8_t, 8>;

  // member variables
  bool                        ext                                  = false;
  bool                        cell_resel_prio_r12_present          = false;
  bool                        thresh_x_q_r12_present               = false;
  bool                        multi_band_info_list_r12_present     = false;
  bool                        reduced_meas_performance_r12_present = false;
  uint16_t                    carrier_freq_r12                     = 0;
  uint8_t                     cell_resel_prio_r12                  = 0;
  uint8_t                     thresh_x_high_r12                    = 0;
  uint8_t                     thresh_x_low_r12                     = 0;
  int8_t                      q_rx_lev_min_r12                     = -60;
  int8_t                      p_max_utra_r12                       = -50;
  int8_t                      q_qual_min_r12                       = -24;
  thresh_x_q_r12_s_           thresh_x_q_r12;
  multi_band_info_list_r12_l_ multi_band_info_list_r12;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqUTRA-TDD ::= SEQUENCE
struct carrier_freq_utra_tdd_s {
  bool     ext                     = false;
  bool     cell_resel_prio_present = false;
  uint16_t carrier_freq            = 0;
  uint8_t  cell_resel_prio         = 0;
  uint8_t  thresh_x_high           = 0;
  uint8_t  thresh_x_low            = 0;
  int8_t   q_rx_lev_min            = -60;
  int8_t   p_max_utra              = -50;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqUTRA-TDD-r12 ::= SEQUENCE
struct carrier_freq_utra_tdd_r12_s {
  bool     ext                                  = false;
  bool     cell_resel_prio_r12_present          = false;
  bool     reduced_meas_performance_r12_present = false;
  uint16_t carrier_freq_r12                     = 0;
  uint8_t  cell_resel_prio_r12                  = 0;
  uint8_t  thresh_x_high_r12                    = 0;
  uint8_t  thresh_x_low_r12                     = 0;
  int8_t   q_rx_lev_min_r12                     = -60;
  int8_t   p_max_utra_r12                       = -50;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqsInfoGERAN ::= SEQUENCE
struct carrier_freqs_info_geran_s {
  struct common_info_s_ {
    bool               cell_resel_prio_present = false;
    bool               p_max_geran_present     = false;
    uint8_t            cell_resel_prio         = 0;
    fixed_bitstring<8> ncc_permitted;
    uint8_t            q_rx_lev_min  = 0;
    uint8_t            p_max_geran   = 0;
    uint8_t            thresh_x_high = 0;
    uint8_t            thresh_x_low  = 0;
  };

  // member variables
  bool                  ext = false;
  carrier_freqs_geran_s carrier_freqs;
  common_info_s_        common_info;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfo-v1130 ::= SEQUENCE
struct cell_sel_info_v1130_s {
  int8_t q_qual_min_wb_r11 = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EAB-Config-r11 ::= SEQUENCE
struct eab_cfg_r11_s {
  struct eab_category_r11_opts {
    enum options { a, b, c, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<eab_category_r11_opts> eab_category_r11_e_;

  // member variables
  eab_category_r11_e_ eab_category_r11;
  fixed_bitstring<10> eab_barr_bitmap_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EUTRA-CarrierList-r15 ::= SEQUENCE (SIZE (1..8)) OF MeasIdleCarrierEUTRA-r15
using eutra_carrier_list_r15_l = dyn_array<meas_idle_carrier_eutra_r15_s>;

// FreqHoppingParameters-r13 ::= SEQUENCE
struct freq_hop_params_r13_s {
  struct dummy_opts {
    enum options { nb2, nb4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<dummy_opts> dummy_e_;
  struct dummy2_c_ {
    struct interv_fdd_r13_opts {
      enum options { int1, int2, int4, int8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
    struct interv_tdd_r13_opts {
      enum options { int1, int5, int10, int20, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
    struct types_opts {
      enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    dummy2_c_() = default;
    dummy2_c_(const dummy2_c_& other);
    dummy2_c_& operator=(const dummy2_c_& other);
    ~dummy2_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    interv_fdd_r13_e_& interv_fdd_r13()
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "dummy2");
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& interv_tdd_r13()
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "dummy2");
      return c.get<interv_tdd_r13_e_>();
    }
    const interv_fdd_r13_e_& interv_fdd_r13() const
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "dummy2");
      return c.get<interv_fdd_r13_e_>();
    }
    const interv_tdd_r13_e_& interv_tdd_r13() const
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "dummy2");
      return c.get<interv_tdd_r13_e_>();
    }
    interv_fdd_r13_e_& set_interv_fdd_r13()
    {
      set(types::interv_fdd_r13);
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& set_interv_tdd_r13()
    {
      set(types::interv_tdd_r13);
      return c.get<interv_tdd_r13_e_>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };
  struct dummy3_c_ {
    struct interv_fdd_r13_opts {
      enum options { int2, int4, int8, int16, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
    struct interv_tdd_r13_opts {
      enum options { int5, int10, int20, int40, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
    struct types_opts {
      enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    dummy3_c_() = default;
    dummy3_c_(const dummy3_c_& other);
    dummy3_c_& operator=(const dummy3_c_& other);
    ~dummy3_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    interv_fdd_r13_e_& interv_fdd_r13()
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "dummy3");
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& interv_tdd_r13()
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "dummy3");
      return c.get<interv_tdd_r13_e_>();
    }
    const interv_fdd_r13_e_& interv_fdd_r13() const
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "dummy3");
      return c.get<interv_fdd_r13_e_>();
    }
    const interv_tdd_r13_e_& interv_tdd_r13() const
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "dummy3");
      return c.get<interv_tdd_r13_e_>();
    }
    interv_fdd_r13_e_& set_interv_fdd_r13()
    {
      set(types::interv_fdd_r13);
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& set_interv_tdd_r13()
    {
      set(types::interv_tdd_r13);
      return c.get<interv_tdd_r13_e_>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };
  struct interv_ul_hop_cfg_common_mode_a_r13_c_ {
    struct interv_fdd_r13_opts {
      enum options { int1, int2, int4, int8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
    struct interv_tdd_r13_opts {
      enum options { int1, int5, int10, int20, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
    struct types_opts {
      enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    interv_ul_hop_cfg_common_mode_a_r13_c_() = default;
    interv_ul_hop_cfg_common_mode_a_r13_c_(const interv_ul_hop_cfg_common_mode_a_r13_c_& other);
    interv_ul_hop_cfg_common_mode_a_r13_c_& operator=(const interv_ul_hop_cfg_common_mode_a_r13_c_& other);
    ~interv_ul_hop_cfg_common_mode_a_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    interv_fdd_r13_e_& interv_fdd_r13()
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeA-r13");
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& interv_tdd_r13()
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeA-r13");
      return c.get<interv_tdd_r13_e_>();
    }
    const interv_fdd_r13_e_& interv_fdd_r13() const
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeA-r13");
      return c.get<interv_fdd_r13_e_>();
    }
    const interv_tdd_r13_e_& interv_tdd_r13() const
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeA-r13");
      return c.get<interv_tdd_r13_e_>();
    }
    interv_fdd_r13_e_& set_interv_fdd_r13()
    {
      set(types::interv_fdd_r13);
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& set_interv_tdd_r13()
    {
      set(types::interv_tdd_r13);
      return c.get<interv_tdd_r13_e_>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };
  struct interv_ul_hop_cfg_common_mode_b_r13_c_ {
    struct interv_fdd_r13_opts {
      enum options { int2, int4, int8, int16, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_fdd_r13_opts> interv_fdd_r13_e_;
    struct interv_tdd_r13_opts {
      enum options { int5, int10, int20, int40, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<interv_tdd_r13_opts> interv_tdd_r13_e_;
    struct types_opts {
      enum options { interv_fdd_r13, interv_tdd_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    interv_ul_hop_cfg_common_mode_b_r13_c_() = default;
    interv_ul_hop_cfg_common_mode_b_r13_c_(const interv_ul_hop_cfg_common_mode_b_r13_c_& other);
    interv_ul_hop_cfg_common_mode_b_r13_c_& operator=(const interv_ul_hop_cfg_common_mode_b_r13_c_& other);
    ~interv_ul_hop_cfg_common_mode_b_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    interv_fdd_r13_e_& interv_fdd_r13()
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeB-r13");
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& interv_tdd_r13()
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeB-r13");
      return c.get<interv_tdd_r13_e_>();
    }
    const interv_fdd_r13_e_& interv_fdd_r13() const
    {
      assert_choice_type("interval-FDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeB-r13");
      return c.get<interv_fdd_r13_e_>();
    }
    const interv_tdd_r13_e_& interv_tdd_r13() const
    {
      assert_choice_type("interval-TDD-r13", type_.to_string(), "interval-ULHoppingConfigCommonModeB-r13");
      return c.get<interv_tdd_r13_e_>();
    }
    interv_fdd_r13_e_& set_interv_fdd_r13()
    {
      set(types::interv_fdd_r13);
      return c.get<interv_fdd_r13_e_>();
    }
    interv_tdd_r13_e_& set_interv_tdd_r13()
    {
      set(types::interv_tdd_r13);
      return c.get<interv_tdd_r13_e_>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };

  // member variables
  bool                                   dummy_present                               = false;
  bool                                   dummy2_present                              = false;
  bool                                   dummy3_present                              = false;
  bool                                   interv_ul_hop_cfg_common_mode_a_r13_present = false;
  bool                                   interv_ul_hop_cfg_common_mode_b_r13_present = false;
  bool                                   dummy4_present                              = false;
  dummy_e_                               dummy;
  dummy2_c_                              dummy2;
  dummy3_c_                              dummy3;
  interv_ul_hop_cfg_common_mode_a_r13_c_ interv_ul_hop_cfg_common_mode_a_r13;
  interv_ul_hop_cfg_common_mode_b_r13_c_ interv_ul_hop_cfg_common_mode_b_r13;
  uint8_t                                dummy4 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// HighSpeedConfig-r14 ::= SEQUENCE
struct high_speed_cfg_r14_s {
  bool high_speed_enhanced_meas_flag_r14_present  = false;
  bool high_speed_enhanced_demod_flag_r14_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// HighSpeedConfig-v1530 ::= SEQUENCE
struct high_speed_cfg_v1530_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo ::= SEQUENCE
struct inter_freq_carrier_freq_info_s {
  struct thresh_x_q_r9_s_ {
    uint8_t thresh_x_high_q_r9 = 0;
    uint8_t thresh_x_low_q_r9  = 0;
  };

  // member variables
  bool                         ext                                = false;
  bool                         p_max_present                      = false;
  bool                         t_resel_eutra_sf_present           = false;
  bool                         cell_resel_prio_present            = false;
  bool                         q_offset_freq_present              = false;
  bool                         inter_freq_neigh_cell_list_present = false;
  bool                         inter_freq_black_cell_list_present = false;
  uint32_t                     dl_carrier_freq                    = 0;
  int8_t                       q_rx_lev_min                       = -70;
  int8_t                       p_max                              = -30;
  uint8_t                      t_resel_eutra                      = 0;
  speed_state_scale_factors_s  t_resel_eutra_sf;
  uint8_t                      thresh_x_high = 0;
  uint8_t                      thresh_x_low  = 0;
  allowed_meas_bw_e            allowed_meas_bw;
  bool                         presence_ant_port1 = false;
  uint8_t                      cell_resel_prio    = 0;
  fixed_bitstring<2>           neigh_cell_cfg;
  q_offset_range_e             q_offset_freq;
  inter_freq_neigh_cell_list_l inter_freq_neigh_cell_list;
  inter_freq_black_cell_list_l inter_freq_black_cell_list;
  // ...
  // group 0
  bool                       q_qual_min_r9_present = false;
  int8_t                     q_qual_min_r9         = -34;
  copy_ptr<thresh_x_q_r9_s_> thresh_x_q_r9;
  // group 1
  bool   q_qual_min_wb_r11_present = false;
  int8_t q_qual_min_wb_r11         = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-r12 ::= SEQUENCE
struct inter_freq_carrier_freq_info_r12_s {
  struct thresh_x_q_r12_s_ {
    uint8_t thresh_x_high_q_r12 = 0;
    uint8_t thresh_x_low_q_r12  = 0;
  };

  // member variables
  bool                         ext                                        = false;
  bool                         p_max_r12_present                          = false;
  bool                         t_resel_eutra_sf_r12_present               = false;
  bool                         cell_resel_prio_r12_present                = false;
  bool                         q_offset_freq_r12_present                  = false;
  bool                         inter_freq_neigh_cell_list_r12_present     = false;
  bool                         inter_freq_black_cell_list_r12_present     = false;
  bool                         q_qual_min_r12_present                     = false;
  bool                         thresh_x_q_r12_present                     = false;
  bool                         q_qual_min_wb_r12_present                  = false;
  bool                         multi_band_info_list_r12_present           = false;
  bool                         reduced_meas_performance_r12_present       = false;
  bool                         q_qual_min_rsrq_on_all_symbols_r12_present = false;
  uint32_t                     dl_carrier_freq_r12                        = 0;
  int8_t                       q_rx_lev_min_r12                           = -70;
  int8_t                       p_max_r12                                  = -30;
  uint8_t                      t_resel_eutra_r12                          = 0;
  speed_state_scale_factors_s  t_resel_eutra_sf_r12;
  uint8_t                      thresh_x_high_r12 = 0;
  uint8_t                      thresh_x_low_r12  = 0;
  allowed_meas_bw_e            allowed_meas_bw_r12;
  bool                         presence_ant_port1_r12 = false;
  uint8_t                      cell_resel_prio_r12    = 0;
  fixed_bitstring<2>           neigh_cell_cfg_r12;
  q_offset_range_e             q_offset_freq_r12;
  inter_freq_neigh_cell_list_l inter_freq_neigh_cell_list_r12;
  inter_freq_black_cell_list_l inter_freq_black_cell_list_r12;
  int8_t                       q_qual_min_r12 = -34;
  thresh_x_q_r12_s_            thresh_x_q_r12;
  int8_t                       q_qual_min_wb_r12 = -34;
  multi_band_info_list_r11_l   multi_band_info_list_r12;
  int8_t                       q_qual_min_rsrq_on_all_symbols_r12 = -34;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-v1250 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v1250_s {
  bool   reduced_meas_performance_r12_present       = false;
  bool   q_qual_min_rsrq_on_all_symbols_r12_present = false;
  int8_t q_qual_min_rsrq_on_all_symbols_r12         = -34;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-v1310 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v1310_s {
  bool                         cell_resel_sub_prio_r13_present    = false;
  bool                         redist_inter_freq_info_r13_present = false;
  bool                         cell_sel_info_ce_r13_present       = false;
  bool                         t_resel_eutra_ce_r13_present       = false;
  cell_resel_sub_prio_r13_e    cell_resel_sub_prio_r13;
  redist_inter_freq_info_r13_s redist_inter_freq_info_r13;
  cell_sel_info_ce_r13_s       cell_sel_info_ce_r13;
  uint8_t                      t_resel_eutra_ce_r13 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-v1350 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v1350_s {
  bool                    cell_sel_info_ce1_r13_present = false;
  cell_sel_info_ce1_r13_s cell_sel_info_ce1_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-v1530 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v1530_s {
  bool                                  inter_freq_neigh_hsdn_cell_list_r15_present = false;
  bool                                  cell_sel_info_ce_v1530_present              = false;
  bool                                  hsdn_ind_r15                                = false;
  inter_freq_neigh_hsdn_cell_list_r15_l inter_freq_neigh_hsdn_cell_list_r15;
  cell_sel_info_ce_v1530_s              cell_sel_info_ce_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqInfo-v8h0 ::= SEQUENCE
struct inter_freq_carrier_freq_info_v8h0_s {
  bool                   multi_band_info_list_present = false;
  multi_band_info_list_l multi_band_info_list;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// IntraFreqNeighCellInfo ::= SEQUENCE
struct intra_freq_neigh_cell_info_s {
  bool             ext = false;
  uint16_t         pci = 0;
  q_offset_range_e q_offset_cell;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBMS-CarrierType-r14 ::= SEQUENCE
struct mbms_carrier_type_r14_s {
  struct carrier_type_r14_opts {
    enum options { mbms, fembms_mixed, fembms_ded, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<carrier_type_r14_opts> carrier_type_r14_e_;

  // member variables
  bool                frame_offset_r14_present = false;
  carrier_type_r14_e_ carrier_type_r14;
  uint8_t             frame_offset_r14 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBMS-SAI-InterFreq-r11 ::= SEQUENCE
struct mbms_sai_inter_freq_r11_s {
  uint32_t            dl_carrier_freq_r11 = 0;
  mbms_sai_list_r11_l mbms_sai_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBMS-SAI-InterFreq-v1140 ::= SEQUENCE
struct mbms_sai_inter_freq_v1140_s {
  bool                       multi_band_info_list_r11_present = false;
  multi_band_info_list_r11_l multi_band_info_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBSFN-AreaInfo-r9 ::= SEQUENCE
struct mbsfn_area_info_r9_s {
  struct non_mbsfn_region_len_opts {
    enum options { s1, s2, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<non_mbsfn_region_len_opts> non_mbsfn_region_len_e_;
  struct mcch_cfg_r9_s_ {
    struct mcch_repeat_period_r9_opts {
      enum options { rf32, rf64, rf128, rf256, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<mcch_repeat_period_r9_opts> mcch_repeat_period_r9_e_;
    struct mcch_mod_period_r9_opts {
      enum options { rf512, rf1024, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<mcch_mod_period_r9_opts> mcch_mod_period_r9_e_;
    struct sig_mcs_r9_opts {
      enum options { n2, n7, n13, n19, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<sig_mcs_r9_opts> sig_mcs_r9_e_;

    // member variables
    mcch_repeat_period_r9_e_ mcch_repeat_period_r9;
    uint8_t                  mcch_offset_r9 = 0;
    mcch_mod_period_r9_e_    mcch_mod_period_r9;
    fixed_bitstring<6>       sf_alloc_info_r9;
    sig_mcs_r9_e_            sig_mcs_r9;
  };
  struct mcch_cfg_r14_s_ {
    struct mcch_repeat_period_v1430_opts {
      enum options { rf1, rf2, rf4, rf8, rf16, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<mcch_repeat_period_v1430_opts> mcch_repeat_period_v1430_e_;
    struct mcch_mod_period_v1430_opts {
      enum options { rf1, rf2, rf4, rf8, rf16, rf32, rf64, rf128, rf256, spare7, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<mcch_mod_period_v1430_opts> mcch_mod_period_v1430_e_;

    // member variables
    bool                        mcch_repeat_period_v1430_present = false;
    bool                        mcch_mod_period_v1430_present    = false;
    mcch_repeat_period_v1430_e_ mcch_repeat_period_v1430;
    mcch_mod_period_v1430_e_    mcch_mod_period_v1430;
  };
  struct subcarrier_spacing_mbms_r14_opts {
    enum options { khz_minus7dot5, khz_minus1dot25, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<subcarrier_spacing_mbms_r14_opts> subcarrier_spacing_mbms_r14_e_;

  // member variables
  bool                    ext              = false;
  uint16_t                mbsfn_area_id_r9 = 0;
  non_mbsfn_region_len_e_ non_mbsfn_region_len;
  uint8_t                 notif_ind_r9 = 0;
  mcch_cfg_r9_s_          mcch_cfg_r9;
  // ...
  // group 0
  bool                           subcarrier_spacing_mbms_r14_present = false;
  copy_ptr<mcch_cfg_r14_s_>      mcch_cfg_r14;
  subcarrier_spacing_mbms_r14_e_ subcarrier_spacing_mbms_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBSFN-SubframeConfig ::= SEQUENCE
struct mbsfn_sf_cfg_s {
  struct radioframe_alloc_period_opts {
    enum options { n1, n2, n4, n8, n16, n32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<radioframe_alloc_period_opts> radioframe_alloc_period_e_;
  struct sf_alloc_c_ {
    struct types_opts {
      enum options { one_frame, four_frames, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sf_alloc_c_() = default;
    sf_alloc_c_(const sf_alloc_c_& other);
    sf_alloc_c_& operator=(const sf_alloc_c_& other);
    ~sf_alloc_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<6>& one_frame()
    {
      assert_choice_type("oneFrame", type_.to_string(), "subframeAllocation");
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<24>& four_frames()
    {
      assert_choice_type("fourFrames", type_.to_string(), "subframeAllocation");
      return c.get<fixed_bitstring<24> >();
    }
    const fixed_bitstring<6>& one_frame() const
    {
      assert_choice_type("oneFrame", type_.to_string(), "subframeAllocation");
      return c.get<fixed_bitstring<6> >();
    }
    const fixed_bitstring<24>& four_frames() const
    {
      assert_choice_type("fourFrames", type_.to_string(), "subframeAllocation");
      return c.get<fixed_bitstring<24> >();
    }
    fixed_bitstring<6>& set_one_frame()
    {
      set(types::one_frame);
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<24>& set_four_frames()
    {
      set(types::four_frames);
      return c.get<fixed_bitstring<24> >();
    }

  private:
    types                                 type_;
    choice_buffer_t<fixed_bitstring<24> > c;

    void destroy_();
  };

  // member variables
  radioframe_alloc_period_e_ radioframe_alloc_period;
  uint8_t                    radioframe_alloc_offset = 0;
  sf_alloc_c_                sf_alloc;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBSFN-SubframeConfig-v1430 ::= SEQUENCE
struct mbsfn_sf_cfg_v1430_s {
  struct sf_alloc_v1430_c_ {
    struct types_opts {
      enum options { one_frame_v1430, four_frames_v1430, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sf_alloc_v1430_c_() = default;
    sf_alloc_v1430_c_(const sf_alloc_v1430_c_& other);
    sf_alloc_v1430_c_& operator=(const sf_alloc_v1430_c_& other);
    ~sf_alloc_v1430_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<2>& one_frame_v1430()
    {
      assert_choice_type("oneFrame-v1430", type_.to_string(), "subframeAllocation-v1430");
      return c.get<fixed_bitstring<2> >();
    }
    fixed_bitstring<8>& four_frames_v1430()
    {
      assert_choice_type("fourFrames-v1430", type_.to_string(), "subframeAllocation-v1430");
      return c.get<fixed_bitstring<8> >();
    }
    const fixed_bitstring<2>& one_frame_v1430() const
    {
      assert_choice_type("oneFrame-v1430", type_.to_string(), "subframeAllocation-v1430");
      return c.get<fixed_bitstring<2> >();
    }
    const fixed_bitstring<8>& four_frames_v1430() const
    {
      assert_choice_type("fourFrames-v1430", type_.to_string(), "subframeAllocation-v1430");
      return c.get<fixed_bitstring<8> >();
    }
    fixed_bitstring<2>& set_one_frame_v1430()
    {
      set(types::one_frame_v1430);
      return c.get<fixed_bitstring<2> >();
    }
    fixed_bitstring<8>& set_four_frames_v1430()
    {
      set(types::four_frames_v1430);
      return c.get<fixed_bitstring<8> >();
    }

  private:
    types                                type_;
    choice_buffer_t<fixed_bitstring<8> > c;

    void destroy_();
  };

  // member variables
  sf_alloc_v1430_c_ sf_alloc_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellListCDMA2000 ::= SEQUENCE (SIZE (1..16)) OF NeighCellCDMA2000
using neigh_cell_list_cdma2000_l = dyn_array<neigh_cell_cdma2000_s>;

// NeighCellListCDMA2000-v920 ::= SEQUENCE (SIZE (1..16)) OF NeighCellCDMA2000-v920
using neigh_cell_list_cdma2000_v920_l = dyn_array<neigh_cell_cdma2000_v920_s>;

// PCCH-Config ::= SEQUENCE
struct pcch_cfg_s {
  struct default_paging_cycle_opts {
    enum options { rf32, rf64, rf128, rf256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<default_paging_cycle_opts> default_paging_cycle_e_;
  struct nb_opts {
    enum options {
      four_t,
      two_t,
      one_t,
      half_t,
      quarter_t,
      one_eighth_t,
      one_sixteenth_t,
      one_thirty_second_t,
      nulltype
    } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<nb_opts> nb_e_;

  // member variables
  default_paging_cycle_e_ default_paging_cycle;
  nb_e_                   nb;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PCCH-Config-v1310 ::= SEQUENCE
struct pcch_cfg_v1310_s {
  struct mpdcch_num_repeat_paging_r13_opts {
    enum options { r1, r2, r4, r8, r16, r32, r64, r128, r256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<mpdcch_num_repeat_paging_r13_opts> mpdcch_num_repeat_paging_r13_e_;
  struct nb_v1310_opts {
    enum options { one64th_t, one128th_t, one256th_t, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<nb_v1310_opts> nb_v1310_e_;

  // member variables
  bool                            nb_v1310_present        = false;
  uint8_t                         paging_narrow_bands_r13 = 1;
  mpdcch_num_repeat_paging_r13_e_ mpdcch_num_repeat_paging_r13;
  nb_v1310_e_                     nb_v1310;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigCommon ::= SEQUENCE
struct pdsch_cfg_common_s {
  int8_t  ref_sig_pwr = -60;
  uint8_t p_b         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigCommon-v1310 ::= SEQUENCE
struct pdsch_cfg_common_v1310_s {
  struct pdsch_max_num_repeat_cemode_a_r13_opts {
    enum options { r16, r32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pdsch_max_num_repeat_cemode_a_r13_opts> pdsch_max_num_repeat_cemode_a_r13_e_;
  struct pdsch_max_num_repeat_cemode_b_r13_opts {
    enum options { r192, r256, r384, r512, r768, r1024, r1536, r2048, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<pdsch_max_num_repeat_cemode_b_r13_opts> pdsch_max_num_repeat_cemode_b_r13_e_;

  // member variables
  bool                                 pdsch_max_num_repeat_cemode_a_r13_present = false;
  bool                                 pdsch_max_num_repeat_cemode_b_r13_present = false;
  pdsch_max_num_repeat_cemode_a_r13_e_ pdsch_max_num_repeat_cemode_a_r13;
  pdsch_max_num_repeat_cemode_b_r13_e_ pdsch_max_num_repeat_cemode_b_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-Info-r15 ::= SEQUENCE
struct plmn_info_r15_s {
  bool upper_layer_ind_r15_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-Config-v1430 ::= SEQUENCE
struct prach_cfg_v1430_s {
  uint16_t root_seq_idx_high_speed_r14              = 0;
  uint8_t  zero_correlation_zone_cfg_high_speed_r14 = 0;
  uint8_t  prach_cfg_idx_high_speed_r14             = 0;
  uint8_t  prach_freq_offset_high_speed_r14         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-ConfigSIB ::= SEQUENCE
struct prach_cfg_sib_s {
  uint16_t         root_seq_idx = 0;
  prach_cfg_info_s prach_cfg_info;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-ConfigSIB-v1310 ::= SEQUENCE
struct prach_cfg_sib_v1310_s {
  struct mpdcch_start_sf_css_ra_r13_c_ {
    struct fdd_r13_opts {
      enum options { v1, v1dot5, v2, v2dot5, v4, v5, v8, v10, nulltype } value;
      typedef float number_type;

      std::string to_string() const;
      float       to_number() const;
      std::string to_number_string() const;
    };
    typedef enumerated<fdd_r13_opts> fdd_r13_e_;
    struct tdd_r13_opts {
      enum options { v1, v2, v4, v5, v8, v10, v20, spare, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tdd_r13_opts> tdd_r13_e_;
    struct types_opts {
      enum options { fdd_r13, tdd_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    mpdcch_start_sf_css_ra_r13_c_() = default;
    mpdcch_start_sf_css_ra_r13_c_(const mpdcch_start_sf_css_ra_r13_c_& other);
    mpdcch_start_sf_css_ra_r13_c_& operator=(const mpdcch_start_sf_css_ra_r13_c_& other);
    ~mpdcch_start_sf_css_ra_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fdd_r13_e_& fdd_r13()
    {
      assert_choice_type("fdd-r13", type_.to_string(), "mpdcch-startSF-CSS-RA-r13");
      return c.get<fdd_r13_e_>();
    }
    tdd_r13_e_& tdd_r13()
    {
      assert_choice_type("tdd-r13", type_.to_string(), "mpdcch-startSF-CSS-RA-r13");
      return c.get<tdd_r13_e_>();
    }
    const fdd_r13_e_& fdd_r13() const
    {
      assert_choice_type("fdd-r13", type_.to_string(), "mpdcch-startSF-CSS-RA-r13");
      return c.get<fdd_r13_e_>();
    }
    const tdd_r13_e_& tdd_r13() const
    {
      assert_choice_type("tdd-r13", type_.to_string(), "mpdcch-startSF-CSS-RA-r13");
      return c.get<tdd_r13_e_>();
    }
    fdd_r13_e_& set_fdd_r13()
    {
      set(types::fdd_r13);
      return c.get<fdd_r13_e_>();
    }
    tdd_r13_e_& set_tdd_r13()
    {
      set(types::tdd_r13);
      return c.get<tdd_r13_e_>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };

  // member variables
  bool                              mpdcch_start_sf_css_ra_r13_present = false;
  bool                              prach_hop_offset_r13_present       = false;
  rsrp_thress_prach_info_list_r13_l rsrp_thress_prach_info_list_r13;
  mpdcch_start_sf_css_ra_r13_c_     mpdcch_start_sf_css_ra_r13;
  uint8_t                           prach_hop_offset_r13 = 0;
  prach_params_list_ce_r13_l        prach_params_list_ce_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-ConfigSIB-v1530 ::= SEQUENCE
struct prach_cfg_sib_v1530_s {
  using edt_prach_params_list_ce_r15_l_ = dyn_array<edt_prach_params_ce_r15_s>;

  // member variables
  edt_prach_params_list_ce_r15_l_ edt_prach_params_list_ce_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigCommon ::= SEQUENCE
struct pucch_cfg_common_s {
  struct delta_pucch_shift_opts {
    enum options { ds1, ds2, ds3, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_pucch_shift_opts> delta_pucch_shift_e_;

  // member variables
  delta_pucch_shift_e_ delta_pucch_shift;
  uint8_t              nrb_cqi     = 0;
  uint8_t              ncs_an      = 0;
  uint16_t             n1_pucch_an = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigCommon-v1310 ::= SEQUENCE
struct pucch_cfg_common_v1310_s {
  struct pucch_num_repeat_ce_msg4_level0_r13_opts {
    enum options { n1, n2, n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_msg4_level0_r13_opts> pucch_num_repeat_ce_msg4_level0_r13_e_;
  struct pucch_num_repeat_ce_msg4_level1_r13_opts {
    enum options { n1, n2, n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_msg4_level1_r13_opts> pucch_num_repeat_ce_msg4_level1_r13_e_;
  struct pucch_num_repeat_ce_msg4_level2_r13_opts {
    enum options { n4, n8, n16, n32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_msg4_level2_r13_opts> pucch_num_repeat_ce_msg4_level2_r13_e_;
  struct pucch_num_repeat_ce_msg4_level3_r13_opts {
    enum options { n4, n8, n16, n32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_msg4_level3_r13_opts> pucch_num_repeat_ce_msg4_level3_r13_e_;

  // member variables
  bool                                   n1_pucch_an_info_list_r13_present           = false;
  bool                                   pucch_num_repeat_ce_msg4_level0_r13_present = false;
  bool                                   pucch_num_repeat_ce_msg4_level1_r13_present = false;
  bool                                   pucch_num_repeat_ce_msg4_level2_r13_present = false;
  bool                                   pucch_num_repeat_ce_msg4_level3_r13_present = false;
  n1_pucch_an_info_list_r13_l            n1_pucch_an_info_list_r13;
  pucch_num_repeat_ce_msg4_level0_r13_e_ pucch_num_repeat_ce_msg4_level0_r13;
  pucch_num_repeat_ce_msg4_level1_r13_e_ pucch_num_repeat_ce_msg4_level1_r13;
  pucch_num_repeat_ce_msg4_level2_r13_e_ pucch_num_repeat_ce_msg4_level2_r13;
  pucch_num_repeat_ce_msg4_level3_r13_e_ pucch_num_repeat_ce_msg4_level3_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigCommon-v1430 ::= SEQUENCE
struct pucch_cfg_common_v1430_s {
  struct pucch_num_repeat_ce_msg4_level3_r14_opts {
    enum options { n64, n128, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_msg4_level3_r14_opts> pucch_num_repeat_ce_msg4_level3_r14_e_;

  // member variables
  bool                                   pucch_num_repeat_ce_msg4_level3_r14_present = false;
  pucch_num_repeat_ce_msg4_level3_r14_e_ pucch_num_repeat_ce_msg4_level3_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigCommon ::= SEQUENCE
struct pusch_cfg_common_s {
  struct pusch_cfg_basic_s_ {
    struct hop_mode_opts {
      enum options { inter_sub_frame, intra_and_inter_sub_frame, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<hop_mode_opts> hop_mode_e_;

    // member variables
    uint8_t     n_sb = 1;
    hop_mode_e_ hop_mode;
    uint8_t     pusch_hop_offset = 0;
    bool        enable64_qam     = false;
  };

  // member variables
  pusch_cfg_basic_s_  pusch_cfg_basic;
  ul_ref_sigs_pusch_s ul_ref_sigs_pusch;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigCommon-v1270 ::= SEQUENCE
struct pusch_cfg_common_v1270_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigCommon-v1310 ::= SEQUENCE
struct pusch_cfg_common_v1310_s {
  struct pusch_max_num_repeat_cemode_a_r13_opts {
    enum options { r8, r16, r32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pusch_max_num_repeat_cemode_a_r13_opts> pusch_max_num_repeat_cemode_a_r13_e_;
  struct pusch_max_num_repeat_cemode_b_r13_opts {
    enum options { r192, r256, r384, r512, r768, r1024, r1536, r2048, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<pusch_max_num_repeat_cemode_b_r13_opts> pusch_max_num_repeat_cemode_b_r13_e_;

  // member variables
  bool                                 pusch_max_num_repeat_cemode_a_r13_present = false;
  bool                                 pusch_max_num_repeat_cemode_b_r13_present = false;
  bool                                 pusch_hop_offset_v1310_present            = false;
  pusch_max_num_repeat_cemode_a_r13_e_ pusch_max_num_repeat_cemode_a_r13;
  pusch_max_num_repeat_cemode_b_r13_e_ pusch_max_num_repeat_cemode_b_r13;
  uint8_t                              pusch_hop_offset_v1310 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-ConfigCommon ::= SEQUENCE
struct rach_cfg_common_s {
  struct preamb_info_s_ {
    struct nof_ra_preambs_opts {
      enum options { n4, n8, n12, n16, n20, n24, n28, n32, n36, n40, n44, n48, n52, n56, n60, n64, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<nof_ra_preambs_opts> nof_ra_preambs_e_;
    struct preambs_group_a_cfg_s_ {
      struct size_of_ra_preambs_group_a_opts {
        enum options { n4, n8, n12, n16, n20, n24, n28, n32, n36, n40, n44, n48, n52, n56, n60, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<size_of_ra_preambs_group_a_opts> size_of_ra_preambs_group_a_e_;
      struct msg_size_group_a_opts {
        enum options { b56, b144, b208, b256, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<msg_size_group_a_opts> msg_size_group_a_e_;
      struct msg_pwr_offset_group_b_opts {
        enum options { minusinfinity, db0, db5, db8, db10, db12, db15, db18, nulltype } value;
        typedef int8_t number_type;

        std::string to_string() const;
        int8_t      to_number() const;
      };
      typedef enumerated<msg_pwr_offset_group_b_opts> msg_pwr_offset_group_b_e_;

      // member variables
      bool                          ext = false;
      size_of_ra_preambs_group_a_e_ size_of_ra_preambs_group_a;
      msg_size_group_a_e_           msg_size_group_a;
      msg_pwr_offset_group_b_e_     msg_pwr_offset_group_b;
      // ...
    };

    // member variables
    bool                   preambs_group_a_cfg_present = false;
    nof_ra_preambs_e_      nof_ra_preambs;
    preambs_group_a_cfg_s_ preambs_group_a_cfg;
  };
  struct ra_supervision_info_s_ {
    struct ra_resp_win_size_opts {
      enum options { sf2, sf3, sf4, sf5, sf6, sf7, sf8, sf10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<ra_resp_win_size_opts> ra_resp_win_size_e_;
    struct mac_contention_resolution_timer_opts {
      enum options { sf8, sf16, sf24, sf32, sf40, sf48, sf56, sf64, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<mac_contention_resolution_timer_opts> mac_contention_resolution_timer_e_;

    // member variables
    preamb_trans_max_e                 preamb_trans_max;
    ra_resp_win_size_e_                ra_resp_win_size;
    mac_contention_resolution_timer_e_ mac_contention_resolution_timer;
  };

  // member variables
  bool                   ext = false;
  preamb_info_s_         preamb_info;
  pwr_ramp_params_s      pwr_ramp_params;
  ra_supervision_info_s_ ra_supervision_info;
  uint8_t                max_harq_msg3_tx = 1;
  // ...
  // group 0
  bool                                    preamb_trans_max_ce_r13_present = false;
  preamb_trans_max_e                      preamb_trans_max_ce_r13;
  copy_ptr<rach_ce_level_info_list_r13_l> rach_ce_level_info_list_r13;
  // group 1
  bool edt_small_tbs_subset_r15_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-ConfigCommon-v1250 ::= SEQUENCE
struct rach_cfg_common_v1250_s {
  struct tx_fail_params_r12_s_ {
    struct conn_est_fail_count_r12_opts {
      enum options { n1, n2, n3, n4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<conn_est_fail_count_r12_opts> conn_est_fail_count_r12_e_;
    struct conn_est_fail_offset_validity_r12_opts {
      enum options { s30, s60, s120, s240, s300, s420, s600, s900, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<conn_est_fail_offset_validity_r12_opts> conn_est_fail_offset_validity_r12_e_;

    // member variables
    bool                                 conn_est_fail_offset_r12_present = false;
    conn_est_fail_count_r12_e_           conn_est_fail_count_r12;
    conn_est_fail_offset_validity_r12_e_ conn_est_fail_offset_validity_r12;
    uint8_t                              conn_est_fail_offset_r12 = 0;
  };

  // member variables
  tx_fail_params_r12_s_ tx_fail_params_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RSS-Config-r15 ::= SEQUENCE
struct rss_cfg_r15_s {
  struct dur_r15_opts {
    enum options { sf8, sf16, sf32, sf40, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<dur_r15_opts> dur_r15_e_;
  struct periodicity_r15_opts {
    enum options { ms160, ms320, ms640, ms1280, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<periodicity_r15_opts> periodicity_r15_e_;
  struct pwr_boost_r15_opts {
    enum options { db0, db3, db4dot8, db6, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<pwr_boost_r15_opts> pwr_boost_r15_e_;

  // member variables
  dur_r15_e_         dur_r15;
  uint8_t            freq_location_r15 = 0;
  periodicity_r15_e_ periodicity_r15;
  pwr_boost_r15_e_   pwr_boost_r15;
  uint8_t            time_offset_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ReferenceTime-r15 ::= SEQUENCE
struct ref_time_r15_s {
  uint32_t ref_days_r15                  = 0;
  uint32_t ref_seconds_r15               = 0;
  uint16_t ref_milli_seconds_r15         = 0;
  uint16_t ref_quarter_micro_seconds_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ReselectionInfoRelay-r13 ::= SEQUENCE
struct resel_info_relay_r13_s {
  struct min_hyst_r13_opts {
    enum options { db0, db3, db6, db9, db12, dbinf, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<min_hyst_r13_opts> min_hyst_r13_e_;

  // member variables
  bool            min_hyst_r13_present = false;
  int8_t          q_rx_lev_min_r13     = -70;
  filt_coef_e     filt_coef_r13;
  min_hyst_r13_e_ min_hyst_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SIB-Type ::= ENUMERATED
struct sib_type_opts {
  enum options {
    sib_type3,
    sib_type4,
    sib_type5,
    sib_type6,
    sib_type7,
    sib_type8,
    sib_type9,
    sib_type10,
    sib_type11,
    sib_type12_v920,
    sib_type13_v920,
    sib_type14_v1130,
    sib_type15_v1130,
    sib_type16_v1130,
    sib_type17_v1250,
    sib_type18_v1250,
    // ...
    sib_type19_v1250,
    sib_type20_v1310,
    sib_type21_v1430,
    sib_type24_v1530,
    sib_type25_v1530,
    sib_type26_v1530,
    nulltype
  } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<sib_type_opts, true, 6> sib_type_e;

// SIB8-PerPLMN-r11 ::= SEQUENCE
struct sib8_per_plmn_r11_s {
  struct params_cdma2000_r11_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    params_cdma2000_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    params_cdma2000_r11_s& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "parametersCDMA2000-r11");
      return c;
    }
    const params_cdma2000_r11_s& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "parametersCDMA2000-r11");
      return c;
    }
    params_cdma2000_r11_s& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types                 type_;
    params_cdma2000_r11_s c;
  };

  // member variables
  uint8_t                plmn_id_r11 = 1;
  params_cdma2000_r11_c_ params_cdma2000_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-AnchorCarrierFreqList-V2X-r14 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (0..262143)
using sl_anchor_carrier_freq_list_v2x_r14_l = bounded_array<uint32_t, 8>;

// SL-CBR-CommonTxConfigList-r14 ::= SEQUENCE
struct sl_cbr_common_tx_cfg_list_r14_s {
  using cbr_range_common_cfg_list_r14_l_ = dyn_array<sl_cbr_levels_cfg_r14_l>;
  using sl_cbr_pssch_tx_cfg_list_r14_l_  = dyn_array<sl_cbr_pssch_tx_cfg_r14_s>;

  // member variables
  cbr_range_common_cfg_list_r14_l_ cbr_range_common_cfg_list_r14;
  sl_cbr_pssch_tx_cfg_list_r14_l_  sl_cbr_pssch_tx_cfg_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CarrierFreqInfo-r12 ::= SEQUENCE
struct sl_carrier_freq_info_r12_s {
  bool                plmn_id_list_r12_present = false;
  uint32_t            carrier_freq_r12         = 0;
  plmn_id_list4_r12_l plmn_id_list_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CarrierFreqInfo-v1310 ::= SEQUENCE
struct sl_carrier_freq_info_v1310_s {
  bool                               ext                         = false;
  bool                               disc_res_non_ps_r13_present = false;
  bool                               disc_res_ps_r13_present     = false;
  bool                               disc_cfg_other_r13_present  = false;
  sl_res_inter_freq_r13_s            disc_res_non_ps_r13;
  sl_res_inter_freq_r13_s            disc_res_ps_r13;
  sl_disc_cfg_other_inter_freq_r13_s disc_cfg_other_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CommResourcePool-r12 ::= SEQUENCE
struct sl_comm_res_pool_r12_s {
  struct ue_sel_res_cfg_r12_s_ {
    bool                    trpt_subset_r12_present = false;
    sl_tf_res_cfg_r12_s     data_tf_res_cfg_r12;
    bounded_bitstring<3, 5> trpt_subset_r12;
  };
  struct rx_params_ncell_r12_s_ {
    bool      tdd_cfg_r12_present = false;
    tdd_cfg_s tdd_cfg_r12;
    uint8_t   sync_cfg_idx_r12 = 0;
  };
  struct tx_params_r12_s_ {
    sl_tx_params_r12_s sc_tx_params_r12;
    sl_tx_params_r12_s data_tx_params_r12;
  };

  // member variables
  bool                   ext                         = false;
  bool                   ue_sel_res_cfg_r12_present  = false;
  bool                   rx_params_ncell_r12_present = false;
  bool                   tx_params_r12_present       = false;
  sl_cp_len_r12_e        sc_cp_len_r12;
  sl_period_comm_r12_e   sc_period_r12;
  sl_tf_res_cfg_r12_s    sc_tf_res_cfg_r12;
  sl_cp_len_r12_e        data_cp_len_r12;
  sl_hop_cfg_comm_r12_s  data_hop_cfg_r12;
  ue_sel_res_cfg_r12_s_  ue_sel_res_cfg_r12;
  rx_params_ncell_r12_s_ rx_params_ncell_r12;
  tx_params_r12_s_       tx_params_r12;
  // ...
  // group 0
  copy_ptr<sl_prio_list_r13_l> prio_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-InterFreqInfoListV2X-r14 ::= SEQUENCE (SIZE (0..7)) OF SL-InterFreqInfoV2X-r14
using sl_inter_freq_info_list_v2x_r14_l = dyn_array<sl_inter_freq_info_v2x_r14_s>;

// SL-PPPP-TxConfigIndex-r15 ::= SEQUENCE
struct sl_pppp_tx_cfg_idx_r15_s {
  using tx_cfg_idx_list_r15_l_      = bounded_array<uint8_t, 16>;
  using mcs_pssch_range_list_r15_l_ = dyn_array<mcs_pssch_range_r15_s>;

  // member variables
  uint8_t                     prio_thres_r15         = 1;
  uint8_t                     default_tx_cfg_idx_r15 = 0;
  uint8_t                     cbr_cfg_idx_r15        = 0;
  tx_cfg_idx_list_r15_l_      tx_cfg_idx_list_r15;
  mcs_pssch_range_list_r15_l_ mcs_pssch_range_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-PPPR-Dest-CarrierFreqList-r15 ::= SEQUENCE (SIZE (1..16)) OF SL-PPPR-Dest-CarrierFreq
using sl_pppr_dest_carrier_freq_list_r15_l = dyn_array<sl_pppr_dest_carrier_freq_s>;

// SL-SyncConfigListV2X-r14 ::= SEQUENCE (SIZE (1..16)) OF SL-SyncConfig-r12
using sl_sync_cfg_list_v2x_r14_l = dyn_array<sl_sync_cfg_r12_s>;

// SoundingRS-UL-ConfigCommon ::= CHOICE
struct srs_ul_cfg_common_c {
  struct setup_s_ {
    struct srs_bw_cfg_opts {
      enum options { bw0, bw1, bw2, bw3, bw4, bw5, bw6, bw7, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_bw_cfg_opts> srs_bw_cfg_e_;
    struct srs_sf_cfg_opts {
      enum options {
        sc0,
        sc1,
        sc2,
        sc3,
        sc4,
        sc5,
        sc6,
        sc7,
        sc8,
        sc9,
        sc10,
        sc11,
        sc12,
        sc13,
        sc14,
        sc15,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_sf_cfg_opts> srs_sf_cfg_e_;

    // member variables
    bool          srs_max_up_pts_present = false;
    srs_bw_cfg_e_ srs_bw_cfg;
    srs_sf_cfg_e_ srs_sf_cfg;
    bool          ack_nack_srs_simul_tx = false;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_common_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigCommon");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigCommon");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SystemInformationBlockType1-v10j0-IEs ::= SEQUENCE
struct sib_type1_v10j0_ies_s {
  bool                         freq_band_info_r10_present         = false;
  bool                         multi_band_info_list_v10j0_present = false;
  bool                         non_crit_ext_present               = false;
  ns_pmax_list_r10_l           freq_band_info_r10;
  multi_band_info_list_v10j0_l multi_band_info_list_v10j0;
  sib_type1_v10l0_ies_s        non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v1250-IEs ::= SEQUENCE
struct sib_type1_v1250_ies_s {
  struct cell_access_related_info_v1250_s_ {
    bool category0_allowed_r12_present = false;
  };

  // member variables
  bool                              cell_sel_info_v1250_present    = false;
  bool                              freq_band_ind_prio_r12_present = false;
  bool                              non_crit_ext_present           = false;
  cell_access_related_info_v1250_s_ cell_access_related_info_v1250;
  cell_sel_info_v1250_s             cell_sel_info_v1250;
  sib_type1_v1310_ies_s             non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType2-v9e0-IEs ::= SEQUENCE
struct sib_type2_v9e0_ies_s {
  bool                 ul_carrier_freq_v9e0_present = false;
  bool                 non_crit_ext_present         = false;
  uint32_t             ul_carrier_freq_v9e0         = 65536;
  sib_type2_v9i0_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType3-v10l0-IEs ::= SEQUENCE
struct sib_type3_v10l0_ies_s {
  bool                         freq_band_info_v10l0_present       = false;
  bool                         multi_band_info_list_v10l0_present = false;
  bool                         non_crit_ext_present               = false;
  ns_pmax_list_v10l0_l         freq_band_info_v10l0;
  multi_band_info_list_v10l0_l multi_band_info_list_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5-v9e0-IEs ::= SEQUENCE
struct sib_type5_v9e0_ies_s {
  using inter_freq_carrier_freq_list_v9e0_l_ = dyn_array<inter_freq_carrier_freq_info_v9e0_s>;

  // member variables
  bool                                 inter_freq_carrier_freq_list_v9e0_present = false;
  bool                                 non_crit_ext_present                      = false;
  inter_freq_carrier_freq_list_v9e0_l_ inter_freq_carrier_freq_list_v9e0;
  sib_type5_v10j0_ies_s                non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UAC-BarringInfoSet-r15 ::= SEQUENCE
struct uac_barr_info_set_r15_s {
  struct uac_barr_factor_r15_opts {
    enum options { p00, p05, p10, p15, p20, p25, p30, p40, p50, p60, p70, p75, p80, p85, p90, p95, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<uac_barr_factor_r15_opts> uac_barr_factor_r15_e_;
  struct uac_barr_time_r15_opts {
    enum options { s4, s8, s16, s32, s64, s128, s256, s512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<uac_barr_time_r15_opts> uac_barr_time_r15_e_;

  // member variables
  uac_barr_factor_r15_e_ uac_barr_factor_r15;
  uac_barr_time_r15_e_   uac_barr_time_r15;
  fixed_bitstring<7>     uac_barr_for_access_id_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UAC-BarringPerPLMN-r15 ::= SEQUENCE
struct uac_barr_per_plmn_r15_s {
  struct uac_ac_barr_list_type_r15_c_ {
    using uac_implicit_ac_barr_list_r15_l_ = std::array<uint8_t, 63>;
    struct types_opts {
      enum options { uac_implicit_ac_barr_list_r15, uac_explicit_ac_barr_list_r15, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    uac_ac_barr_list_type_r15_c_() = default;
    uac_ac_barr_list_type_r15_c_(const uac_ac_barr_list_type_r15_c_& other);
    uac_ac_barr_list_type_r15_c_& operator=(const uac_ac_barr_list_type_r15_c_& other);
    ~uac_ac_barr_list_type_r15_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uac_implicit_ac_barr_list_r15_l_& uac_implicit_ac_barr_list_r15()
    {
      assert_choice_type("uac-ImplicitAC-BarringList-r15", type_.to_string(), "uac-AC-BarringListType-r15");
      return c.get<uac_implicit_ac_barr_list_r15_l_>();
    }
    uac_barr_per_cat_list_r15_l& uac_explicit_ac_barr_list_r15()
    {
      assert_choice_type("uac-ExplicitAC-BarringList-r15", type_.to_string(), "uac-AC-BarringListType-r15");
      return c.get<uac_barr_per_cat_list_r15_l>();
    }
    const uac_implicit_ac_barr_list_r15_l_& uac_implicit_ac_barr_list_r15() const
    {
      assert_choice_type("uac-ImplicitAC-BarringList-r15", type_.to_string(), "uac-AC-BarringListType-r15");
      return c.get<uac_implicit_ac_barr_list_r15_l_>();
    }
    const uac_barr_per_cat_list_r15_l& uac_explicit_ac_barr_list_r15() const
    {
      assert_choice_type("uac-ExplicitAC-BarringList-r15", type_.to_string(), "uac-AC-BarringListType-r15");
      return c.get<uac_barr_per_cat_list_r15_l>();
    }
    uac_implicit_ac_barr_list_r15_l_& set_uac_implicit_ac_barr_list_r15()
    {
      set(types::uac_implicit_ac_barr_list_r15);
      return c.get<uac_implicit_ac_barr_list_r15_l_>();
    }
    uac_barr_per_cat_list_r15_l& set_uac_explicit_ac_barr_list_r15()
    {
      set(types::uac_explicit_ac_barr_list_r15);
      return c.get<uac_barr_per_cat_list_r15_l>();
    }

  private:
    types                                                                          type_;
    choice_buffer_t<uac_barr_per_cat_list_r15_l, uac_implicit_ac_barr_list_r15_l_> c;

    void destroy_();
  };

  // member variables
  bool                         uac_ac_barr_list_type_r15_present = false;
  uint8_t                      plmn_id_idx_r15                   = 1;
  uac_ac_barr_list_type_r15_c_ uac_ac_barr_list_type_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UDT-RestrictingPerPLMN-r13 ::= SEQUENCE
struct udt_restricting_per_plmn_r13_s {
  bool                  udt_restricting_r13_present = false;
  uint8_t               plmn_id_idx_r13             = 1;
  udt_restricting_r13_s udt_restricting_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UL-CyclicPrefixLength ::= ENUMERATED
struct ul_cp_len_opts {
  enum options { len1, len2, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<ul_cp_len_opts> ul_cp_len_e;

// UplinkPowerControlCommon ::= SEQUENCE
struct ul_pwr_ctrl_common_s {
  int8_t              p0_nominal_pusch = -126;
  alpha_r12_e         alpha;
  int8_t              p0_nominal_pucch = -127;
  delta_flist_pucch_s delta_flist_pucch;
  int8_t              delta_preamb_msg3 = -1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommon-v1020 ::= SEQUENCE
struct ul_pwr_ctrl_common_v1020_s {
  struct delta_f_pucch_format3_r10_opts {
    enum options {
      delta_f_minus1,
      delta_f0,
      delta_f1,
      delta_f2,
      delta_f3,
      delta_f4,
      delta_f5,
      delta_f6,
      nulltype
    } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format3_r10_opts> delta_f_pucch_format3_r10_e_;
  struct delta_f_pucch_format1b_cs_r10_opts {
    enum options { delta_f1, delta_f2, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format1b_cs_r10_opts> delta_f_pucch_format1b_cs_r10_e_;

  // member variables
  delta_f_pucch_format3_r10_e_     delta_f_pucch_format3_r10;
  delta_f_pucch_format1b_cs_r10_e_ delta_f_pucch_format1b_cs_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-Id-List-r12 ::= SEQUENCE (SIZE (1..16)) OF WLAN-Identifiers-r12
using wlan_id_list_r12_l = dyn_array<wlan_ids_r12_s>;

// WLAN-OffloadConfig-r12 ::= SEQUENCE
struct wlan_offload_cfg_r12_s {
  struct thres_rsrp_r12_s_ {
    uint8_t thres_rsrp_low_r12  = 0;
    uint8_t thres_rsrp_high_r12 = 0;
  };
  struct thres_rsrq_r12_s_ {
    uint8_t thres_rsrq_low_r12  = 0;
    uint8_t thres_rsrq_high_r12 = 0;
  };
  struct thres_rsrq_on_all_symbols_with_wb_r12_s_ {
    uint8_t thres_rsrq_on_all_symbols_with_wb_low_r12  = 0;
    uint8_t thres_rsrq_on_all_symbols_with_wb_high_r12 = 0;
  };
  struct thres_rsrq_on_all_symbols_r12_s_ {
    uint8_t thres_rsrq_on_all_symbols_low_r12  = 0;
    uint8_t thres_rsrq_on_all_symbols_high_r12 = 0;
  };
  struct thres_rsrq_wb_r12_s_ {
    uint8_t thres_rsrq_wb_low_r12  = 0;
    uint8_t thres_rsrq_wb_high_r12 = 0;
  };
  struct thres_ch_utilization_r12_s_ {
    uint16_t thres_ch_utilization_low_r12  = 0;
    uint16_t thres_ch_utilization_high_r12 = 0;
  };
  struct thres_backhaul_bw_r12_s_ {
    wlan_backhaul_rate_r12_e thres_backhaul_dl_bw_low_r12;
    wlan_backhaul_rate_r12_e thres_backhaul_dl_bw_high_r12;
    wlan_backhaul_rate_r12_e thres_backhaul_ul_bw_low_r12;
    wlan_backhaul_rate_r12_e thres_backhaul_ul_bw_high_r12;
  };
  struct thres_wlan_rssi_r12_s_ {
    uint16_t thres_wlan_rssi_low_r12  = 0;
    uint16_t thres_wlan_rssi_high_r12 = 0;
  };

  // member variables
  bool                                     ext                                           = false;
  bool                                     thres_rsrp_r12_present                        = false;
  bool                                     thres_rsrq_r12_present                        = false;
  bool                                     thres_rsrq_on_all_symbols_with_wb_r12_present = false;
  bool                                     thres_rsrq_on_all_symbols_r12_present         = false;
  bool                                     thres_rsrq_wb_r12_present                     = false;
  bool                                     thres_ch_utilization_r12_present              = false;
  bool                                     thres_backhaul_bw_r12_present                 = false;
  bool                                     thres_wlan_rssi_r12_present                   = false;
  bool                                     offload_pref_ind_r12_present                  = false;
  bool                                     t_steering_wlan_r12_present                   = false;
  thres_rsrp_r12_s_                        thres_rsrp_r12;
  thres_rsrq_r12_s_                        thres_rsrq_r12;
  thres_rsrq_on_all_symbols_with_wb_r12_s_ thres_rsrq_on_all_symbols_with_wb_r12;
  thres_rsrq_on_all_symbols_r12_s_         thres_rsrq_on_all_symbols_r12;
  thres_rsrq_wb_r12_s_                     thres_rsrq_wb_r12;
  thres_ch_utilization_r12_s_              thres_ch_utilization_r12;
  thres_backhaul_bw_r12_s_                 thres_backhaul_bw_r12;
  thres_wlan_rssi_r12_s_                   thres_wlan_rssi_r12;
  fixed_bitstring<16>                      offload_pref_ind_r12;
  uint8_t                                  t_steering_wlan_r12 = 0;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WUS-Config-r15 ::= SEQUENCE
struct wus_cfg_r15_s {
  struct max_dur_factor_r15_opts {
    enum options { one32th, one16th, one8th, one4th, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_dur_factor_r15_opts> max_dur_factor_r15_e_;
  struct num_pos_r15_opts {
    enum options { n1, n2, n4, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<num_pos_r15_opts> num_pos_r15_e_;
  struct freq_location_r15_opts {
    enum options { n0, n2, n4, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<freq_location_r15_opts> freq_location_r15_e_;
  struct time_offset_drx_r15_opts {
    enum options { ms40, ms80, ms160, ms240, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<time_offset_drx_r15_opts> time_offset_drx_r15_e_;
  struct time_offset_e_drx_short_r15_opts {
    enum options { ms40, ms80, ms160, ms240, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<time_offset_e_drx_short_r15_opts> time_offset_e_drx_short_r15_e_;
  struct time_offset_e_drx_long_r15_opts {
    enum options { ms1000, ms2000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<time_offset_e_drx_long_r15_opts> time_offset_e_drx_long_r15_e_;

  // member variables
  bool                           num_pos_r15_present                = false;
  bool                           time_offset_e_drx_long_r15_present = false;
  max_dur_factor_r15_e_          max_dur_factor_r15;
  num_pos_r15_e_                 num_pos_r15;
  freq_location_r15_e_           freq_location_r15;
  time_offset_drx_r15_e_         time_offset_drx_r15;
  time_offset_e_drx_short_r15_e_ time_offset_e_drx_short_r15;
  time_offset_e_drx_long_r15_e_  time_offset_e_drx_long_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AC-BarringPerPLMN-List-r12 ::= SEQUENCE (SIZE (1..6)) OF AC-BarringPerPLMN-r12
using ac_barr_per_plmn_list_r12_l = dyn_array<ac_barr_per_plmn_r12_s>;

// ACDC-BarringForCommon-r13 ::= SEQUENCE
struct acdc_barr_for_common_r13_s {
  bool                              acdc_hplm_nonly_r13 = false;
  barr_per_acdc_category_list_r13_l barr_per_acdc_category_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ACDC-BarringPerPLMN-List-r13 ::= SEQUENCE (SIZE (1..6)) OF ACDC-BarringPerPLMN-r13
using acdc_barr_per_plmn_list_r13_l = dyn_array<acdc_barr_per_plmn_r13_s>;

// CIOT-EPS-OptimisationInfo-r13 ::= SEQUENCE (SIZE (1..6)) OF CIOT-OptimisationPLMN-r13
using ciot_eps_optim_info_r13_l = dyn_array<ciot_optim_plmn_r13_s>;

// CarrierFreqInfoUTRA-v1250 ::= SEQUENCE
struct carrier_freq_info_utra_v1250_s {
  bool reduced_meas_performance_r12_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqListNR-r15 ::= SEQUENCE (SIZE (1..8)) OF CarrierFreqNR-r15
using carrier_freq_list_nr_r15_l = dyn_array<carrier_freq_nr_r15_s>;

// CarrierFreqListUTRA-FDD ::= SEQUENCE (SIZE (1..16)) OF CarrierFreqUTRA-FDD
using carrier_freq_list_utra_fdd_l = dyn_array<carrier_freq_utra_fdd_s>;

// CarrierFreqListUTRA-FDD-Ext-r12 ::= SEQUENCE (SIZE (1..16)) OF CarrierFreqUTRA-FDD-Ext-r12
using carrier_freq_list_utra_fdd_ext_r12_l = dyn_array<carrier_freq_utra_fdd_ext_r12_s>;

// CarrierFreqListUTRA-TDD ::= SEQUENCE (SIZE (1..16)) OF CarrierFreqUTRA-TDD
using carrier_freq_list_utra_tdd_l = dyn_array<carrier_freq_utra_tdd_s>;

// CarrierFreqListUTRA-TDD-Ext-r12 ::= SEQUENCE (SIZE (1..16)) OF CarrierFreqUTRA-TDD-r12
using carrier_freq_list_utra_tdd_ext_r12_l = dyn_array<carrier_freq_utra_tdd_r12_s>;

// CarrierFreqsInfoListGERAN ::= SEQUENCE (SIZE (1..16)) OF CarrierFreqsInfoGERAN
using carrier_freqs_info_list_geran_l = dyn_array<carrier_freqs_info_geran_s>;

// CellReselectionInfoCommon-v1460 ::= SEQUENCE
struct cell_resel_info_common_v1460_s {
  struct s_search_delta_p_r14_opts {
    enum options { db6, db9, db12, db15, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<s_search_delta_p_r14_opts> s_search_delta_p_r14_e_;

  // member variables
  s_search_delta_p_r14_e_ s_search_delta_p_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellReselectionInfoHSDN-r15 ::= SEQUENCE
struct cell_resel_info_hsdn_r15_s {
  uint8_t cell_equivalent_size_r15 = 2;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellReselectionParametersCDMA2000 ::= SEQUENCE
struct cell_resel_params_cdma2000_s {
  bool                        t_resel_cdma2000_sf_present = false;
  band_class_list_cdma2000_l  band_class_list;
  neigh_cell_list_cdma2000_l  neigh_cell_list;
  uint8_t                     t_resel_cdma2000 = 0;
  speed_state_scale_factors_s t_resel_cdma2000_sf;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellReselectionParametersCDMA2000-v920 ::= SEQUENCE
struct cell_resel_params_cdma2000_v920_s {
  neigh_cell_list_cdma2000_v920_l neigh_cell_list_v920;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellReselectionServingFreqInfo-v1310 ::= SEQUENCE
struct cell_resel_serving_freq_info_v1310_s {
  cell_resel_sub_prio_r13_e cell_resel_sub_prio_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellSelectionInfo-v920 ::= SEQUENCE
struct cell_sel_info_v920_s {
  bool    q_qual_min_offset_r9_present = false;
  int8_t  q_qual_min_r9                = -34;
  uint8_t q_qual_min_offset_r9         = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EAB-ConfigPLMN-r11 ::= SEQUENCE
struct eab_cfg_plmn_r11_s {
  bool          eab_cfg_r11_present = false;
  eab_cfg_r11_s eab_cfg_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// InterFreqCarrierFreqList ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo
using inter_freq_carrier_freq_list_l = dyn_array<inter_freq_carrier_freq_info_s>;

// InterFreqCarrierFreqList-v1250 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1250
using inter_freq_carrier_freq_list_v1250_l = dyn_array<inter_freq_carrier_freq_info_v1250_s>;

// InterFreqCarrierFreqList-v1310 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1310
using inter_freq_carrier_freq_list_v1310_l = dyn_array<inter_freq_carrier_freq_info_v1310_s>;

// InterFreqCarrierFreqList-v1350 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1350
using inter_freq_carrier_freq_list_v1350_l = dyn_array<inter_freq_carrier_freq_info_v1350_s>;

// InterFreqCarrierFreqList-v1530 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1530
using inter_freq_carrier_freq_list_v1530_l = dyn_array<inter_freq_carrier_freq_info_v1530_s>;

// InterFreqCarrierFreqListExt-r12 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-r12
using inter_freq_carrier_freq_list_ext_r12_l = dyn_array<inter_freq_carrier_freq_info_r12_s>;

// InterFreqCarrierFreqListExt-v1280 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v10j0
using inter_freq_carrier_freq_list_ext_v1280_l = dyn_array<inter_freq_carrier_freq_info_v10j0_s>;

// InterFreqCarrierFreqListExt-v1310 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1310
using inter_freq_carrier_freq_list_ext_v1310_l = dyn_array<inter_freq_carrier_freq_info_v1310_s>;

// InterFreqCarrierFreqListExt-v1350 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1350
using inter_freq_carrier_freq_list_ext_v1350_l = dyn_array<inter_freq_carrier_freq_info_v1350_s>;

// InterFreqCarrierFreqListExt-v1360 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1360
using inter_freq_carrier_freq_list_ext_v1360_l = dyn_array<inter_freq_carrier_freq_info_v1360_s>;

// InterFreqCarrierFreqListExt-v1530 ::= SEQUENCE (SIZE (1..8)) OF InterFreqCarrierFreqInfo-v1530
using inter_freq_carrier_freq_list_ext_v1530_l = dyn_array<inter_freq_carrier_freq_info_v1530_s>;

// IntraFreqBlackCellList ::= SEQUENCE (SIZE (1..16)) OF PhysCellIdRange
using intra_freq_black_cell_list_l = dyn_array<pci_range_s>;

// IntraFreqNeighCellList ::= SEQUENCE (SIZE (1..16)) OF IntraFreqNeighCellInfo
using intra_freq_neigh_cell_list_l = dyn_array<intra_freq_neigh_cell_info_s>;

// IntraFreqNeighHSDN-CellList-r15 ::= SEQUENCE (SIZE (1..16)) OF PhysCellIdRange
using intra_freq_neigh_hsdn_cell_list_r15_l = dyn_array<pci_range_s>;

// MBMS-InterFreqCarrierTypeList-r14 ::= SEQUENCE (SIZE (1..8)) OF MBMS-CarrierType-r14
using mbms_inter_freq_carrier_type_list_r14_l = dyn_array<mbms_carrier_type_r14_s>;

// MBMS-NotificationConfig-r9 ::= SEQUENCE
struct mbms_notif_cfg_r9_s {
  struct notif_repeat_coeff_r9_opts {
    enum options { n2, n4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<notif_repeat_coeff_r9_opts> notif_repeat_coeff_r9_e_;

  // member variables
  notif_repeat_coeff_r9_e_ notif_repeat_coeff_r9;
  uint8_t                  notif_offset_r9 = 0;
  uint8_t                  notif_sf_idx_r9 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBMS-NotificationConfig-v1430 ::= SEQUENCE
struct mbms_notif_cfg_v1430_s {
  uint8_t notif_sf_idx_v1430 = 7;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MBMS-SAI-InterFreqList-r11 ::= SEQUENCE (SIZE (1..8)) OF MBMS-SAI-InterFreq-r11
using mbms_sai_inter_freq_list_r11_l = dyn_array<mbms_sai_inter_freq_r11_s>;

// MBMS-SAI-InterFreqList-v1140 ::= SEQUENCE (SIZE (1..8)) OF MBMS-SAI-InterFreq-v1140
using mbms_sai_inter_freq_list_v1140_l = dyn_array<mbms_sai_inter_freq_v1140_s>;

// MBSFN-AreaInfoList-r9 ::= SEQUENCE (SIZE (1..8)) OF MBSFN-AreaInfo-r9
using mbsfn_area_info_list_r9_l = dyn_array<mbsfn_area_info_r9_s>;

// MBSFN-SubframeConfigList ::= SEQUENCE (SIZE (1..8)) OF MBSFN-SubframeConfig
using mbsfn_sf_cfg_list_l = dyn_array<mbsfn_sf_cfg_s>;

// MBSFN-SubframeConfigList-v1430 ::= SEQUENCE (SIZE (1..8)) OF MBSFN-SubframeConfig-v1430
using mbsfn_sf_cfg_list_v1430_l = dyn_array<mbsfn_sf_cfg_v1430_s>;

// MeasIdleConfigSIB-r15 ::= SEQUENCE
struct meas_idle_cfg_sib_r15_s {
  bool                     ext = false;
  eutra_carrier_list_r15_l meas_idle_carrier_list_eutra_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MobilityStateParameters ::= SEQUENCE
struct mob_state_params_s {
  struct t_eval_opts {
    enum options { s30, s60, s120, s180, s240, spare3, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<t_eval_opts> t_eval_e_;
  struct t_hyst_normal_opts {
    enum options { s30, s60, s120, s180, s240, spare3, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<t_hyst_normal_opts> t_hyst_normal_e_;

  // member variables
  t_eval_e_        t_eval;
  t_hyst_normal_e_ t_hyst_normal;
  uint8_t          n_cell_change_medium = 1;
  uint8_t          n_cell_change_high   = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-InfoList-r15 ::= SEQUENCE (SIZE (1..6)) OF PLMN-Info-r15
using plmn_info_list_r15_l = dyn_array<plmn_info_r15_s>;

// RadioResourceConfigCommonSIB ::= SEQUENCE
struct rr_cfg_common_sib_s {
  bool                 ext = false;
  rach_cfg_common_s    rach_cfg_common;
  bcch_cfg_s           bcch_cfg;
  pcch_cfg_s           pcch_cfg;
  prach_cfg_sib_s      prach_cfg;
  pdsch_cfg_common_s   pdsch_cfg_common;
  pusch_cfg_common_s   pusch_cfg_common;
  pucch_cfg_common_s   pucch_cfg_common;
  srs_ul_cfg_common_c  srs_ul_cfg_common;
  ul_pwr_ctrl_common_s ul_pwr_ctrl_common;
  ul_cp_len_e          ul_cp_len;
  // ...
  // group 0
  copy_ptr<ul_pwr_ctrl_common_v1020_s> ul_pwr_ctrl_common_v1020;
  // group 1
  copy_ptr<rach_cfg_common_v1250_s> rach_cfg_common_v1250;
  // group 2
  copy_ptr<pusch_cfg_common_v1270_s> pusch_cfg_common_v1270;
  // group 3
  copy_ptr<bcch_cfg_v1310_s>         bcch_cfg_v1310;
  copy_ptr<pcch_cfg_v1310_s>         pcch_cfg_v1310;
  copy_ptr<freq_hop_params_r13_s>    freq_hop_params_r13;
  copy_ptr<pdsch_cfg_common_v1310_s> pdsch_cfg_common_v1310;
  copy_ptr<pusch_cfg_common_v1310_s> pusch_cfg_common_v1310;
  copy_ptr<prach_cfg_sib_v1310_s>    prach_cfg_common_v1310;
  copy_ptr<pucch_cfg_common_v1310_s> pucch_cfg_common_v1310;
  // group 4
  copy_ptr<high_speed_cfg_r14_s>     high_speed_cfg_r14;
  copy_ptr<prach_cfg_v1430_s>        prach_cfg_v1430;
  copy_ptr<pucch_cfg_common_v1430_s> pucch_cfg_common_v1430;
  // group 5
  copy_ptr<prach_cfg_sib_v1530_s>  prach_cfg_v1530;
  copy_ptr<rss_cfg_r15_s>          ce_rss_cfg_r15;
  copy_ptr<wus_cfg_r15_s>          wus_cfg_r15;
  copy_ptr<high_speed_cfg_v1530_s> high_speed_cfg_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RedistributionServingInfo-r13 ::= SEQUENCE
struct redist_serving_info_r13_s {
  struct t360_r13_opts {
    enum options { min4, min8, min16, min32, infinity, spare3, spare2, spare1, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<t360_r13_opts> t360_r13_e_;

  // member variables
  bool        redist_factor_cell_r13_present     = false;
  bool        redistr_on_paging_only_r13_present = false;
  uint8_t     redist_factor_serving_r13          = 0;
  t360_r13_e_ t360_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SC-MCCH-SchedulingInfo-r14 ::= SEQUENCE
struct sc_mcch_sched_info_r14_s {
  struct on_dur_timer_scptm_r14_opts {
    enum options { psf10, psf20, psf100, psf300, psf500, psf1000, psf1200, psf1600, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<on_dur_timer_scptm_r14_opts> on_dur_timer_scptm_r14_e_;
  struct drx_inactivity_timer_scptm_r14_opts {
    enum options {
      psf0,
      psf1,
      psf2,
      psf4,
      psf8,
      psf16,
      psf32,
      psf64,
      psf128,
      psf256,
      ps512,
      psf1024,
      psf2048,
      psf4096,
      psf8192,
      psf16384,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<drx_inactivity_timer_scptm_r14_opts> drx_inactivity_timer_scptm_r14_e_;
  struct sched_period_start_offset_scptm_r14_c_ {
    struct types_opts {
      enum options {
        sf10,
        sf20,
        sf32,
        sf40,
        sf64,
        sf80,
        sf128,
        sf160,
        sf256,
        sf320,
        sf512,
        sf640,
        sf1024,
        sf2048,
        sf4096,
        sf8192,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sched_period_start_offset_scptm_r14_c_() = default;
    sched_period_start_offset_scptm_r14_c_(const sched_period_start_offset_scptm_r14_c_& other);
    sched_period_start_offset_scptm_r14_c_& operator=(const sched_period_start_offset_scptm_r14_c_& other);
    ~sched_period_start_offset_scptm_r14_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& sf10()
    {
      assert_choice_type("sf10", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf20()
    {
      assert_choice_type("sf20", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf32()
    {
      assert_choice_type("sf32", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf40()
    {
      assert_choice_type("sf40", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf64()
    {
      assert_choice_type("sf64", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf80()
    {
      assert_choice_type("sf80", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf128()
    {
      assert_choice_type("sf128", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint8_t& sf160()
    {
      assert_choice_type("sf160", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    uint16_t& sf256()
    {
      assert_choice_type("sf256", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf320()
    {
      assert_choice_type("sf320", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf512()
    {
      assert_choice_type("sf512", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf640()
    {
      assert_choice_type("sf640", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf1024()
    {
      assert_choice_type("sf1024", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf2048()
    {
      assert_choice_type("sf2048", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf4096()
    {
      assert_choice_type("sf4096", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint16_t& sf8192()
    {
      assert_choice_type("sf8192", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint8_t& sf10() const
    {
      assert_choice_type("sf10", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf20() const
    {
      assert_choice_type("sf20", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf32() const
    {
      assert_choice_type("sf32", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf40() const
    {
      assert_choice_type("sf40", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf64() const
    {
      assert_choice_type("sf64", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf80() const
    {
      assert_choice_type("sf80", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf128() const
    {
      assert_choice_type("sf128", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& sf160() const
    {
      assert_choice_type("sf160", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint8_t>();
    }
    const uint16_t& sf256() const
    {
      assert_choice_type("sf256", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf320() const
    {
      assert_choice_type("sf320", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf512() const
    {
      assert_choice_type("sf512", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf640() const
    {
      assert_choice_type("sf640", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf1024() const
    {
      assert_choice_type("sf1024", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf2048() const
    {
      assert_choice_type("sf2048", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf4096() const
    {
      assert_choice_type("sf4096", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    const uint16_t& sf8192() const
    {
      assert_choice_type("sf8192", type_.to_string(), "schedulingPeriodStartOffsetSCPTM-r14");
      return c.get<uint16_t>();
    }
    uint8_t& set_sf10()
    {
      set(types::sf10);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf20()
    {
      set(types::sf20);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf32()
    {
      set(types::sf32);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf40()
    {
      set(types::sf40);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf64()
    {
      set(types::sf64);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf80()
    {
      set(types::sf80);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf128()
    {
      set(types::sf128);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf160()
    {
      set(types::sf160);
      return c.get<uint8_t>();
    }
    uint16_t& set_sf256()
    {
      set(types::sf256);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf320()
    {
      set(types::sf320);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf512()
    {
      set(types::sf512);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf640()
    {
      set(types::sf640);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf1024()
    {
      set(types::sf1024);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf2048()
    {
      set(types::sf2048);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf4096()
    {
      set(types::sf4096);
      return c.get<uint16_t>();
    }
    uint16_t& set_sf8192()
    {
      set(types::sf8192);
      return c.get<uint16_t>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };

  // member variables
  bool                                   ext = false;
  on_dur_timer_scptm_r14_e_              on_dur_timer_scptm_r14;
  drx_inactivity_timer_scptm_r14_e_      drx_inactivity_timer_scptm_r14;
  sched_period_start_offset_scptm_r14_c_ sched_period_start_offset_scptm_r14;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SIB-MappingInfo ::= SEQUENCE (SIZE (0..31)) OF SIB-Type
using sib_map_info_l = bounded_array<sib_type_e, 31>;

// SIB8-PerPLMN-List-r11 ::= SEQUENCE (SIZE (1..6)) OF SIB8-PerPLMN-r11
using sib8_per_plmn_list_r11_l = dyn_array<sib8_per_plmn_r11_s>;

// SL-CBR-PPPP-TxConfigList-r15 ::= SEQUENCE (SIZE (1..8)) OF SL-PPPP-TxConfigIndex-r15
using sl_cbr_pppp_tx_cfg_list_r15_l = dyn_array<sl_pppp_tx_cfg_idx_r15_s>;

// SL-CarrierFreqInfoList-r12 ::= SEQUENCE (SIZE (1..8)) OF SL-CarrierFreqInfo-r12
using sl_carrier_freq_info_list_r12_l = dyn_array<sl_carrier_freq_info_r12_s>;

// SL-CarrierFreqInfoList-v1310 ::= SEQUENCE (SIZE (1..8)) OF SL-CarrierFreqInfo-v1310
using sl_carrier_freq_info_list_v1310_l = dyn_array<sl_carrier_freq_info_v1310_s>;

// SL-CommRxPoolList-r12 ::= SEQUENCE (SIZE (1..16)) OF SL-CommResourcePool-r12
using sl_comm_rx_pool_list_r12_l = dyn_array<sl_comm_res_pool_r12_s>;

// SL-CommTxPoolList-r12 ::= SEQUENCE (SIZE (1..4)) OF SL-CommResourcePool-r12
using sl_comm_tx_pool_list_r12_l = dyn_array<sl_comm_res_pool_r12_s>;

// SL-CommTxPoolListExt-r13 ::= SEQUENCE (SIZE (1..4)) OF SL-CommResourcePool-r12
using sl_comm_tx_pool_list_ext_r13_l = dyn_array<sl_comm_res_pool_r12_s>;

// SL-DiscConfigRelayUE-r13 ::= SEQUENCE
struct sl_disc_cfg_relay_ue_r13_s {
  struct hyst_max_r13_opts {
    enum options { db0, db3, db6, db9, db12, dbinf, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<hyst_max_r13_opts> hyst_max_r13_e_;
  struct hyst_min_r13_opts {
    enum options { db0, db3, db6, db9, db12, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<hyst_min_r13_opts> hyst_min_r13_e_;

  // member variables
  bool            thresh_high_r13_present = false;
  bool            thresh_low_r13_present  = false;
  bool            hyst_max_r13_present    = false;
  bool            hyst_min_r13_present    = false;
  uint8_t         thresh_high_r13         = 0;
  uint8_t         thresh_low_r13          = 0;
  hyst_max_r13_e_ hyst_max_r13;
  hyst_min_r13_e_ hyst_min_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscConfigRemoteUE-r13 ::= SEQUENCE
struct sl_disc_cfg_remote_ue_r13_s {
  struct hyst_max_r13_opts {
    enum options { db0, db3, db6, db9, db12, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<hyst_max_r13_opts> hyst_max_r13_e_;

  // member variables
  bool                   thresh_high_r13_present = false;
  bool                   hyst_max_r13_present    = false;
  uint8_t                thresh_high_r13         = 0;
  hyst_max_r13_e_        hyst_max_r13;
  resel_info_relay_r13_s resel_info_ic_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-SyncConfigList-r12 ::= SEQUENCE (SIZE (1..16)) OF SL-SyncConfig-r12
using sl_sync_cfg_list_r12_l = dyn_array<sl_sync_cfg_r12_s>;

// SL-V2X-ConfigCommon-r14 ::= SEQUENCE
struct sl_v2x_cfg_common_r14_s {
  bool                                  v2x_comm_rx_pool_r14_present               = false;
  bool                                  v2x_comm_tx_pool_normal_common_r14_present = false;
  bool                                  p2x_comm_tx_pool_normal_common_r14_present = false;
  bool                                  v2x_comm_tx_pool_exceptional_r14_present   = false;
  bool                                  v2x_sync_cfg_r14_present                   = false;
  bool                                  v2x_inter_freq_info_list_r14_present       = false;
  bool                                  v2x_res_sel_cfg_r14_present                = false;
  bool                                  zone_cfg_r14_present                       = false;
  bool                                  type_tx_sync_r14_present                   = false;
  bool                                  thres_sl_tx_prioritization_r14_present     = false;
  bool                                  anchor_carrier_freq_list_r14_present       = false;
  bool                                  offset_dfn_r14_present                     = false;
  bool                                  cbr_common_tx_cfg_list_r14_present         = false;
  sl_comm_rx_pool_list_v2x_r14_l        v2x_comm_rx_pool_r14;
  sl_comm_tx_pool_list_v2x_r14_l        v2x_comm_tx_pool_normal_common_r14;
  sl_comm_tx_pool_list_v2x_r14_l        p2x_comm_tx_pool_normal_common_r14;
  sl_comm_res_pool_v2x_r14_s            v2x_comm_tx_pool_exceptional_r14;
  sl_sync_cfg_list_v2x_r14_l            v2x_sync_cfg_r14;
  sl_inter_freq_info_list_v2x_r14_l     v2x_inter_freq_info_list_r14;
  sl_comm_tx_pool_sensing_cfg_r14_s     v2x_res_sel_cfg_r14;
  sl_zone_cfg_r14_s                     zone_cfg_r14;
  sl_type_tx_sync_r14_e                 type_tx_sync_r14;
  uint8_t                               thres_sl_tx_prioritization_r14 = 1;
  sl_anchor_carrier_freq_list_v2x_r14_l anchor_carrier_freq_list_r14;
  uint16_t                              offset_dfn_r14 = 0;
  sl_cbr_common_tx_cfg_list_r14_s       cbr_common_tx_cfg_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-V2X-PacketDuplicationConfig-r15 ::= SEQUENCE
struct sl_v2x_packet_dupl_cfg_r15_s {
  bool                                 ext                                  = false;
  bool                                 allowed_carrier_freq_cfg_r15_present = false;
  uint8_t                              thresh_sl_reliability_r15            = 1;
  sl_pppr_dest_carrier_freq_list_r15_l allowed_carrier_freq_cfg_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-V2X-SyncFreqList-r15 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (0..262143)
using sl_v2x_sync_freq_list_r15_l = bounded_array<uint32_t, 8>;

// SystemInformationBlockType1-v1130-IEs ::= SEQUENCE
struct sib_type1_v1130_ies_s {
  bool                  tdd_cfg_v1130_present       = false;
  bool                  cell_sel_info_v1130_present = false;
  bool                  non_crit_ext_present        = false;
  tdd_cfg_v1130_s       tdd_cfg_v1130;
  cell_sel_info_v1130_s cell_sel_info_v1130;
  sib_type1_v1250_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v9e0-IEs ::= SEQUENCE
struct sib_type1_v9e0_ies_s {
  bool                        freq_band_ind_v9e0_present        = false;
  bool                        multi_band_info_list_v9e0_present = false;
  bool                        non_crit_ext_present              = false;
  uint16_t                    freq_band_ind_v9e0                = 65;
  multi_band_info_list_v9e0_l multi_band_info_list_v9e0;
  sib_type1_v10j0_ies_s       non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType2-v8h0-IEs ::= SEQUENCE
struct sib_type2_v8h0_ies_s {
  using multi_band_info_list_l_ = bounded_array<uint8_t, 8>;

  // member variables
  bool                    multi_band_info_list_present = false;
  bool                    non_crit_ext_present         = false;
  multi_band_info_list_l_ multi_band_info_list;
  sib_type2_v9e0_ies_s    non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType3-v10j0-IEs ::= SEQUENCE
struct sib_type3_v10j0_ies_s {
  bool                         freq_band_info_r10_present         = false;
  bool                         multi_band_info_list_v10j0_present = false;
  bool                         non_crit_ext_present               = false;
  ns_pmax_list_r10_l           freq_band_info_r10;
  multi_band_info_list_v10j0_l multi_band_info_list_v10j0;
  sib_type3_v10l0_ies_s        non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5-v8h0-IEs ::= SEQUENCE
struct sib_type5_v8h0_ies_s {
  using inter_freq_carrier_freq_list_v8h0_l_ = dyn_array<inter_freq_carrier_freq_info_v8h0_s>;

  // member variables
  bool                                 inter_freq_carrier_freq_list_v8h0_present = false;
  bool                                 non_crit_ext_present                      = false;
  inter_freq_carrier_freq_list_v8h0_l_ inter_freq_carrier_freq_list_v8h0;
  sib_type5_v9e0_ies_s                 non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType6-v8h0-IEs ::= SEQUENCE
struct sib_type6_v8h0_ies_s {
  using carrier_freq_list_utra_fdd_v8h0_l_ = dyn_array<carrier_freq_info_utra_fdd_v8h0_s>;

  // member variables
  bool                               carrier_freq_list_utra_fdd_v8h0_present = false;
  bool                               non_crit_ext_present                    = false;
  carrier_freq_list_utra_fdd_v8h0_l_ carrier_freq_list_utra_fdd_v8h0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TimeAlignmentTimer ::= ENUMERATED
struct time_align_timer_opts {
  enum options { sf500, sf750, sf1280, sf1920, sf2560, sf5120, sf10240, infinity, nulltype } value;
  typedef int16_t number_type;

  std::string to_string() const;
  int16_t     to_number() const;
};
typedef enumerated<time_align_timer_opts> time_align_timer_e;

// TimeReferenceInfo-r15 ::= SEQUENCE
struct time_ref_info_r15_s {
  bool           uncertainty_r15_present    = false;
  bool           time_info_type_r15_present = false;
  bool           ref_sfn_r15_present        = false;
  ref_time_r15_s time_r15;
  uint8_t        uncertainty_r15 = 0;
  uint16_t       ref_sfn_r15     = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UAC-AC1-SelectAssistInfo-r15 ::= ENUMERATED
struct uac_ac1_select_assist_info_r15_opts {
  enum options { a, b, c, nulltype } value;

  std::string to_string() const;
};
typedef enumerated<uac_ac1_select_assist_info_r15_opts> uac_ac1_select_assist_info_r15_e;

// UAC-BarringInfoSetList-r15 ::= SEQUENCE (SIZE (1..8)) OF UAC-BarringInfoSet-r15
using uac_barr_info_set_list_r15_l = dyn_array<uac_barr_info_set_r15_s>;

// UAC-BarringPerPLMN-List-r15 ::= SEQUENCE (SIZE (1..6)) OF UAC-BarringPerPLMN-r15
using uac_barr_per_plmn_list_r15_l = dyn_array<uac_barr_per_plmn_r15_s>;

// UDT-RestrictingPerPLMN-List-r13 ::= SEQUENCE (SIZE (1..6)) OF UDT-RestrictingPerPLMN-r13
using udt_restricting_per_plmn_list_r13_l = dyn_array<udt_restricting_per_plmn_r13_s>;

// UE-TimersAndConstants ::= SEQUENCE
struct ue_timers_and_consts_s {
  struct t300_opts {
    enum options { ms100, ms200, ms300, ms400, ms600, ms1000, ms1500, ms2000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t300_opts> t300_e_;
  struct t301_opts {
    enum options { ms100, ms200, ms300, ms400, ms600, ms1000, ms1500, ms2000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t301_opts> t301_e_;
  struct t310_opts {
    enum options { ms0, ms50, ms100, ms200, ms500, ms1000, ms2000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t310_opts> t310_e_;
  struct n310_opts {
    enum options { n1, n2, n3, n4, n6, n8, n10, n20, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<n310_opts> n310_e_;
  struct t311_opts {
    enum options { ms1000, ms3000, ms5000, ms10000, ms15000, ms20000, ms30000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t311_opts> t311_e_;
  struct n311_opts {
    enum options { n1, n2, n3, n4, n5, n6, n8, n10, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<n311_opts> n311_e_;
  struct t300_v1310_opts {
    enum options { ms2500, ms3000, ms3500, ms4000, ms5000, ms6000, ms8000, ms10000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t300_v1310_opts> t300_v1310_e_;
  struct t301_v1310_opts {
    enum options { ms2500, ms3000, ms3500, ms4000, ms5000, ms6000, ms8000, ms10000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t301_v1310_opts> t301_v1310_e_;
  struct t310_v1330_opts {
    enum options { ms4000, ms6000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t310_v1330_opts> t310_v1330_e_;
  struct t300_r15_opts {
    enum options { ms4000, ms6000, ms8000, ms10000, ms15000, ms25000, ms40000, ms60000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t300_r15_opts> t300_r15_e_;

  // member variables
  bool    ext = false;
  t300_e_ t300;
  t301_e_ t301;
  t310_e_ t310;
  n310_e_ n310;
  t311_e_ t311;
  n311_e_ n311;
  // ...
  // group 0
  bool          t300_v1310_present = false;
  bool          t301_v1310_present = false;
  t300_v1310_e_ t300_v1310;
  t301_v1310_e_ t301_v1310;
  // group 1
  bool          t310_v1330_present = false;
  t310_v1330_e_ t310_v1330;
  // group 2
  bool        t300_r15_present = false;
  t300_r15_e_ t300_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-OffloadInfoPerPLMN-r12 ::= SEQUENCE
struct wlan_offload_info_per_plmn_r12_s {
  bool                   ext                                 = false;
  bool                   wlan_offload_cfg_common_r12_present = false;
  bool                   wlan_id_list_r12_present            = false;
  wlan_offload_cfg_r12_s wlan_offload_cfg_common_r12;
  wlan_id_list_r12_l     wlan_id_list_r12;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SchedulingInfo ::= SEQUENCE
struct sched_info_s {
  struct si_periodicity_opts {
    enum options { rf8, rf16, rf32, rf64, rf128, rf256, rf512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<si_periodicity_opts> si_periodicity_e_;

  // member variables
  si_periodicity_e_ si_periodicity;
  sib_map_info_l    sib_map_info;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformation-v8a0-IEs ::= SEQUENCE
struct sys_info_v8a0_ies_s {
  bool          late_non_crit_ext_present = false;
  bool          non_crit_ext_present      = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockPos-r15 ::= SEQUENCE
struct sib_pos_r15_s {
  bool          ext                       = false;
  bool          late_non_crit_ext_present = false;
  dyn_octstring assist_data_sib_elem_r15;
  dyn_octstring late_non_crit_ext;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v8h0-IEs ::= SEQUENCE
struct sib_type1_v8h0_ies_s {
  bool                   multi_band_info_list_present = false;
  bool                   non_crit_ext_present         = false;
  multi_band_info_list_l multi_band_info_list;
  sib_type1_v9e0_ies_s   non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v920-IEs ::= SEQUENCE
struct sib_type1_v920_ies_s {
  bool                  ims_emergency_support_r9_present = false;
  bool                  cell_sel_info_v920_present       = false;
  bool                  non_crit_ext_present             = false;
  cell_sel_info_v920_s  cell_sel_info_v920;
  sib_type1_v1130_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType10 ::= SEQUENCE
struct sib_type10_s {
  bool                ext           = false;
  bool                dummy_present = false;
  fixed_bitstring<16> msg_id;
  fixed_bitstring<16> serial_num;
  fixed_octstring<2>  warning_type;
  fixed_octstring<50> dummy;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType11 ::= SEQUENCE
struct sib_type11_s {
  struct warning_msg_segment_type_opts {
    enum options { not_last_segment, last_segment, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<warning_msg_segment_type_opts> warning_msg_segment_type_e_;

  // member variables
  bool                        ext                        = false;
  bool                        data_coding_scheme_present = false;
  fixed_bitstring<16>         msg_id;
  fixed_bitstring<16>         serial_num;
  warning_msg_segment_type_e_ warning_msg_segment_type;
  uint8_t                     warning_msg_segment_num = 0;
  dyn_octstring               warning_msg_segment;
  fixed_octstring<1>          data_coding_scheme;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType12-r9 ::= SEQUENCE
struct sib_type12_r9_s {
  struct warning_msg_segment_type_r9_opts {
    enum options { not_last_segment, last_segment, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<warning_msg_segment_type_r9_opts> warning_msg_segment_type_r9_e_;

  // member variables
  bool                           ext                           = false;
  bool                           data_coding_scheme_r9_present = false;
  bool                           late_non_crit_ext_present     = false;
  fixed_bitstring<16>            msg_id_r9;
  fixed_bitstring<16>            serial_num_r9;
  warning_msg_segment_type_r9_e_ warning_msg_segment_type_r9;
  uint8_t                        warning_msg_segment_num_r9 = 0;
  dyn_octstring                  warning_msg_segment_r9;
  fixed_octstring<1>             data_coding_scheme_r9;
  dyn_octstring                  late_non_crit_ext;
  // ...
  // group 0
  bool          warning_area_coordinates_segment_r15_present = false;
  dyn_octstring warning_area_coordinates_segment_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType13-r9 ::= SEQUENCE
struct sib_type13_r9_s {
  bool                      ext                       = false;
  bool                      late_non_crit_ext_present = false;
  mbsfn_area_info_list_r9_l mbsfn_area_info_list_r9;
  mbms_notif_cfg_r9_s       notif_cfg_r9;
  dyn_octstring             late_non_crit_ext;
  // ...
  // group 0
  copy_ptr<mbms_notif_cfg_v1430_s> notif_cfg_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType14-r11 ::= SEQUENCE
struct sib_type14_r11_s {
  struct eab_param_r11_c_ {
    using eab_per_plmn_list_r11_l_ = dyn_array<eab_cfg_plmn_r11_s>;
    struct types_opts {
      enum options { eab_common_r11, eab_per_plmn_list_r11, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    eab_param_r11_c_() = default;
    eab_param_r11_c_(const eab_param_r11_c_& other);
    eab_param_r11_c_& operator=(const eab_param_r11_c_& other);
    ~eab_param_r11_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    eab_cfg_r11_s& eab_common_r11()
    {
      assert_choice_type("eab-Common-r11", type_.to_string(), "eab-Param-r11");
      return c.get<eab_cfg_r11_s>();
    }
    eab_per_plmn_list_r11_l_& eab_per_plmn_list_r11()
    {
      assert_choice_type("eab-PerPLMN-List-r11", type_.to_string(), "eab-Param-r11");
      return c.get<eab_per_plmn_list_r11_l_>();
    }
    const eab_cfg_r11_s& eab_common_r11() const
    {
      assert_choice_type("eab-Common-r11", type_.to_string(), "eab-Param-r11");
      return c.get<eab_cfg_r11_s>();
    }
    const eab_per_plmn_list_r11_l_& eab_per_plmn_list_r11() const
    {
      assert_choice_type("eab-PerPLMN-List-r11", type_.to_string(), "eab-Param-r11");
      return c.get<eab_per_plmn_list_r11_l_>();
    }
    eab_cfg_r11_s& set_eab_common_r11()
    {
      set(types::eab_common_r11);
      return c.get<eab_cfg_r11_s>();
    }
    eab_per_plmn_list_r11_l_& set_eab_per_plmn_list_r11()
    {
      set(types::eab_per_plmn_list_r11);
      return c.get<eab_per_plmn_list_r11_l_>();
    }

  private:
    types                                                    type_;
    choice_buffer_t<eab_cfg_r11_s, eab_per_plmn_list_r11_l_> c;

    void destroy_();
  };
  struct eab_per_rsrp_r15_opts {
    enum options { thresh0, thresh1, thresh2, thresh3, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<eab_per_rsrp_r15_opts> eab_per_rsrp_r15_e_;

  // member variables
  bool             ext                       = false;
  bool             eab_param_r11_present     = false;
  bool             late_non_crit_ext_present = false;
  eab_param_r11_c_ eab_param_r11;
  dyn_octstring    late_non_crit_ext;
  // ...
  // group 0
  bool                eab_per_rsrp_r15_present = false;
  eab_per_rsrp_r15_e_ eab_per_rsrp_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType15-r11 ::= SEQUENCE
struct sib_type15_r11_s {
  bool                           ext                                  = false;
  bool                           mbms_sai_intra_freq_r11_present      = false;
  bool                           mbms_sai_inter_freq_list_r11_present = false;
  bool                           late_non_crit_ext_present            = false;
  mbms_sai_list_r11_l            mbms_sai_intra_freq_r11;
  mbms_sai_inter_freq_list_r11_l mbms_sai_inter_freq_list_r11;
  dyn_octstring                  late_non_crit_ext;
  // ...
  // group 0
  copy_ptr<mbms_sai_inter_freq_list_v1140_l> mbms_sai_inter_freq_list_v1140;
  // group 1
  copy_ptr<mbms_carrier_type_r14_s>                 mbms_intra_freq_carrier_type_r14;
  copy_ptr<mbms_inter_freq_carrier_type_list_r14_l> mbms_inter_freq_carrier_type_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType16-r11 ::= SEQUENCE
struct sib_type16_r11_s {
  struct time_info_r11_s_ {
    bool               day_light_saving_time_r11_present = false;
    bool               leap_seconds_r11_present          = false;
    bool               local_time_offset_r11_present     = false;
    uint64_t           time_info_utc_r11                 = 0;
    fixed_bitstring<2> day_light_saving_time_r11;
    int16_t            leap_seconds_r11      = -127;
    int8_t             local_time_offset_r11 = -63;
  };

  // member variables
  bool             ext                       = false;
  bool             time_info_r11_present     = false;
  bool             late_non_crit_ext_present = false;
  time_info_r11_s_ time_info_r11;
  dyn_octstring    late_non_crit_ext;
  // ...
  // group 0
  copy_ptr<time_ref_info_r15_s> time_ref_info_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType17-r12 ::= SEQUENCE
struct sib_type17_r12_s {
  using wlan_offload_info_per_plmn_list_r12_l_ = dyn_array<wlan_offload_info_per_plmn_r12_s>;

  // member variables
  bool                                   ext                                         = false;
  bool                                   wlan_offload_info_per_plmn_list_r12_present = false;
  bool                                   late_non_crit_ext_present                   = false;
  wlan_offload_info_per_plmn_list_r12_l_ wlan_offload_info_per_plmn_list_r12;
  dyn_octstring                          late_non_crit_ext;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType18-r12 ::= SEQUENCE
struct sib_type18_r12_s {
  struct comm_cfg_r12_s_ {
    bool                       comm_tx_pool_normal_common_r12_present = false;
    bool                       comm_tx_pool_exceptional_r12_present   = false;
    bool                       comm_sync_cfg_r12_present              = false;
    sl_comm_rx_pool_list_r12_l comm_rx_pool_r12;
    sl_comm_tx_pool_list_r12_l comm_tx_pool_normal_common_r12;
    sl_comm_tx_pool_list_r12_l comm_tx_pool_exceptional_r12;
    sl_sync_cfg_list_r12_l     comm_sync_cfg_r12;
  };

  // member variables
  bool            ext                       = false;
  bool            comm_cfg_r12_present      = false;
  bool            late_non_crit_ext_present = false;
  comm_cfg_r12_s_ comm_cfg_r12;
  dyn_octstring   late_non_crit_ext;
  // ...
  // group 0
  bool                                     comm_tx_res_uc_req_allowed_r13_present = false;
  bool                                     comm_tx_allow_relay_common_r13_present = false;
  copy_ptr<sl_comm_tx_pool_list_ext_r13_l> comm_tx_pool_normal_common_ext_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType19-r12 ::= SEQUENCE
struct sib_type19_r12_s {
  struct disc_cfg_r12_s_ {
    bool                           disc_tx_pool_common_r12_present = false;
    bool                           disc_tx_pwr_info_r12_present    = false;
    bool                           disc_sync_cfg_r12_present       = false;
    sl_disc_rx_pool_list_r12_l     disc_rx_pool_r12;
    sl_disc_tx_pool_list_r12_l     disc_tx_pool_common_r12;
    sl_disc_tx_pwr_info_list_r12_l disc_tx_pwr_info_r12;
    sl_sync_cfg_list_r12_l         disc_sync_cfg_r12;
  };
  struct disc_cfg_v1310_s_ {
    bool                              disc_inter_freq_list_v1310_present  = false;
    bool                              gap_requests_allowed_common_present = false;
    sl_carrier_freq_info_list_v1310_l disc_inter_freq_list_v1310;
  };
  struct disc_cfg_relay_r13_s_ {
    sl_disc_cfg_relay_ue_r13_s  relay_ue_cfg_r13;
    sl_disc_cfg_remote_ue_r13_s remote_ue_cfg_r13;
  };
  struct disc_cfg_ps_minus13_s_ {
    bool                       disc_tx_pool_ps_common_r13_present = false;
    sl_disc_rx_pool_list_r12_l disc_rx_pool_ps_r13;
    sl_disc_tx_pool_list_r12_l disc_tx_pool_ps_common_r13;
  };

  // member variables
  bool                            ext                              = false;
  bool                            disc_cfg_r12_present             = false;
  bool                            disc_inter_freq_list_r12_present = false;
  bool                            late_non_crit_ext_present        = false;
  disc_cfg_r12_s_                 disc_cfg_r12;
  sl_carrier_freq_info_list_r12_l disc_inter_freq_list_r12;
  dyn_octstring                   late_non_crit_ext;
  // ...
  // group 0
  copy_ptr<disc_cfg_v1310_s_>      disc_cfg_v1310;
  copy_ptr<disc_cfg_relay_r13_s_>  disc_cfg_relay_r13;
  copy_ptr<disc_cfg_ps_minus13_s_> disc_cfg_ps_minus13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType2 ::= SEQUENCE
struct sib_type2_s {
  struct ac_barr_info_s_ {
    bool          ac_barr_for_mo_sig_present  = false;
    bool          ac_barr_for_mo_data_present = false;
    bool          ac_barr_for_emergency       = false;
    ac_barr_cfg_s ac_barr_for_mo_sig;
    ac_barr_cfg_s ac_barr_for_mo_data;
  };
  struct freq_info_s_ {
    struct ul_bw_opts {
      enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<ul_bw_opts> ul_bw_e_;

    // member variables
    bool     ul_carrier_freq_present = false;
    bool     ul_bw_present           = false;
    uint32_t ul_carrier_freq         = 0;
    ul_bw_e_ ul_bw;
    uint8_t  add_spec_emission = 1;
  };

  // member variables
  bool                   ext                       = false;
  bool                   ac_barr_info_present      = false;
  bool                   mbsfn_sf_cfg_list_present = false;
  ac_barr_info_s_        ac_barr_info;
  rr_cfg_common_sib_s    rr_cfg_common;
  ue_timers_and_consts_s ue_timers_and_consts;
  freq_info_s_           freq_info;
  mbsfn_sf_cfg_list_l    mbsfn_sf_cfg_list;
  time_align_timer_e     time_align_timer_common;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  copy_ptr<ac_barr_cfg_s> ssac_barr_for_mmtel_voice_r9;
  copy_ptr<ac_barr_cfg_s> ssac_barr_for_mmtel_video_r9;
  // group 1
  copy_ptr<ac_barr_cfg_s> ac_barr_for_csfb_r10;
  // group 2
  bool                                  ac_barr_skip_for_mmtel_voice_r12_present = false;
  bool                                  ac_barr_skip_for_mmtel_video_r12_present = false;
  bool                                  ac_barr_skip_for_sms_r12_present         = false;
  copy_ptr<ac_barr_per_plmn_list_r12_l> ac_barr_per_plmn_list_r12;
  // group 3
  bool voice_service_cause_ind_r12_present = false;
  // group 4
  copy_ptr<acdc_barr_for_common_r13_s>    acdc_barr_for_common_r13;
  copy_ptr<acdc_barr_per_plmn_list_r13_l> acdc_barr_per_plmn_list_r13;
  // group 5
  bool                                          use_full_resume_id_r13_present = false;
  copy_ptr<udt_restricting_r13_s>               udt_restricting_for_common_r13;
  copy_ptr<udt_restricting_per_plmn_list_r13_l> udt_restricting_per_plmn_list_r13;
  copy_ptr<ciot_eps_optim_info_r13_l>           cio_t_eps_optim_info_r13;
  // group 6
  bool unicast_freq_hop_ind_r13_present = false;
  // group 7
  bool                                video_service_cause_ind_r14_present = false;
  copy_ptr<mbsfn_sf_cfg_list_v1430_l> mbsfn_sf_cfg_list_v1430;
  // group 8
  copy_ptr<plmn_info_list_r15_l> plmn_info_list_r15;
  // group 9
  bool cp_edt_r15_present                     = false;
  bool up_edt_r15_present                     = false;
  bool idle_mode_meass_r15_present            = false;
  bool reduced_cp_latency_enabled_r15_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType20-r13 ::= SEQUENCE
struct sib_type20_r13_s {
  struct sc_mcch_repeat_period_r13_opts {
    enum options { rf2, rf4, rf8, rf16, rf32, rf64, rf128, rf256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<sc_mcch_repeat_period_r13_opts> sc_mcch_repeat_period_r13_e_;
  struct sc_mcch_mod_period_r13_opts {
    enum options {
      rf2,
      rf4,
      rf8,
      rf16,
      rf32,
      rf64,
      rf128,
      rf256,
      rf512,
      rf1024,
      r2048,
      rf4096,
      rf8192,
      rf16384,
      rf32768,
      rf65536,
      nulltype
    } value;
    typedef uint32_t number_type;

    std::string to_string() const;
    uint32_t    to_number() const;
  };
  typedef enumerated<sc_mcch_mod_period_r13_opts> sc_mcch_mod_period_r13_e_;
  struct br_bcch_cfg_r14_s_ {
    struct mpdcch_num_repeat_sc_mcch_r14_opts {
      enum options { r1, r2, r4, r8, r16, r32, r64, r128, r256, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<mpdcch_num_repeat_sc_mcch_r14_opts> mpdcch_num_repeat_sc_mcch_r14_e_;
    struct mpdcch_start_sf_sc_mcch_r14_c_ {
      struct fdd_r14_opts {
        enum options { v1, v1dot5, v2, v2dot5, v4, v5, v8, v10, nulltype } value;
        typedef float number_type;

        std::string to_string() const;
        float       to_number() const;
        std::string to_number_string() const;
      };
      typedef enumerated<fdd_r14_opts> fdd_r14_e_;
      struct tdd_r14_opts {
        enum options { v1, v2, v4, v5, v8, v10, v20, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<tdd_r14_opts> tdd_r14_e_;
      struct types_opts {
        enum options { fdd_r14, tdd_r14, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      mpdcch_start_sf_sc_mcch_r14_c_() = default;
      mpdcch_start_sf_sc_mcch_r14_c_(const mpdcch_start_sf_sc_mcch_r14_c_& other);
      mpdcch_start_sf_sc_mcch_r14_c_& operator=(const mpdcch_start_sf_sc_mcch_r14_c_& other);
      ~mpdcch_start_sf_sc_mcch_r14_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      fdd_r14_e_& fdd_r14()
      {
        assert_choice_type("fdd-r14", type_.to_string(), "mpdcch-StartSF-SC-MCCH-r14");
        return c.get<fdd_r14_e_>();
      }
      tdd_r14_e_& tdd_r14()
      {
        assert_choice_type("tdd-r14", type_.to_string(), "mpdcch-StartSF-SC-MCCH-r14");
        return c.get<tdd_r14_e_>();
      }
      const fdd_r14_e_& fdd_r14() const
      {
        assert_choice_type("fdd-r14", type_.to_string(), "mpdcch-StartSF-SC-MCCH-r14");
        return c.get<fdd_r14_e_>();
      }
      const tdd_r14_e_& tdd_r14() const
      {
        assert_choice_type("tdd-r14", type_.to_string(), "mpdcch-StartSF-SC-MCCH-r14");
        return c.get<tdd_r14_e_>();
      }
      fdd_r14_e_& set_fdd_r14()
      {
        set(types::fdd_r14);
        return c.get<fdd_r14_e_>();
      }
      tdd_r14_e_& set_tdd_r14()
      {
        set(types::tdd_r14);
        return c.get<tdd_r14_e_>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };
    struct mpdcch_pdsch_hop_cfg_sc_mcch_r14_opts {
      enum options { off, ce_mode_a, ce_mode_b, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<mpdcch_pdsch_hop_cfg_sc_mcch_r14_opts> mpdcch_pdsch_hop_cfg_sc_mcch_r14_e_;
    struct sc_mcch_repeat_period_br_r14_opts {
      enum options { rf32, rf128, rf512, rf1024, rf2048, rf4096, rf8192, rf16384, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<sc_mcch_repeat_period_br_r14_opts> sc_mcch_repeat_period_br_r14_e_;
    struct sc_mcch_mod_period_br_r14_opts {
      enum options {
        rf32,
        rf128,
        rf256,
        rf512,
        rf1024,
        rf2048,
        rf4096,
        rf8192,
        rf16384,
        rf32768,
        rf65536,
        rf131072,
        rf262144,
        rf524288,
        rf1048576,
        nulltype
      } value;
      typedef uint32_t number_type;

      std::string to_string() const;
      uint32_t    to_number() const;
    };
    typedef enumerated<sc_mcch_mod_period_br_r14_opts> sc_mcch_mod_period_br_r14_e_;

    // member variables
    uint8_t                             mpdcch_nb_sc_mcch_r14 = 1;
    mpdcch_num_repeat_sc_mcch_r14_e_    mpdcch_num_repeat_sc_mcch_r14;
    mpdcch_start_sf_sc_mcch_r14_c_      mpdcch_start_sf_sc_mcch_r14;
    mpdcch_pdsch_hop_cfg_sc_mcch_r14_e_ mpdcch_pdsch_hop_cfg_sc_mcch_r14;
    uint32_t                            sc_mcch_carrier_freq_r14 = 0;
    uint8_t                             sc_mcch_offset_br_r14    = 0;
    sc_mcch_repeat_period_br_r14_e_     sc_mcch_repeat_period_br_r14;
    sc_mcch_mod_period_br_r14_e_        sc_mcch_mod_period_br_r14;
  };
  struct pdsch_max_num_repeat_cemode_a_sc_mtch_r14_opts {
    enum options { r16, r32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pdsch_max_num_repeat_cemode_a_sc_mtch_r14_opts> pdsch_max_num_repeat_cemode_a_sc_mtch_r14_e_;
  struct pdsch_max_num_repeat_cemode_b_sc_mtch_r14_opts {
    enum options { r192, r256, r384, r512, r768, r1024, r1536, r2048, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<pdsch_max_num_repeat_cemode_b_sc_mtch_r14_opts> pdsch_max_num_repeat_cemode_b_sc_mtch_r14_e_;

  // member variables
  bool                         ext                       = false;
  bool                         sc_mcch_dur_r13_present   = false;
  bool                         late_non_crit_ext_present = false;
  sc_mcch_repeat_period_r13_e_ sc_mcch_repeat_period_r13;
  uint8_t                      sc_mcch_offset_r13   = 0;
  uint8_t                      sc_mcch_first_sf_r13 = 0;
  uint8_t                      sc_mcch_dur_r13      = 2;
  sc_mcch_mod_period_r13_e_    sc_mcch_mod_period_r13;
  dyn_octstring                late_non_crit_ext;
  // ...
  // group 0
  bool                                         pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present = false;
  bool                                         pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present = false;
  copy_ptr<br_bcch_cfg_r14_s_>                 br_bcch_cfg_r14;
  copy_ptr<sc_mcch_sched_info_r14_s>           sc_mcch_sched_info_r14;
  pdsch_max_num_repeat_cemode_a_sc_mtch_r14_e_ pdsch_max_num_repeat_cemode_a_sc_mtch_r14;
  pdsch_max_num_repeat_cemode_b_sc_mtch_r14_e_ pdsch_max_num_repeat_cemode_b_sc_mtch_r14;
  // group 1
  bool sc_mcch_repeat_period_v1470_present = false;
  bool sc_mcch_mod_period_v1470_present    = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType21-r14 ::= SEQUENCE
struct sib_type21_r14_s {
  bool                    ext                           = false;
  bool                    sl_v2x_cfg_common_r14_present = false;
  bool                    late_non_crit_ext_present     = false;
  sl_v2x_cfg_common_r14_s sl_v2x_cfg_common_r14;
  dyn_octstring           late_non_crit_ext;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType24-r15 ::= SEQUENCE
struct sib_type24_r15_s {
  bool                        ext                              = false;
  bool                        carrier_freq_list_nr_r15_present = false;
  bool                        t_resel_nr_sf_r15_present        = false;
  bool                        late_non_crit_ext_present        = false;
  carrier_freq_list_nr_r15_l  carrier_freq_list_nr_r15;
  uint8_t                     t_resel_nr_r15 = 0;
  speed_state_scale_factors_s t_resel_nr_sf_r15;
  dyn_octstring               late_non_crit_ext;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType25-r15 ::= SEQUENCE
struct sib_type25_r15_s {
  struct uac_ac1_select_assist_info_r15_c_ {
    using individual_plmn_list_r15_l_ = bounded_array<uac_ac1_select_assist_info_r15_e, 6>;
    struct types_opts {
      enum options { plmn_common_r15, individual_plmn_list_r15, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    uac_ac1_select_assist_info_r15_c_() = default;
    uac_ac1_select_assist_info_r15_c_(const uac_ac1_select_assist_info_r15_c_& other);
    uac_ac1_select_assist_info_r15_c_& operator=(const uac_ac1_select_assist_info_r15_c_& other);
    ~uac_ac1_select_assist_info_r15_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uac_ac1_select_assist_info_r15_e& plmn_common_r15()
    {
      assert_choice_type("plmnCommon-r15", type_.to_string(), "uac-AC1-SelectAssistInfo-r15");
      return c.get<uac_ac1_select_assist_info_r15_e>();
    }
    individual_plmn_list_r15_l_& individual_plmn_list_r15()
    {
      assert_choice_type("individualPLMNList-r15", type_.to_string(), "uac-AC1-SelectAssistInfo-r15");
      return c.get<individual_plmn_list_r15_l_>();
    }
    const uac_ac1_select_assist_info_r15_e& plmn_common_r15() const
    {
      assert_choice_type("plmnCommon-r15", type_.to_string(), "uac-AC1-SelectAssistInfo-r15");
      return c.get<uac_ac1_select_assist_info_r15_e>();
    }
    const individual_plmn_list_r15_l_& individual_plmn_list_r15() const
    {
      assert_choice_type("individualPLMNList-r15", type_.to_string(), "uac-AC1-SelectAssistInfo-r15");
      return c.get<individual_plmn_list_r15_l_>();
    }
    uac_ac1_select_assist_info_r15_e& set_plmn_common_r15()
    {
      set(types::plmn_common_r15);
      return c.get<uac_ac1_select_assist_info_r15_e>();
    }
    individual_plmn_list_r15_l_& set_individual_plmn_list_r15()
    {
      set(types::individual_plmn_list_r15);
      return c.get<individual_plmn_list_r15_l_>();
    }

  private:
    types                                        type_;
    choice_buffer_t<individual_plmn_list_r15_l_> c;

    void destroy_();
  };

  // member variables
  bool                              ext                                    = false;
  bool                              uac_barr_for_common_r15_present        = false;
  bool                              uac_barr_per_plmn_list_r15_present     = false;
  bool                              uac_ac1_select_assist_info_r15_present = false;
  bool                              late_non_crit_ext_present              = false;
  uac_barr_per_cat_list_r15_l       uac_barr_for_common_r15;
  uac_barr_per_plmn_list_r15_l      uac_barr_per_plmn_list_r15;
  uac_barr_info_set_list_r15_l      uac_barr_info_set_list_r15;
  uac_ac1_select_assist_info_r15_c_ uac_ac1_select_assist_info_r15;
  dyn_octstring                     late_non_crit_ext;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType26-r15 ::= SEQUENCE
struct sib_type26_r15_s {
  bool                              ext                                  = false;
  bool                              v2x_inter_freq_info_list_r15_present = false;
  bool                              cbr_pssch_tx_cfg_list_r15_present    = false;
  bool                              v2x_packet_dupl_cfg_r15_present      = false;
  bool                              sync_freq_list_r15_present           = false;
  bool                              slss_tx_multi_freq_r15_present       = false;
  bool                              v2x_freq_sel_cfg_list_r15_present    = false;
  bool                              thresh_s_rssi_cbr_r14_present        = false;
  sl_inter_freq_info_list_v2x_r14_l v2x_inter_freq_info_list_r15;
  sl_cbr_pppp_tx_cfg_list_r15_l     cbr_pssch_tx_cfg_list_r15;
  sl_v2x_packet_dupl_cfg_r15_s      v2x_packet_dupl_cfg_r15;
  sl_v2x_sync_freq_list_r15_l       sync_freq_list_r15;
  sl_v2x_freq_sel_cfg_list_r15_l    v2x_freq_sel_cfg_list_r15;
  uint8_t                           thresh_s_rssi_cbr_r14 = 0;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType3 ::= SEQUENCE
struct sib_type3_s {
  struct cell_resel_info_common_s_ {
    struct q_hyst_opts {
      enum options {
        db0,
        db1,
        db2,
        db3,
        db4,
        db5,
        db6,
        db8,
        db10,
        db12,
        db14,
        db16,
        db18,
        db20,
        db22,
        db24,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<q_hyst_opts> q_hyst_e_;
    struct speed_state_resel_pars_s_ {
      struct q_hyst_sf_s_ {
        struct sf_medium_opts {
          enum options { db_minus6, db_minus4, db_minus2, db0, nulltype } value;
          typedef int8_t number_type;

          std::string to_string() const;
          int8_t      to_number() const;
        };
        typedef enumerated<sf_medium_opts> sf_medium_e_;
        struct sf_high_opts {
          enum options { db_minus6, db_minus4, db_minus2, db0, nulltype } value;
          typedef int8_t number_type;

          std::string to_string() const;
          int8_t      to_number() const;
        };
        typedef enumerated<sf_high_opts> sf_high_e_;

        // member variables
        sf_medium_e_ sf_medium;
        sf_high_e_   sf_high;
      };

      // member variables
      mob_state_params_s mob_state_params;
      q_hyst_sf_s_       q_hyst_sf;
    };

    // member variables
    bool                      speed_state_resel_pars_present = false;
    q_hyst_e_                 q_hyst;
    speed_state_resel_pars_s_ speed_state_resel_pars;
  };
  struct cell_resel_serving_freq_info_s_ {
    bool    s_non_intra_search_present = false;
    uint8_t s_non_intra_search         = 0;
    uint8_t thresh_serving_low         = 0;
    uint8_t cell_resel_prio            = 0;
  };
  struct intra_freq_cell_resel_info_s_ {
    bool                        p_max_present            = false;
    bool                        s_intra_search_present   = false;
    bool                        allowed_meas_bw_present  = false;
    bool                        t_resel_eutra_sf_present = false;
    int8_t                      q_rx_lev_min             = -70;
    int8_t                      p_max                    = -30;
    uint8_t                     s_intra_search           = 0;
    allowed_meas_bw_e           allowed_meas_bw;
    bool                        presence_ant_port1 = false;
    fixed_bitstring<2>          neigh_cell_cfg;
    uint8_t                     t_resel_eutra = 0;
    speed_state_scale_factors_s t_resel_eutra_sf;
  };
  struct s_intra_search_v920_s_ {
    uint8_t s_intra_search_p_r9 = 0;
    uint8_t s_intra_search_q_r9 = 0;
  };
  struct s_non_intra_search_v920_s_ {
    uint8_t s_non_intra_search_p_r9 = 0;
    uint8_t s_non_intra_search_q_r9 = 0;
  };

  // member variables
  bool                            ext = false;
  cell_resel_info_common_s_       cell_resel_info_common;
  cell_resel_serving_freq_info_s_ cell_resel_serving_freq_info;
  intra_freq_cell_resel_info_s_   intra_freq_cell_resel_info;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  bool                                 q_qual_min_r9_present           = false;
  bool                                 thresh_serving_low_q_r9_present = false;
  copy_ptr<s_intra_search_v920_s_>     s_intra_search_v920;
  copy_ptr<s_non_intra_search_v920_s_> s_non_intra_search_v920;
  int8_t                               q_qual_min_r9           = -34;
  uint8_t                              thresh_serving_low_q_r9 = 0;
  // group 1
  bool   q_qual_min_wb_r11_present = false;
  int8_t q_qual_min_wb_r11         = -34;
  // group 2
  bool   q_qual_min_rsrq_on_all_symbols_r12_present = false;
  int8_t q_qual_min_rsrq_on_all_symbols_r12         = -34;
  // group 3
  bool                                           t_resel_eutra_ce_r13_present = false;
  copy_ptr<cell_resel_serving_freq_info_v1310_s> cell_resel_serving_freq_info_v1310;
  copy_ptr<redist_serving_info_r13_s>            redist_serving_info_r13;
  copy_ptr<cell_sel_info_ce_r13_s>               cell_sel_info_ce_r13;
  uint8_t                                        t_resel_eutra_ce_r13 = 0;
  // group 4
  copy_ptr<cell_sel_info_ce1_r13_s> cell_sel_info_ce1_r13;
  // group 5
  copy_ptr<cell_sel_info_ce1_v1360_s> cell_sel_info_ce1_v1360;
  // group 6
  copy_ptr<cell_resel_info_common_v1460_s> cell_resel_info_common_v1460;
  // group 7
  bool                                 crs_intf_mitig_neigh_cells_ce_r15_present = false;
  copy_ptr<cell_resel_info_hsdn_r15_s> cell_resel_info_hsdn_r15;
  copy_ptr<cell_sel_info_ce_v1530_s>   cell_sel_info_ce_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType4 ::= SEQUENCE
struct sib_type4_s {
  bool                         ext                                = false;
  bool                         intra_freq_neigh_cell_list_present = false;
  bool                         intra_freq_black_cell_list_present = false;
  bool                         csg_pci_range_present              = false;
  intra_freq_neigh_cell_list_l intra_freq_neigh_cell_list;
  intra_freq_black_cell_list_l intra_freq_black_cell_list;
  pci_range_s                  csg_pci_range;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  copy_ptr<intra_freq_neigh_hsdn_cell_list_r15_l> intra_freq_neigh_hsdn_cell_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType5 ::= SEQUENCE
struct sib_type5_s {
  bool                           ext = false;
  inter_freq_carrier_freq_list_l inter_freq_carrier_freq_list;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  copy_ptr<inter_freq_carrier_freq_list_v1250_l>   inter_freq_carrier_freq_list_v1250;
  copy_ptr<inter_freq_carrier_freq_list_ext_r12_l> inter_freq_carrier_freq_list_ext_r12;
  // group 1
  copy_ptr<inter_freq_carrier_freq_list_ext_v1280_l> inter_freq_carrier_freq_list_ext_v1280;
  // group 2
  copy_ptr<inter_freq_carrier_freq_list_v1310_l>     inter_freq_carrier_freq_list_v1310;
  copy_ptr<inter_freq_carrier_freq_list_ext_v1310_l> inter_freq_carrier_freq_list_ext_v1310;
  // group 3
  copy_ptr<inter_freq_carrier_freq_list_v1350_l>     inter_freq_carrier_freq_list_v1350;
  copy_ptr<inter_freq_carrier_freq_list_ext_v1350_l> inter_freq_carrier_freq_list_ext_v1350;
  // group 4
  copy_ptr<inter_freq_carrier_freq_list_ext_v1360_l> inter_freq_carrier_freq_list_ext_v1360;
  // group 5
  bool    scptm_freq_offset_r14_present = false;
  uint8_t scptm_freq_offset_r14         = 1;
  // group 6
  copy_ptr<inter_freq_carrier_freq_list_v1530_l>     inter_freq_carrier_freq_list_v1530;
  copy_ptr<inter_freq_carrier_freq_list_ext_v1530_l> inter_freq_carrier_freq_list_ext_v1530;
  copy_ptr<meas_idle_cfg_sib_r15_s>                  meas_idle_cfg_sib_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType6 ::= SEQUENCE
struct sib_type6_s {
  using carrier_freq_list_utra_fdd_v1250_l_ = dyn_array<carrier_freq_info_utra_v1250_s>;
  using carrier_freq_list_utra_tdd_v1250_l_ = dyn_array<carrier_freq_info_utra_v1250_s>;

  // member variables
  bool                         ext                                = false;
  bool                         carrier_freq_list_utra_fdd_present = false;
  bool                         carrier_freq_list_utra_tdd_present = false;
  bool                         t_resel_utra_sf_present            = false;
  carrier_freq_list_utra_fdd_l carrier_freq_list_utra_fdd;
  carrier_freq_list_utra_tdd_l carrier_freq_list_utra_tdd;
  uint8_t                      t_resel_utra = 0;
  speed_state_scale_factors_s  t_resel_utra_sf;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  copy_ptr<carrier_freq_list_utra_fdd_v1250_l_>  carrier_freq_list_utra_fdd_v1250;
  copy_ptr<carrier_freq_list_utra_tdd_v1250_l_>  carrier_freq_list_utra_tdd_v1250;
  copy_ptr<carrier_freq_list_utra_fdd_ext_r12_l> carrier_freq_list_utra_fdd_ext_r12;
  copy_ptr<carrier_freq_list_utra_tdd_ext_r12_l> carrier_freq_list_utra_tdd_ext_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType7 ::= SEQUENCE
struct sib_type7_s {
  bool                            ext                             = false;
  bool                            t_resel_geran_sf_present        = false;
  bool                            carrier_freqs_info_list_present = false;
  uint8_t                         t_resel_geran                   = 0;
  speed_state_scale_factors_s     t_resel_geran_sf;
  carrier_freqs_info_list_geran_l carrier_freqs_info_list;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType8 ::= SEQUENCE
struct sib_type8_s {
  struct params_hrpd_s_ {
    bool                         cell_resel_params_hrpd_present = false;
    pre_regist_info_hrpd_s       pre_regist_info_hrpd;
    cell_resel_params_cdma2000_s cell_resel_params_hrpd;
  };
  struct params1_xrtt_s_ {
    bool                         csfb_regist_param1_xrtt_present = false;
    bool                         long_code_state1_xrtt_present   = false;
    bool                         cell_resel_params1_xrtt_present = false;
    csfb_regist_param1_xrtt_s    csfb_regist_param1_xrtt;
    fixed_bitstring<42>          long_code_state1_xrtt;
    cell_resel_params_cdma2000_s cell_resel_params1_xrtt;
  };

  // member variables
  bool                     ext                     = false;
  bool                     sys_time_info_present   = false;
  bool                     search_win_size_present = false;
  bool                     params_hrpd_present     = false;
  bool                     params1_xrtt_present    = false;
  sys_time_info_cdma2000_s sys_time_info;
  uint8_t                  search_win_size = 0;
  params_hrpd_s_           params_hrpd;
  params1_xrtt_s_          params1_xrtt;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;
  // group 0
  bool                                        csfb_support_for_dual_rx_ues_r9_present = false;
  bool                                        csfb_support_for_dual_rx_ues_r9         = false;
  copy_ptr<cell_resel_params_cdma2000_v920_s> cell_resel_params_hrpd_v920;
  copy_ptr<cell_resel_params_cdma2000_v920_s> cell_resel_params1_xrtt_v920;
  copy_ptr<csfb_regist_param1_xrtt_v920_s>    csfb_regist_param1_xrtt_v920;
  copy_ptr<ac_barr_cfg1_xrtt_r9_s>            ac_barr_cfg1_xrtt_r9;
  // group 1
  bool csfb_dual_rx_tx_support_r10_present = false;
  // group 2
  copy_ptr<sib8_per_plmn_list_r11_l> sib8_per_plmn_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType9 ::= SEQUENCE
struct sib_type9_s {
  bool          ext              = false;
  bool          hnb_name_present = false;
  dyn_octstring hnb_name;
  // ...
  bool          late_non_crit_ext_present = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PosSystemInformation-r15-IEs ::= SEQUENCE
struct pos_sys_info_r15_ies_s {
  struct pos_sib_type_and_info_r15_item_c_ {
    struct types_opts {
      enum options {
        pos_sib1_minus1_r15,
        pos_sib1_minus2_r15,
        pos_sib1_minus3_r15,
        pos_sib1_minus4_r15,
        pos_sib1_minus5_r15,
        pos_sib1_minus6_r15,
        pos_sib1_minus7_r15,
        pos_sib2_minus1_r15,
        pos_sib2_minus2_r15,
        pos_sib2_minus3_r15,
        pos_sib2_minus4_r15,
        pos_sib2_minus5_r15,
        pos_sib2_minus6_r15,
        pos_sib2_minus7_r15,
        pos_sib2_minus8_r15,
        pos_sib2_minus9_r15,
        pos_sib2_minus10_r15,
        pos_sib2_minus11_r15,
        pos_sib2_minus12_r15,
        pos_sib2_minus13_r15,
        pos_sib2_minus14_r15,
        pos_sib2_minus15_r15,
        pos_sib2_minus16_r15,
        pos_sib2_minus17_r15,
        pos_sib2_minus18_r15,
        pos_sib2_minus19_r15,
        pos_sib3_minus1_r15,
        // ...
        nulltype
      } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts, true> types;

    // choice methods
    pos_sib_type_and_info_r15_item_c_() = default;
    pos_sib_type_and_info_r15_item_c_(const pos_sib_type_and_info_r15_item_c_& other);
    pos_sib_type_and_info_r15_item_c_& operator=(const pos_sib_type_and_info_r15_item_c_& other);
    ~pos_sib_type_and_info_r15_item_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sib_pos_r15_s& pos_sib1_minus1_r15()
    {
      assert_choice_type("posSib1-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus2_r15()
    {
      assert_choice_type("posSib1-2-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus3_r15()
    {
      assert_choice_type("posSib1-3-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus4_r15()
    {
      assert_choice_type("posSib1-4-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus5_r15()
    {
      assert_choice_type("posSib1-5-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus6_r15()
    {
      assert_choice_type("posSib1-6-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib1_minus7_r15()
    {
      assert_choice_type("posSib1-7-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus1_r15()
    {
      assert_choice_type("posSib2-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus2_r15()
    {
      assert_choice_type("posSib2-2-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus3_r15()
    {
      assert_choice_type("posSib2-3-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus4_r15()
    {
      assert_choice_type("posSib2-4-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus5_r15()
    {
      assert_choice_type("posSib2-5-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus6_r15()
    {
      assert_choice_type("posSib2-6-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus7_r15()
    {
      assert_choice_type("posSib2-7-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus8_r15()
    {
      assert_choice_type("posSib2-8-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus9_r15()
    {
      assert_choice_type("posSib2-9-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus10_r15()
    {
      assert_choice_type("posSib2-10-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus11_r15()
    {
      assert_choice_type("posSib2-11-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus12_r15()
    {
      assert_choice_type("posSib2-12-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus13_r15()
    {
      assert_choice_type("posSib2-13-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus14_r15()
    {
      assert_choice_type("posSib2-14-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus15_r15()
    {
      assert_choice_type("posSib2-15-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus16_r15()
    {
      assert_choice_type("posSib2-16-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus17_r15()
    {
      assert_choice_type("posSib2-17-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus18_r15()
    {
      assert_choice_type("posSib2-18-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib2_minus19_r15()
    {
      assert_choice_type("posSib2-19-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& pos_sib3_minus1_r15()
    {
      assert_choice_type("posSib3-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus1_r15() const
    {
      assert_choice_type("posSib1-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus2_r15() const
    {
      assert_choice_type("posSib1-2-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus3_r15() const
    {
      assert_choice_type("posSib1-3-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus4_r15() const
    {
      assert_choice_type("posSib1-4-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus5_r15() const
    {
      assert_choice_type("posSib1-5-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus6_r15() const
    {
      assert_choice_type("posSib1-6-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib1_minus7_r15() const
    {
      assert_choice_type("posSib1-7-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus1_r15() const
    {
      assert_choice_type("posSib2-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus2_r15() const
    {
      assert_choice_type("posSib2-2-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus3_r15() const
    {
      assert_choice_type("posSib2-3-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus4_r15() const
    {
      assert_choice_type("posSib2-4-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus5_r15() const
    {
      assert_choice_type("posSib2-5-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus6_r15() const
    {
      assert_choice_type("posSib2-6-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus7_r15() const
    {
      assert_choice_type("posSib2-7-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus8_r15() const
    {
      assert_choice_type("posSib2-8-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus9_r15() const
    {
      assert_choice_type("posSib2-9-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus10_r15() const
    {
      assert_choice_type("posSib2-10-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus11_r15() const
    {
      assert_choice_type("posSib2-11-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus12_r15() const
    {
      assert_choice_type("posSib2-12-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus13_r15() const
    {
      assert_choice_type("posSib2-13-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus14_r15() const
    {
      assert_choice_type("posSib2-14-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus15_r15() const
    {
      assert_choice_type("posSib2-15-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus16_r15() const
    {
      assert_choice_type("posSib2-16-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus17_r15() const
    {
      assert_choice_type("posSib2-17-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus18_r15() const
    {
      assert_choice_type("posSib2-18-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib2_minus19_r15() const
    {
      assert_choice_type("posSib2-19-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    const sib_pos_r15_s& pos_sib3_minus1_r15() const
    {
      assert_choice_type("posSib3-1-r15", type_.to_string(), "posSIB-TypeAndInfo-r15-item");
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus1_r15()
    {
      set(types::pos_sib1_minus1_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus2_r15()
    {
      set(types::pos_sib1_minus2_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus3_r15()
    {
      set(types::pos_sib1_minus3_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus4_r15()
    {
      set(types::pos_sib1_minus4_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus5_r15()
    {
      set(types::pos_sib1_minus5_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus6_r15()
    {
      set(types::pos_sib1_minus6_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib1_minus7_r15()
    {
      set(types::pos_sib1_minus7_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus1_r15()
    {
      set(types::pos_sib2_minus1_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus2_r15()
    {
      set(types::pos_sib2_minus2_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus3_r15()
    {
      set(types::pos_sib2_minus3_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus4_r15()
    {
      set(types::pos_sib2_minus4_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus5_r15()
    {
      set(types::pos_sib2_minus5_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus6_r15()
    {
      set(types::pos_sib2_minus6_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus7_r15()
    {
      set(types::pos_sib2_minus7_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus8_r15()
    {
      set(types::pos_sib2_minus8_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus9_r15()
    {
      set(types::pos_sib2_minus9_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus10_r15()
    {
      set(types::pos_sib2_minus10_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus11_r15()
    {
      set(types::pos_sib2_minus11_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus12_r15()
    {
      set(types::pos_sib2_minus12_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus13_r15()
    {
      set(types::pos_sib2_minus13_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus14_r15()
    {
      set(types::pos_sib2_minus14_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus15_r15()
    {
      set(types::pos_sib2_minus15_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus16_r15()
    {
      set(types::pos_sib2_minus16_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus17_r15()
    {
      set(types::pos_sib2_minus17_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus18_r15()
    {
      set(types::pos_sib2_minus18_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib2_minus19_r15()
    {
      set(types::pos_sib2_minus19_r15);
      return c.get<sib_pos_r15_s>();
    }
    sib_pos_r15_s& set_pos_sib3_minus1_r15()
    {
      set(types::pos_sib3_minus1_r15);
      return c.get<sib_pos_r15_s>();
    }

  private:
    types                          type_;
    choice_buffer_t<sib_pos_r15_s> c;

    void destroy_();
  };
  using pos_sib_type_and_info_r15_l_ = dyn_array<pos_sib_type_and_info_r15_item_c_>;

  // member variables
  bool                         late_non_crit_ext_present = false;
  bool                         non_crit_ext_present      = false;
  pos_sib_type_and_info_r15_l_ pos_sib_type_and_info_r15;
  dyn_octstring                late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SchedulingInfoList ::= SEQUENCE (SIZE (1..32)) OF SchedulingInfo
using sched_info_list_l = dyn_array<sched_info_s>;

struct sib_info_item_c {
  struct types_opts {
    enum options {
      sib2,
      sib3,
      sib4,
      sib5,
      sib6,
      sib7,
      sib8,
      sib9,
      sib10,
      sib11,
      // ...
      sib12_v920,
      sib13_v920,
      sib14_v1130,
      sib15_v1130,
      sib16_v1130,
      sib17_v1250,
      sib18_v1250,
      sib19_v1250,
      sib20_v1310,
      sib21_v1430,
      sib24_v1530,
      sib25_v1530,
      sib26_v1530,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts, true, 13> types;

  // choice methods
  sib_info_item_c() = default;
  sib_info_item_c(const sib_info_item_c& other);
  sib_info_item_c& operator=(const sib_info_item_c& other);
  ~sib_info_item_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  sib_type2_s& sib2()
  {
    assert_choice_type("sib2", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type2_s>();
  }
  sib_type3_s& sib3()
  {
    assert_choice_type("sib3", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type3_s>();
  }
  sib_type4_s& sib4()
  {
    assert_choice_type("sib4", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type4_s>();
  }
  sib_type5_s& sib5()
  {
    assert_choice_type("sib5", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type5_s>();
  }
  sib_type6_s& sib6()
  {
    assert_choice_type("sib6", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type6_s>();
  }
  sib_type7_s& sib7()
  {
    assert_choice_type("sib7", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type7_s>();
  }
  sib_type8_s& sib8()
  {
    assert_choice_type("sib8", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type8_s>();
  }
  sib_type9_s& sib9()
  {
    assert_choice_type("sib9", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type9_s>();
  }
  sib_type10_s& sib10()
  {
    assert_choice_type("sib10", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type10_s>();
  }
  sib_type11_s& sib11()
  {
    assert_choice_type("sib11", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type11_s>();
  }
  sib_type12_r9_s& sib12_v920()
  {
    assert_choice_type("sib12-v920", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type12_r9_s>();
  }
  sib_type13_r9_s& sib13_v920()
  {
    assert_choice_type("sib13-v920", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type13_r9_s>();
  }
  sib_type14_r11_s& sib14_v1130()
  {
    assert_choice_type("sib14-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type14_r11_s>();
  }
  sib_type15_r11_s& sib15_v1130()
  {
    assert_choice_type("sib15-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type15_r11_s>();
  }
  sib_type16_r11_s& sib16_v1130()
  {
    assert_choice_type("sib16-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type16_r11_s>();
  }
  sib_type17_r12_s& sib17_v1250()
  {
    assert_choice_type("sib17-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type17_r12_s>();
  }
  sib_type18_r12_s& sib18_v1250()
  {
    assert_choice_type("sib18-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type18_r12_s>();
  }
  sib_type19_r12_s& sib19_v1250()
  {
    assert_choice_type("sib19-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type19_r12_s>();
  }
  sib_type20_r13_s& sib20_v1310()
  {
    assert_choice_type("sib20-v1310", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type20_r13_s>();
  }
  sib_type21_r14_s& sib21_v1430()
  {
    assert_choice_type("sib21-v1430", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type21_r14_s>();
  }
  sib_type24_r15_s& sib24_v1530()
  {
    assert_choice_type("sib24-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type24_r15_s>();
  }
  sib_type25_r15_s& sib25_v1530()
  {
    assert_choice_type("sib25-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type25_r15_s>();
  }
  sib_type26_r15_s& sib26_v1530()
  {
    assert_choice_type("sib26-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type26_r15_s>();
  }
  const sib_type2_s& sib2() const
  {
    assert_choice_type("sib2", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type2_s>();
  }
  const sib_type3_s& sib3() const
  {
    assert_choice_type("sib3", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type3_s>();
  }
  const sib_type4_s& sib4() const
  {
    assert_choice_type("sib4", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type4_s>();
  }
  const sib_type5_s& sib5() const
  {
    assert_choice_type("sib5", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type5_s>();
  }
  const sib_type6_s& sib6() const
  {
    assert_choice_type("sib6", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type6_s>();
  }
  const sib_type7_s& sib7() const
  {
    assert_choice_type("sib7", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type7_s>();
  }
  const sib_type8_s& sib8() const
  {
    assert_choice_type("sib8", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type8_s>();
  }
  const sib_type9_s& sib9() const
  {
    assert_choice_type("sib9", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type9_s>();
  }
  const sib_type10_s& sib10() const
  {
    assert_choice_type("sib10", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type10_s>();
  }
  const sib_type11_s& sib11() const
  {
    assert_choice_type("sib11", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type11_s>();
  }
  const sib_type12_r9_s& sib12_v920() const
  {
    assert_choice_type("sib12-v920", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type12_r9_s>();
  }
  const sib_type13_r9_s& sib13_v920() const
  {
    assert_choice_type("sib13-v920", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type13_r9_s>();
  }
  const sib_type14_r11_s& sib14_v1130() const
  {
    assert_choice_type("sib14-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type14_r11_s>();
  }
  const sib_type15_r11_s& sib15_v1130() const
  {
    assert_choice_type("sib15-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type15_r11_s>();
  }
  const sib_type16_r11_s& sib16_v1130() const
  {
    assert_choice_type("sib16-v1130", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type16_r11_s>();
  }
  const sib_type17_r12_s& sib17_v1250() const
  {
    assert_choice_type("sib17-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type17_r12_s>();
  }
  const sib_type18_r12_s& sib18_v1250() const
  {
    assert_choice_type("sib18-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type18_r12_s>();
  }
  const sib_type19_r12_s& sib19_v1250() const
  {
    assert_choice_type("sib19-v1250", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type19_r12_s>();
  }
  const sib_type20_r13_s& sib20_v1310() const
  {
    assert_choice_type("sib20-v1310", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type20_r13_s>();
  }
  const sib_type21_r14_s& sib21_v1430() const
  {
    assert_choice_type("sib21-v1430", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type21_r14_s>();
  }
  const sib_type24_r15_s& sib24_v1530() const
  {
    assert_choice_type("sib24-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type24_r15_s>();
  }
  const sib_type25_r15_s& sib25_v1530() const
  {
    assert_choice_type("sib25-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type25_r15_s>();
  }
  const sib_type26_r15_s& sib26_v1530() const
  {
    assert_choice_type("sib26-v1530", type_.to_string(), "sib-TypeAndInfo-item");
    return c.get<sib_type26_r15_s>();
  }
  sib_type2_s& set_sib2()
  {
    set(types::sib2);
    return c.get<sib_type2_s>();
  }
  sib_type3_s& set_sib3()
  {
    set(types::sib3);
    return c.get<sib_type3_s>();
  }
  sib_type4_s& set_sib4()
  {
    set(types::sib4);
    return c.get<sib_type4_s>();
  }
  sib_type5_s& set_sib5()
  {
    set(types::sib5);
    return c.get<sib_type5_s>();
  }
  sib_type6_s& set_sib6()
  {
    set(types::sib6);
    return c.get<sib_type6_s>();
  }
  sib_type7_s& set_sib7()
  {
    set(types::sib7);
    return c.get<sib_type7_s>();
  }
  sib_type8_s& set_sib8()
  {
    set(types::sib8);
    return c.get<sib_type8_s>();
  }
  sib_type9_s& set_sib9()
  {
    set(types::sib9);
    return c.get<sib_type9_s>();
  }
  sib_type10_s& set_sib10()
  {
    set(types::sib10);
    return c.get<sib_type10_s>();
  }
  sib_type11_s& set_sib11()
  {
    set(types::sib11);
    return c.get<sib_type11_s>();
  }
  sib_type12_r9_s& set_sib12_v920()
  {
    set(types::sib12_v920);
    return c.get<sib_type12_r9_s>();
  }
  sib_type13_r9_s& set_sib13_v920()
  {
    set(types::sib13_v920);
    return c.get<sib_type13_r9_s>();
  }
  sib_type14_r11_s& set_sib14_v1130()
  {
    set(types::sib14_v1130);
    return c.get<sib_type14_r11_s>();
  }
  sib_type15_r11_s& set_sib15_v1130()
  {
    set(types::sib15_v1130);
    return c.get<sib_type15_r11_s>();
  }
  sib_type16_r11_s& set_sib16_v1130()
  {
    set(types::sib16_v1130);
    return c.get<sib_type16_r11_s>();
  }
  sib_type17_r12_s& set_sib17_v1250()
  {
    set(types::sib17_v1250);
    return c.get<sib_type17_r12_s>();
  }
  sib_type18_r12_s& set_sib18_v1250()
  {
    set(types::sib18_v1250);
    return c.get<sib_type18_r12_s>();
  }
  sib_type19_r12_s& set_sib19_v1250()
  {
    set(types::sib19_v1250);
    return c.get<sib_type19_r12_s>();
  }
  sib_type20_r13_s& set_sib20_v1310()
  {
    set(types::sib20_v1310);
    return c.get<sib_type20_r13_s>();
  }
  sib_type21_r14_s& set_sib21_v1430()
  {
    set(types::sib21_v1430);
    return c.get<sib_type21_r14_s>();
  }
  sib_type24_r15_s& set_sib24_v1530()
  {
    set(types::sib24_v1530);
    return c.get<sib_type24_r15_s>();
  }
  sib_type25_r15_s& set_sib25_v1530()
  {
    set(types::sib25_v1530);
    return c.get<sib_type25_r15_s>();
  }
  sib_type26_r15_s& set_sib26_v1530()
  {
    set(types::sib26_v1530);
    return c.get<sib_type26_r15_s>();
  }

private:
  types type_;
  choice_buffer_t<sib_type10_s,
                  sib_type11_s,
                  sib_type12_r9_s,
                  sib_type13_r9_s,
                  sib_type14_r11_s,
                  sib_type15_r11_s,
                  sib_type16_r11_s,
                  sib_type17_r12_s,
                  sib_type18_r12_s,
                  sib_type19_r12_s,
                  sib_type20_r13_s,
                  sib_type21_r14_s,
                  sib_type24_r15_s,
                  sib_type25_r15_s,
                  sib_type26_r15_s,
                  sib_type2_s,
                  sib_type3_s,
                  sib_type4_s,
                  sib_type5_s,
                  sib_type6_s,
                  sib_type7_s,
                  sib_type8_s,
                  sib_type9_s>
      c;

  void destroy_();
};

// SystemInformation-r8-IEs ::= SEQUENCE
struct sys_info_r8_ies_s {
  typedef sib_info_item_c sib_type_and_info_item_c_;
  using sib_type_and_info_l_ = dyn_array<sib_type_and_info_item_c_>;

  // member variables
  bool                 non_crit_ext_present = false;
  sib_type_and_info_l_ sib_type_and_info;
  sys_info_v8a0_ies_s  non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1-v890-IEs ::= SEQUENCE
struct sib_type1_v890_ies_s {
  bool                 late_non_crit_ext_present = false;
  bool                 non_crit_ext_present      = false;
  dyn_octstring        late_non_crit_ext;
  sib_type1_v920_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformation ::= SEQUENCE
struct sys_info_s {
  struct crit_exts_c_ {
    struct crit_exts_future_r15_c_ {
      struct types_opts {
        enum options { pos_sys_info_r15, crit_exts_future, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      crit_exts_future_r15_c_() = default;
      crit_exts_future_r15_c_(const crit_exts_future_r15_c_& other);
      crit_exts_future_r15_c_& operator=(const crit_exts_future_r15_c_& other);
      ~crit_exts_future_r15_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      pos_sys_info_r15_ies_s& pos_sys_info_r15()
      {
        assert_choice_type("posSystemInformation-r15", type_.to_string(), "criticalExtensionsFuture-r15");
        return c.get<pos_sys_info_r15_ies_s>();
      }
      const pos_sys_info_r15_ies_s& pos_sys_info_r15() const
      {
        assert_choice_type("posSystemInformation-r15", type_.to_string(), "criticalExtensionsFuture-r15");
        return c.get<pos_sys_info_r15_ies_s>();
      }
      pos_sys_info_r15_ies_s& set_pos_sys_info_r15()
      {
        set(types::pos_sys_info_r15);
        return c.get<pos_sys_info_r15_ies_s>();
      }

    private:
      types                                   type_;
      choice_buffer_t<pos_sys_info_r15_ies_s> c;

      void destroy_();
    };
    struct types_opts {
      enum options { sys_info_r8, crit_exts_future_r15, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sys_info_r8_ies_s& sys_info_r8()
    {
      assert_choice_type("systemInformation-r8", type_.to_string(), "criticalExtensions");
      return c.get<sys_info_r8_ies_s>();
    }
    crit_exts_future_r15_c_& crit_exts_future_r15()
    {
      assert_choice_type("criticalExtensionsFuture-r15", type_.to_string(), "criticalExtensions");
      return c.get<crit_exts_future_r15_c_>();
    }
    const sys_info_r8_ies_s& sys_info_r8() const
    {
      assert_choice_type("systemInformation-r8", type_.to_string(), "criticalExtensions");
      return c.get<sys_info_r8_ies_s>();
    }
    const crit_exts_future_r15_c_& crit_exts_future_r15() const
    {
      assert_choice_type("criticalExtensionsFuture-r15", type_.to_string(), "criticalExtensions");
      return c.get<crit_exts_future_r15_c_>();
    }
    sys_info_r8_ies_s& set_sys_info_r8()
    {
      set(types::sys_info_r8);
      return c.get<sys_info_r8_ies_s>();
    }
    crit_exts_future_r15_c_& set_crit_exts_future_r15()
    {
      set(types::crit_exts_future_r15);
      return c.get<crit_exts_future_r15_c_>();
    }

  private:
    types                                                       type_;
    choice_buffer_t<crit_exts_future_r15_c_, sys_info_r8_ies_s> c;

    void destroy_();
  };

  // member variables
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformationBlockType1 ::= SEQUENCE
struct sib_type1_s {
  struct cell_access_related_info_s_ {
    struct cell_barred_opts {
      enum options { barred, not_barred, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<cell_barred_opts> cell_barred_e_;
    struct intra_freq_resel_opts {
      enum options { allowed, not_allowed, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<intra_freq_resel_opts> intra_freq_resel_e_;

    // member variables
    bool                csg_id_present = false;
    plmn_id_list_l      plmn_id_list;
    fixed_bitstring<16> tac;
    fixed_bitstring<28> cell_id;
    cell_barred_e_      cell_barred;
    intra_freq_resel_e_ intra_freq_resel;
    bool                csg_ind = false;
    fixed_bitstring<27> csg_id;
  };
  struct cell_sel_info_s_ {
    bool    q_rx_lev_min_offset_present = false;
    int8_t  q_rx_lev_min                = -70;
    uint8_t q_rx_lev_min_offset         = 1;
  };
  struct si_win_len_opts {
    enum options { ms1, ms2, ms5, ms10, ms15, ms20, ms40, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<si_win_len_opts> si_win_len_e_;

  // member variables
  bool                        p_max_present        = false;
  bool                        tdd_cfg_present      = false;
  bool                        non_crit_ext_present = false;
  cell_access_related_info_s_ cell_access_related_info;
  cell_sel_info_s_            cell_sel_info;
  int8_t                      p_max         = -30;
  uint8_t                     freq_band_ind = 1;
  sched_info_list_l           sched_info_list;
  tdd_cfg_s                   tdd_cfg;
  si_win_len_e_               si_win_len;
  uint8_t                     sys_info_value_tag = 0;
  sib_type1_v890_ies_s        non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-DL-SCH-MessageType ::= CHOICE
struct bcch_dl_sch_msg_type_c {
  struct c1_c_ {
    struct types_opts {
      enum options { sys_info, sib_type1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    c1_c_() = default;
    c1_c_(const c1_c_& other);
    c1_c_& operator=(const c1_c_& other);
    ~c1_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sys_info_s& sys_info()
    {
      assert_choice_type("systemInformation", type_.to_string(), "c1");
      return c.get<sys_info_s>();
    }
    sib_type1_s& sib_type1()
    {
      assert_choice_type("systemInformationBlockType1", type_.to_string(), "c1");
      return c.get<sib_type1_s>();
    }
    const sys_info_s& sys_info() const
    {
      assert_choice_type("systemInformation", type_.to_string(), "c1");
      return c.get<sys_info_s>();
    }
    const sib_type1_s& sib_type1() const
    {
      assert_choice_type("systemInformationBlockType1", type_.to_string(), "c1");
      return c.get<sib_type1_s>();
    }
    sys_info_s& set_sys_info()
    {
      set(types::sys_info);
      return c.get<sys_info_s>();
    }
    sib_type1_s& set_sib_type1()
    {
      set(types::sib_type1);
      return c.get<sib_type1_s>();
    }

  private:
    types                                    type_;
    choice_buffer_t<sib_type1_s, sys_info_s> c;

    void destroy_();
  };
  struct types_opts {
    enum options { c1, msg_class_ext, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  bcch_dl_sch_msg_type_c() = default;
  bcch_dl_sch_msg_type_c(const bcch_dl_sch_msg_type_c& other);
  bcch_dl_sch_msg_type_c& operator=(const bcch_dl_sch_msg_type_c& other);
  ~bcch_dl_sch_msg_type_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  c1_c_& c1()
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType");
    return c.get<c1_c_>();
  }
  const c1_c_& c1() const
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType");
    return c.get<c1_c_>();
  }
  c1_c_& set_c1()
  {
    set(types::c1);
    return c.get<c1_c_>();
  }

private:
  types                  type_;
  choice_buffer_t<c1_c_> c;

  void destroy_();
};

// BCCH-DL-SCH-Message ::= SEQUENCE
struct bcch_dl_sch_msg_s {
  bcch_dl_sch_msg_type_c msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SystemInformation-BR-r13 ::= SystemInformation
typedef sys_info_s sys_info_br_r13_s;

// SystemInformationBlockType1-BR-r13 ::= SystemInformationBlockType1
typedef sib_type1_s sib_type1_br_r13_s;

// BCCH-DL-SCH-MessageType-BR-r13 ::= CHOICE
struct bcch_dl_sch_msg_type_br_r13_c {
  struct c1_c_ {
    struct types_opts {
      enum options { sys_info_br_r13, sib_type1_br_r13, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    c1_c_() = default;
    c1_c_(const c1_c_& other);
    c1_c_& operator=(const c1_c_& other);
    ~c1_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sys_info_br_r13_s& sys_info_br_r13()
    {
      assert_choice_type("systemInformation-BR-r13", type_.to_string(), "c1");
      return c.get<sys_info_br_r13_s>();
    }
    sib_type1_br_r13_s& sib_type1_br_r13()
    {
      assert_choice_type("systemInformationBlockType1-BR-r13", type_.to_string(), "c1");
      return c.get<sib_type1_br_r13_s>();
    }
    const sys_info_br_r13_s& sys_info_br_r13() const
    {
      assert_choice_type("systemInformation-BR-r13", type_.to_string(), "c1");
      return c.get<sys_info_br_r13_s>();
    }
    const sib_type1_br_r13_s& sib_type1_br_r13() const
    {
      assert_choice_type("systemInformationBlockType1-BR-r13", type_.to_string(), "c1");
      return c.get<sib_type1_br_r13_s>();
    }
    sys_info_br_r13_s& set_sys_info_br_r13()
    {
      set(types::sys_info_br_r13);
      return c.get<sys_info_br_r13_s>();
    }
    sib_type1_br_r13_s& set_sib_type1_br_r13()
    {
      set(types::sib_type1_br_r13);
      return c.get<sib_type1_br_r13_s>();
    }

  private:
    types                                                  type_;
    choice_buffer_t<sib_type1_br_r13_s, sys_info_br_r13_s> c;

    void destroy_();
  };
  struct types_opts {
    enum options { c1, msg_class_ext, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  bcch_dl_sch_msg_type_br_r13_c() = default;
  bcch_dl_sch_msg_type_br_r13_c(const bcch_dl_sch_msg_type_br_r13_c& other);
  bcch_dl_sch_msg_type_br_r13_c& operator=(const bcch_dl_sch_msg_type_br_r13_c& other);
  ~bcch_dl_sch_msg_type_br_r13_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  c1_c_& c1()
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType-BR-r13");
    return c.get<c1_c_>();
  }
  const c1_c_& c1() const
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType-BR-r13");
    return c.get<c1_c_>();
  }
  c1_c_& set_c1()
  {
    set(types::c1);
    return c.get<c1_c_>();
  }

private:
  types                  type_;
  choice_buffer_t<c1_c_> c;

  void destroy_();
};

// BCCH-DL-SCH-Message-BR ::= SEQUENCE
struct bcch_dl_sch_msg_br_s {
  bcch_dl_sch_msg_type_br_r13_c msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SIB-Type-MBMS-r14 ::= ENUMERATED
struct sib_type_mbms_r14_opts {
  enum options {
    sib_type10,
    sib_type11,
    sib_type12_v920,
    sib_type13_v920,
    sib_type15_v1130,
    sib_type16_v1130,
    // ...
    nulltype
  } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<sib_type_mbms_r14_opts, true> sib_type_mbms_r14_e;

// SIB-MappingInfo-MBMS-r14 ::= SEQUENCE (SIZE (0..31)) OF SIB-Type-MBMS-r14
using sib_map_info_mbms_r14_l = bounded_array<sib_type_mbms_r14_e, 31>;

// SchedulingInfo-MBMS-r14 ::= SEQUENCE
struct sched_info_mbms_r14_s {
  struct si_periodicity_r14_opts {
    enum options { rf16, rf32, rf64, rf128, rf256, rf512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<si_periodicity_r14_opts> si_periodicity_r14_e_;

  // member variables
  si_periodicity_r14_e_   si_periodicity_r14;
  sib_map_info_mbms_r14_l sib_map_info_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NonMBSFN-SubframeConfig-r14 ::= SEQUENCE
struct non_mbsfn_sf_cfg_r14_s {
  struct radio_frame_alloc_period_r14_opts {
    enum options { rf4, rf8, rf16, rf32, rf64, rf128, rf512, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<radio_frame_alloc_period_r14_opts> radio_frame_alloc_period_r14_e_;

  // member variables
  radio_frame_alloc_period_r14_e_ radio_frame_alloc_period_r14;
  uint8_t                         radio_frame_alloc_offset_r14 = 0;
  fixed_bitstring<9>              sf_alloc_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-IdentityList-MBMS-r14 ::= SEQUENCE (SIZE (1..6)) OF PLMN-Identity
using plmn_id_list_mbms_r14_l = dyn_array<plmn_id_s>;

// SchedulingInfoList-MBMS-r14 ::= SEQUENCE (SIZE (1..32)) OF SchedulingInfo-MBMS-r14
using sched_info_list_mbms_r14_l = dyn_array<sched_info_mbms_r14_s>;

// SystemInformation-MBMS-r14 ::= SystemInformation
typedef sys_info_s sys_info_mbms_r14_s;

// SystemInformationBlockType1-MBMS-r14 ::= SEQUENCE
struct sib_type1_mbms_r14_s {
  struct cell_access_related_info_r14_s_ {
    plmn_id_list_mbms_r14_l plmn_id_list_r14;
    fixed_bitstring<16>     tac_r14;
    fixed_bitstring<28>     cell_id_r14;
  };
  struct si_win_len_r14_opts {
    enum options { ms1, ms2, ms5, ms10, ms15, ms20, ms40, ms80, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<si_win_len_r14_opts> si_win_len_r14_e_;
  using cell_access_related_info_list_r14_l_ = dyn_array<cell_access_related_info_r14_s>;

  // member variables
  bool                                 multi_band_info_list_r14_present          = false;
  bool                                 non_mbsfn_sf_cfg_r14_present              = false;
  bool                                 sib_type13_r14_present                    = false;
  bool                                 cell_access_related_info_list_r14_present = false;
  bool                                 non_crit_ext_present                      = false;
  cell_access_related_info_r14_s_      cell_access_related_info_r14;
  uint16_t                             freq_band_ind_r14 = 1;
  multi_band_info_list_r11_l           multi_band_info_list_r14;
  sched_info_list_mbms_r14_l           sched_info_list_mbms_r14;
  si_win_len_r14_e_                    si_win_len_r14;
  uint8_t                              sys_info_value_tag_r14 = 0;
  non_mbsfn_sf_cfg_r14_s               non_mbsfn_sf_cfg_r14;
  pdsch_cfg_common_s                   pdsch_cfg_common_r14;
  sib_type13_r9_s                      sib_type13_r14;
  cell_access_related_info_list_r14_l_ cell_access_related_info_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BCCH-DL-SCH-MessageType-MBMS-r14 ::= CHOICE
struct bcch_dl_sch_msg_type_mbms_r14_c {
  struct c1_c_ {
    struct types_opts {
      enum options { sys_info_mbms_r14, sib_type1_mbms_r14, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    c1_c_() = default;
    c1_c_(const c1_c_& other);
    c1_c_& operator=(const c1_c_& other);
    ~c1_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sys_info_mbms_r14_s& sys_info_mbms_r14()
    {
      assert_choice_type("systemInformation-MBMS-r14", type_.to_string(), "c1");
      return c.get<sys_info_mbms_r14_s>();
    }
    sib_type1_mbms_r14_s& sib_type1_mbms_r14()
    {
      assert_choice_type("systemInformationBlockType1-MBMS-r14", type_.to_string(), "c1");
      return c.get<sib_type1_mbms_r14_s>();
    }
    const sys_info_mbms_r14_s& sys_info_mbms_r14() const
    {
      assert_choice_type("systemInformation-MBMS-r14", type_.to_string(), "c1");
      return c.get<sys_info_mbms_r14_s>();
    }
    const sib_type1_mbms_r14_s& sib_type1_mbms_r14() const
    {
      assert_choice_type("systemInformationBlockType1-MBMS-r14", type_.to_string(), "c1");
      return c.get<sib_type1_mbms_r14_s>();
    }
    sys_info_mbms_r14_s& set_sys_info_mbms_r14()
    {
      set(types::sys_info_mbms_r14);
      return c.get<sys_info_mbms_r14_s>();
    }
    sib_type1_mbms_r14_s& set_sib_type1_mbms_r14()
    {
      set(types::sib_type1_mbms_r14);
      return c.get<sib_type1_mbms_r14_s>();
    }

  private:
    types                                                      type_;
    choice_buffer_t<sib_type1_mbms_r14_s, sys_info_mbms_r14_s> c;

    void destroy_();
  };
  struct types_opts {
    enum options { c1, msg_class_ext, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  bcch_dl_sch_msg_type_mbms_r14_c() = default;
  bcch_dl_sch_msg_type_mbms_r14_c(const bcch_dl_sch_msg_type_mbms_r14_c& other);
  bcch_dl_sch_msg_type_mbms_r14_c& operator=(const bcch_dl_sch_msg_type_mbms_r14_c& other);
  ~bcch_dl_sch_msg_type_mbms_r14_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  c1_c_& c1()
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType-MBMS-r14");
    return c.get<c1_c_>();
  }
  const c1_c_& c1() const
  {
    assert_choice_type("c1", type_.to_string(), "BCCH-DL-SCH-MessageType-MBMS-r14");
    return c.get<c1_c_>();
  }
  c1_c_& set_c1()
  {
    set(types::c1);
    return c.get<c1_c_>();
  }

private:
  types                  type_;
  choice_buffer_t<c1_c_> c;

  void destroy_();
};

// BCCH-DL-SCH-Message-MBMS ::= SEQUENCE
struct bcch_dl_sch_msg_mbms_s {
  bcch_dl_sch_msg_type_mbms_r14_c msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NZP-FrequencyDensity-r14 ::= ENUMERATED
struct nzp_freq_density_r14_opts {
  enum options { d1, d2, d3, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<nzp_freq_density_r14_opts> nzp_freq_density_r14_e;

// P-C-AndCBSR-r13 ::= SEQUENCE
struct p_c_and_cbsr_r13_s {
  struct cbsr_sel_r13_c_ {
    struct non_precoded_r13_s_ {
      dyn_bitstring codebook_subset_restrict1_r13;
      dyn_bitstring codebook_subset_restrict2_r13;
    };
    struct bf_k1a_r13_s_ {
      dyn_bitstring codebook_subset_restrict3_r13;
    };
    struct bf_kn_r13_s_ {
      dyn_bitstring codebook_subset_restrict_r13;
    };
    struct types_opts {
      enum options { non_precoded_r13, bf_k1a_r13, bf_kn_r13, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    cbsr_sel_r13_c_() = default;
    cbsr_sel_r13_c_(const cbsr_sel_r13_c_& other);
    cbsr_sel_r13_c_& operator=(const cbsr_sel_r13_c_& other);
    ~cbsr_sel_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    non_precoded_r13_s_& non_precoded_r13()
    {
      assert_choice_type("nonPrecoded-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<non_precoded_r13_s_>();
    }
    bf_k1a_r13_s_& bf_k1a_r13()
    {
      assert_choice_type("beamformedK1a-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<bf_k1a_r13_s_>();
    }
    bf_kn_r13_s_& bf_kn_r13()
    {
      assert_choice_type("beamformedKN-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<bf_kn_r13_s_>();
    }
    const non_precoded_r13_s_& non_precoded_r13() const
    {
      assert_choice_type("nonPrecoded-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<non_precoded_r13_s_>();
    }
    const bf_k1a_r13_s_& bf_k1a_r13() const
    {
      assert_choice_type("beamformedK1a-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<bf_k1a_r13_s_>();
    }
    const bf_kn_r13_s_& bf_kn_r13() const
    {
      assert_choice_type("beamformedKN-r13", type_.to_string(), "cbsr-Selection-r13");
      return c.get<bf_kn_r13_s_>();
    }
    non_precoded_r13_s_& set_non_precoded_r13()
    {
      set(types::non_precoded_r13);
      return c.get<non_precoded_r13_s_>();
    }
    bf_k1a_r13_s_& set_bf_k1a_r13()
    {
      set(types::bf_k1a_r13);
      return c.get<bf_k1a_r13_s_>();
    }
    bf_kn_r13_s_& set_bf_kn_r13()
    {
      set(types::bf_kn_r13);
      return c.get<bf_kn_r13_s_>();
    }

  private:
    types                                                             type_;
    choice_buffer_t<bf_k1a_r13_s_, bf_kn_r13_s_, non_precoded_r13_s_> c;

    void destroy_();
  };

  // member variables
  bool            ext     = false;
  int8_t          p_c_r13 = -8;
  cbsr_sel_r13_c_ cbsr_sel_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-NZP-v1430 ::= SEQUENCE
struct csi_rs_cfg_nzp_v1430_s {
  bool                   tx_comb_r14_present      = false;
  bool                   freq_density_r14_present = false;
  uint8_t                tx_comb_r14              = 0;
  nzp_freq_density_r14_e freq_density_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNZP-Activation-r14 ::= SEQUENCE
struct csi_rs_cfg_nzp_activation_r14_s {
  struct csi_rs_nzp_mode_r14_opts {
    enum options { semi_persistent, aperiodic, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<csi_rs_nzp_mode_r14_opts> csi_rs_nzp_mode_r14_e_;

  // member variables
  csi_rs_nzp_mode_r14_e_ csi_rs_nzp_mode_r14;
  uint8_t                activ_res_r14 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNZP-r11 ::= SEQUENCE
struct csi_rs_cfg_nzp_r11_s {
  struct ant_ports_count_r11_opts {
    enum options { an1, an2, an4, an8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ant_ports_count_r11_opts> ant_ports_count_r11_e_;
  struct qcl_crs_info_r11_s_ {
    struct crs_ports_count_r11_opts {
      enum options { n1, n2, n4, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<crs_ports_count_r11_opts> crs_ports_count_r11_e_;
    struct mbsfn_sf_cfg_list_r11_c_ {
      struct setup_s_ {
        mbsfn_sf_cfg_list_l sf_cfg_list;
      };
      typedef setup_e types;

      // choice methods
      mbsfn_sf_cfg_list_r11_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-r11");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-r11");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                     mbsfn_sf_cfg_list_r11_present = false;
    uint16_t                 qcl_scrambling_id_r11         = 0;
    crs_ports_count_r11_e_   crs_ports_count_r11;
    mbsfn_sf_cfg_list_r11_c_ mbsfn_sf_cfg_list_r11;
  };
  struct mbsfn_sf_cfg_list_v1430_c_ {
    struct setup_s_ {
      mbsfn_sf_cfg_list_v1430_l sf_cfg_list_v1430;
    };
    typedef setup_e types;

    // choice methods
    mbsfn_sf_cfg_list_v1430_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1430");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1430");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                   ext                      = false;
  bool                   qcl_crs_info_r11_present = false;
  uint8_t                csi_rs_cfg_nzp_id_r11    = 1;
  ant_ports_count_r11_e_ ant_ports_count_r11;
  uint8_t                res_cfg_r11       = 0;
  uint8_t                sf_cfg_r11        = 0;
  uint16_t               scrambling_id_r11 = 0;
  qcl_crs_info_r11_s_    qcl_crs_info_r11;
  // ...
  // group 0
  bool    csi_rs_cfg_nzp_id_v1310_present = false;
  uint8_t csi_rs_cfg_nzp_id_v1310         = 4;
  // group 1
  bool                   tx_comb_r14_present      = false;
  bool                   freq_density_r14_present = false;
  uint8_t                tx_comb_r14              = 0;
  nzp_freq_density_r14_e freq_density_r14;
  // group 2
  copy_ptr<mbsfn_sf_cfg_list_v1430_c_> mbsfn_sf_cfg_list_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NZP-ResourceConfig-r13 ::= SEQUENCE
struct nzp_res_cfg_r13_s {
  bool    ext         = false;
  uint8_t res_cfg_r13 = 0;
  // ...
  // group 0
  bool                   tx_comb_r14_present      = false;
  bool                   freq_density_r14_present = false;
  uint8_t                tx_comb_r14              = 0;
  nzp_freq_density_r14_e freq_density_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// P-C-AndCBSR-Pair-r13 ::= SEQUENCE (SIZE (1..2)) OF P-C-AndCBSR-r13
using p_c_and_cbsr_pair_r13_l = dyn_array<p_c_and_cbsr_r13_s>;

// P-C-AndCBSR-r15 ::= SEQUENCE
struct p_c_and_cbsr_r15_s {
  int8_t        p_c_r15 = -8;
  dyn_bitstring codebook_subset_restrict4_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigBeamformed-r14 ::= SEQUENCE
struct csi_rs_cfg_bf_r14_s {
  using csi_rs_cfg_nzp_id_list_ext_r14_l_    = bounded_array<uint8_t, 7>;
  using csi_im_cfg_id_list_r14_l_            = bounded_array<uint8_t, 8>;
  using p_c_and_cbsr_per_res_cfg_list_r14_l_ = dyn_array<p_c_and_cbsr_pair_r13_l>;
  using ace_for4_tx_per_res_cfg_list_r14_l_  = bounded_array<bool, 7>;
  using csi_rs_cfg_nzp_ap_list_r14_l_        = dyn_array<csi_rs_cfg_nzp_r11_s>;

  // member variables
  bool                                 csi_rs_cfg_nzp_id_list_ext_r14_present    = false;
  bool                                 csi_im_cfg_id_list_r14_present            = false;
  bool                                 p_c_and_cbsr_per_res_cfg_list_r14_present = false;
  bool                                 ace_for4_tx_per_res_cfg_list_r14_present  = false;
  bool                                 alt_codebook_enabled_bf_r14_present       = false;
  bool                                 ch_meas_restrict_r14_present              = false;
  bool                                 csi_rs_cfg_nzp_ap_list_r14_present        = false;
  bool                                 nzp_res_cfg_original_v1430_present        = false;
  bool                                 csi_rs_nzp_activation_r14_present         = false;
  csi_rs_cfg_nzp_id_list_ext_r14_l_    csi_rs_cfg_nzp_id_list_ext_r14;
  csi_im_cfg_id_list_r14_l_            csi_im_cfg_id_list_r14;
  p_c_and_cbsr_per_res_cfg_list_r14_l_ p_c_and_cbsr_per_res_cfg_list_r14;
  ace_for4_tx_per_res_cfg_list_r14_l_  ace_for4_tx_per_res_cfg_list_r14;
  csi_rs_cfg_nzp_ap_list_r14_l_        csi_rs_cfg_nzp_ap_list_r14;
  csi_rs_cfg_nzp_v1430_s               nzp_res_cfg_original_v1430;
  csi_rs_cfg_nzp_activation_r14_s      csi_rs_nzp_activation_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNZP-EMIMO-r13 ::= CHOICE
struct csi_rs_cfg_nzp_emimo_r13_c {
  struct setup_s_ {
    using nzp_res_cfg_list_r13_l_ = dyn_array<nzp_res_cfg_r13_s>;
    struct cdm_type_r13_opts {
      enum options { cdm2, cdm4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cdm_type_r13_opts> cdm_type_r13_e_;

    // member variables
    bool                    cdm_type_r13_present = false;
    nzp_res_cfg_list_r13_l_ nzp_res_cfg_list_r13;
    cdm_type_r13_e_         cdm_type_r13;
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_nzp_emimo_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigNZP-EMIMO-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigNZP-EMIMO-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CSI-RS-ConfigNZP-EMIMO-v1430 ::= SEQUENCE
struct csi_rs_cfg_nzp_emimo_v1430_s {
  using nzp_res_cfg_list_ext_r14_l_ = dyn_array<nzp_res_cfg_r13_s>;

  // member variables
  bool                        cdm_type_v1430_present = false;
  nzp_res_cfg_list_ext_r14_l_ nzp_res_cfg_list_ext_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// P-C-AndCBSR-Pair-r15 ::= SEQUENCE (SIZE (1..2)) OF P-C-AndCBSR-r15
using p_c_and_cbsr_pair_r15_l = dyn_array<p_c_and_cbsr_r15_s>;

// CQI-ReportModeAperiodic ::= ENUMERATED
struct cqi_report_mode_aperiodic_opts {
  enum options { rm12, rm20, rm22, rm30, rm31, rm32_v1250, rm10_v1310, rm11_v1310, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<cqi_report_mode_aperiodic_opts> cqi_report_mode_aperiodic_e;

// CSI-RS-ConfigBeamformed-r13 ::= SEQUENCE
struct csi_rs_cfg_bf_r13_s {
  using csi_rs_cfg_nzp_id_list_ext_r13_l_    = bounded_array<uint8_t, 7>;
  using csi_im_cfg_id_list_r13_l_            = bounded_array<uint8_t, 8>;
  using p_c_and_cbsr_per_res_cfg_list_r13_l_ = dyn_array<p_c_and_cbsr_pair_r13_l>;
  using ace_for4_tx_per_res_cfg_list_r13_l_  = bounded_array<bool, 7>;

  // member variables
  bool                                 csi_rs_cfg_nzp_id_list_ext_r13_present    = false;
  bool                                 csi_im_cfg_id_list_r13_present            = false;
  bool                                 p_c_and_cbsr_per_res_cfg_list_r13_present = false;
  bool                                 ace_for4_tx_per_res_cfg_list_r13_present  = false;
  bool                                 alt_codebook_enabled_bf_r13_present       = false;
  bool                                 ch_meas_restrict_r13_present              = false;
  csi_rs_cfg_nzp_id_list_ext_r13_l_    csi_rs_cfg_nzp_id_list_ext_r13;
  csi_im_cfg_id_list_r13_l_            csi_im_cfg_id_list_r13;
  p_c_and_cbsr_per_res_cfg_list_r13_l_ p_c_and_cbsr_per_res_cfg_list_r13;
  ace_for4_tx_per_res_cfg_list_r13_l_  ace_for4_tx_per_res_cfg_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigBeamformed-v1430 ::= SEQUENCE
struct csi_rs_cfg_bf_v1430_s {
  using csi_rs_cfg_nzp_ap_list_r14_l_ = dyn_array<csi_rs_cfg_nzp_r11_s>;

  // member variables
  bool                            csi_rs_cfg_nzp_ap_list_r14_present = false;
  bool                            nzp_res_cfg_original_v1430_present = false;
  bool                            csi_rs_nzp_activation_r14_present  = false;
  csi_rs_cfg_nzp_ap_list_r14_l_   csi_rs_cfg_nzp_ap_list_r14;
  csi_rs_cfg_nzp_v1430_s          nzp_res_cfg_original_v1430;
  csi_rs_cfg_nzp_activation_r14_s csi_rs_nzp_activation_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigEMIMO2-r14 ::= CHOICE
struct csi_rs_cfg_emimo2_r14_c {
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo2_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  csi_rs_cfg_bf_r14_s& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO2-r14");
    return c;
  }
  const csi_rs_cfg_bf_r14_s& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO2-r14");
    return c;
  }
  csi_rs_cfg_bf_r14_s& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types               type_;
  csi_rs_cfg_bf_r14_s c;
};

// CSI-RS-ConfigNonPrecoded-r13 ::= SEQUENCE
struct csi_rs_cfg_non_precoded_r13_s {
  struct codebook_cfg_n1_r13_opts {
    enum options { n1, n2, n3, n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n1_r13_opts> codebook_cfg_n1_r13_e_;
  struct codebook_cfg_n2_r13_opts {
    enum options { n1, n2, n3, n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n2_r13_opts> codebook_cfg_n2_r13_e_;
  struct codebook_over_sampling_rate_cfg_o1_r13_opts {
    enum options { n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_over_sampling_rate_cfg_o1_r13_opts> codebook_over_sampling_rate_cfg_o1_r13_e_;
  struct codebook_over_sampling_rate_cfg_o2_r13_opts {
    enum options { n4, n8, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_over_sampling_rate_cfg_o2_r13_opts> codebook_over_sampling_rate_cfg_o2_r13_e_;
  using csi_im_cfg_id_list_r13_l_ = bounded_array<uint8_t, 2>;

  // member variables
  bool                                      p_c_and_cbsr_list_r13_present                  = false;
  bool                                      codebook_over_sampling_rate_cfg_o1_r13_present = false;
  bool                                      codebook_over_sampling_rate_cfg_o2_r13_present = false;
  bool                                      csi_im_cfg_id_list_r13_present                 = false;
  bool                                      csi_rs_cfg_nzp_emimo_r13_present               = false;
  p_c_and_cbsr_pair_r13_l                   p_c_and_cbsr_list_r13;
  codebook_cfg_n1_r13_e_                    codebook_cfg_n1_r13;
  codebook_cfg_n2_r13_e_                    codebook_cfg_n2_r13;
  codebook_over_sampling_rate_cfg_o1_r13_e_ codebook_over_sampling_rate_cfg_o1_r13;
  codebook_over_sampling_rate_cfg_o2_r13_e_ codebook_over_sampling_rate_cfg_o2_r13;
  uint8_t                                   codebook_cfg_r13 = 1;
  csi_im_cfg_id_list_r13_l_                 csi_im_cfg_id_list_r13;
  csi_rs_cfg_nzp_emimo_r13_c                csi_rs_cfg_nzp_emimo_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNonPrecoded-v1430 ::= SEQUENCE
struct csi_rs_cfg_non_precoded_v1430_s {
  struct codebook_cfg_n1_v1430_opts {
    enum options { n5, n6, n7, n10, n12, n14, n16, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n1_v1430_opts> codebook_cfg_n1_v1430_e_;
  struct codebook_cfg_n2_v1430_opts {
    enum options { n5, n6, n7, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n2_v1430_opts> codebook_cfg_n2_v1430_e_;

  // member variables
  bool                         csi_rs_cfg_nzp_emimo_v1430_present = false;
  csi_rs_cfg_nzp_emimo_v1430_s csi_rs_cfg_nzp_emimo_v1430;
  codebook_cfg_n1_v1430_e_     codebook_cfg_n1_v1430;
  codebook_cfg_n2_v1430_e_     codebook_cfg_n2_v1430;
  csi_rs_cfg_nzp_v1430_s       nzp_res_cfg_tm9_original_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNonPrecoded-v1480 ::= SEQUENCE
struct csi_rs_cfg_non_precoded_v1480_s {
  struct codebook_cfg_n1_v1480_opts {
    enum options { n5, n6, n7, n10, n12, n14, n16, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n1_v1480_opts> codebook_cfg_n1_v1480_e_;
  struct codebook_cfg_n2_r1480_opts {
    enum options { n5, n6, n7, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<codebook_cfg_n2_r1480_opts> codebook_cfg_n2_r1480_e_;

  // member variables
  bool                         csi_rs_cfg_nzp_emimo_v1480_present = false;
  bool                         codebook_cfg_n1_v1480_present      = false;
  bool                         codebook_cfg_n2_r1480_present      = false;
  csi_rs_cfg_nzp_emimo_v1430_s csi_rs_cfg_nzp_emimo_v1480;
  codebook_cfg_n1_v1480_e_     codebook_cfg_n1_v1480;
  codebook_cfg_n2_r1480_e_     codebook_cfg_n2_r1480;
  csi_rs_cfg_nzp_v1430_s       nzp_res_cfg_tm9_original_v1480;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNonPrecoded-v1530 ::= SEQUENCE
struct csi_rs_cfg_non_precoded_v1530_s {
  bool                    p_c_and_cbsr_list_r15_present = false;
  p_c_and_cbsr_pair_r15_l p_c_and_cbsr_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// P-C-AndCBSR-r11 ::= SEQUENCE
struct p_c_and_cbsr_r11_s {
  int8_t        p_c_r11 = -8;
  dyn_bitstring codebook_subset_restrict_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportAperiodicProc-r11 ::= SEQUENCE
struct cqi_report_aperiodic_proc_r11_s {
  cqi_report_mode_aperiodic_e cqi_report_mode_aperiodic_r11;
  bool                        trigger01_r11 = false;
  bool                        trigger10_r11 = false;
  bool                        trigger11_r11 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportAperiodicProc-v1310 ::= SEQUENCE
struct cqi_report_aperiodic_proc_v1310_s {
  bool trigger001_r13 = false;
  bool trigger010_r13 = false;
  bool trigger011_r13 = false;
  bool trigger100_r13 = false;
  bool trigger101_r13 = false;
  bool trigger110_r13 = false;
  bool trigger111_r13 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportBothProc-r11 ::= SEQUENCE
struct cqi_report_both_proc_r11_s {
  bool    ri_ref_csi_process_id_r11_present = false;
  bool    pmi_ri_report_r11_present         = false;
  uint8_t ri_ref_csi_process_id_r11         = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CRI-ReportConfig-r13 ::= CHOICE
struct cri_report_cfg_r13_c {
  struct setup_s_ {
    bool     cri_cfg_idx2_r13_present = false;
    uint16_t cri_cfg_idx_r13          = 0;
    uint16_t cri_cfg_idx2_r13         = 0;
  };
  typedef setup_e types;

  // choice methods
  cri_report_cfg_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CRI-ReportConfig-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CRI-ReportConfig-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CSI-RS-ConfigEMIMO-Hybrid-r14 ::= CHOICE
struct csi_rs_cfg_emimo_hybrid_r14_c {
  struct setup_s_ {
    bool                    periodicity_offset_idx_r14_present = false;
    bool                    emimo_type2_r14_present            = false;
    uint16_t                periodicity_offset_idx_r14         = 0;
    csi_rs_cfg_emimo2_r14_c emimo_type2_r14;
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo_hybrid_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-Hybrid-r14");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-Hybrid-r14");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CSI-RS-ConfigEMIMO-r13 ::= CHOICE
struct csi_rs_cfg_emimo_r13_c {
  struct setup_c_ {
    struct types_opts {
      enum options { non_precoded_r13, bf_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    setup_c_(const setup_c_& other);
    setup_c_& operator=(const setup_c_& other);
    ~setup_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    csi_rs_cfg_non_precoded_r13_s& non_precoded_r13()
    {
      assert_choice_type("nonPrecoded-r13", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_r13_s>();
    }
    csi_rs_cfg_bf_r13_s& bf_r13()
    {
      assert_choice_type("beamformed-r13", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_r13_s>();
    }
    const csi_rs_cfg_non_precoded_r13_s& non_precoded_r13() const
    {
      assert_choice_type("nonPrecoded-r13", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_r13_s>();
    }
    const csi_rs_cfg_bf_r13_s& bf_r13() const
    {
      assert_choice_type("beamformed-r13", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_r13_s>();
    }
    csi_rs_cfg_non_precoded_r13_s& set_non_precoded_r13()
    {
      set(types::non_precoded_r13);
      return c.get<csi_rs_cfg_non_precoded_r13_s>();
    }
    csi_rs_cfg_bf_r13_s& set_bf_r13()
    {
      set(types::bf_r13);
      return c.get<csi_rs_cfg_bf_r13_s>();
    }

  private:
    types                                                               type_;
    choice_buffer_t<csi_rs_cfg_bf_r13_s, csi_rs_cfg_non_precoded_r13_s> c;

    void destroy_();
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-r13");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-r13");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// CSI-RS-ConfigEMIMO-v1430 ::= CHOICE
struct csi_rs_cfg_emimo_v1430_c {
  struct setup_c_ {
    struct types_opts {
      enum options { non_precoded_v1430, bf_v1430, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    setup_c_(const setup_c_& other);
    setup_c_& operator=(const setup_c_& other);
    ~setup_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    csi_rs_cfg_non_precoded_v1430_s& non_precoded_v1430()
    {
      assert_choice_type("nonPrecoded-v1430", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_v1430_s>();
    }
    csi_rs_cfg_bf_v1430_s& bf_v1430()
    {
      assert_choice_type("beamformed-v1430", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }
    const csi_rs_cfg_non_precoded_v1430_s& non_precoded_v1430() const
    {
      assert_choice_type("nonPrecoded-v1430", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_v1430_s>();
    }
    const csi_rs_cfg_bf_v1430_s& bf_v1430() const
    {
      assert_choice_type("beamformed-v1430", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }
    csi_rs_cfg_non_precoded_v1430_s& set_non_precoded_v1430()
    {
      set(types::non_precoded_v1430);
      return c.get<csi_rs_cfg_non_precoded_v1430_s>();
    }
    csi_rs_cfg_bf_v1430_s& set_bf_v1430()
    {
      set(types::bf_v1430);
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }

  private:
    types                                                                   type_;
    choice_buffer_t<csi_rs_cfg_bf_v1430_s, csi_rs_cfg_non_precoded_v1430_s> c;

    void destroy_();
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo_v1430_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1430");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1430");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// CSI-RS-ConfigEMIMO-v1480 ::= CHOICE
struct csi_rs_cfg_emimo_v1480_c {
  struct setup_c_ {
    struct types_opts {
      enum options { non_precoded_v1480, bf_v1480, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    setup_c_(const setup_c_& other);
    setup_c_& operator=(const setup_c_& other);
    ~setup_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    csi_rs_cfg_non_precoded_v1480_s& non_precoded_v1480()
    {
      assert_choice_type("nonPrecoded-v1480", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_v1480_s>();
    }
    csi_rs_cfg_bf_v1430_s& bf_v1480()
    {
      assert_choice_type("beamformed-v1480", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }
    const csi_rs_cfg_non_precoded_v1480_s& non_precoded_v1480() const
    {
      assert_choice_type("nonPrecoded-v1480", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_non_precoded_v1480_s>();
    }
    const csi_rs_cfg_bf_v1430_s& bf_v1480() const
    {
      assert_choice_type("beamformed-v1480", type_.to_string(), "setup");
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }
    csi_rs_cfg_non_precoded_v1480_s& set_non_precoded_v1480()
    {
      set(types::non_precoded_v1480);
      return c.get<csi_rs_cfg_non_precoded_v1480_s>();
    }
    csi_rs_cfg_bf_v1430_s& set_bf_v1480()
    {
      set(types::bf_v1480);
      return c.get<csi_rs_cfg_bf_v1430_s>();
    }

  private:
    types                                                                   type_;
    choice_buffer_t<csi_rs_cfg_bf_v1430_s, csi_rs_cfg_non_precoded_v1480_s> c;

    void destroy_();
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo_v1480_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1480");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1480");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// CSI-RS-ConfigEMIMO-v1530 ::= CHOICE
struct csi_rs_cfg_emimo_v1530_c {
  struct setup_c_ {
    struct types_opts {
      enum options { non_precoded_v1530, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    types       type() const { return types::non_precoded_v1530; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    csi_rs_cfg_non_precoded_v1530_s&       non_precoded_v1530() { return c; }
    const csi_rs_cfg_non_precoded_v1530_s& non_precoded_v1530() const { return c; }

  private:
    csi_rs_cfg_non_precoded_v1530_s c;
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_emimo_v1530_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1530");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigEMIMO-v1530");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// P-C-AndCBSR-Pair-r13a ::= SEQUENCE (SIZE (1..2)) OF P-C-AndCBSR-r11
using p_c_and_cbsr_pair_r13a_l = dyn_array<p_c_and_cbsr_r11_s>;

// CQI-ReportPeriodicProcExt-r11 ::= SEQUENCE
struct cqi_report_periodic_proc_ext_r11_s {
  struct cqi_format_ind_periodic_r11_c_ {
    struct wideband_cqi_r11_s_ {
      struct csi_report_mode_r11_opts {
        enum options { submode1, submode2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<csi_report_mode_r11_opts> csi_report_mode_r11_e_;

      // member variables
      bool                   csi_report_mode_r11_present = false;
      csi_report_mode_r11_e_ csi_report_mode_r11;
    };
    struct subband_cqi_r11_s_ {
      struct periodicity_factor_r11_opts {
        enum options { n2, n4, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<periodicity_factor_r11_opts> periodicity_factor_r11_e_;

      // member variables
      uint8_t                   k = 1;
      periodicity_factor_r11_e_ periodicity_factor_r11;
    };
    struct types_opts {
      enum options { wideband_cqi_r11, subband_cqi_r11, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    cqi_format_ind_periodic_r11_c_() = default;
    cqi_format_ind_periodic_r11_c_(const cqi_format_ind_periodic_r11_c_& other);
    cqi_format_ind_periodic_r11_c_& operator=(const cqi_format_ind_periodic_r11_c_& other);
    ~cqi_format_ind_periodic_r11_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    wideband_cqi_r11_s_& wideband_cqi_r11()
    {
      assert_choice_type("widebandCQI-r11", type_.to_string(), "cqi-FormatIndicatorPeriodic-r11");
      return c.get<wideband_cqi_r11_s_>();
    }
    subband_cqi_r11_s_& subband_cqi_r11()
    {
      assert_choice_type("subbandCQI-r11", type_.to_string(), "cqi-FormatIndicatorPeriodic-r11");
      return c.get<subband_cqi_r11_s_>();
    }
    const wideband_cqi_r11_s_& wideband_cqi_r11() const
    {
      assert_choice_type("widebandCQI-r11", type_.to_string(), "cqi-FormatIndicatorPeriodic-r11");
      return c.get<wideband_cqi_r11_s_>();
    }
    const subband_cqi_r11_s_& subband_cqi_r11() const
    {
      assert_choice_type("subbandCQI-r11", type_.to_string(), "cqi-FormatIndicatorPeriodic-r11");
      return c.get<subband_cqi_r11_s_>();
    }
    wideband_cqi_r11_s_& set_wideband_cqi_r11()
    {
      set(types::wideband_cqi_r11);
      return c.get<wideband_cqi_r11_s_>();
    }
    subband_cqi_r11_s_& set_subband_cqi_r11()
    {
      set(types::subband_cqi_r11);
      return c.get<subband_cqi_r11_s_>();
    }

  private:
    types                                                    type_;
    choice_buffer_t<subband_cqi_r11_s_, wideband_cqi_r11_s_> c;

    void destroy_();
  };
  struct csi_cfg_idx_r11_c_ {
    struct setup_s_ {
      bool     ri_cfg_idx2_r11_present = false;
      uint16_t cqi_pmi_cfg_idx2_r11    = 0;
      uint16_t ri_cfg_idx2_r11         = 0;
    };
    typedef setup_e types;

    // choice methods
    csi_cfg_idx_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-ConfigIndex-r11");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-ConfigIndex-r11");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct periodicity_factor_wb_r13_opts {
    enum options { n2, n4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<periodicity_factor_wb_r13_opts> periodicity_factor_wb_r13_e_;

  // member variables
  bool                           ext                                 = false;
  bool                           ri_cfg_idx_r11_present              = false;
  bool                           csi_cfg_idx_r11_present             = false;
  uint8_t                        cqi_report_periodic_proc_ext_id_r11 = 1;
  uint16_t                       cqi_pmi_cfg_idx_r11                 = 0;
  cqi_format_ind_periodic_r11_c_ cqi_format_ind_periodic_r11;
  uint16_t                       ri_cfg_idx_r11 = 0;
  csi_cfg_idx_r11_c_             csi_cfg_idx_r11;
  // ...
  // group 0
  copy_ptr<cri_report_cfg_r13_c> cri_report_cfg_r13;
  // group 1
  bool                         periodicity_factor_wb_r13_present = false;
  periodicity_factor_wb_r13_e_ periodicity_factor_wb_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-IM-Config-r11 ::= SEQUENCE
struct csi_im_cfg_r11_s {
  bool    ext               = false;
  uint8_t csi_im_cfg_id_r11 = 1;
  uint8_t res_cfg_r11       = 0;
  uint8_t sf_cfg_r11        = 0;
  // ...
  // group 0
  bool interference_meas_restrict_r13_present = false;
  bool interference_meas_restrict_r13         = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-IM-ConfigExt-r12 ::= SEQUENCE
struct csi_im_cfg_ext_r12_s {
  bool    ext                 = false;
  uint8_t csi_im_cfg_id_v1250 = 4;
  uint8_t res_cfg_r12         = 0;
  uint8_t sf_cfg_r12          = 0;
  // ...
  // group 0
  bool    interference_meas_restrict_r13_present = false;
  bool    csi_im_cfg_id_v1310_present            = false;
  bool    interference_meas_restrict_r13         = false;
  uint8_t csi_im_cfg_id_v1310                    = 5;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-Process-r11 ::= SEQUENCE
struct csi_process_r11_s {
  struct csi_im_cfg_id_list_r12_c_ {
    using setup_l_ = bounded_array<uint8_t, 2>;
    typedef setup_e types;

    // choice methods
    csi_im_cfg_id_list_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_l_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-IM-ConfigIdList-r12");
      return c;
    }
    const setup_l_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-IM-ConfigIdList-r12");
      return c;
    }
    setup_l_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_l_ c;
  };
  struct cqi_report_aperiodic_proc2_r12_c_ {
    typedef setup_e types;

    // choice methods
    cqi_report_aperiodic_proc2_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    cqi_report_aperiodic_proc_r11_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc2-r12");
      return c;
    }
    const cqi_report_aperiodic_proc_r11_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc2-r12");
      return c;
    }
    cqi_report_aperiodic_proc_r11_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types                           type_;
    cqi_report_aperiodic_proc_r11_s c;
  };
  struct cqi_report_aperiodic_proc_v1310_c_ {
    typedef setup_e types;

    // choice methods
    cqi_report_aperiodic_proc_v1310_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    cqi_report_aperiodic_proc_v1310_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc-v1310");
      return c;
    }
    const cqi_report_aperiodic_proc_v1310_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc-v1310");
      return c;
    }
    cqi_report_aperiodic_proc_v1310_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types                             type_;
    cqi_report_aperiodic_proc_v1310_s c;
  };
  struct cqi_report_aperiodic_proc2_v1310_c_ {
    typedef setup_e types;

    // choice methods
    cqi_report_aperiodic_proc2_v1310_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    cqi_report_aperiodic_proc_v1310_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc2-v1310");
      return c;
    }
    const cqi_report_aperiodic_proc_v1310_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "cqi-ReportAperiodicProc2-v1310");
      return c;
    }
    cqi_report_aperiodic_proc_v1310_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types                             type_;
    cqi_report_aperiodic_proc_v1310_s c;
  };

  // member variables
  bool                            ext                                     = false;
  bool                            cqi_report_both_proc_r11_present        = false;
  bool                            cqi_report_periodic_proc_id_r11_present = false;
  bool                            cqi_report_aperiodic_proc_r11_present   = false;
  uint8_t                         csi_process_id_r11                      = 1;
  uint8_t                         csi_rs_cfg_nzp_id_r11                   = 1;
  uint8_t                         csi_im_cfg_id_r11                       = 1;
  p_c_and_cbsr_pair_r13a_l        p_c_and_cbsr_list_r11;
  cqi_report_both_proc_r11_s      cqi_report_both_proc_r11;
  uint8_t                         cqi_report_periodic_proc_id_r11 = 0;
  cqi_report_aperiodic_proc_r11_s cqi_report_aperiodic_proc_r11;
  // ...
  // group 0
  bool                                        alt_codebook_enabled_for4_tx_proc_r12_present = false;
  copy_ptr<csi_im_cfg_id_list_r12_c_>         csi_im_cfg_id_list_r12;
  copy_ptr<cqi_report_aperiodic_proc2_r12_c_> cqi_report_aperiodic_proc2_r12;
  // group 1
  copy_ptr<cqi_report_aperiodic_proc_v1310_c_>  cqi_report_aperiodic_proc_v1310;
  copy_ptr<cqi_report_aperiodic_proc2_v1310_c_> cqi_report_aperiodic_proc2_v1310;
  copy_ptr<csi_rs_cfg_emimo_r13_c>              emimo_type_r13;
  // group 2
  bool                                    advanced_codebook_enabled_r14_present = false;
  copy_ptr<csi_rs_cfg_emimo_v1430_c>      dummy;
  copy_ptr<csi_rs_cfg_emimo_hybrid_r14_c> emimo_hybrid_r14;
  bool                                    advanced_codebook_enabled_r14 = false;
  // group 3
  copy_ptr<csi_rs_cfg_emimo_v1480_c> emimo_type_v1480;
  // group 4
  bool                               fe_comp_csi_enabled_v1530_present = false;
  bool                               fe_comp_csi_enabled_v1530         = false;
  copy_ptr<csi_rs_cfg_emimo_v1530_c> emimo_type_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportPeriodicProcExtToAddModList-r11 ::= SEQUENCE (SIZE (1..3)) OF CQI-ReportPeriodicProcExt-r11
using cqi_report_periodic_proc_ext_to_add_mod_list_r11_l = dyn_array<cqi_report_periodic_proc_ext_r11_s>;

// CQI-ReportPeriodicProcExtToReleaseList-r11 ::= SEQUENCE (SIZE (1..3)) OF INTEGER (1..3)
using cqi_report_periodic_proc_ext_to_release_list_r11_l = bounded_array<uint8_t, 3>;

// CSI-IM-ConfigToAddModList-r11 ::= SEQUENCE (SIZE (1..3)) OF CSI-IM-Config-r11
using csi_im_cfg_to_add_mod_list_r11_l = dyn_array<csi_im_cfg_r11_s>;

// CSI-IM-ConfigToAddModListExt-r13 ::= SEQUENCE (SIZE (1..20)) OF CSI-IM-ConfigExt-r12
using csi_im_cfg_to_add_mod_list_ext_r13_l = dyn_array<csi_im_cfg_ext_r12_s>;

// CSI-IM-ConfigToReleaseList-r11 ::= SEQUENCE (SIZE (1..3)) OF INTEGER (1..3)
using csi_im_cfg_to_release_list_r11_l = bounded_array<uint8_t, 3>;

// CSI-IM-ConfigToReleaseListExt-r13 ::= SEQUENCE (SIZE (1..20)) OF INTEGER (5..24)
using csi_im_cfg_to_release_list_ext_r13_l = bounded_array<uint8_t, 20>;

// CSI-ProcessToAddModList-r11 ::= SEQUENCE (SIZE (1..4)) OF CSI-Process-r11
using csi_process_to_add_mod_list_r11_l = dyn_array<csi_process_r11_s>;

// CSI-ProcessToReleaseList-r11 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..4)
using csi_process_to_release_list_r11_l = bounded_array<uint8_t, 4>;

// DCI7-CandidatesPerAL-SPDCCH-r15 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (0..6)
using dci7_candidates_per_al_spdcch_r15_l = bounded_array<uint8_t, 4>;

// N4SPUCCH-Resource-r15 ::= SEQUENCE
struct n4_spucch_res_r15_s {
  uint8_t n4start_prb_r15 = 0;
  uint8_t n4nof_prb_r15   = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PollByte-r14 ::= ENUMERATED
struct poll_byte_r14_opts {
  enum options {
    kb1,
    kb2,
    kb5,
    kb8,
    kb10,
    kb15,
    kb3500,
    kb4000,
    kb4500,
    kb5000,
    kb5500,
    kb6000,
    kb6500,
    kb7000,
    kb7500,
    kb8000,
    kb9000,
    kb10000,
    kb11000,
    kb12000,
    kb13000,
    kb14000,
    kb15000,
    kb16000,
    kb17000,
    kb18000,
    kb19000,
    kb20000,
    kb25000,
    kb30000,
    kb35000,
    kb40000,
    nulltype
  } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<poll_byte_r14_opts> poll_byte_r14_e;

// PollPDU-r15 ::= ENUMERATED
struct poll_pdu_r15_opts {
  enum options {
    p4,
    p8,
    p16,
    p32,
    p64,
    p128,
    p256,
    p512,
    p1024,
    p2048_r15,
    p4096_r15,
    p6144_r15,
    p8192_r15,
    p12288_r15,
    p16384_r15,
    pinfinity,
    nulltype
  } value;
  typedef int16_t number_type;

  std::string to_string() const;
  int16_t     to_number() const;
};
typedef enumerated<poll_pdu_r15_opts> poll_pdu_r15_e;

// SN-FieldLength ::= ENUMERATED
struct sn_field_len_opts {
  enum options { size5, size10, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<sn_field_len_opts> sn_field_len_e;

// SN-FieldLength-r15 ::= ENUMERATED
struct sn_field_len_r15_opts {
  enum options { size5, size10, size16_r15, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<sn_field_len_r15_opts> sn_field_len_r15_e;

// T-PollRetransmit ::= ENUMERATED
struct t_poll_retx_opts {
  enum options {
    ms5,
    ms10,
    ms15,
    ms20,
    ms25,
    ms30,
    ms35,
    ms40,
    ms45,
    ms50,
    ms55,
    ms60,
    ms65,
    ms70,
    ms75,
    ms80,
    ms85,
    ms90,
    ms95,
    ms100,
    ms105,
    ms110,
    ms115,
    ms120,
    ms125,
    ms130,
    ms135,
    ms140,
    ms145,
    ms150,
    ms155,
    ms160,
    ms165,
    ms170,
    ms175,
    ms180,
    ms185,
    ms190,
    ms195,
    ms200,
    ms205,
    ms210,
    ms215,
    ms220,
    ms225,
    ms230,
    ms235,
    ms240,
    ms245,
    ms250,
    ms300,
    ms350,
    ms400,
    ms450,
    ms500,
    ms800_v1310,
    ms1000_v1310,
    ms2000_v1310,
    ms4000_v1310,
    spare5,
    spare4,
    spare3,
    spare2,
    spare1,
    nulltype
  } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<t_poll_retx_opts> t_poll_retx_e;

// T-Reordering ::= ENUMERATED
struct t_reordering_opts {
  enum options {
    ms0,
    ms5,
    ms10,
    ms15,
    ms20,
    ms25,
    ms30,
    ms35,
    ms40,
    ms45,
    ms50,
    ms55,
    ms60,
    ms65,
    ms70,
    ms75,
    ms80,
    ms85,
    ms90,
    ms95,
    ms100,
    ms110,
    ms120,
    ms130,
    ms140,
    ms150,
    ms160,
    ms170,
    ms180,
    ms190,
    ms200,
    ms1600_v1310,
    nulltype
  } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<t_reordering_opts> t_reordering_e;

// T-StatusProhibit ::= ENUMERATED
struct t_status_prohibit_opts {
  enum options {
    ms0,
    ms5,
    ms10,
    ms15,
    ms20,
    ms25,
    ms30,
    ms35,
    ms40,
    ms45,
    ms50,
    ms55,
    ms60,
    ms65,
    ms70,
    ms75,
    ms80,
    ms85,
    ms90,
    ms95,
    ms100,
    ms105,
    ms110,
    ms115,
    ms120,
    ms125,
    ms130,
    ms135,
    ms140,
    ms145,
    ms150,
    ms155,
    ms160,
    ms165,
    ms170,
    ms175,
    ms180,
    ms185,
    ms190,
    ms195,
    ms200,
    ms205,
    ms210,
    ms215,
    ms220,
    ms225,
    ms230,
    ms235,
    ms240,
    ms245,
    ms250,
    ms300,
    ms350,
    ms400,
    ms450,
    ms500,
    ms800_v1310,
    ms1000_v1310,
    ms1200_v1310,
    ms1600_v1310,
    ms2000_v1310,
    ms2400_v1310,
    spare2,
    spare1,
    nulltype
  } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<t_status_prohibit_opts> t_status_prohibit_e;

// ZeroTxPowerCSI-RS-r12 ::= SEQUENCE
struct zero_tx_pwr_csi_rs_r12_s {
  fixed_bitstring<16> zero_tx_pwr_res_cfg_list_r12;
  uint8_t             zero_tx_pwr_sf_cfg_r12 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportAperiodic-r10 ::= CHOICE
struct cqi_report_aperiodic_r10_c {
  struct setup_s_ {
    struct aperiodic_csi_trigger_r10_s_ {
      fixed_bitstring<8> trigger1_r10;
      fixed_bitstring<8> trigger2_r10;
    };

    // member variables
    bool                         aperiodic_csi_trigger_r10_present = false;
    cqi_report_mode_aperiodic_e  cqi_report_mode_aperiodic_r10;
    aperiodic_csi_trigger_r10_s_ aperiodic_csi_trigger_r10;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_aperiodic_r10_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-r10");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-r10");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportAperiodic-v1250 ::= CHOICE
struct cqi_report_aperiodic_v1250_c {
  struct setup_s_ {
    struct aperiodic_csi_trigger_v1250_s_ {
      struct trigger_sf_set_ind_r12_opts {
        enum options { s1, s2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<trigger_sf_set_ind_r12_opts> trigger_sf_set_ind_r12_e_;

      // member variables
      trigger_sf_set_ind_r12_e_ trigger_sf_set_ind_r12;
      fixed_bitstring<8>        trigger1_sf_set_ind_r12;
      fixed_bitstring<8>        trigger2_sf_set_ind_r12;
    };

    // member variables
    aperiodic_csi_trigger_v1250_s_ aperiodic_csi_trigger_v1250;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_aperiodic_v1250_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-v1250");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-v1250");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportAperiodic-v1310 ::= CHOICE
struct cqi_report_aperiodic_v1310_c {
  struct setup_s_ {
    struct aperiodic_csi_trigger_v1310_s_ {
      fixed_bitstring<32> trigger1_r13;
      fixed_bitstring<32> trigger2_r13;
      fixed_bitstring<32> trigger3_r13;
      fixed_bitstring<32> trigger4_r13;
      fixed_bitstring<32> trigger5_r13;
      fixed_bitstring<32> trigger6_r13;
    };
    struct aperiodic_csi_trigger2_r13_c_ {
      struct setup_s_ {
        fixed_bitstring<32> trigger1_sf_set_ind_r13;
        fixed_bitstring<32> trigger2_sf_set_ind_r13;
        fixed_bitstring<32> trigger3_sf_set_ind_r13;
        fixed_bitstring<32> trigger4_sf_set_ind_r13;
        fixed_bitstring<32> trigger5_sf_set_ind_r13;
        fixed_bitstring<32> trigger6_sf_set_ind_r13;
      };
      typedef setup_e types;

      // choice methods
      aperiodic_csi_trigger2_r13_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "aperiodicCSI-Trigger2-r13");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "aperiodicCSI-Trigger2-r13");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                           aperiodic_csi_trigger_v1310_present = false;
    bool                           aperiodic_csi_trigger2_r13_present  = false;
    aperiodic_csi_trigger_v1310_s_ aperiodic_csi_trigger_v1310;
    aperiodic_csi_trigger2_r13_c_  aperiodic_csi_trigger2_r13;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_aperiodic_v1310_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-v1310");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportAperiodic-v1310");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportAperiodicHybrid-r14 ::= SEQUENCE
struct cqi_report_aperiodic_hybrid_r14_s {
  struct triggers_r14_c_ {
    struct one_bit_r14_s_ {
      fixed_bitstring<8> trigger1_ind_r14;
    };
    struct two_bit_r14_s_ {
      fixed_bitstring<8> trigger01_ind_r14;
      fixed_bitstring<8> trigger10_ind_r14;
      fixed_bitstring<8> trigger11_ind_r14;
    };
    struct three_bit_r14_s_ {
      fixed_bitstring<32> trigger001_ind_r14;
      fixed_bitstring<32> trigger010_ind_r14;
      fixed_bitstring<32> trigger011_ind_r14;
      fixed_bitstring<32> trigger100_ind_r14;
      fixed_bitstring<32> trigger101_ind_r14;
      fixed_bitstring<32> trigger110_ind_r14;
      fixed_bitstring<32> trigger111_ind_r14;
    };
    struct types_opts {
      enum options { one_bit_r14, two_bit_r14, three_bit_r14, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    triggers_r14_c_() = default;
    triggers_r14_c_(const triggers_r14_c_& other);
    triggers_r14_c_& operator=(const triggers_r14_c_& other);
    ~triggers_r14_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    one_bit_r14_s_& one_bit_r14()
    {
      assert_choice_type("oneBit-r14", type_.to_string(), "triggers-r14");
      return c.get<one_bit_r14_s_>();
    }
    two_bit_r14_s_& two_bit_r14()
    {
      assert_choice_type("twoBit-r14", type_.to_string(), "triggers-r14");
      return c.get<two_bit_r14_s_>();
    }
    three_bit_r14_s_& three_bit_r14()
    {
      assert_choice_type("threeBit-r14", type_.to_string(), "triggers-r14");
      return c.get<three_bit_r14_s_>();
    }
    const one_bit_r14_s_& one_bit_r14() const
    {
      assert_choice_type("oneBit-r14", type_.to_string(), "triggers-r14");
      return c.get<one_bit_r14_s_>();
    }
    const two_bit_r14_s_& two_bit_r14() const
    {
      assert_choice_type("twoBit-r14", type_.to_string(), "triggers-r14");
      return c.get<two_bit_r14_s_>();
    }
    const three_bit_r14_s_& three_bit_r14() const
    {
      assert_choice_type("threeBit-r14", type_.to_string(), "triggers-r14");
      return c.get<three_bit_r14_s_>();
    }
    one_bit_r14_s_& set_one_bit_r14()
    {
      set(types::one_bit_r14);
      return c.get<one_bit_r14_s_>();
    }
    two_bit_r14_s_& set_two_bit_r14()
    {
      set(types::two_bit_r14);
      return c.get<two_bit_r14_s_>();
    }
    three_bit_r14_s_& set_three_bit_r14()
    {
      set(types::three_bit_r14);
      return c.get<three_bit_r14_s_>();
    }

  private:
    types                                                             type_;
    choice_buffer_t<one_bit_r14_s_, three_bit_r14_s_, two_bit_r14_s_> c;

    void destroy_();
  };

  // member variables
  bool            triggers_r14_present = false;
  triggers_r14_c_ triggers_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportBoth-r11 ::= SEQUENCE
struct cqi_report_both_r11_s {
  bool                              csi_im_cfg_to_release_list_r11_present  = false;
  bool                              csi_im_cfg_to_add_mod_list_r11_present  = false;
  bool                              csi_process_to_release_list_r11_present = false;
  bool                              csi_process_to_add_mod_list_r11_present = false;
  csi_im_cfg_to_release_list_r11_l  csi_im_cfg_to_release_list_r11;
  csi_im_cfg_to_add_mod_list_r11_l  csi_im_cfg_to_add_mod_list_r11;
  csi_process_to_release_list_r11_l csi_process_to_release_list_r11;
  csi_process_to_add_mod_list_r11_l csi_process_to_add_mod_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportBoth-v1250 ::= SEQUENCE
struct cqi_report_both_v1250_s {
  bool                 csi_im_cfg_to_release_list_ext_r12_present = false;
  bool                 csi_im_cfg_to_add_mod_list_ext_r12_present = false;
  uint8_t              csi_im_cfg_to_release_list_ext_r12         = 4;
  csi_im_cfg_ext_r12_s csi_im_cfg_to_add_mod_list_ext_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportBoth-v1310 ::= SEQUENCE
struct cqi_report_both_v1310_s {
  bool                                 csi_im_cfg_to_release_list_ext_r13_present = false;
  bool                                 csi_im_cfg_to_add_mod_list_ext_r13_present = false;
  csi_im_cfg_to_release_list_ext_r13_l csi_im_cfg_to_release_list_ext_r13;
  csi_im_cfg_to_add_mod_list_ext_r13_l csi_im_cfg_to_add_mod_list_ext_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportPeriodic-r10 ::= CHOICE
struct cqi_report_periodic_r10_c {
  struct setup_s_ {
    struct cqi_format_ind_periodic_r10_c_ {
      struct wideband_cqi_r10_s_ {
        struct csi_report_mode_r10_opts {
          enum options { submode1, submode2, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<csi_report_mode_r10_opts> csi_report_mode_r10_e_;

        // member variables
        bool                   csi_report_mode_r10_present = false;
        csi_report_mode_r10_e_ csi_report_mode_r10;
      };
      struct subband_cqi_r10_s_ {
        struct periodicity_factor_r10_opts {
          enum options { n2, n4, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<periodicity_factor_r10_opts> periodicity_factor_r10_e_;

        // member variables
        uint8_t                   k = 1;
        periodicity_factor_r10_e_ periodicity_factor_r10;
      };
      struct types_opts {
        enum options { wideband_cqi_r10, subband_cqi_r10, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      cqi_format_ind_periodic_r10_c_() = default;
      cqi_format_ind_periodic_r10_c_(const cqi_format_ind_periodic_r10_c_& other);
      cqi_format_ind_periodic_r10_c_& operator=(const cqi_format_ind_periodic_r10_c_& other);
      ~cqi_format_ind_periodic_r10_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      wideband_cqi_r10_s_& wideband_cqi_r10()
      {
        assert_choice_type("widebandCQI-r10", type_.to_string(), "cqi-FormatIndicatorPeriodic-r10");
        return c.get<wideband_cqi_r10_s_>();
      }
      subband_cqi_r10_s_& subband_cqi_r10()
      {
        assert_choice_type("subbandCQI-r10", type_.to_string(), "cqi-FormatIndicatorPeriodic-r10");
        return c.get<subband_cqi_r10_s_>();
      }
      const wideband_cqi_r10_s_& wideband_cqi_r10() const
      {
        assert_choice_type("widebandCQI-r10", type_.to_string(), "cqi-FormatIndicatorPeriodic-r10");
        return c.get<wideband_cqi_r10_s_>();
      }
      const subband_cqi_r10_s_& subband_cqi_r10() const
      {
        assert_choice_type("subbandCQI-r10", type_.to_string(), "cqi-FormatIndicatorPeriodic-r10");
        return c.get<subband_cqi_r10_s_>();
      }
      wideband_cqi_r10_s_& set_wideband_cqi_r10()
      {
        set(types::wideband_cqi_r10);
        return c.get<wideband_cqi_r10_s_>();
      }
      subband_cqi_r10_s_& set_subband_cqi_r10()
      {
        set(types::subband_cqi_r10);
        return c.get<subband_cqi_r10_s_>();
      }

    private:
      types                                                    type_;
      choice_buffer_t<subband_cqi_r10_s_, wideband_cqi_r10_s_> c;

      void destroy_();
    };
    struct csi_cfg_idx_r10_c_ {
      struct setup_s_ {
        bool     ri_cfg_idx2_r10_present = false;
        uint16_t cqi_pmi_cfg_idx2_r10    = 0;
        uint16_t ri_cfg_idx2_r10         = 0;
      };
      typedef setup_e types;

      // choice methods
      csi_cfg_idx_r10_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "csi-ConfigIndex-r10");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "csi-ConfigIndex-r10");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                           cqi_pucch_res_idx_p1_r10_present = false;
    bool                           ri_cfg_idx_present               = false;
    bool                           cqi_mask_r9_present              = false;
    bool                           csi_cfg_idx_r10_present          = false;
    uint16_t                       cqi_pucch_res_idx_r10            = 0;
    uint16_t                       cqi_pucch_res_idx_p1_r10         = 0;
    uint16_t                       cqi_pmi_cfg_idx                  = 0;
    cqi_format_ind_periodic_r10_c_ cqi_format_ind_periodic_r10;
    uint16_t                       ri_cfg_idx             = 0;
    bool                           simul_ack_nack_and_cqi = false;
    csi_cfg_idx_r10_c_             csi_cfg_idx_r10;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_periodic_r10_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodic-r10");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodic-r10");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportPeriodic-v1130 ::= SEQUENCE
struct cqi_report_periodic_v1130_s {
  bool                                               simul_ack_nack_and_cqi_format3_r11_present               = false;
  bool                                               cqi_report_periodic_proc_ext_to_release_list_r11_present = false;
  bool                                               cqi_report_periodic_proc_ext_to_add_mod_list_r11_present = false;
  cqi_report_periodic_proc_ext_to_release_list_r11_l cqi_report_periodic_proc_ext_to_release_list_r11;
  cqi_report_periodic_proc_ext_to_add_mod_list_r11_l cqi_report_periodic_proc_ext_to_add_mod_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportPeriodic-v1310 ::= SEQUENCE
struct cqi_report_periodic_v1310_s {
  bool                 cri_report_cfg_r13_present                         = false;
  bool                 simul_ack_nack_and_cqi_format4_format5_r13_present = false;
  cri_report_cfg_r13_c cri_report_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportPeriodic-v1320 ::= SEQUENCE
struct cqi_report_periodic_v1320_s {
  struct periodicity_factor_wb_r13_opts {
    enum options { n2, n4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<periodicity_factor_wb_r13_opts> periodicity_factor_wb_r13_e_;

  // member variables
  bool                         periodicity_factor_wb_r13_present = false;
  periodicity_factor_wb_r13_e_ periodicity_factor_wb_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DL-AM-RLC-r15 ::= SEQUENCE
struct dl_am_rlc_r15_s {
  t_reordering_e      t_reordering_r15;
  t_status_prohibit_e t_status_prohibit_r15;
  bool                extended_rlc_li_field_r15 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DL-UM-RLC-r15 ::= SEQUENCE
struct dl_um_rlc_r15_s {
  sn_field_len_r15_e sn_field_len_r15;
  t_reordering_e     t_reordering_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasSubframePattern-r10 ::= CHOICE
struct meas_sf_pattern_r10_c {
  struct sf_pattern_tdd_r10_c_ {
    struct types_opts {
      enum options { sf_cfg1_minus5_r10, sf_cfg0_r10, sf_cfg6_r10, /*...*/ nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts, true> types;

    // choice methods
    sf_pattern_tdd_r10_c_() = default;
    sf_pattern_tdd_r10_c_(const sf_pattern_tdd_r10_c_& other);
    sf_pattern_tdd_r10_c_& operator=(const sf_pattern_tdd_r10_c_& other);
    ~sf_pattern_tdd_r10_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<20>& sf_cfg1_minus5_r10()
    {
      assert_choice_type("subframeConfig1-5-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<20> >();
    }
    fixed_bitstring<70>& sf_cfg0_r10()
    {
      assert_choice_type("subframeConfig0-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<70> >();
    }
    fixed_bitstring<60>& sf_cfg6_r10()
    {
      assert_choice_type("subframeConfig6-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<60> >();
    }
    const fixed_bitstring<20>& sf_cfg1_minus5_r10() const
    {
      assert_choice_type("subframeConfig1-5-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<20> >();
    }
    const fixed_bitstring<70>& sf_cfg0_r10() const
    {
      assert_choice_type("subframeConfig0-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<70> >();
    }
    const fixed_bitstring<60>& sf_cfg6_r10() const
    {
      assert_choice_type("subframeConfig6-r10", type_.to_string(), "subframePatternTDD-r10");
      return c.get<fixed_bitstring<60> >();
    }
    fixed_bitstring<20>& set_sf_cfg1_minus5_r10()
    {
      set(types::sf_cfg1_minus5_r10);
      return c.get<fixed_bitstring<20> >();
    }
    fixed_bitstring<70>& set_sf_cfg0_r10()
    {
      set(types::sf_cfg0_r10);
      return c.get<fixed_bitstring<70> >();
    }
    fixed_bitstring<60>& set_sf_cfg6_r10()
    {
      set(types::sf_cfg6_r10);
      return c.get<fixed_bitstring<60> >();
    }

  private:
    types                                 type_;
    choice_buffer_t<fixed_bitstring<70> > c;

    void destroy_();
  };
  struct types_opts {
    enum options { sf_pattern_fdd_r10, sf_pattern_tdd_r10, /*...*/ nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts, true> types;

  // choice methods
  meas_sf_pattern_r10_c() = default;
  meas_sf_pattern_r10_c(const meas_sf_pattern_r10_c& other);
  meas_sf_pattern_r10_c& operator=(const meas_sf_pattern_r10_c& other);
  ~meas_sf_pattern_r10_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  fixed_bitstring<40>& sf_pattern_fdd_r10()
  {
    assert_choice_type("subframePatternFDD-r10", type_.to_string(), "MeasSubframePattern-r10");
    return c.get<fixed_bitstring<40> >();
  }
  sf_pattern_tdd_r10_c_& sf_pattern_tdd_r10()
  {
    assert_choice_type("subframePatternTDD-r10", type_.to_string(), "MeasSubframePattern-r10");
    return c.get<sf_pattern_tdd_r10_c_>();
  }
  const fixed_bitstring<40>& sf_pattern_fdd_r10() const
  {
    assert_choice_type("subframePatternFDD-r10", type_.to_string(), "MeasSubframePattern-r10");
    return c.get<fixed_bitstring<40> >();
  }
  const sf_pattern_tdd_r10_c_& sf_pattern_tdd_r10() const
  {
    assert_choice_type("subframePatternTDD-r10", type_.to_string(), "MeasSubframePattern-r10");
    return c.get<sf_pattern_tdd_r10_c_>();
  }
  fixed_bitstring<40>& set_sf_pattern_fdd_r10()
  {
    set(types::sf_pattern_fdd_r10);
    return c.get<fixed_bitstring<40> >();
  }
  sf_pattern_tdd_r10_c_& set_sf_pattern_tdd_r10()
  {
    set(types::sf_pattern_tdd_r10);
    return c.get<sf_pattern_tdd_r10_c_>();
  }

private:
  types                                                       type_;
  choice_buffer_t<fixed_bitstring<40>, sf_pattern_tdd_r10_c_> c;

  void destroy_();
};

// PollByte ::= ENUMERATED
struct poll_byte_opts {
  enum options {
    kb25,
    kb50,
    kb75,
    kb100,
    kb125,
    kb250,
    kb375,
    kb500,
    kb750,
    kb1000,
    kb1250,
    kb1500,
    kb2000,
    kb3000,
    kbinfinity,
    spare1,
    nulltype
  } value;
  typedef int16_t number_type;

  std::string to_string() const;
  int16_t     to_number() const;
};
typedef enumerated<poll_byte_opts> poll_byte_e;

// PollPDU ::= ENUMERATED
struct poll_pdu_opts {
  enum options { p4, p8, p16, p32, p64, p128, p256, pinfinity, nulltype } value;
  typedef int16_t number_type;

  std::string to_string() const;
  int16_t     to_number() const;
};
typedef enumerated<poll_pdu_opts> poll_pdu_e;

// SPDCCH-Elements-r15 ::= CHOICE
struct spdcch_elems_r15_c {
  struct setup_s_ {
    struct spdcch_set_ref_sig_r15_opts {
      enum options { crs, dmrs, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<spdcch_set_ref_sig_r15_opts> spdcch_set_ref_sig_r15_e_;
    struct tx_type_r15_opts {
      enum options { localised, distributed, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<tx_type_r15_opts> tx_type_r15_e_;
    using dci7_candidates_per_al_pdcch_r15_l_      = bounded_array<uint8_t, 4>;
    using dci7_candidate_sets_per_al_spdcch_r15_l_ = dyn_array<dci7_candidates_per_al_spdcch_r15_l>;
    struct res_block_assign_r15_s_ {
      uint8_t             num_rb_in_freq_domain_r15 = 2;
      fixed_bitstring<98> res_block_assign_r15;
    };
    using al_start_point_spdcch_r15_l_ = bounded_array<uint8_t, 4>;
    struct sf_type_r15_opts {
      enum options { mbsfn, nonmbsfn, all, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<sf_type_r15_opts> sf_type_r15_e_;
    struct rate_matching_mode_r15_opts {
      enum options { m1, m2, m3, m4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<rate_matching_mode_r15_opts> rate_matching_mode_r15_e_;

    // member variables
    bool                                     ext                                           = false;
    bool                                     spdcch_set_cfg_id_r15_present                 = false;
    bool                                     spdcch_set_ref_sig_r15_present                = false;
    bool                                     tx_type_r15_present                           = false;
    bool                                     spdcch_no_of_symbols_r15_present              = false;
    bool                                     dmrs_scrambling_seq_int_r15_present           = false;
    bool                                     dci7_candidates_per_al_pdcch_r15_present      = false;
    bool                                     dci7_candidate_sets_per_al_spdcch_r15_present = false;
    bool                                     res_block_assign_r15_present                  = false;
    bool                                     subslot_applicability_r15_present             = false;
    bool                                     al_start_point_spdcch_r15_present             = false;
    bool                                     sf_type_r15_present                           = false;
    bool                                     rate_matching_mode_r15_present                = false;
    uint8_t                                  spdcch_set_cfg_id_r15                         = 0;
    spdcch_set_ref_sig_r15_e_                spdcch_set_ref_sig_r15;
    tx_type_r15_e_                           tx_type_r15;
    uint8_t                                  spdcch_no_of_symbols_r15    = 1;
    uint16_t                                 dmrs_scrambling_seq_int_r15 = 0;
    dci7_candidates_per_al_pdcch_r15_l_      dci7_candidates_per_al_pdcch_r15;
    dci7_candidate_sets_per_al_spdcch_r15_l_ dci7_candidate_sets_per_al_spdcch_r15;
    res_block_assign_r15_s_                  res_block_assign_r15;
    fixed_bitstring<5>                       subslot_applicability_r15;
    al_start_point_spdcch_r15_l_             al_start_point_spdcch_r15;
    sf_type_r15_e_                           sf_type_r15;
    rate_matching_mode_r15_e_                rate_matching_mode_r15;
    // ...
  };
  typedef setup_e types;

  // choice methods
  spdcch_elems_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPDCCH-Elements-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPDCCH-Elements-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SPUCCH-Elements-r15 ::= CHOICE
struct spucch_elems_r15_c {
  struct setup_s_ {
    using n1_subslot_spucch_an_list_r15_l_ = bounded_array<uint16_t, 4>;
    using n4_spucch_slot_res_r15_l_        = dyn_array<n4_spucch_res_r15_s>;
    using n4_spucch_subslot_res_r15_l_     = dyn_array<n4_spucch_res_r15_s>;

    // member variables
    bool                             n1_subslot_spucch_an_list_r15_present              = false;
    bool                             n1_slot_spucch_fh_an_list_r15_present              = false;
    bool                             n1_slot_spucch_no_fh_an_list_r15_present           = false;
    bool                             n3_spucch_an_list_r15_present                      = false;
    bool                             n4_spucch_slot_res_r15_present                     = false;
    bool                             n4_spucch_subslot_res_r15_present                  = false;
    bool                             n4max_coderate_slot_pucch_r15_present              = false;
    bool                             n4max_coderate_subslot_pucch_r15_present           = false;
    bool                             n4max_coderate_multi_res_slot_pucch_r15_present    = false;
    bool                             n4max_coderate_multi_res_subslot_pucch_r15_present = false;
    n1_subslot_spucch_an_list_r15_l_ n1_subslot_spucch_an_list_r15;
    uint16_t                         n1_slot_spucch_fh_an_list_r15    = 0;
    uint16_t                         n1_slot_spucch_no_fh_an_list_r15 = 0;
    uint16_t                         n3_spucch_an_list_r15            = 0;
    n4_spucch_slot_res_r15_l_        n4_spucch_slot_res_r15;
    n4_spucch_subslot_res_r15_l_     n4_spucch_subslot_res_r15;
    uint8_t                          n4max_coderate_slot_pucch_r15              = 0;
    uint8_t                          n4max_coderate_subslot_pucch_r15           = 0;
    uint8_t                          n4max_coderate_multi_res_slot_pucch_r15    = 0;
    uint8_t                          n4max_coderate_multi_res_subslot_pucch_r15 = 0;
  };
  typedef setup_e types;

  // choice methods
  spucch_elems_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPUCCH-Elements-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPUCCH-Elements-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// TPC-Index ::= CHOICE
struct tpc_idx_c {
  struct types_opts {
    enum options { idx_of_format3, idx_of_format3_a, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  tpc_idx_c() = default;
  tpc_idx_c(const tpc_idx_c& other);
  tpc_idx_c& operator=(const tpc_idx_c& other);
  ~tpc_idx_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  uint8_t& idx_of_format3()
  {
    assert_choice_type("indexOfFormat3", type_.to_string(), "TPC-Index");
    return c.get<uint8_t>();
  }
  uint8_t& idx_of_format3_a()
  {
    assert_choice_type("indexOfFormat3A", type_.to_string(), "TPC-Index");
    return c.get<uint8_t>();
  }
  const uint8_t& idx_of_format3() const
  {
    assert_choice_type("indexOfFormat3", type_.to_string(), "TPC-Index");
    return c.get<uint8_t>();
  }
  const uint8_t& idx_of_format3_a() const
  {
    assert_choice_type("indexOfFormat3A", type_.to_string(), "TPC-Index");
    return c.get<uint8_t>();
  }
  uint8_t& set_idx_of_format3()
  {
    set(types::idx_of_format3);
    return c.get<uint8_t>();
  }
  uint8_t& set_idx_of_format3_a()
  {
    set(types::idx_of_format3_a);
    return c.get<uint8_t>();
  }

private:
  types               type_;
  pod_choice_buffer_t c;

  void destroy_();
};

// UL-AM-RLC-r15 ::= SEQUENCE
struct ul_am_rlc_r15_s {
  struct max_retx_thres_r15_opts {
    enum options { t1, t2, t3, t4, t6, t8, t16, t32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_retx_thres_r15_opts> max_retx_thres_r15_e_;

  // member variables
  t_poll_retx_e         t_poll_retx_r15;
  poll_pdu_r15_e        poll_pdu_r15;
  poll_byte_r14_e       poll_byte_r15;
  max_retx_thres_r15_e_ max_retx_thres_r15;
  bool                  extended_rlc_li_field_r15 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UL-UM-RLC ::= SEQUENCE
struct ul_um_rlc_s {
  sn_field_len_e sn_field_len;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// ZeroTxPowerCSI-RS-Conf-r12 ::= CHOICE
struct zero_tx_pwr_csi_rs_conf_r12_c {
  typedef setup_e types;

  // choice methods
  zero_tx_pwr_csi_rs_conf_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  zero_tx_pwr_csi_rs_r12_s& setup()
  {
    assert_choice_type("setup", type_.to_string(), "ZeroTxPowerCSI-RS-Conf-r12");
    return c;
  }
  const zero_tx_pwr_csi_rs_r12_s& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "ZeroTxPowerCSI-RS-Conf-r12");
    return c;
  }
  zero_tx_pwr_csi_rs_r12_s& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types                    type_;
  zero_tx_pwr_csi_rs_r12_s c;
};

// CQI-ReportConfig-r10 ::= SEQUENCE
struct cqi_report_cfg_r10_s {
  struct csi_sf_pattern_cfg_r10_c_ {
    struct setup_s_ {
      meas_sf_pattern_r10_c csi_meas_sf_set1_r10;
      meas_sf_pattern_r10_c csi_meas_sf_set2_r10;
    };
    typedef setup_e types;

    // choice methods
    csi_sf_pattern_cfg_r10_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-SubframePatternConfig-r10");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-SubframePatternConfig-r10");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                       cqi_report_aperiodic_r10_present = false;
  bool                       cqi_report_periodic_r10_present  = false;
  bool                       pmi_ri_report_r9_present         = false;
  bool                       csi_sf_pattern_cfg_r10_present   = false;
  cqi_report_aperiodic_r10_c cqi_report_aperiodic_r10;
  int8_t                     nom_pdsch_rs_epre_offset = -1;
  cqi_report_periodic_r10_c  cqi_report_periodic_r10;
  csi_sf_pattern_cfg_r10_c_  csi_sf_pattern_cfg_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1130 ::= SEQUENCE
struct cqi_report_cfg_v1130_s {
  cqi_report_periodic_v1130_s cqi_report_periodic_v1130;
  cqi_report_both_r11_s       cqi_report_both_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1250 ::= SEQUENCE
struct cqi_report_cfg_v1250_s {
  struct csi_sf_pattern_cfg_r12_c_ {
    struct setup_s_ {
      fixed_bitstring<10> csi_meas_sf_sets_r12;
    };
    typedef setup_e types;

    // choice methods
    csi_sf_pattern_cfg_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-SubframePatternConfig-r12");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-SubframePatternConfig-r12");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct alt_cqi_table_r12_opts {
    enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<alt_cqi_table_r12_opts> alt_cqi_table_r12_e_;

  // member variables
  bool                         csi_sf_pattern_cfg_r12_present     = false;
  bool                         cqi_report_both_v1250_present      = false;
  bool                         cqi_report_aperiodic_v1250_present = false;
  bool                         alt_cqi_table_r12_present          = false;
  csi_sf_pattern_cfg_r12_c_    csi_sf_pattern_cfg_r12;
  cqi_report_both_v1250_s      cqi_report_both_v1250;
  cqi_report_aperiodic_v1250_c cqi_report_aperiodic_v1250;
  alt_cqi_table_r12_e_         alt_cqi_table_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1310 ::= SEQUENCE
struct cqi_report_cfg_v1310_s {
  bool                         cqi_report_both_v1310_present      = false;
  bool                         cqi_report_aperiodic_v1310_present = false;
  bool                         cqi_report_periodic_v1310_present  = false;
  cqi_report_both_v1310_s      cqi_report_both_v1310;
  cqi_report_aperiodic_v1310_c cqi_report_aperiodic_v1310;
  cqi_report_periodic_v1310_s  cqi_report_periodic_v1310;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1320 ::= SEQUENCE
struct cqi_report_cfg_v1320_s {
  bool                        cqi_report_periodic_v1320_present = false;
  cqi_report_periodic_v1320_s cqi_report_periodic_v1320;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1430 ::= SEQUENCE
struct cqi_report_cfg_v1430_s {
  bool                              cqi_report_aperiodic_hybrid_r14_present = false;
  cqi_report_aperiodic_hybrid_r14_s cqi_report_aperiodic_hybrid_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-r10 ::= SEQUENCE
struct csi_rs_cfg_r10_s {
  struct csi_rs_r10_c_ {
    struct setup_s_ {
      struct ant_ports_count_r10_opts {
        enum options { an1, an2, an4, an8, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<ant_ports_count_r10_opts> ant_ports_count_r10_e_;

      // member variables
      ant_ports_count_r10_e_ ant_ports_count_r10;
      uint8_t                res_cfg_r10 = 0;
      uint8_t                sf_cfg_r10  = 0;
      int8_t                 p_c_r10     = -8;
    };
    typedef setup_e types;

    // choice methods
    csi_rs_r10_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-RS-r10");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-RS-r10");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                          csi_rs_r10_present             = false;
  bool                          zero_tx_pwr_csi_rs_r10_present = false;
  csi_rs_r10_c_                 csi_rs_r10;
  zero_tx_pwr_csi_rs_conf_r12_c zero_tx_pwr_csi_rs_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-v1250 ::= SEQUENCE
struct csi_rs_cfg_v1250_s {
  struct ds_zero_tx_pwr_csi_rs_r12_c_ {
    struct setup_s_ {
      using zero_tx_pwr_csi_rs_list_r12_l_ = dyn_array<zero_tx_pwr_csi_rs_r12_s>;

      // member variables
      zero_tx_pwr_csi_rs_list_r12_l_ zero_tx_pwr_csi_rs_list_r12;
    };
    typedef setup_e types;

    // choice methods
    ds_zero_tx_pwr_csi_rs_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ds-ZeroTxPowerCSI-RS-r12");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ds-ZeroTxPowerCSI-RS-r12");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                          zero_tx_pwr_csi_rs2_r12_present   = false;
  bool                          ds_zero_tx_pwr_csi_rs_r12_present = false;
  zero_tx_pwr_csi_rs_conf_r12_c zero_tx_pwr_csi_rs2_r12;
  ds_zero_tx_pwr_csi_rs_r12_c_  ds_zero_tx_pwr_csi_rs_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-v1310 ::= SEQUENCE
struct csi_rs_cfg_v1310_s {
  bool                   emimo_type_r13_present = false;
  csi_rs_cfg_emimo_r13_c emimo_type_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-v1430 ::= SEQUENCE
struct csi_rs_cfg_v1430_s {
  bool                          dummy_present                         = false;
  bool                          emimo_hybrid_r14_present              = false;
  bool                          advanced_codebook_enabled_r14_present = false;
  csi_rs_cfg_emimo_v1430_c      dummy;
  csi_rs_cfg_emimo_hybrid_r14_c emimo_hybrid_r14;
  bool                          advanced_codebook_enabled_r14 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigZP-r11 ::= SEQUENCE
struct csi_rs_cfg_zp_r11_s {
  bool                ext                  = false;
  uint8_t             csi_rs_cfg_zp_id_r11 = 1;
  fixed_bitstring<16> res_cfg_list_r11;
  uint8_t             sf_cfg_r11 = 0;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DL-AM-RLC ::= SEQUENCE
struct dl_am_rlc_s {
  t_reordering_e      t_reordering;
  t_status_prohibit_e t_status_prohibit;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DL-UM-RLC ::= SEQUENCE
struct dl_um_rlc_s {
  sn_field_len_e sn_field_len;
  t_reordering_e t_reordering;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DeltaTxD-OffsetListSPUCCH-r15 ::= SEQUENCE
struct delta_tx_d_offset_list_spucch_r15_s {
  struct delta_tx_d_offset_spucch_format1_r15_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_spucch_format1_r15_opts> delta_tx_d_offset_spucch_format1_r15_e_;
  struct delta_tx_d_offset_spucch_format1a_r15_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_spucch_format1a_r15_opts> delta_tx_d_offset_spucch_format1a_r15_e_;
  struct delta_tx_d_offset_spucch_format1b_r15_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_spucch_format1b_r15_opts> delta_tx_d_offset_spucch_format1b_r15_e_;
  struct delta_tx_d_offset_spucch_format3_r15_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_spucch_format3_r15_opts> delta_tx_d_offset_spucch_format3_r15_e_;

  // member variables
  bool                                     ext = false;
  delta_tx_d_offset_spucch_format1_r15_e_  delta_tx_d_offset_spucch_format1_r15;
  delta_tx_d_offset_spucch_format1a_r15_e_ delta_tx_d_offset_spucch_format1a_r15;
  delta_tx_d_offset_spucch_format1b_r15_e_ delta_tx_d_offset_spucch_format1b_r15;
  delta_tx_d_offset_spucch_format3_r15_e_  delta_tx_d_offset_spucch_format3_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EPDCCH-SetConfig-r11 ::= SEQUENCE
struct epdcch_set_cfg_r11_s {
  struct tx_type_r11_opts {
    enum options { localised, distributed, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<tx_type_r11_opts> tx_type_r11_e_;
  struct res_block_assign_r11_s_ {
    struct num_prb_pairs_r11_opts {
      enum options { n2, n4, n8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<num_prb_pairs_r11_opts> num_prb_pairs_r11_e_;

    // member variables
    num_prb_pairs_r11_e_     num_prb_pairs_r11;
    bounded_bitstring<4, 38> res_block_assign_r11;
  };
  struct csi_rs_cfg_zp_id2_r12_c_ {
    typedef setup_e types;

    // choice methods
    csi_rs_cfg_zp_id2_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& setup()
    {
      assert_choice_type("setup", type_.to_string(), "csi-RS-ConfigZPId2-r12");
      return c;
    }
    const uint8_t& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "csi-RS-ConfigZPId2-r12");
      return c;
    }
    uint8_t& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types   type_;
    uint8_t c;
  };
  struct num_prb_pairs_v1310_c_ {
    typedef setup_e types;

    // choice methods
    num_prb_pairs_v1310_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;

  private:
    types type_;
  };
  struct mpdcch_cfg_r13_c_ {
    struct setup_s_ {
      struct csi_num_repeat_ce_r13_opts {
        enum options { sf1, sf2, sf4, sf8, sf16, sf32, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<csi_num_repeat_ce_r13_opts> csi_num_repeat_ce_r13_e_;
      struct mpdcch_pdsch_hop_cfg_r13_opts {
        enum options { on, off, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<mpdcch_pdsch_hop_cfg_r13_opts> mpdcch_pdsch_hop_cfg_r13_e_;
      struct mpdcch_start_sf_uess_r13_c_ {
        struct fdd_r13_opts {
          enum options { v1, v1dot5, v2, v2dot5, v4, v5, v8, v10, nulltype } value;
          typedef float number_type;

          std::string to_string() const;
          float       to_number() const;
          std::string to_number_string() const;
        };
        typedef enumerated<fdd_r13_opts> fdd_r13_e_;
        struct tdd_r13_opts {
          enum options { v1, v2, v4, v5, v8, v10, v20, spare1, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<tdd_r13_opts> tdd_r13_e_;
        struct types_opts {
          enum options { fdd_r13, tdd_r13, nulltype } value;

          std::string to_string() const;
        };
        typedef enumerated<types_opts> types;

        // choice methods
        mpdcch_start_sf_uess_r13_c_() = default;
        mpdcch_start_sf_uess_r13_c_(const mpdcch_start_sf_uess_r13_c_& other);
        mpdcch_start_sf_uess_r13_c_& operator=(const mpdcch_start_sf_uess_r13_c_& other);
        ~mpdcch_start_sf_uess_r13_c_() { destroy_(); }
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        fdd_r13_e_& fdd_r13()
        {
          assert_choice_type("fdd-r13", type_.to_string(), "mpdcch-StartSF-UESS-r13");
          return c.get<fdd_r13_e_>();
        }
        tdd_r13_e_& tdd_r13()
        {
          assert_choice_type("tdd-r13", type_.to_string(), "mpdcch-StartSF-UESS-r13");
          return c.get<tdd_r13_e_>();
        }
        const fdd_r13_e_& fdd_r13() const
        {
          assert_choice_type("fdd-r13", type_.to_string(), "mpdcch-StartSF-UESS-r13");
          return c.get<fdd_r13_e_>();
        }
        const tdd_r13_e_& tdd_r13() const
        {
          assert_choice_type("tdd-r13", type_.to_string(), "mpdcch-StartSF-UESS-r13");
          return c.get<tdd_r13_e_>();
        }
        fdd_r13_e_& set_fdd_r13()
        {
          set(types::fdd_r13);
          return c.get<fdd_r13_e_>();
        }
        tdd_r13_e_& set_tdd_r13()
        {
          set(types::tdd_r13);
          return c.get<tdd_r13_e_>();
        }

      private:
        types               type_;
        pod_choice_buffer_t c;

        void destroy_();
      };
      struct mpdcch_num_repeat_r13_opts {
        enum options { r1, r2, r4, r8, r16, r32, r64, r128, r256, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<mpdcch_num_repeat_r13_opts> mpdcch_num_repeat_r13_e_;

      // member variables
      csi_num_repeat_ce_r13_e_    csi_num_repeat_ce_r13;
      mpdcch_pdsch_hop_cfg_r13_e_ mpdcch_pdsch_hop_cfg_r13;
      mpdcch_start_sf_uess_r13_c_ mpdcch_start_sf_uess_r13;
      mpdcch_num_repeat_r13_e_    mpdcch_num_repeat_r13;
      uint8_t                     mpdcch_nb_r13 = 1;
    };
    typedef setup_e types;

    // choice methods
    mpdcch_cfg_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "mpdcch-config-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "mpdcch-config-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                    ext                           = false;
  bool                    re_map_qcl_cfg_id_r11_present = false;
  uint8_t                 set_cfg_id_r11                = 0;
  tx_type_r11_e_          tx_type_r11;
  res_block_assign_r11_s_ res_block_assign_r11;
  uint16_t                dmrs_scrambling_seq_int_r11 = 0;
  uint16_t                pucch_res_start_offset_r11  = 0;
  uint8_t                 re_map_qcl_cfg_id_r11       = 1;
  // ...
  // group 0
  copy_ptr<csi_rs_cfg_zp_id2_r12_c_> csi_rs_cfg_zp_id2_r12;
  // group 1
  copy_ptr<num_prb_pairs_v1310_c_> num_prb_pairs_v1310;
  copy_ptr<mpdcch_cfg_r13_c_>      mpdcch_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// Enable256QAM-r14 ::= CHOICE
struct enable256_qam_r14_c {
  struct setup_c_ {
    struct tpc_sf_set_cfgured_r14_s_ {
      bool sf_set1_dci_format0_r14 = false;
      bool sf_set1_dci_format4_r14 = false;
      bool sf_set2_dci_format0_r14 = false;
      bool sf_set2_dci_format4_r14 = false;
    };
    struct tpc_sf_set_not_cfgured_r14_s_ {
      bool dci_format0_r14 = false;
      bool dci_format4_r14 = false;
    };
    struct types_opts {
      enum options { tpc_sf_set_cfgured_r14, tpc_sf_set_not_cfgured_r14, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    setup_c_(const setup_c_& other);
    setup_c_& operator=(const setup_c_& other);
    ~setup_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    tpc_sf_set_cfgured_r14_s_& tpc_sf_set_cfgured_r14()
    {
      assert_choice_type("tpc-SubframeSet-Configured-r14", type_.to_string(), "setup");
      return c.get<tpc_sf_set_cfgured_r14_s_>();
    }
    tpc_sf_set_not_cfgured_r14_s_& tpc_sf_set_not_cfgured_r14()
    {
      assert_choice_type("tpc-SubframeSet-NotConfigured-r14", type_.to_string(), "setup");
      return c.get<tpc_sf_set_not_cfgured_r14_s_>();
    }
    const tpc_sf_set_cfgured_r14_s_& tpc_sf_set_cfgured_r14() const
    {
      assert_choice_type("tpc-SubframeSet-Configured-r14", type_.to_string(), "setup");
      return c.get<tpc_sf_set_cfgured_r14_s_>();
    }
    const tpc_sf_set_not_cfgured_r14_s_& tpc_sf_set_not_cfgured_r14() const
    {
      assert_choice_type("tpc-SubframeSet-NotConfigured-r14", type_.to_string(), "setup");
      return c.get<tpc_sf_set_not_cfgured_r14_s_>();
    }
    tpc_sf_set_cfgured_r14_s_& set_tpc_sf_set_cfgured_r14()
    {
      set(types::tpc_sf_set_cfgured_r14);
      return c.get<tpc_sf_set_cfgured_r14_s_>();
    }
    tpc_sf_set_not_cfgured_r14_s_& set_tpc_sf_set_not_cfgured_r14()
    {
      set(types::tpc_sf_set_not_cfgured_r14);
      return c.get<tpc_sf_set_not_cfgured_r14_s_>();
    }

  private:
    types                                                                     type_;
    choice_buffer_t<tpc_sf_set_cfgured_r14_s_, tpc_sf_set_not_cfgured_r14_s_> c;

    void destroy_();
  };
  typedef setup_e types;

  // choice methods
  enable256_qam_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "Enable256QAM-r14");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "Enable256QAM-r14");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// LogicalChannelConfig ::= SEQUENCE
struct lc_ch_cfg_s {
  struct ul_specific_params_s_ {
    struct prioritised_bit_rate_opts {
      enum options {
        kbps0,
        kbps8,
        kbps16,
        kbps32,
        kbps64,
        kbps128,
        kbps256,
        infinity,
        kbps512_v1020,
        kbps1024_v1020,
        kbps2048_v1020,
        spare5,
        spare4,
        spare3,
        spare2,
        spare1,
        nulltype
      } value;
      typedef int16_t number_type;

      std::string to_string() const;
      int16_t     to_number() const;
    };
    typedef enumerated<prioritised_bit_rate_opts> prioritised_bit_rate_e_;
    struct bucket_size_dur_opts {
      enum options { ms50, ms100, ms150, ms300, ms500, ms1000, spare2, spare1, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<bucket_size_dur_opts> bucket_size_dur_e_;

    // member variables
    bool                    lc_ch_group_present = false;
    uint8_t                 prio                = 1;
    prioritised_bit_rate_e_ prioritised_bit_rate;
    bucket_size_dur_e_      bucket_size_dur;
    uint8_t                 lc_ch_group = 0;
  };
  struct bit_rate_query_prohibit_timer_r14_opts {
    enum options { s0, s0dot4, s0dot8, s1dot6, s3, s6, s12, s30, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<bit_rate_query_prohibit_timer_r14_opts> bit_rate_query_prohibit_timer_r14_e_;
  struct allowed_tti_lens_r15_c_ {
    struct setup_s_ {
      bool short_tti_r15 = false;
      bool sf_tti_r15    = false;
    };
    typedef setup_e types;

    // choice methods
    allowed_tti_lens_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "allowedTTI-Lengths-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "allowedTTI-Lengths-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct lc_ch_sr_restrict_r15_c_ {
    struct setup_opts {
      enum options { spucch, pucch, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    lc_ch_sr_restrict_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "logicalChannelSR-Restriction-r15");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "logicalChannelSR-Restriction-r15");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };
  struct chl_access_prio_r15_c_ {
    typedef setup_e types;

    // choice methods
    chl_access_prio_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& setup()
    {
      assert_choice_type("setup", type_.to_string(), "channellAccessPriority-r15");
      return c;
    }
    const uint8_t& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "channellAccessPriority-r15");
      return c;
    }
    uint8_t& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types   type_;
    uint8_t c;
  };

  // member variables
  bool                  ext                        = false;
  bool                  ul_specific_params_present = false;
  ul_specific_params_s_ ul_specific_params;
  // ...
  // group 0
  bool lc_ch_sr_mask_r9_present = false;
  // group 1
  bool lc_ch_sr_prohibit_r12_present = false;
  bool lc_ch_sr_prohibit_r12         = false;
  // group 2
  bool                                 laa_ul_allowed_r14_present                = false;
  bool                                 bit_rate_query_prohibit_timer_r14_present = false;
  bool                                 laa_ul_allowed_r14                        = false;
  bit_rate_query_prohibit_timer_r14_e_ bit_rate_query_prohibit_timer_r14;
  // group 3
  bool                               lch_cell_restrict_r15_present = false;
  copy_ptr<allowed_tti_lens_r15_c_>  allowed_tti_lens_r15;
  copy_ptr<lc_ch_sr_restrict_r15_c_> lc_ch_sr_restrict_r15;
  copy_ptr<chl_access_prio_r15_c_>   chl_access_prio_r15;
  fixed_bitstring<32>                lch_cell_restrict_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// P-a ::= ENUMERATED
struct p_a_opts {
  enum options { db_minus6, db_minus4dot77, db_minus3, db_minus1dot77, db0, db1, db2, db3, nulltype } value;
  typedef float number_type;

  std::string to_string() const;
  float       to_number() const;
  std::string to_number_string() const;
};
typedef enumerated<p_a_opts> p_a_e;

// PDSCH-RE-MappingQCL-Config-r11 ::= SEQUENCE
struct pdsch_re_map_qcl_cfg_r11_s {
  struct optional_set_of_fields_r11_s_ {
    struct crs_ports_count_r11_opts {
      enum options { n1, n2, n4, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<crs_ports_count_r11_opts> crs_ports_count_r11_e_;
    struct mbsfn_sf_cfg_list_r11_c_ {
      struct setup_s_ {
        mbsfn_sf_cfg_list_l sf_cfg_list;
      };
      typedef setup_e types;

      // choice methods
      mbsfn_sf_cfg_list_r11_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-r11");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-r11");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };
    struct pdsch_start_r11_opts {
      enum options { reserved, n1, n2, n3, n4, assigned, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<pdsch_start_r11_opts> pdsch_start_r11_e_;

    // member variables
    bool                     mbsfn_sf_cfg_list_r11_present = false;
    crs_ports_count_r11_e_   crs_ports_count_r11;
    uint8_t                  crs_freq_shift_r11 = 0;
    mbsfn_sf_cfg_list_r11_c_ mbsfn_sf_cfg_list_r11;
    pdsch_start_r11_e_       pdsch_start_r11;
  };
  struct mbsfn_sf_cfg_list_v1430_c_ {
    struct setup_s_ {
      mbsfn_sf_cfg_list_v1430_l sf_cfg_list_v1430;
    };
    typedef setup_e types;

    // choice methods
    mbsfn_sf_cfg_list_v1430_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1430");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1430");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct codeword_one_cfg_v1530_c_ {
    struct setup_s_ {
      struct crs_ports_count_v1530_opts {
        enum options { n1, n2, n4, spare1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<crs_ports_count_v1530_opts> crs_ports_count_v1530_e_;
      struct pdsch_start_v1530_opts {
        enum options { reserved, n1, n2, n3, n4, assigned, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<pdsch_start_v1530_opts> pdsch_start_v1530_e_;

      // member variables
      bool                      mbsfn_sf_cfg_list_v1530_present     = false;
      bool                      mbsfn_sf_cfg_list_ext_v1530_present = false;
      bool                      qcl_csi_rs_cfg_nzp_id_v1530_present = false;
      crs_ports_count_v1530_e_  crs_ports_count_v1530;
      uint8_t                   crs_freq_shift_v1530 = 0;
      mbsfn_sf_cfg_list_l       mbsfn_sf_cfg_list_v1530;
      mbsfn_sf_cfg_list_v1430_l mbsfn_sf_cfg_list_ext_v1530;
      pdsch_start_v1530_e_      pdsch_start_v1530;
      uint8_t                   csi_rs_cfg_zp_id_v1530      = 1;
      uint8_t                   qcl_csi_rs_cfg_nzp_id_v1530 = 1;
    };
    typedef setup_e types;

    // choice methods
    codeword_one_cfg_v1530_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "codewordOneConfig-v1530");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "codewordOneConfig-v1530");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                          ext                                = false;
  bool                          optional_set_of_fields_r11_present = false;
  bool                          qcl_csi_rs_cfg_nzp_id_r11_present  = false;
  uint8_t                       pdsch_re_map_qcl_cfg_id_r11        = 1;
  optional_set_of_fields_r11_s_ optional_set_of_fields_r11;
  uint8_t                       csi_rs_cfg_zp_id_r11      = 1;
  uint8_t                       qcl_csi_rs_cfg_nzp_id_r11 = 1;
  // ...
  // group 0
  copy_ptr<mbsfn_sf_cfg_list_v1430_c_> mbsfn_sf_cfg_list_v1430;
  // group 1
  copy_ptr<codeword_one_cfg_v1530_c_> codeword_one_cfg_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PollPDU-v1310 ::= ENUMERATED
struct poll_pdu_v1310_opts {
  enum options { p512, p1024, p2048, p4096, p6144, p8192, p12288, p16384, nulltype } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<poll_pdu_v1310_opts> poll_pdu_v1310_e;

// RLC-Config-r15 ::= SEQUENCE
struct rlc_cfg_r15_s {
  struct mode_r15_c_ {
    struct am_r15_s_ {
      ul_am_rlc_r15_s ul_am_rlc_r15;
      dl_am_rlc_r15_s dl_am_rlc_r15;
    };
    struct um_bi_dir_r15_s_ {
      ul_um_rlc_s     ul_um_rlc_r15;
      dl_um_rlc_r15_s dl_um_rlc_r15;
    };
    struct um_uni_dir_ul_r15_s_ {
      ul_um_rlc_s ul_um_rlc_r15;
    };
    struct um_uni_dir_dl_r15_s_ {
      dl_um_rlc_r15_s dl_um_rlc_r15;
    };
    struct types_opts {
      enum options { am_r15, um_bi_dir_r15, um_uni_dir_ul_r15, um_uni_dir_dl_r15, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    mode_r15_c_() = default;
    mode_r15_c_(const mode_r15_c_& other);
    mode_r15_c_& operator=(const mode_r15_c_& other);
    ~mode_r15_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    am_r15_s_& am_r15()
    {
      assert_choice_type("am-r15", type_.to_string(), "mode-r15");
      return c.get<am_r15_s_>();
    }
    um_bi_dir_r15_s_& um_bi_dir_r15()
    {
      assert_choice_type("um-Bi-Directional-r15", type_.to_string(), "mode-r15");
      return c.get<um_bi_dir_r15_s_>();
    }
    um_uni_dir_ul_r15_s_& um_uni_dir_ul_r15()
    {
      assert_choice_type("um-Uni-Directional-UL-r15", type_.to_string(), "mode-r15");
      return c.get<um_uni_dir_ul_r15_s_>();
    }
    um_uni_dir_dl_r15_s_& um_uni_dir_dl_r15()
    {
      assert_choice_type("um-Uni-Directional-DL-r15", type_.to_string(), "mode-r15");
      return c.get<um_uni_dir_dl_r15_s_>();
    }
    const am_r15_s_& am_r15() const
    {
      assert_choice_type("am-r15", type_.to_string(), "mode-r15");
      return c.get<am_r15_s_>();
    }
    const um_bi_dir_r15_s_& um_bi_dir_r15() const
    {
      assert_choice_type("um-Bi-Directional-r15", type_.to_string(), "mode-r15");
      return c.get<um_bi_dir_r15_s_>();
    }
    const um_uni_dir_ul_r15_s_& um_uni_dir_ul_r15() const
    {
      assert_choice_type("um-Uni-Directional-UL-r15", type_.to_string(), "mode-r15");
      return c.get<um_uni_dir_ul_r15_s_>();
    }
    const um_uni_dir_dl_r15_s_& um_uni_dir_dl_r15() const
    {
      assert_choice_type("um-Uni-Directional-DL-r15", type_.to_string(), "mode-r15");
      return c.get<um_uni_dir_dl_r15_s_>();
    }
    am_r15_s_& set_am_r15()
    {
      set(types::am_r15);
      return c.get<am_r15_s_>();
    }
    um_bi_dir_r15_s_& set_um_bi_dir_r15()
    {
      set(types::um_bi_dir_r15);
      return c.get<um_bi_dir_r15_s_>();
    }
    um_uni_dir_ul_r15_s_& set_um_uni_dir_ul_r15()
    {
      set(types::um_uni_dir_ul_r15);
      return c.get<um_uni_dir_ul_r15_s_>();
    }
    um_uni_dir_dl_r15_s_& set_um_uni_dir_dl_r15()
    {
      set(types::um_uni_dir_dl_r15);
      return c.get<um_uni_dir_dl_r15_s_>();
    }

  private:
    types                                                                                    type_;
    choice_buffer_t<am_r15_s_, um_bi_dir_r15_s_, um_uni_dir_dl_r15_s_, um_uni_dir_ul_r15_s_> c;

    void destroy_();
  };

  // member variables
  bool        ext                                   = false;
  bool        reestablish_rlc_r15_present           = false;
  bool        rlc_out_of_order_delivery_r15_present = false;
  mode_r15_c_ mode_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPDCCH-Set-r15 ::= SEQUENCE (SIZE (1..4)) OF SPDCCH-Elements-r15
using spdcch_set_r15_l = dyn_array<spdcch_elems_r15_c>;

// SPUCCH-Set-r15 ::= SEQUENCE (SIZE (1..4)) OF SPUCCH-Elements-r15
using spucch_set_r15_l = dyn_array<spucch_elems_r15_c>;

// SR-SubslotSPUCCH-ResourceList-r15 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (0..1319)
using sr_subslot_spucch_res_list_r15_l = bounded_array<uint16_t, 4>;

// SRS-AntennaPort ::= ENUMERATED
struct srs_ant_port_opts {
  enum options { an1, an2, an4, spare1, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<srs_ant_port_opts> srs_ant_port_e;

// ShortTTI-Length-r15 ::= ENUMERATED
struct short_tti_len_r15_opts {
  enum options { slot, subslot, nulltype } value;

  std::string to_string() const;
};
typedef enumerated<short_tti_len_r15_opts> short_tti_len_r15_e;

// TPC-PDCCH-Config ::= CHOICE
struct tpc_pdcch_cfg_c {
  struct setup_s_ {
    fixed_bitstring<16> tpc_rnti;
    tpc_idx_c           tpc_idx;
  };
  typedef setup_e types;

  // choice methods
  tpc_pdcch_cfg_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "TPC-PDCCH-Config");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "TPC-PDCCH-Config");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// UL-AM-RLC ::= SEQUENCE
struct ul_am_rlc_s {
  struct max_retx_thres_opts {
    enum options { t1, t2, t3, t4, t6, t8, t16, t32, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_retx_thres_opts> max_retx_thres_e_;

  // member variables
  t_poll_retx_e     t_poll_retx;
  poll_pdu_e        poll_pdu;
  poll_byte_e       poll_byte;
  max_retx_thres_e_ max_retx_thres;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicatedSTTI-r15 ::= CHOICE
struct ant_info_ded_stti_r15_c {
  struct setup_s_ {
    struct tx_mode_dl_mbsfn_r15_opts {
      enum options { tm9, tm10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tx_mode_dl_mbsfn_r15_opts> tx_mode_dl_mbsfn_r15_e_;
    struct tx_mode_dl_non_mbsfn_r15_opts {
      enum options { tm1, tm2, tm3, tm4, tm6, tm8, tm9, tm10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tx_mode_dl_non_mbsfn_r15_opts> tx_mode_dl_non_mbsfn_r15_e_;
    struct codebook_subset_restrict_c_ {
      struct types_opts {
        enum options {
          n2_tx_ant_tm3_r15,
          n4_tx_ant_tm3_r15,
          n2_tx_ant_tm4_r15,
          n4_tx_ant_tm4_r15,
          n2_tx_ant_tm5_r15,
          n4_tx_ant_tm5_r15,
          n2_tx_ant_tm6_r15,
          n4_tx_ant_tm6_r15,
          n2_tx_ant_tm8_r15,
          n4_tx_ant_tm8_r15,
          n2_tx_ant_tm9and10_r15,
          n4_tx_ant_tm9and10_r15,
          n8_tx_ant_tm9and10_r15,
          nulltype
        } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      codebook_subset_restrict_c_() = default;
      codebook_subset_restrict_c_(const codebook_subset_restrict_c_& other);
      codebook_subset_restrict_c_& operator=(const codebook_subset_restrict_c_& other);
      ~codebook_subset_restrict_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      fixed_bitstring<2>& n2_tx_ant_tm3_r15()
      {
        assert_choice_type("n2TxAntenna-tm3-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<2> >();
      }
      fixed_bitstring<4>& n4_tx_ant_tm3_r15()
      {
        assert_choice_type("n4TxAntenna-tm3-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<6>& n2_tx_ant_tm4_r15()
      {
        assert_choice_type("n2TxAntenna-tm4-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<64>& n4_tx_ant_tm4_r15()
      {
        assert_choice_type("n4TxAntenna-tm4-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<64> >();
      }
      fixed_bitstring<4>& n2_tx_ant_tm5_r15()
      {
        assert_choice_type("n2TxAntenna-tm5-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<16>& n4_tx_ant_tm5_r15()
      {
        assert_choice_type("n4TxAntenna-tm5-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<16> >();
      }
      fixed_bitstring<4>& n2_tx_ant_tm6_r15()
      {
        assert_choice_type("n2TxAntenna-tm6-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<16>& n4_tx_ant_tm6_r15()
      {
        assert_choice_type("n4TxAntenna-tm6-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<16> >();
      }
      fixed_bitstring<6>& n2_tx_ant_tm8_r15()
      {
        assert_choice_type("n2TxAntenna-tm8-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<64>& n4_tx_ant_tm8_r15()
      {
        assert_choice_type("n4TxAntenna-tm8-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<64> >();
      }
      fixed_bitstring<6>& n2_tx_ant_tm9and10_r15()
      {
        assert_choice_type("n2TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<96>& n4_tx_ant_tm9and10_r15()
      {
        assert_choice_type("n4TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<96> >();
      }
      fixed_bitstring<109>& n8_tx_ant_tm9and10_r15()
      {
        assert_choice_type("n8TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<109> >();
      }
      const fixed_bitstring<2>& n2_tx_ant_tm3_r15() const
      {
        assert_choice_type("n2TxAntenna-tm3-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<2> >();
      }
      const fixed_bitstring<4>& n4_tx_ant_tm3_r15() const
      {
        assert_choice_type("n4TxAntenna-tm3-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      const fixed_bitstring<6>& n2_tx_ant_tm4_r15() const
      {
        assert_choice_type("n2TxAntenna-tm4-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      const fixed_bitstring<64>& n4_tx_ant_tm4_r15() const
      {
        assert_choice_type("n4TxAntenna-tm4-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<64> >();
      }
      const fixed_bitstring<4>& n2_tx_ant_tm5_r15() const
      {
        assert_choice_type("n2TxAntenna-tm5-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      const fixed_bitstring<16>& n4_tx_ant_tm5_r15() const
      {
        assert_choice_type("n4TxAntenna-tm5-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<16> >();
      }
      const fixed_bitstring<4>& n2_tx_ant_tm6_r15() const
      {
        assert_choice_type("n2TxAntenna-tm6-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<4> >();
      }
      const fixed_bitstring<16>& n4_tx_ant_tm6_r15() const
      {
        assert_choice_type("n4TxAntenna-tm6-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<16> >();
      }
      const fixed_bitstring<6>& n2_tx_ant_tm8_r15() const
      {
        assert_choice_type("n2TxAntenna-tm8-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      const fixed_bitstring<64>& n4_tx_ant_tm8_r15() const
      {
        assert_choice_type("n4TxAntenna-tm8-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<64> >();
      }
      const fixed_bitstring<6>& n2_tx_ant_tm9and10_r15() const
      {
        assert_choice_type("n2TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<6> >();
      }
      const fixed_bitstring<96>& n4_tx_ant_tm9and10_r15() const
      {
        assert_choice_type("n4TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<96> >();
      }
      const fixed_bitstring<109>& n8_tx_ant_tm9and10_r15() const
      {
        assert_choice_type("n8TxAntenna-tm9and10-r15", type_.to_string(), "codebookSubsetRestriction");
        return c.get<fixed_bitstring<109> >();
      }
      fixed_bitstring<2>& set_n2_tx_ant_tm3_r15()
      {
        set(types::n2_tx_ant_tm3_r15);
        return c.get<fixed_bitstring<2> >();
      }
      fixed_bitstring<4>& set_n4_tx_ant_tm3_r15()
      {
        set(types::n4_tx_ant_tm3_r15);
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<6>& set_n2_tx_ant_tm4_r15()
      {
        set(types::n2_tx_ant_tm4_r15);
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<64>& set_n4_tx_ant_tm4_r15()
      {
        set(types::n4_tx_ant_tm4_r15);
        return c.get<fixed_bitstring<64> >();
      }
      fixed_bitstring<4>& set_n2_tx_ant_tm5_r15()
      {
        set(types::n2_tx_ant_tm5_r15);
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<16>& set_n4_tx_ant_tm5_r15()
      {
        set(types::n4_tx_ant_tm5_r15);
        return c.get<fixed_bitstring<16> >();
      }
      fixed_bitstring<4>& set_n2_tx_ant_tm6_r15()
      {
        set(types::n2_tx_ant_tm6_r15);
        return c.get<fixed_bitstring<4> >();
      }
      fixed_bitstring<16>& set_n4_tx_ant_tm6_r15()
      {
        set(types::n4_tx_ant_tm6_r15);
        return c.get<fixed_bitstring<16> >();
      }
      fixed_bitstring<6>& set_n2_tx_ant_tm8_r15()
      {
        set(types::n2_tx_ant_tm8_r15);
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<64>& set_n4_tx_ant_tm8_r15()
      {
        set(types::n4_tx_ant_tm8_r15);
        return c.get<fixed_bitstring<64> >();
      }
      fixed_bitstring<6>& set_n2_tx_ant_tm9and10_r15()
      {
        set(types::n2_tx_ant_tm9and10_r15);
        return c.get<fixed_bitstring<6> >();
      }
      fixed_bitstring<96>& set_n4_tx_ant_tm9and10_r15()
      {
        set(types::n4_tx_ant_tm9and10_r15);
        return c.get<fixed_bitstring<96> >();
      }
      fixed_bitstring<109>& set_n8_tx_ant_tm9and10_r15()
      {
        set(types::n8_tx_ant_tm9and10_r15);
        return c.get<fixed_bitstring<109> >();
      }

    private:
      types                                  type_;
      choice_buffer_t<fixed_bitstring<109> > c;

      void destroy_();
    };
    struct max_layers_mimo_stti_r15_opts {
      enum options { two_layers, four_layers, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<max_layers_mimo_stti_r15_opts> max_layers_mimo_stti_r15_e_;

    // member variables
    bool                        tx_mode_dl_mbsfn_r15_present     = false;
    bool                        tx_mode_dl_non_mbsfn_r15_present = false;
    bool                        codebook_subset_restrict_present = false;
    bool                        max_layers_mimo_stti_r15_present = false;
    tx_mode_dl_mbsfn_r15_e_     tx_mode_dl_mbsfn_r15;
    tx_mode_dl_non_mbsfn_r15_e_ tx_mode_dl_non_mbsfn_r15;
    codebook_subset_restrict_c_ codebook_subset_restrict;
    max_layers_mimo_stti_r15_e_ max_layers_mimo_stti_r15;
    bool                        slot_subslot_pdsch_tx_div_minus2_layer_r15 = false;
    bool                        slot_subslot_pdsch_tx_div_minus4_layer_r15 = false;
  };
  typedef setup_e types;

  // choice methods
  ant_info_ded_stti_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "AntennaInfoDedicatedSTTI-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "AntennaInfoDedicatedSTTI-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// AntennaInfoUL-STTI-r15 ::= SEQUENCE
struct ant_info_ul_stti_r15_s {
  struct tx_mode_ul_stti_r15_opts {
    enum options { tm1, tm2, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_mode_ul_stti_r15_opts> tx_mode_ul_stti_r15_e_;

  // member variables
  bool                   tx_mode_ul_stti_r15_present = false;
  tx_mode_ul_stti_r15_e_ tx_mode_ul_stti_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-r15 ::= CHOICE
struct cqi_report_cfg_r15_c {
  struct setup_s_ {
    struct alt_cqi_table_minus1024_qam_r15_opts {
      enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<alt_cqi_table_minus1024_qam_r15_opts> alt_cqi_table_minus1024_qam_r15_e_;

    // member variables
    bool                               cqi_report_cfg_r10_present              = false;
    bool                               cqi_report_cfg_v1130_present            = false;
    bool                               cqi_report_cfg_pcell_v1250_present      = false;
    bool                               cqi_report_cfg_v1310_present            = false;
    bool                               cqi_report_cfg_v1320_present            = false;
    bool                               cqi_report_cfg_v1430_present            = false;
    bool                               alt_cqi_table_minus1024_qam_r15_present = false;
    cqi_report_cfg_r10_s               cqi_report_cfg_r10;
    cqi_report_cfg_v1130_s             cqi_report_cfg_v1130;
    cqi_report_cfg_v1250_s             cqi_report_cfg_pcell_v1250;
    cqi_report_cfg_v1310_s             cqi_report_cfg_v1310;
    cqi_report_cfg_v1320_s             cqi_report_cfg_v1320;
    cqi_report_cfg_v1430_s             cqi_report_cfg_v1430;
    alt_cqi_table_minus1024_qam_r15_e_ alt_cqi_table_minus1024_qam_r15;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportConfig-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportConfig-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportPeriodic ::= CHOICE
struct cqi_report_periodic_c {
  struct setup_s_ {
    struct cqi_format_ind_periodic_c_ {
      struct subband_cqi_s_ {
        uint8_t k = 1;
      };
      struct types_opts {
        enum options { wideband_cqi, subband_cqi, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      cqi_format_ind_periodic_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      subband_cqi_s_& subband_cqi()
      {
        assert_choice_type("subbandCQI", type_.to_string(), "cqi-FormatIndicatorPeriodic");
        return c;
      }
      const subband_cqi_s_& subband_cqi() const
      {
        assert_choice_type("subbandCQI", type_.to_string(), "cqi-FormatIndicatorPeriodic");
        return c;
      }
      subband_cqi_s_& set_subband_cqi()
      {
        set(types::subband_cqi);
        return c;
      }

    private:
      types          type_;
      subband_cqi_s_ c;
    };

    // member variables
    bool                       ri_cfg_idx_present = false;
    uint16_t                   cqi_pucch_res_idx  = 0;
    uint16_t                   cqi_pmi_cfg_idx    = 0;
    cqi_format_ind_periodic_c_ cqi_format_ind_periodic;
    uint16_t                   ri_cfg_idx             = 0;
    bool                       simul_ack_nack_and_cqi = false;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_periodic_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodic");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodic");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CRS-AssistanceInfo-r11 ::= SEQUENCE
struct crs_assist_info_r11_s {
  struct ant_ports_count_r11_opts {
    enum options { an1, an2, an4, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ant_ports_count_r11_opts> ant_ports_count_r11_e_;

  // member variables
  bool                   ext     = false;
  uint16_t               pci_r11 = 0;
  ant_ports_count_r11_e_ ant_ports_count_r11;
  mbsfn_sf_cfg_list_l    mbsfn_sf_cfg_list_r11;
  // ...
  // group 0
  copy_ptr<mbsfn_sf_cfg_list_v1430_l> mbsfn_sf_cfg_list_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CRS-AssistanceInfo-r13 ::= SEQUENCE
struct crs_assist_info_r13_s {
  struct ant_ports_count_r13_opts {
    enum options { an1, an2, an4, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ant_ports_count_r13_opts> ant_ports_count_r13_e_;

  // member variables
  bool                   ext                           = false;
  bool                   mbsfn_sf_cfg_list_r13_present = false;
  uint16_t               pci_r13                       = 0;
  ant_ports_count_r13_e_ ant_ports_count_r13;
  mbsfn_sf_cfg_list_l    mbsfn_sf_cfg_list_r13;
  // ...
  // group 0
  copy_ptr<mbsfn_sf_cfg_list_v1430_l> mbsfn_sf_cfg_list_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CRS-AssistanceInfo-r15 ::= SEQUENCE
struct crs_assist_info_r15_s {
  bool     crs_intf_mitig_enabled_minus15_present = false;
  uint16_t pci_r15                                = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-r15 ::= CHOICE
struct csi_rs_cfg_r15_c {
  struct setup_s_ {
    bool               csi_rs_cfg_r10_present   = false;
    bool               csi_rs_cfg_v1250_present = false;
    bool               csi_rs_cfg_v1310_present = false;
    bool               csi_rs_cfg_v1430_present = false;
    csi_rs_cfg_r10_s   csi_rs_cfg_r10;
    csi_rs_cfg_v1250_s csi_rs_cfg_v1250;
    csi_rs_cfg_v1310_s csi_rs_cfg_v1310;
    csi_rs_cfg_v1430_s csi_rs_cfg_v1430;
  };
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CSI-RS-ConfigNZPToAddModList-r15 ::= SEQUENCE (SIZE (1..24)) OF CSI-RS-ConfigNZP-r11
using csi_rs_cfg_nzp_to_add_mod_list_r15_l = dyn_array<csi_rs_cfg_nzp_r11_s>;

// CSI-RS-ConfigNZPToReleaseList-r15 ::= SEQUENCE (SIZE (1..24)) OF INTEGER (1..24)
using csi_rs_cfg_nzp_to_release_list_r15_l = bounded_array<uint8_t, 24>;

// CSI-RS-ConfigZP-ApList-r14 ::= CHOICE
struct csi_rs_cfg_zp_ap_list_r14_c {
  using setup_l_ = dyn_array<csi_rs_cfg_zp_r11_s>;
  typedef setup_e types;

  // choice methods
  csi_rs_cfg_zp_ap_list_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_l_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigZP-ApList-r14");
    return c;
  }
  const setup_l_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CSI-RS-ConfigZP-ApList-r14");
    return c;
  }
  setup_l_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_l_ c;
};

// CSI-RS-ConfigZPToAddModList-r11 ::= SEQUENCE (SIZE (1..4)) OF CSI-RS-ConfigZP-r11
using csi_rs_cfg_zp_to_add_mod_list_r11_l = dyn_array<csi_rs_cfg_zp_r11_s>;

// CSI-RS-ConfigZPToReleaseList-r11 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..4)
using csi_rs_cfg_zp_to_release_list_r11_l = bounded_array<uint8_t, 4>;

// DMRS-Config-r11 ::= CHOICE
struct dmrs_cfg_r11_c {
  struct setup_s_ {
    uint16_t scrambling_id_r11  = 0;
    uint16_t scrambling_id2_r11 = 0;
  };
  typedef setup_e types;

  // choice methods
  dmrs_cfg_r11_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "DMRS-Config-r11");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "DMRS-Config-r11");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// DMRS-Config-v1310 ::= SEQUENCE
struct dmrs_cfg_v1310_s {
  bool dmrs_table_alt_r13_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DeltaTxD-OffsetListPUCCH-r10 ::= SEQUENCE
struct delta_tx_d_offset_list_pucch_r10_s {
  struct delta_tx_d_offset_pucch_format1_r10_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_pucch_format1_r10_opts> delta_tx_d_offset_pucch_format1_r10_e_;
  struct delta_tx_d_offset_pucch_format1a1b_r10_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_pucch_format1a1b_r10_opts> delta_tx_d_offset_pucch_format1a1b_r10_e_;
  struct delta_tx_d_offset_pucch_format22a2b_r10_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_pucch_format22a2b_r10_opts> delta_tx_d_offset_pucch_format22a2b_r10_e_;
  struct delta_tx_d_offset_pucch_format3_r10_opts {
    enum options { db0, db_minus2, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_pucch_format3_r10_opts> delta_tx_d_offset_pucch_format3_r10_e_;

  // member variables
  bool                                       ext = false;
  delta_tx_d_offset_pucch_format1_r10_e_     delta_tx_d_offset_pucch_format1_r10;
  delta_tx_d_offset_pucch_format1a1b_r10_e_  delta_tx_d_offset_pucch_format1a1b_r10;
  delta_tx_d_offset_pucch_format22a2b_r10_e_ delta_tx_d_offset_pucch_format22a2b_r10;
  delta_tx_d_offset_pucch_format3_r10_e_     delta_tx_d_offset_pucch_format3_r10;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DeltaTxD-OffsetListPUCCH-v1130 ::= SEQUENCE
struct delta_tx_d_offset_list_pucch_v1130_s {
  struct delta_tx_d_offset_pucch_format1b_cs_r11_opts {
    enum options { db0, db_minus1, nulltype } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_tx_d_offset_pucch_format1b_cs_r11_opts> delta_tx_d_offset_pucch_format1b_cs_r11_e_;

  // member variables
  delta_tx_d_offset_pucch_format1b_cs_r11_e_ delta_tx_d_offset_pucch_format1b_cs_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// EIMTA-MainConfig-r12 ::= CHOICE
struct eimta_main_cfg_r12_c {
  struct setup_s_ {
    struct eimta_cmd_periodicity_r12_opts {
      enum options { sf10, sf20, sf40, sf80, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<eimta_cmd_periodicity_r12_opts> eimta_cmd_periodicity_r12_e_;

    // member variables
    fixed_bitstring<16>          eimta_rnti_r12;
    eimta_cmd_periodicity_r12_e_ eimta_cmd_periodicity_r12;
    fixed_bitstring<10>          eimta_cmd_sf_set_r12;
  };
  typedef setup_e types;

  // choice methods
  eimta_main_cfg_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "EIMTA-MainConfig-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "EIMTA-MainConfig-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// EIMTA-MainConfigServCell-r12 ::= CHOICE
struct eimta_main_cfg_serv_cell_r12_c {
  struct setup_s_ {
    struct eimta_harq_ref_cfg_r12_opts {
      enum options { sa2, sa4, sa5, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<eimta_harq_ref_cfg_r12_opts> eimta_harq_ref_cfg_r12_e_;
    struct mbsfn_sf_cfg_list_v1250_c_ {
      struct setup_s_ {
        mbsfn_sf_cfg_list_l sf_cfg_list_r12;
      };
      typedef setup_e types;

      // choice methods
      mbsfn_sf_cfg_list_v1250_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1250");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "mbsfn-SubframeConfigList-v1250");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    uint8_t                    eimta_ul_dl_cfg_idx_r12 = 1;
    eimta_harq_ref_cfg_r12_e_  eimta_harq_ref_cfg_r12;
    mbsfn_sf_cfg_list_v1250_c_ mbsfn_sf_cfg_list_v1250;
  };
  typedef setup_e types;

  // choice methods
  eimta_main_cfg_serv_cell_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "EIMTA-MainConfigServCell-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "EIMTA-MainConfigServCell-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// EPDCCH-SetConfigToAddModList-r11 ::= SEQUENCE (SIZE (1..2)) OF EPDCCH-SetConfig-r11
using epdcch_set_cfg_to_add_mod_list_r11_l = dyn_array<epdcch_set_cfg_r11_s>;

// EPDCCH-SetConfigToReleaseList-r11 ::= SEQUENCE (SIZE (1..2)) OF INTEGER (0..1)
using epdcch_set_cfg_to_release_list_r11_l = bounded_array<uint8_t, 2>;

// Format4-resource-r13 ::= SEQUENCE
struct format4_res_r13_s {
  uint8_t start_prb_format4_r13 = 0;
  uint8_t nof_prb_format4_r13   = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// Format5-resource-r13 ::= SEQUENCE
struct format5_res_r13_s {
  uint8_t start_prb_format5_r13 = 0;
  uint8_t cdm_idx_format5_r13   = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// N1PUCCH-AN-CS-r10 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (0..2047)
using n1_pucch_an_cs_r10_l = bounded_array<uint16_t, 4>;

// N1PUCCH-AN-PersistentList ::= SEQUENCE (SIZE (1..4)) OF INTEGER (0..2047)
using n1_pucch_an_persistent_list_l = bounded_array<uint16_t, 4>;

// NeighCellsInfo-r12 ::= SEQUENCE
struct neigh_cells_info_r12_s {
  struct crs_ports_count_r12_opts {
    enum options { n1, n2, n4, spare, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<crs_ports_count_r12_opts> crs_ports_count_r12_e_;
  using p_a_list_r12_l_ = bounded_array<p_a_e, 3>;

  // member variables
  bool                   ext                      = false;
  bool                   mbsfn_sf_cfg_r12_present = false;
  uint16_t               pci_r12                  = 0;
  uint8_t                p_b_r12                  = 0;
  crs_ports_count_r12_e_ crs_ports_count_r12;
  mbsfn_sf_cfg_list_l    mbsfn_sf_cfg_r12;
  p_a_list_r12_l_        p_a_list_r12;
  fixed_bitstring<8>     tx_mode_list_r12;
  uint8_t                res_alloc_granularity_r12 = 1;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDCCH-CandidateReductionValue-r13 ::= ENUMERATED
struct pdcch_candidate_reduction_value_r13_opts {
  enum options { n0, n33, n66, n100, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<pdcch_candidate_reduction_value_r13_opts> pdcch_candidate_reduction_value_r13_e;

// PDCP-Config ::= SEQUENCE
struct pdcp_cfg_s {
  struct discard_timer_opts {
    enum options { ms50, ms100, ms150, ms300, ms500, ms750, ms1500, infinity, nulltype } value;
    typedef int16_t number_type;

    std::string to_string() const;
    int16_t     to_number() const;
  };
  typedef enumerated<discard_timer_opts> discard_timer_e_;
  struct rlc_am_s_ {
    bool status_report_required = false;
  };
  struct rlc_um_s_ {
    struct pdcp_sn_size_opts {
      enum options { len7bits, len12bits, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<pdcp_sn_size_opts> pdcp_sn_size_e_;

    // member variables
    pdcp_sn_size_e_ pdcp_sn_size;
  };
  struct hdr_compress_c_ {
    struct rohc_s_ {
      struct profiles_s_ {
        bool profile0x0001 = false;
        bool profile0x0002 = false;
        bool profile0x0003 = false;
        bool profile0x0004 = false;
        bool profile0x0006 = false;
        bool profile0x0101 = false;
        bool profile0x0102 = false;
        bool profile0x0103 = false;
        bool profile0x0104 = false;
      };

      // member variables
      bool        ext             = false;
      bool        max_cid_present = false;
      uint16_t    max_cid         = 1;
      profiles_s_ profiles;
      // ...
    };
    struct types_opts {
      enum options { not_used, rohc, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    hdr_compress_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rohc_s_& rohc()
    {
      assert_choice_type("rohc", type_.to_string(), "headerCompression");
      return c;
    }
    const rohc_s_& rohc() const
    {
      assert_choice_type("rohc", type_.to_string(), "headerCompression");
      return c;
    }
    rohc_s_& set_rohc()
    {
      set(types::rohc);
      return c;
    }

  private:
    types   type_;
    rohc_s_ c;
  };
  struct t_reordering_r12_opts {
    enum options {
      ms0,
      ms20,
      ms40,
      ms60,
      ms80,
      ms100,
      ms120,
      ms140,
      ms160,
      ms180,
      ms200,
      ms220,
      ms240,
      ms260,
      ms280,
      ms300,
      ms500,
      ms750,
      spare14,
      spare13,
      spare12,
      spare11,
      spare10,
      spare9,
      spare8,
      spare7,
      spare6,
      spare5,
      spare4,
      spare3,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t_reordering_r12_opts> t_reordering_r12_e_;
  struct ul_data_split_thres_r13_c_ {
    struct setup_opts {
      enum options {
        b0,
        b100,
        b200,
        b400,
        b800,
        b1600,
        b3200,
        b6400,
        b12800,
        b25600,
        b51200,
        b102400,
        b204800,
        b409600,
        b819200,
        spare1,
        nulltype
      } value;
      typedef uint32_t number_type;

      std::string to_string() const;
      uint32_t    to_number() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    ul_data_split_thres_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ul-DataSplitThreshold-r13");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ul-DataSplitThreshold-r13");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };
  struct status_feedback_r13_c_ {
    struct setup_s_ {
      struct status_pdu_type_for_polling_r13_opts {
        enum options { type1, type2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<status_pdu_type_for_polling_r13_opts> status_pdu_type_for_polling_r13_e_;
      struct status_pdu_periodicity_type1_r13_opts {
        enum options {
          ms5,
          ms10,
          ms20,
          ms30,
          ms40,
          ms50,
          ms60,
          ms70,
          ms80,
          ms90,
          ms100,
          ms150,
          ms200,
          ms300,
          ms500,
          ms1000,
          ms2000,
          ms5000,
          ms10000,
          ms20000,
          ms50000,
          nulltype
        } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<status_pdu_periodicity_type1_r13_opts> status_pdu_periodicity_type1_r13_e_;
      struct status_pdu_periodicity_type2_r13_opts {
        enum options {
          ms5,
          ms10,
          ms20,
          ms30,
          ms40,
          ms50,
          ms60,
          ms70,
          ms80,
          ms90,
          ms100,
          ms150,
          ms200,
          ms300,
          ms500,
          ms1000,
          ms2000,
          ms5000,
          ms10000,
          ms20000,
          ms50000,
          nulltype
        } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<status_pdu_periodicity_type2_r13_opts> status_pdu_periodicity_type2_r13_e_;
      struct status_pdu_periodicity_offset_r13_opts {
        enum options { ms1, ms2, ms5, ms10, ms25, ms50, ms100, ms250, ms500, ms2500, ms5000, ms25000, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<status_pdu_periodicity_offset_r13_opts> status_pdu_periodicity_offset_r13_e_;

      // member variables
      bool                                 status_pdu_type_for_polling_r13_present   = false;
      bool                                 status_pdu_periodicity_type1_r13_present  = false;
      bool                                 status_pdu_periodicity_type2_r13_present  = false;
      bool                                 status_pdu_periodicity_offset_r13_present = false;
      status_pdu_type_for_polling_r13_e_   status_pdu_type_for_polling_r13;
      status_pdu_periodicity_type1_r13_e_  status_pdu_periodicity_type1_r13;
      status_pdu_periodicity_type2_r13_e_  status_pdu_periodicity_type2_r13;
      status_pdu_periodicity_offset_r13_e_ status_pdu_periodicity_offset_r13;
    };
    typedef setup_e types;

    // choice methods
    status_feedback_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "statusFeedback-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "statusFeedback-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct ul_lwa_cfg_r14_c_ {
    struct setup_s_ {
      struct ul_lwa_data_split_thres_r14_opts {
        enum options {
          b0,
          b100,
          b200,
          b400,
          b800,
          b1600,
          b3200,
          b6400,
          b12800,
          b25600,
          b51200,
          b102400,
          b204800,
          b409600,
          b819200,
          nulltype
        } value;
        typedef uint32_t number_type;

        std::string to_string() const;
        uint32_t    to_number() const;
      };
      typedef enumerated<ul_lwa_data_split_thres_r14_opts> ul_lwa_data_split_thres_r14_e_;

      // member variables
      bool                           ul_lwa_data_split_thres_r14_present = false;
      bool                           ul_lwa_drb_via_wlan_r14             = false;
      ul_lwa_data_split_thres_r14_e_ ul_lwa_data_split_thres_r14;
    };
    typedef setup_e types;

    // choice methods
    ul_lwa_cfg_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ul-LWA-Config-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ul-LWA-Config-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct ul_only_hdr_compress_r14_c_ {
    struct rohc_r14_s_ {
      struct profiles_r14_s_ {
        bool profile0x0006_r14 = false;
      };

      // member variables
      bool            ext                 = false;
      bool            max_cid_r14_present = false;
      uint16_t        max_cid_r14         = 1;
      profiles_r14_s_ profiles_r14;
      // ...
    };
    struct types_opts {
      enum options { not_used_r14, rohc_r14, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    ul_only_hdr_compress_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rohc_r14_s_& rohc_r14()
    {
      assert_choice_type("rohc-r14", type_.to_string(), "uplinkOnlyHeaderCompression-r14");
      return c;
    }
    const rohc_r14_s_& rohc_r14() const
    {
      assert_choice_type("rohc-r14", type_.to_string(), "uplinkOnlyHeaderCompression-r14");
      return c;
    }
    rohc_r14_s_& set_rohc_r14()
    {
      set(types::rohc_r14);
      return c;
    }

  private:
    types       type_;
    rohc_r14_s_ c;
  };
  struct ul_data_compress_r15_s_ {
    struct buffer_size_r15_opts {
      enum options { kbyte2, kbyte4, kbyte8, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<buffer_size_r15_opts> buffer_size_r15_e_;
    struct dictionary_r15_opts {
      enum options { sip_sdp, operator_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<dictionary_r15_opts> dictionary_r15_e_;

    // member variables
    bool               ext                    = false;
    bool               dictionary_r15_present = false;
    buffer_size_r15_e_ buffer_size_r15;
    dictionary_r15_e_  dictionary_r15;
    // ...
  };
  struct pdcp_dupl_cfg_r15_c_ {
    struct setup_s_ {
      struct pdcp_dupl_r15_opts {
        enum options { cfgured, activ, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<pdcp_dupl_r15_opts> pdcp_dupl_r15_e_;

      // member variables
      pdcp_dupl_r15_e_ pdcp_dupl_r15;
    };
    typedef setup_e types;

    // choice methods
    pdcp_dupl_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pdcp-DuplicationConfig-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pdcp-DuplicationConfig-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool             ext                   = false;
  bool             discard_timer_present = false;
  bool             rlc_am_present        = false;
  bool             rlc_um_present        = false;
  discard_timer_e_ discard_timer;
  rlc_am_s_        rlc_am;
  rlc_um_s_        rlc_um;
  hdr_compress_c_  hdr_compress;
  // ...
  // group 0
  bool rn_integrity_protection_r10_present = false;
  // group 1
  bool pdcp_sn_size_v1130_present = false;
  // group 2
  bool                ul_data_split_drb_via_scg_r12_present = false;
  bool                t_reordering_r12_present              = false;
  bool                ul_data_split_drb_via_scg_r12         = false;
  t_reordering_r12_e_ t_reordering_r12;
  // group 3
  bool                                 pdcp_sn_size_v1310_present = false;
  copy_ptr<ul_data_split_thres_r13_c_> ul_data_split_thres_r13;
  copy_ptr<status_feedback_r13_c_>     status_feedback_r13;
  // group 4
  copy_ptr<ul_lwa_cfg_r14_c_>           ul_lwa_cfg_r14;
  copy_ptr<ul_only_hdr_compress_r14_c_> ul_only_hdr_compress_r14;
  // group 5
  copy_ptr<ul_data_compress_r15_s_> ul_data_compress_r15;
  copy_ptr<pdcp_dupl_cfg_r15_c_>    pdcp_dupl_cfg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1530 ::= SEQUENCE
struct pucch_cfg_ded_v1530_s {
  struct codebooksize_determination_stti_r15_opts {
    enum options { dai, cc, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<codebooksize_determination_stti_r15_opts> codebooksize_determination_stti_r15_e_;

  // member variables
  bool                                   n1_pucch_an_spt_r15_present                 = false;
  bool                                   codebooksize_determination_stti_r15_present = false;
  uint16_t                               n1_pucch_an_spt_r15                         = 0;
  codebooksize_determination_stti_r15_e_ codebooksize_determination_stti_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-Format3-Conf-r13 ::= SEQUENCE
struct pucch_format3_conf_r13_s {
  using n3_pucch_an_list_r13_l_ = bounded_array<uint16_t, 4>;
  struct two_ant_port_activ_pucch_format3_r13_c_ {
    struct setup_s_ {
      using n3_pucch_an_list_p1_r13_l_ = bounded_array<uint16_t, 4>;

      // member variables
      n3_pucch_an_list_p1_r13_l_ n3_pucch_an_list_p1_r13;
    };
    typedef setup_e types;

    // choice methods
    two_ant_port_activ_pucch_format3_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivatedPUCCH-Format3-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivatedPUCCH-Format3-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                                    n3_pucch_an_list_r13_present                 = false;
  bool                                    two_ant_port_activ_pucch_format3_r13_present = false;
  n3_pucch_an_list_r13_l_                 n3_pucch_an_list_r13;
  two_ant_port_activ_pucch_format3_r13_c_ two_ant_port_activ_pucch_format3_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RE-MappingQCLConfigToAddModList-r11 ::= SEQUENCE (SIZE (1..4)) OF PDSCH-RE-MappingQCL-Config-r11
using re_map_qcl_cfg_to_add_mod_list_r11_l = dyn_array<pdsch_re_map_qcl_cfg_r11_s>;

// RE-MappingQCLConfigToReleaseList-r11 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..4)
using re_map_qcl_cfg_to_release_list_r11_l = bounded_array<uint8_t, 4>;

// RLC-BearerConfig-r15 ::= CHOICE
struct rlc_bearer_cfg_r15_c {
  struct setup_s_ {
    struct lc_ch_id_cfg_r15_c_ {
      struct types_opts {
        enum options { lc_ch_id_r15, lc_ch_id_ext_r15, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      lc_ch_id_cfg_r15_c_() = default;
      lc_ch_id_cfg_r15_c_(const lc_ch_id_cfg_r15_c_& other);
      lc_ch_id_cfg_r15_c_& operator=(const lc_ch_id_cfg_r15_c_& other);
      ~lc_ch_id_cfg_r15_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& lc_ch_id_r15()
      {
        assert_choice_type("logicalChannelIdentity-r15", type_.to_string(), "logicalChannelIdentityConfig-r15");
        return c.get<uint8_t>();
      }
      uint8_t& lc_ch_id_ext_r15()
      {
        assert_choice_type("logicalChannelIdentityExt-r15", type_.to_string(), "logicalChannelIdentityConfig-r15");
        return c.get<uint8_t>();
      }
      const uint8_t& lc_ch_id_r15() const
      {
        assert_choice_type("logicalChannelIdentity-r15", type_.to_string(), "logicalChannelIdentityConfig-r15");
        return c.get<uint8_t>();
      }
      const uint8_t& lc_ch_id_ext_r15() const
      {
        assert_choice_type("logicalChannelIdentityExt-r15", type_.to_string(), "logicalChannelIdentityConfig-r15");
        return c.get<uint8_t>();
      }
      uint8_t& set_lc_ch_id_r15()
      {
        set(types::lc_ch_id_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_lc_ch_id_ext_r15()
      {
        set(types::lc_ch_id_ext_r15);
        return c.get<uint8_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };

    // member variables
    bool                rlc_cfg_r15_present   = false;
    bool                lc_ch_cfg_r15_present = false;
    rlc_cfg_r15_s       rlc_cfg_r15;
    lc_ch_id_cfg_r15_c_ lc_ch_id_cfg_r15;
    lc_ch_cfg_s         lc_ch_cfg_r15;
  };
  typedef setup_e types;

  // choice methods
  rlc_bearer_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLC-BearerConfig-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLC-BearerConfig-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RLC-Config ::= CHOICE
struct rlc_cfg_c {
  struct am_s_ {
    ul_am_rlc_s ul_am_rlc;
    dl_am_rlc_s dl_am_rlc;
  };
  struct um_bi_dir_s_ {
    ul_um_rlc_s ul_um_rlc;
    dl_um_rlc_s dl_um_rlc;
  };
  struct um_uni_dir_ul_s_ {
    ul_um_rlc_s ul_um_rlc;
  };
  struct um_uni_dir_dl_s_ {
    dl_um_rlc_s dl_um_rlc;
  };
  struct types_opts {
    enum options { am, um_bi_dir, um_uni_dir_ul, um_uni_dir_dl, /*...*/ nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts, true> types;

  // choice methods
  rlc_cfg_c() = default;
  rlc_cfg_c(const rlc_cfg_c& other);
  rlc_cfg_c& operator=(const rlc_cfg_c& other);
  ~rlc_cfg_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  am_s_& am()
  {
    assert_choice_type("am", type_.to_string(), "RLC-Config");
    return c.get<am_s_>();
  }
  um_bi_dir_s_& um_bi_dir()
  {
    assert_choice_type("um-Bi-Directional", type_.to_string(), "RLC-Config");
    return c.get<um_bi_dir_s_>();
  }
  um_uni_dir_ul_s_& um_uni_dir_ul()
  {
    assert_choice_type("um-Uni-Directional-UL", type_.to_string(), "RLC-Config");
    return c.get<um_uni_dir_ul_s_>();
  }
  um_uni_dir_dl_s_& um_uni_dir_dl()
  {
    assert_choice_type("um-Uni-Directional-DL", type_.to_string(), "RLC-Config");
    return c.get<um_uni_dir_dl_s_>();
  }
  const am_s_& am() const
  {
    assert_choice_type("am", type_.to_string(), "RLC-Config");
    return c.get<am_s_>();
  }
  const um_bi_dir_s_& um_bi_dir() const
  {
    assert_choice_type("um-Bi-Directional", type_.to_string(), "RLC-Config");
    return c.get<um_bi_dir_s_>();
  }
  const um_uni_dir_ul_s_& um_uni_dir_ul() const
  {
    assert_choice_type("um-Uni-Directional-UL", type_.to_string(), "RLC-Config");
    return c.get<um_uni_dir_ul_s_>();
  }
  const um_uni_dir_dl_s_& um_uni_dir_dl() const
  {
    assert_choice_type("um-Uni-Directional-DL", type_.to_string(), "RLC-Config");
    return c.get<um_uni_dir_dl_s_>();
  }
  am_s_& set_am()
  {
    set(types::am);
    return c.get<am_s_>();
  }
  um_bi_dir_s_& set_um_bi_dir()
  {
    set(types::um_bi_dir);
    return c.get<um_bi_dir_s_>();
  }
  um_uni_dir_ul_s_& set_um_uni_dir_ul()
  {
    set(types::um_uni_dir_ul);
    return c.get<um_uni_dir_ul_s_>();
  }
  um_uni_dir_dl_s_& set_um_uni_dir_dl()
  {
    set(types::um_uni_dir_dl);
    return c.get<um_uni_dir_dl_s_>();
  }

private:
  types                                                                    type_;
  choice_buffer_t<am_s_, um_bi_dir_s_, um_uni_dir_dl_s_, um_uni_dir_ul_s_> c;

  void destroy_();
};

// RLC-Config-v1250 ::= SEQUENCE
struct rlc_cfg_v1250_s {
  bool ul_extended_rlc_li_field_r12 = false;
  bool dl_extended_rlc_li_field_r12 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RLC-Config-v1310 ::= SEQUENCE
struct rlc_cfg_v1310_s {
  bool             poll_pdu_v1310_present    = false;
  bool             ul_extended_rlc_am_sn_r13 = false;
  bool             dl_extended_rlc_am_sn_r13 = false;
  poll_pdu_v1310_e poll_pdu_v1310;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RLC-Config-v1430 ::= CHOICE
struct rlc_cfg_v1430_c {
  struct setup_s_ {
    poll_byte_r14_e poll_byte_r14;
  };
  typedef setup_e types;

  // choice methods
  rlc_cfg_v1430_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLC-Config-v1430");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLC-Config-v1430");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RLC-Config-v1510 ::= SEQUENCE
struct rlc_cfg_v1510_s {
  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RLC-Config-v1530 ::= CHOICE
struct rlc_cfg_v1530_c {
  struct setup_s_ {
  };
  typedef setup_e types;

  // choice methods
  rlc_cfg_v1530_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLC-Config-v1530");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLC-Config-v1530");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReject-v1320-IEs ::= SEQUENCE
struct rrc_conn_reject_v1320_ies_s {
  bool rrc_suspend_ind_r13_present = false;
  bool non_crit_ext_present        = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPDCCH-Config-r15 ::= CHOICE
struct spdcch_cfg_r15_c {
  struct setup_s_ {
    struct spdcch_l1_reuse_ind_r15_opts {
      enum options { n0, n1, n2, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<spdcch_l1_reuse_ind_r15_opts> spdcch_l1_reuse_ind_r15_e_;

    // member variables
    bool                       spdcch_l1_reuse_ind_r15_present = false;
    bool                       spdcch_set_cfg_r15_present      = false;
    spdcch_l1_reuse_ind_r15_e_ spdcch_l1_reuse_ind_r15;
    spdcch_set_r15_l           spdcch_set_cfg_r15;
  };
  typedef setup_e types;

  // choice methods
  spdcch_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPDCCH-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPDCCH-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SPS-ConfigSL-r14 ::= SEQUENCE
struct sps_cfg_sl_r14_s {
  struct semi_persist_sched_interv_sl_r14_opts {
    enum options {
      sf20,
      sf50,
      sf100,
      sf200,
      sf300,
      sf400,
      sf500,
      sf600,
      sf700,
      sf800,
      sf900,
      sf1000,
      spare4,
      spare3,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<semi_persist_sched_interv_sl_r14_opts> semi_persist_sched_interv_sl_r14_e_;

  // member variables
  uint8_t                             sps_cfg_idx_r14 = 1;
  semi_persist_sched_interv_sl_r14_e_ semi_persist_sched_interv_sl_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPS-ConfigUL ::= CHOICE
struct sps_cfg_ul_c {
  struct setup_s_ {
    struct semi_persist_sched_interv_ul_opts {
      enum options {
        sf10,
        sf20,
        sf32,
        sf40,
        sf64,
        sf80,
        sf128,
        sf160,
        sf320,
        sf640,
        sf1_v1430,
        sf2_v1430,
        sf3_v1430,
        sf4_v1430,
        sf5_v1430,
        spare1,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<semi_persist_sched_interv_ul_opts> semi_persist_sched_interv_ul_e_;
    struct implicit_release_after_opts {
      enum options { e2, e3, e4, e8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<implicit_release_after_opts> implicit_release_after_e_;
    struct p0_persistent_s_ {
      int8_t p0_nominal_pusch_persistent = -126;
      int8_t p0_ue_pusch_persistent      = -8;
    };
    struct p0_persistent_sf_set2_r12_c_ {
      struct setup_s_ {
        int8_t p0_nominal_pusch_persistent_sf_set2_r12 = -126;
        int8_t p0_ue_pusch_persistent_sf_set2_r12      = -8;
      };
      typedef setup_e types;

      // choice methods
      p0_persistent_sf_set2_r12_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "p0-PersistentSubframeSet2-r12");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "p0-PersistentSubframeSet2-r12");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };
    struct semi_persist_sched_interv_ul_v1430_opts {
      enum options {
        sf50,
        sf100,
        sf200,
        sf300,
        sf400,
        sf500,
        sf600,
        sf700,
        sf800,
        sf900,
        sf1000,
        spare5,
        spare4,
        spare3,
        spare2,
        spare1,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<semi_persist_sched_interv_ul_v1430_opts> semi_persist_sched_interv_ul_v1430_e_;
    struct cyclic_shift_sps_r15_opts {
      enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cyclic_shift_sps_r15_opts> cyclic_shift_sps_r15_e_;
    struct rv_sps_ul_repeats_r15_opts {
      enum options { ulrvseq1, ulrvseq2, ulrvseq3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<rv_sps_ul_repeats_r15_opts> rv_sps_ul_repeats_r15_e_;
    struct total_num_pusch_sps_ul_repeats_r15_opts {
      enum options { n2, n3, n4, n6, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<total_num_pusch_sps_ul_repeats_r15_opts> total_num_pusch_sps_ul_repeats_r15_e_;

    // member variables
    bool                            ext                     = false;
    bool                            p0_persistent_present   = false;
    bool                            two_intervs_cfg_present = false;
    semi_persist_sched_interv_ul_e_ semi_persist_sched_interv_ul;
    implicit_release_after_e_       implicit_release_after;
    p0_persistent_s_                p0_persistent;
    // ...
    // group 0
    copy_ptr<p0_persistent_sf_set2_r12_c_> p0_persistent_sf_set2_r12;
    // group 1
    bool    nof_conf_ul_sps_processes_r13_present = false;
    uint8_t nof_conf_ul_sps_processes_r13         = 1;
    // group 2
    bool                                  fixed_rv_non_adaptive_r14_present          = false;
    bool                                  sps_cfg_idx_r14_present                    = false;
    bool                                  semi_persist_sched_interv_ul_v1430_present = false;
    uint8_t                               sps_cfg_idx_r14                            = 1;
    semi_persist_sched_interv_ul_v1430_e_ semi_persist_sched_interv_ul_v1430;
    // group 3
    bool                                  cyclic_shift_sps_r15_present               = false;
    bool                                  harq_proc_id_offset_r15_present            = false;
    bool                                  rv_sps_ul_repeats_r15_present              = false;
    bool                                  total_num_pusch_sps_ul_repeats_r15_present = false;
    bool                                  sps_cfg_idx_r15_present                    = false;
    cyclic_shift_sps_r15_e_               cyclic_shift_sps_r15;
    uint8_t                               harq_proc_id_offset_r15 = 0;
    rv_sps_ul_repeats_r15_e_              rv_sps_ul_repeats_r15;
    copy_ptr<tpc_pdcch_cfg_c>             tpc_pdcch_cfg_pusch_sps_r15;
    total_num_pusch_sps_ul_repeats_r15_e_ total_num_pusch_sps_ul_repeats_r15;
    uint8_t                               sps_cfg_idx_r15 = 1;

    // sequence methods
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
  };
  typedef setup_e types;

  // choice methods
  sps_cfg_ul_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigUL");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigUL");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SPS-ConfigUL-STTI-r15 ::= CHOICE
struct sps_cfg_ul_stti_r15_c {
  struct setup_s_ {
    struct semi_persist_sched_interv_ul_stti_r15_opts {
      enum options {
        stti1,
        stti2,
        stti3,
        stti4,
        stti6,
        stti8,
        stti12,
        stti16,
        stti20,
        stti40,
        stti60,
        stti80,
        stti120,
        stti240,
        spare2,
        spare1,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<semi_persist_sched_interv_ul_stti_r15_opts> semi_persist_sched_interv_ul_stti_r15_e_;
    struct implicit_release_after_opts {
      enum options { e2, e3, e4, e8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<implicit_release_after_opts> implicit_release_after_e_;
    struct p0_persistent_r15_s_ {
      int8_t p0_nominal_spusch_persistent_r15 = -126;
      int8_t p0_ue_spusch_persistent_r15      = -8;
    };
    struct p0_persistent_sf_set2_r15_c_ {
      struct setup_s_ {
        int8_t p0_nominal_spusch_persistent_sf_set2_r15 = -126;
        int8_t p0_ue_spusch_persistent_sf_set2_r15      = -8;
      };
      typedef setup_e types;

      // choice methods
      p0_persistent_sf_set2_r15_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "p0-PersistentSubframeSet2-r15");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "p0-PersistentSubframeSet2-r15");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };
    struct cyclic_shift_sps_s_tti_r15_opts {
      enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cyclic_shift_sps_s_tti_r15_opts> cyclic_shift_sps_s_tti_r15_e_;
    struct rv_sps_stti_ul_repeats_r15_opts {
      enum options { ulrvseq1, ulrvseq2, ulrvseq3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<rv_sps_stti_ul_repeats_r15_opts> rv_sps_stti_ul_repeats_r15_e_;
    struct tbs_scaling_factor_subslot_sps_ul_repeats_r15_opts {
      enum options { n6, n12, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tbs_scaling_factor_subslot_sps_ul_repeats_r15_opts>
        tbs_scaling_factor_subslot_sps_ul_repeats_r15_e_;
    struct total_num_pusch_sps_stti_ul_repeats_r15_opts {
      enum options { n2, n3, n4, n6, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<total_num_pusch_sps_stti_ul_repeats_r15_opts> total_num_pusch_sps_stti_ul_repeats_r15_e_;

    // member variables
    bool                                             ext                                                   = false;
    bool                                             p0_persistent_r15_present                             = false;
    bool                                             two_intervs_cfg_r15_present                           = false;
    bool                                             p0_persistent_sf_set2_r15_present                     = false;
    bool                                             nof_conf_ul_sps_processes_stti_r15_present            = false;
    bool                                             tpc_pdcch_cfg_pusch_sps_r15_present                   = false;
    bool                                             cyclic_shift_sps_s_tti_r15_present                    = false;
    bool                                             ifdma_cfg_sps_r15_present                             = false;
    bool                                             harq_proc_id_offset_r15_present                       = false;
    bool                                             rv_sps_stti_ul_repeats_r15_present                    = false;
    bool                                             sps_cfg_idx_r15_present                               = false;
    bool                                             tbs_scaling_factor_subslot_sps_ul_repeats_r15_present = false;
    bool                                             total_num_pusch_sps_stti_ul_repeats_r15_present       = false;
    semi_persist_sched_interv_ul_stti_r15_e_         semi_persist_sched_interv_ul_stti_r15;
    implicit_release_after_e_                        implicit_release_after;
    p0_persistent_r15_s_                             p0_persistent_r15;
    p0_persistent_sf_set2_r15_c_                     p0_persistent_sf_set2_r15;
    uint8_t                                          nof_conf_ul_sps_processes_stti_r15 = 1;
    uint8_t                                          stti_start_time_ul_r15             = 0;
    tpc_pdcch_cfg_c                                  tpc_pdcch_cfg_pusch_sps_r15;
    cyclic_shift_sps_s_tti_r15_e_                    cyclic_shift_sps_s_tti_r15;
    bool                                             ifdma_cfg_sps_r15       = false;
    uint8_t                                          harq_proc_id_offset_r15 = 0;
    rv_sps_stti_ul_repeats_r15_e_                    rv_sps_stti_ul_repeats_r15;
    uint8_t                                          sps_cfg_idx_r15 = 1;
    tbs_scaling_factor_subslot_sps_ul_repeats_r15_e_ tbs_scaling_factor_subslot_sps_ul_repeats_r15;
    total_num_pusch_sps_stti_ul_repeats_r15_e_       total_num_pusch_sps_stti_ul_repeats_r15;
    // ...
  };
  typedef setup_e types;

  // choice methods
  sps_cfg_ul_stti_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigUL-STTI-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigUL-STTI-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SPUCCH-Config-r15 ::= CHOICE
struct spucch_cfg_r15_c {
  struct setup_s_ {
    struct two_ant_port_activ_spucch_format3_r15_s_ {
      using n3_spucch_an_list_r15_l_ = bounded_array<uint16_t, 4>;

      // member variables
      n3_spucch_an_list_r15_l_ n3_spucch_an_list_r15;
    };

    // member variables
    bool                                     spucch_set_r15_present                           = false;
    bool                                     two_ant_port_activ_spucch_format1a1b_r15_present = false;
    spucch_set_r15_l                         spucch_set_r15;
    two_ant_port_activ_spucch_format3_r15_s_ two_ant_port_activ_spucch_format3_r15;
  };
  typedef setup_e types;

  // choice methods
  spucch_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPUCCH-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPUCCH-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SRS-CC-SetIndex-r14 ::= SEQUENCE
struct srs_cc_set_idx_r14_s {
  uint8_t cc_set_idx_r14           = 0;
  uint8_t cc_idx_in_one_cc_set_r14 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SRS-ConfigAp-r10 ::= SEQUENCE
struct srs_cfg_ap_r10_s {
  struct srs_bw_ap_r10_opts {
    enum options { bw0, bw1, bw2, bw3, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<srs_bw_ap_r10_opts> srs_bw_ap_r10_e_;
  struct cyclic_shift_ap_r10_opts {
    enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<cyclic_shift_ap_r10_opts> cyclic_shift_ap_r10_e_;

  // member variables
  srs_ant_port_e         srs_ant_port_ap_r10;
  srs_bw_ap_r10_e_       srs_bw_ap_r10;
  uint8_t                freq_domain_position_ap_r10 = 0;
  uint8_t                tx_comb_ap_r10              = 0;
  cyclic_shift_ap_r10_e_ cyclic_shift_ap_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SRS-ConfigAp-r13 ::= SEQUENCE
struct srs_cfg_ap_r13_s {
  struct srs_bw_ap_r13_opts {
    enum options { bw0, bw1, bw2, bw3, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<srs_bw_ap_r13_opts> srs_bw_ap_r13_e_;
  struct cyclic_shift_ap_r13_opts {
    enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, cs8, cs9, cs10, cs11, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<cyclic_shift_ap_r13_opts> cyclic_shift_ap_r13_e_;
  struct tx_comb_num_r13_opts {
    enum options { n2, n4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_comb_num_r13_opts> tx_comb_num_r13_e_;

  // member variables
  srs_ant_port_e         srs_ant_port_ap_r13;
  srs_bw_ap_r13_e_       srs_bw_ap_r13;
  uint8_t                freq_domain_position_ap_r13 = 0;
  uint8_t                tx_comb_ap_r13              = 0;
  cyclic_shift_ap_r13_e_ cyclic_shift_ap_r13;
  tx_comb_num_r13_e_     tx_comb_num_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SRS-ConfigAp-v1310 ::= SEQUENCE
struct srs_cfg_ap_v1310_s {
  struct cyclic_shift_ap_v1310_opts {
    enum options { cs8, cs9, cs10, cs11, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<cyclic_shift_ap_v1310_opts> cyclic_shift_ap_v1310_e_;
  struct tx_comb_num_r13_opts {
    enum options { n2, n4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_comb_num_r13_opts> tx_comb_num_r13_e_;

  // member variables
  bool                     tx_comb_ap_v1310_present      = false;
  bool                     cyclic_shift_ap_v1310_present = false;
  bool                     tx_comb_num_r13_present       = false;
  uint8_t                  tx_comb_ap_v1310              = 2;
  cyclic_shift_ap_v1310_e_ cyclic_shift_ap_v1310;
  tx_comb_num_r13_e_       tx_comb_num_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// STAG-ToAddMod-r11 ::= SEQUENCE
struct stag_to_add_mod_r11_s {
  bool               ext         = false;
  uint8_t            stag_id_r11 = 1;
  time_align_timer_e time_align_timer_stag_r11;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SchedulingRequestConfig-v1530 ::= CHOICE
struct sched_request_cfg_v1530_c {
  struct setup_s_ {
    struct dssr_trans_max_r15_opts {
      enum options { n4, n8, n16, n32, n64, spare3, spare2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<dssr_trans_max_r15_opts> dssr_trans_max_r15_e_;

    // member variables
    bool                             sr_slot_spucch_idx_fh_r15_present      = false;
    bool                             sr_slot_spucch_idx_no_fh_r15_present   = false;
    bool                             sr_subslot_spucch_res_list_r15_present = false;
    bool                             sr_cfg_idx_slot_r15_present            = false;
    bool                             sr_cfg_idx_subslot_r15_present         = false;
    uint16_t                         sr_slot_spucch_idx_fh_r15              = 0;
    uint16_t                         sr_slot_spucch_idx_no_fh_r15           = 0;
    sr_subslot_spucch_res_list_r15_l sr_subslot_spucch_res_list_r15;
    uint8_t                          sr_cfg_idx_slot_r15    = 0;
    uint8_t                          sr_cfg_idx_subslot_r15 = 0;
    dssr_trans_max_r15_e_            dssr_trans_max_r15;
  };
  typedef setup_e types;

  // choice methods
  sched_request_cfg_v1530_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfig-v1530");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfig-v1530");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// ShortTTI-r15 ::= SEQUENCE
struct short_tti_r15_s {
  bool                dl_stti_len_r15_present = false;
  bool                ul_stti_len_r15_present = false;
  short_tti_len_r15_e dl_stti_len_r15;
  short_tti_len_r15_e ul_stti_len_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SlotOrSubslotPDSCH-Config-r15 ::= CHOICE
struct slot_or_subslot_pdsch_cfg_r15_c {
  struct setup_s_ {
    struct alt_cqi_table_stti_r15_opts {
      enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<alt_cqi_table_stti_r15_opts> alt_cqi_table_stti_r15_e_;
    struct alt_cqi_table1024_qam_stti_r15_opts {
      enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<alt_cqi_table1024_qam_stti_r15_opts> alt_cqi_table1024_qam_stti_r15_e_;
    struct res_alloc_r15_opts {
      enum options { res_alloc_type0, res_alloc_type2, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<res_alloc_r15_opts> res_alloc_r15_e_;

    // member variables
    bool                              ext                                    = false;
    bool                              alt_cqi_table_stti_r15_present         = false;
    bool                              alt_cqi_table1024_qam_stti_r15_present = false;
    bool                              res_alloc_r15_present                  = false;
    bool                              tbs_idx_alt_stti_r15_present           = false;
    bool                              tbs_idx_alt2_stti_r15_present          = false;
    bool                              tbs_idx_alt3_stti_r15_present          = false;
    alt_cqi_table_stti_r15_e_         alt_cqi_table_stti_r15;
    alt_cqi_table1024_qam_stti_r15_e_ alt_cqi_table1024_qam_stti_r15;
    res_alloc_r15_e_                  res_alloc_r15;
    // ...
  };
  typedef setup_e types;

  // choice methods
  slot_or_subslot_pdsch_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SlotOrSubslotPDSCH-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SlotOrSubslotPDSCH-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SlotOrSubslotPUSCH-Config-r15 ::= CHOICE
struct slot_or_subslot_pusch_cfg_r15_c {
  struct setup_s_ {
    using beta_offset_subslot_ack_idx_r15_l_  = bounded_array<uint8_t, 2>;
    using beta_offset2_subslot_ack_idx_r15_l_ = bounded_array<uint8_t, 2>;
    using beta_offset_subslot_ri_idx_r15_l_   = bounded_array<uint8_t, 2>;

    // member variables
    bool                                ext                                       = false;
    bool                                beta_offset_slot_ack_idx_r15_present      = false;
    bool                                beta_offset2_slot_ack_idx_r15_present     = false;
    bool                                beta_offset_subslot_ack_idx_r15_present   = false;
    bool                                beta_offset2_subslot_ack_idx_r15_present  = false;
    bool                                beta_offset_slot_ri_idx_r15_present       = false;
    bool                                beta_offset_subslot_ri_idx_r15_present    = false;
    bool                                beta_offset_slot_cqi_idx_r15_present      = false;
    bool                                beta_offset_subslot_cqi_idx_r15_present   = false;
    bool                                enable256_qam_slot_or_subslot_r15_present = false;
    bool                                res_alloc_offset_r15_present              = false;
    uint8_t                             beta_offset_slot_ack_idx_r15              = 0;
    uint8_t                             beta_offset2_slot_ack_idx_r15             = 0;
    beta_offset_subslot_ack_idx_r15_l_  beta_offset_subslot_ack_idx_r15;
    beta_offset2_subslot_ack_idx_r15_l_ beta_offset2_subslot_ack_idx_r15;
    uint8_t                             beta_offset_slot_ri_idx_r15 = 0;
    beta_offset_subslot_ri_idx_r15_l_   beta_offset_subslot_ri_idx_r15;
    uint8_t                             beta_offset_slot_cqi_idx_r15    = 0;
    uint8_t                             beta_offset_subslot_cqi_idx_r15 = 0;
    enable256_qam_r14_c                 enable256_qam_slot_or_subslot_r15;
    uint8_t                             res_alloc_offset_r15              = 1;
    bool                                ul_dmrs_ifdma_slot_or_subslot_r15 = false;
    // ...
  };
  typedef setup_e types;

  // choice methods
  slot_or_subslot_pusch_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SlotOrSubslotPUSCH-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SlotOrSubslotPUSCH-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// TDD-PUSCH-UpPTS-r14 ::= CHOICE
struct tdd_pusch_up_pts_r14_c {
  struct setup_s_ {
    struct sym_pusch_up_pts_r14_opts {
      enum options { sym1, sym2, sym3, sym4, sym5, sym6, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<sym_pusch_up_pts_r14_opts> sym_pusch_up_pts_r14_e_;

    // member variables
    bool                    sym_pusch_up_pts_r14_present     = false;
    bool                    dmrs_less_up_pts_cfg_r14_present = false;
    sym_pusch_up_pts_r14_e_ sym_pusch_up_pts_r14;
  };
  typedef setup_e types;

  // choice methods
  tdd_pusch_up_pts_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "TDD-PUSCH-UpPTS-r14");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "TDD-PUSCH-UpPTS-r14");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// UplinkPowerControlDedicatedSTTI-r15 ::= SEQUENCE
struct ul_pwr_ctrl_ded_stti_r15_s {
  bool                                delta_tx_d_offset_list_spucch_r15_present = false;
  bool                                accumulation_enabled_stti_r15             = false;
  delta_tx_d_offset_list_spucch_r15_s delta_tx_d_offset_list_spucch_r15;
  bool                                ul_pwr_csi_payload = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated ::= SEQUENCE
struct ant_info_ded_s {
  struct tx_mode_opts {
    enum options { tm1, tm2, tm3, tm4, tm5, tm6, tm7, tm8_v920, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_mode_opts> tx_mode_e_;
  struct codebook_subset_restrict_c_ {
    struct types_opts {
      enum options {
        n2_tx_ant_tm3,
        n4_tx_ant_tm3,
        n2_tx_ant_tm4,
        n4_tx_ant_tm4,
        n2_tx_ant_tm5,
        n4_tx_ant_tm5,
        n2_tx_ant_tm6,
        n4_tx_ant_tm6,
        nulltype
      } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    codebook_subset_restrict_c_() = default;
    codebook_subset_restrict_c_(const codebook_subset_restrict_c_& other);
    codebook_subset_restrict_c_& operator=(const codebook_subset_restrict_c_& other);
    ~codebook_subset_restrict_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<2>& n2_tx_ant_tm3()
    {
      assert_choice_type("n2TxAntenna-tm3", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<2> >();
    }
    fixed_bitstring<4>& n4_tx_ant_tm3()
    {
      assert_choice_type("n4TxAntenna-tm3", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<6>& n2_tx_ant_tm4()
    {
      assert_choice_type("n2TxAntenna-tm4", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<64>& n4_tx_ant_tm4()
    {
      assert_choice_type("n4TxAntenna-tm4", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<64> >();
    }
    fixed_bitstring<4>& n2_tx_ant_tm5()
    {
      assert_choice_type("n2TxAntenna-tm5", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<16>& n4_tx_ant_tm5()
    {
      assert_choice_type("n4TxAntenna-tm5", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<16> >();
    }
    fixed_bitstring<4>& n2_tx_ant_tm6()
    {
      assert_choice_type("n2TxAntenna-tm6", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<16>& n4_tx_ant_tm6()
    {
      assert_choice_type("n4TxAntenna-tm6", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<16> >();
    }
    const fixed_bitstring<2>& n2_tx_ant_tm3() const
    {
      assert_choice_type("n2TxAntenna-tm3", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<2> >();
    }
    const fixed_bitstring<4>& n4_tx_ant_tm3() const
    {
      assert_choice_type("n4TxAntenna-tm3", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    const fixed_bitstring<6>& n2_tx_ant_tm4() const
    {
      assert_choice_type("n2TxAntenna-tm4", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<6> >();
    }
    const fixed_bitstring<64>& n4_tx_ant_tm4() const
    {
      assert_choice_type("n4TxAntenna-tm4", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<64> >();
    }
    const fixed_bitstring<4>& n2_tx_ant_tm5() const
    {
      assert_choice_type("n2TxAntenna-tm5", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    const fixed_bitstring<16>& n4_tx_ant_tm5() const
    {
      assert_choice_type("n4TxAntenna-tm5", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<16> >();
    }
    const fixed_bitstring<4>& n2_tx_ant_tm6() const
    {
      assert_choice_type("n2TxAntenna-tm6", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<4> >();
    }
    const fixed_bitstring<16>& n4_tx_ant_tm6() const
    {
      assert_choice_type("n4TxAntenna-tm6", type_.to_string(), "codebookSubsetRestriction");
      return c.get<fixed_bitstring<16> >();
    }
    fixed_bitstring<2>& set_n2_tx_ant_tm3()
    {
      set(types::n2_tx_ant_tm3);
      return c.get<fixed_bitstring<2> >();
    }
    fixed_bitstring<4>& set_n4_tx_ant_tm3()
    {
      set(types::n4_tx_ant_tm3);
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<6>& set_n2_tx_ant_tm4()
    {
      set(types::n2_tx_ant_tm4);
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<64>& set_n4_tx_ant_tm4()
    {
      set(types::n4_tx_ant_tm4);
      return c.get<fixed_bitstring<64> >();
    }
    fixed_bitstring<4>& set_n2_tx_ant_tm5()
    {
      set(types::n2_tx_ant_tm5);
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<16>& set_n4_tx_ant_tm5()
    {
      set(types::n4_tx_ant_tm5);
      return c.get<fixed_bitstring<16> >();
    }
    fixed_bitstring<4>& set_n2_tx_ant_tm6()
    {
      set(types::n2_tx_ant_tm6);
      return c.get<fixed_bitstring<4> >();
    }
    fixed_bitstring<16>& set_n4_tx_ant_tm6()
    {
      set(types::n4_tx_ant_tm6);
      return c.get<fixed_bitstring<16> >();
    }

  private:
    types                                 type_;
    choice_buffer_t<fixed_bitstring<64> > c;

    void destroy_();
  };
  struct ue_tx_ant_sel_c_ {
    struct setup_opts {
      enum options { closed_loop, open_loop, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    ue_tx_ant_sel_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ue-TransmitAntennaSelection");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ue-TransmitAntennaSelection");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };

  // member variables
  bool                        codebook_subset_restrict_present = false;
  tx_mode_e_                  tx_mode;
  codebook_subset_restrict_c_ codebook_subset_restrict;
  ue_tx_ant_sel_c_            ue_tx_ant_sel;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated-r10 ::= SEQUENCE
struct ant_info_ded_r10_s {
  struct tx_mode_r10_opts {
    enum options {
      tm1,
      tm2,
      tm3,
      tm4,
      tm5,
      tm6,
      tm7,
      tm8_v920,
      tm9_v1020,
      tm10_v1130,
      spare6,
      spare5,
      spare4,
      spare3,
      spare2,
      spare1,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_mode_r10_opts> tx_mode_r10_e_;
  struct ue_tx_ant_sel_c_ {
    struct setup_opts {
      enum options { closed_loop, open_loop, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    ue_tx_ant_sel_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ue-TransmitAntennaSelection");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ue-TransmitAntennaSelection");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };

  // member variables
  bool             codebook_subset_restrict_r10_present = false;
  tx_mode_r10_e_   tx_mode_r10;
  dyn_bitstring    codebook_subset_restrict_r10;
  ue_tx_ant_sel_c_ ue_tx_ant_sel;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated-v1250 ::= SEQUENCE
struct ant_info_ded_v1250_s {
  bool alt_codebook_enabled_for4_tx_r12 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated-v1430 ::= SEQUENCE
struct ant_info_ded_v1430_s {
  bool ce_ue_tx_ant_sel_cfg_r14_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated-v1530 ::= CHOICE
struct ant_info_ded_v1530_c {
  struct setup_c_ {
    struct ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_opts {
      enum options { two, three, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_opts>
        ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_e_;
    struct types_opts {
      enum options {
        ue_tx_ant_sel_srs_minus1_t4_r_cfg_r15,
        ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15,
        nulltype
      } value;
      typedef int8_t number_type;

      std::string to_string() const;
      int8_t      to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_e_& ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15()
    {
      assert_choice_type("ue-TxAntennaSelection-SRS-2T4R-NrOfPairs-r15", type_.to_string(), "setup");
      return c;
    }
    const ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_e_& ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15() const
    {
      assert_choice_type("ue-TxAntennaSelection-SRS-2T4R-NrOfPairs-r15", type_.to_string(), "setup");
      return c;
    }
    ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_e_& set_ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15()
    {
      set(types::ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15);
      return c;
    }

  private:
    types                                            type_;
    ue_tx_ant_sel_srs_minus2_t4_r_nr_of_pairs_r15_e_ c;
  };
  typedef setup_e types;

  // choice methods
  ant_info_ded_v1530_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "AntennaInfoDedicated-v1530");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "AntennaInfoDedicated-v1530");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// AntennaInfoDedicated-v920 ::= SEQUENCE
struct ant_info_ded_v920_s {
  struct codebook_subset_restrict_v920_c_ {
    struct types_opts {
      enum options { n2_tx_ant_tm8_r9, n4_tx_ant_tm8_r9, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    codebook_subset_restrict_v920_c_() = default;
    codebook_subset_restrict_v920_c_(const codebook_subset_restrict_v920_c_& other);
    codebook_subset_restrict_v920_c_& operator=(const codebook_subset_restrict_v920_c_& other);
    ~codebook_subset_restrict_v920_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    fixed_bitstring<6>& n2_tx_ant_tm8_r9()
    {
      assert_choice_type("n2TxAntenna-tm8-r9", type_.to_string(), "codebookSubsetRestriction-v920");
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<32>& n4_tx_ant_tm8_r9()
    {
      assert_choice_type("n4TxAntenna-tm8-r9", type_.to_string(), "codebookSubsetRestriction-v920");
      return c.get<fixed_bitstring<32> >();
    }
    const fixed_bitstring<6>& n2_tx_ant_tm8_r9() const
    {
      assert_choice_type("n2TxAntenna-tm8-r9", type_.to_string(), "codebookSubsetRestriction-v920");
      return c.get<fixed_bitstring<6> >();
    }
    const fixed_bitstring<32>& n4_tx_ant_tm8_r9() const
    {
      assert_choice_type("n4TxAntenna-tm8-r9", type_.to_string(), "codebookSubsetRestriction-v920");
      return c.get<fixed_bitstring<32> >();
    }
    fixed_bitstring<6>& set_n2_tx_ant_tm8_r9()
    {
      set(types::n2_tx_ant_tm8_r9);
      return c.get<fixed_bitstring<6> >();
    }
    fixed_bitstring<32>& set_n4_tx_ant_tm8_r9()
    {
      set(types::n4_tx_ant_tm8_r9);
      return c.get<fixed_bitstring<32> >();
    }

  private:
    types                                 type_;
    choice_buffer_t<fixed_bitstring<32> > c;

    void destroy_();
  };

  // member variables
  bool                             codebook_subset_restrict_v920_present = false;
  codebook_subset_restrict_v920_c_ codebook_subset_restrict_v920;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoUL-r10 ::= SEQUENCE
struct ant_info_ul_r10_s {
  struct tx_mode_ul_r10_opts {
    enum options { tm1, tm2, spare6, spare5, spare4, spare3, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tx_mode_ul_r10_opts> tx_mode_ul_r10_e_;

  // member variables
  bool              tx_mode_ul_r10_present          = false;
  bool              four_ant_port_activ_r10_present = false;
  tx_mode_ul_r10_e_ tx_mode_ul_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandClassPriority1XRTT ::= SEQUENCE
struct band_class_prio1_xrtt_s {
  bandclass_cdma2000_e band_class;
  uint8_t              cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandClassPriorityHRPD ::= SEQUENCE
struct band_class_prio_hrpd_s {
  bandclass_cdma2000_e band_class;
  uint8_t              cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CFI-Config-r15 ::= SEQUENCE
struct cfi_cfg_r15_s {
  bool    cfi_sf_non_mbsfn_r15_present           = false;
  bool    cfi_slot_subslot_non_mbsfn_r15_present = false;
  bool    cfi_sf_mbsfn_r15_present               = false;
  bool    cfi_slot_subslot_mbsfn_r15_present     = false;
  uint8_t cfi_sf_non_mbsfn_r15                   = 1;
  uint8_t cfi_slot_subslot_non_mbsfn_r15         = 1;
  uint8_t cfi_sf_mbsfn_r15                       = 1;
  uint8_t cfi_slot_subslot_mbsfn_r15             = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CFI-PatternConfig-r15 ::= SEQUENCE
struct cfi_pattern_cfg_r15_s {
  using cfi_pattern_sf_r15_l_           = std::array<uint8_t, 10>;
  using cfi_pattern_slot_subslot_r15_l_ = std::array<uint8_t, 10>;

  // member variables
  bool                            cfi_pattern_sf_r15_present           = false;
  bool                            cfi_pattern_slot_subslot_r15_present = false;
  cfi_pattern_sf_r15_l_           cfi_pattern_sf_r15;
  cfi_pattern_slot_subslot_r15_l_ cfi_pattern_slot_subslot_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig ::= SEQUENCE
struct cqi_report_cfg_s {
  bool                        cqi_report_mode_aperiodic_present = false;
  bool                        cqi_report_periodic_present       = false;
  cqi_report_mode_aperiodic_e cqi_report_mode_aperiodic;
  int8_t                      nom_pdsch_rs_epre_offset = -1;
  cqi_report_periodic_c       cqi_report_periodic;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v1530 ::= SEQUENCE
struct cqi_report_cfg_v1530_s {
  struct alt_cqi_table_minus1024_qam_r15_opts {
    enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<alt_cqi_table_minus1024_qam_r15_opts> alt_cqi_table_minus1024_qam_r15_e_;

  // member variables
  bool                               alt_cqi_table_minus1024_qam_r15_present = false;
  alt_cqi_table_minus1024_qam_r15_e_ alt_cqi_table_minus1024_qam_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfig-v920 ::= SEQUENCE
struct cqi_report_cfg_v920_s {
  bool cqi_mask_r9_present      = false;
  bool pmi_ri_report_r9_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CRS-AssistanceInfoList-r11 ::= SEQUENCE (SIZE (1..8)) OF CRS-AssistanceInfo-r11
using crs_assist_info_list_r11_l = dyn_array<crs_assist_info_r11_s>;

// CRS-AssistanceInfoList-r13 ::= SEQUENCE (SIZE (1..8)) OF CRS-AssistanceInfo-r13
using crs_assist_info_list_r13_l = dyn_array<crs_assist_info_r13_s>;

// CRS-AssistanceInfoList-r15 ::= SEQUENCE (SIZE (1..8)) OF CRS-AssistanceInfo-r15
using crs_assist_info_list_r15_l = dyn_array<crs_assist_info_r15_s>;

// CSI-RS-Config-v1480 ::= SEQUENCE
struct csi_rs_cfg_v1480_s {
  bool                     emimo_type_v1480_present = false;
  csi_rs_cfg_emimo_v1480_c emimo_type_v1480;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-Config-v1530 ::= SEQUENCE
struct csi_rs_cfg_v1530_s {
  bool                     emimo_type_v1530_present = false;
  csi_rs_cfg_emimo_v1530_c emimo_type_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CSI-RS-ConfigNZPToAddModList-r11 ::= SEQUENCE (SIZE (1..3)) OF CSI-RS-ConfigNZP-r11
using csi_rs_cfg_nzp_to_add_mod_list_r11_l = dyn_array<csi_rs_cfg_nzp_r11_s>;

// CSI-RS-ConfigNZPToAddModListExt-r13 ::= SEQUENCE (SIZE (1..21)) OF CSI-RS-ConfigNZP-r11
using csi_rs_cfg_nzp_to_add_mod_list_ext_r13_l = dyn_array<csi_rs_cfg_nzp_r11_s>;

// CSI-RS-ConfigNZPToReleaseList-r11 ::= SEQUENCE (SIZE (1..3)) OF INTEGER (1..3)
using csi_rs_cfg_nzp_to_release_list_r11_l = bounded_array<uint8_t, 3>;

// CSI-RS-ConfigNZPToReleaseListExt-r13 ::= SEQUENCE (SIZE (1..21)) OF INTEGER (4..24)
using csi_rs_cfg_nzp_to_release_list_ext_r13_l = bounded_array<uint8_t, 21>;

// DRB-ToAddMod ::= SEQUENCE
struct drb_to_add_mod_s {
  struct drb_type_lwip_r13_opts {
    enum options { lwip, lwip_dl_only, lwip_ul_only, eutran, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<drb_type_lwip_r13_opts> drb_type_lwip_r13_e_;
  struct lwa_wlan_ac_r14_opts {
    enum options { ac_bk, ac_be, ac_vi, ac_vo, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<lwa_wlan_ac_r14_opts> lwa_wlan_ac_r14_e_;

  // member variables
  bool        ext                   = false;
  bool        eps_bearer_id_present = false;
  bool        pdcp_cfg_present      = false;
  bool        rlc_cfg_present       = false;
  bool        lc_ch_id_present      = false;
  bool        lc_ch_cfg_present     = false;
  uint8_t     eps_bearer_id         = 0;
  uint8_t     drb_id                = 1;
  pdcp_cfg_s  pdcp_cfg;
  rlc_cfg_c   rlc_cfg;
  uint8_t     lc_ch_id = 3;
  lc_ch_cfg_s lc_ch_cfg;
  // ...
  // group 0
  bool                      drb_type_change_r12_present = false;
  copy_ptr<rlc_cfg_v1250_s> rlc_cfg_v1250;
  // group 1
  bool                      drb_type_lwa_r13_present  = false;
  bool                      drb_type_lwip_r13_present = false;
  copy_ptr<rlc_cfg_v1310_s> rlc_cfg_v1310;
  bool                      drb_type_lwa_r13 = false;
  drb_type_lwip_r13_e_      drb_type_lwip_r13;
  // group 2
  bool                      lwip_ul_aggregation_r14_present = false;
  bool                      lwip_dl_aggregation_r14_present = false;
  bool                      lwa_wlan_ac_r14_present         = false;
  copy_ptr<rlc_cfg_v1430_c> rlc_cfg_v1430;
  bool                      lwip_ul_aggregation_r14 = false;
  bool                      lwip_dl_aggregation_r14 = false;
  lwa_wlan_ac_r14_e_        lwa_wlan_ac_r14;
  // group 3
  copy_ptr<rlc_cfg_v1510_s> rlc_cfg_v1510;
  // group 4
  bool                           lc_ch_id_r15_present = false;
  copy_ptr<rlc_cfg_v1530_c>      rlc_cfg_v1530;
  copy_ptr<rlc_bearer_cfg_r15_c> rlc_bearer_cfg_dupl_r15;
  uint8_t                        lc_ch_id_r15 = 32;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRX-Config ::= CHOICE
struct drx_cfg_c {
  struct setup_s_ {
    struct on_dur_timer_opts {
      enum options {
        psf1,
        psf2,
        psf3,
        psf4,
        psf5,
        psf6,
        psf8,
        psf10,
        psf20,
        psf30,
        psf40,
        psf50,
        psf60,
        psf80,
        psf100,
        psf200,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<on_dur_timer_opts> on_dur_timer_e_;
    struct drx_inactivity_timer_opts {
      enum options {
        psf1,
        psf2,
        psf3,
        psf4,
        psf5,
        psf6,
        psf8,
        psf10,
        psf20,
        psf30,
        psf40,
        psf50,
        psf60,
        psf80,
        psf100,
        psf200,
        psf300,
        psf500,
        psf750,
        psf1280,
        psf1920,
        psf2560,
        psf0_v1020,
        spare9,
        spare8,
        spare7,
        spare6,
        spare5,
        spare4,
        spare3,
        spare2,
        spare1,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<drx_inactivity_timer_opts> drx_inactivity_timer_e_;
    struct drx_retx_timer_opts {
      enum options { psf1, psf2, psf4, psf6, psf8, psf16, psf24, psf33, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<drx_retx_timer_opts> drx_retx_timer_e_;
    struct long_drx_cycle_start_offset_c_ {
      struct types_opts {
        enum options {
          sf10,
          sf20,
          sf32,
          sf40,
          sf64,
          sf80,
          sf128,
          sf160,
          sf256,
          sf320,
          sf512,
          sf640,
          sf1024,
          sf1280,
          sf2048,
          sf2560,
          nulltype
        } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      long_drx_cycle_start_offset_c_() = default;
      long_drx_cycle_start_offset_c_(const long_drx_cycle_start_offset_c_& other);
      long_drx_cycle_start_offset_c_& operator=(const long_drx_cycle_start_offset_c_& other);
      ~long_drx_cycle_start_offset_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& sf10()
      {
        assert_choice_type("sf10", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf20()
      {
        assert_choice_type("sf20", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf32()
      {
        assert_choice_type("sf32", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf40()
      {
        assert_choice_type("sf40", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf64()
      {
        assert_choice_type("sf64", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf80()
      {
        assert_choice_type("sf80", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf128()
      {
        assert_choice_type("sf128", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint8_t& sf160()
      {
        assert_choice_type("sf160", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      uint16_t& sf256()
      {
        assert_choice_type("sf256", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf320()
      {
        assert_choice_type("sf320", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf512()
      {
        assert_choice_type("sf512", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf640()
      {
        assert_choice_type("sf640", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf1024()
      {
        assert_choice_type("sf1024", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf1280()
      {
        assert_choice_type("sf1280", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf2048()
      {
        assert_choice_type("sf2048", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint16_t& sf2560()
      {
        assert_choice_type("sf2560", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint8_t& sf10() const
      {
        assert_choice_type("sf10", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf20() const
      {
        assert_choice_type("sf20", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf32() const
      {
        assert_choice_type("sf32", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf40() const
      {
        assert_choice_type("sf40", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf64() const
      {
        assert_choice_type("sf64", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf80() const
      {
        assert_choice_type("sf80", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf128() const
      {
        assert_choice_type("sf128", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& sf160() const
      {
        assert_choice_type("sf160", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint8_t>();
      }
      const uint16_t& sf256() const
      {
        assert_choice_type("sf256", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf320() const
      {
        assert_choice_type("sf320", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf512() const
      {
        assert_choice_type("sf512", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf640() const
      {
        assert_choice_type("sf640", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf1024() const
      {
        assert_choice_type("sf1024", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf1280() const
      {
        assert_choice_type("sf1280", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf2048() const
      {
        assert_choice_type("sf2048", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& sf2560() const
      {
        assert_choice_type("sf2560", type_.to_string(), "longDRX-CycleStartOffset");
        return c.get<uint16_t>();
      }
      uint8_t& set_sf10()
      {
        set(types::sf10);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf20()
      {
        set(types::sf20);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf32()
      {
        set(types::sf32);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf40()
      {
        set(types::sf40);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf64()
      {
        set(types::sf64);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf80()
      {
        set(types::sf80);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf128()
      {
        set(types::sf128);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf160()
      {
        set(types::sf160);
        return c.get<uint8_t>();
      }
      uint16_t& set_sf256()
      {
        set(types::sf256);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf320()
      {
        set(types::sf320);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf512()
      {
        set(types::sf512);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf640()
      {
        set(types::sf640);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf1024()
      {
        set(types::sf1024);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf1280()
      {
        set(types::sf1280);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf2048()
      {
        set(types::sf2048);
        return c.get<uint16_t>();
      }
      uint16_t& set_sf2560()
      {
        set(types::sf2560);
        return c.get<uint16_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };
    struct short_drx_s_ {
      struct short_drx_cycle_opts {
        enum options {
          sf2,
          sf5,
          sf8,
          sf10,
          sf16,
          sf20,
          sf32,
          sf40,
          sf64,
          sf80,
          sf128,
          sf160,
          sf256,
          sf320,
          sf512,
          sf640,
          nulltype
        } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<short_drx_cycle_opts> short_drx_cycle_e_;

      // member variables
      short_drx_cycle_e_ short_drx_cycle;
      uint8_t            drx_short_cycle_timer = 1;
    };

    // member variables
    bool                           short_drx_present = false;
    on_dur_timer_e_                on_dur_timer;
    drx_inactivity_timer_e_        drx_inactivity_timer;
    drx_retx_timer_e_              drx_retx_timer;
    long_drx_cycle_start_offset_c_ long_drx_cycle_start_offset;
    short_drx_s_                   short_drx;
  };
  typedef setup_e types;

  // choice methods
  drx_cfg_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "DRX-Config");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "DRX-Config");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// DRX-Config-r13 ::= SEQUENCE
struct drx_cfg_r13_s {
  struct on_dur_timer_v1310_opts {
    enum options { psf300, psf400, psf500, psf600, psf800, psf1000, psf1200, psf1600, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<on_dur_timer_v1310_opts> on_dur_timer_v1310_e_;
  struct drx_retx_timer_v1310_opts {
    enum options { psf40, psf64, psf80, psf96, psf112, psf128, psf160, psf320, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<drx_retx_timer_v1310_opts> drx_retx_timer_v1310_e_;
  struct drx_ul_retx_timer_r13_opts {
    enum options {
      psf0,
      psf1,
      psf2,
      psf4,
      psf6,
      psf8,
      psf16,
      psf24,
      psf33,
      psf40,
      psf64,
      psf80,
      psf96,
      psf112,
      psf128,
      psf160,
      psf320,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<drx_ul_retx_timer_r13_opts> drx_ul_retx_timer_r13_e_;

  // member variables
  bool                     on_dur_timer_v1310_present    = false;
  bool                     drx_retx_timer_v1310_present  = false;
  bool                     drx_ul_retx_timer_r13_present = false;
  on_dur_timer_v1310_e_    on_dur_timer_v1310;
  drx_retx_timer_v1310_e_  drx_retx_timer_v1310;
  drx_ul_retx_timer_r13_e_ drx_ul_retx_timer_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRX-Config-r15 ::= SEQUENCE
struct drx_cfg_r15_s {
  struct drx_retx_timer_short_tti_r15_opts {
    enum options { tti10, tti20, tti40, tti64, tti80, tti96, tti112, tti128, tti160, tti320, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<drx_retx_timer_short_tti_r15_opts> drx_retx_timer_short_tti_r15_e_;
  struct drx_ul_retx_timer_short_tti_r15_opts {
    enum options {
      tti0,
      tti1,
      tti2,
      tti4,
      tti6,
      tti8,
      tti16,
      tti24,
      tti33,
      tti40,
      tti64,
      tti80,
      tti96,
      tti112,
      tti128,
      tti160,
      tti320,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<drx_ul_retx_timer_short_tti_r15_opts> drx_ul_retx_timer_short_tti_r15_e_;

  // member variables
  bool                               drx_retx_timer_short_tti_r15_present    = false;
  bool                               drx_ul_retx_timer_short_tti_r15_present = false;
  drx_retx_timer_short_tti_r15_e_    drx_retx_timer_short_tti_r15;
  drx_ul_retx_timer_short_tti_r15_e_ drx_ul_retx_timer_short_tti_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRX-Config-v1130 ::= SEQUENCE
struct drx_cfg_v1130_s {
  struct long_drx_cycle_start_offset_v1130_c_ {
    struct types_opts {
      enum options { sf60_v1130, sf70_v1130, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    long_drx_cycle_start_offset_v1130_c_() = default;
    long_drx_cycle_start_offset_v1130_c_(const long_drx_cycle_start_offset_v1130_c_& other);
    long_drx_cycle_start_offset_v1130_c_& operator=(const long_drx_cycle_start_offset_v1130_c_& other);
    ~long_drx_cycle_start_offset_v1130_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& sf60_v1130()
    {
      assert_choice_type("sf60-v1130", type_.to_string(), "longDRX-CycleStartOffset-v1130");
      return c.get<uint8_t>();
    }
    uint8_t& sf70_v1130()
    {
      assert_choice_type("sf70-v1130", type_.to_string(), "longDRX-CycleStartOffset-v1130");
      return c.get<uint8_t>();
    }
    const uint8_t& sf60_v1130() const
    {
      assert_choice_type("sf60-v1130", type_.to_string(), "longDRX-CycleStartOffset-v1130");
      return c.get<uint8_t>();
    }
    const uint8_t& sf70_v1130() const
    {
      assert_choice_type("sf70-v1130", type_.to_string(), "longDRX-CycleStartOffset-v1130");
      return c.get<uint8_t>();
    }
    uint8_t& set_sf60_v1130()
    {
      set(types::sf60_v1130);
      return c.get<uint8_t>();
    }
    uint8_t& set_sf70_v1130()
    {
      set(types::sf70_v1130);
      return c.get<uint8_t>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };

  // member variables
  bool                                 drx_retx_timer_v1130_present              = false;
  bool                                 long_drx_cycle_start_offset_v1130_present = false;
  bool                                 short_drx_cycle_v1130_present             = false;
  long_drx_cycle_start_offset_v1130_c_ long_drx_cycle_start_offset_v1130;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRX-Config-v1310 ::= SEQUENCE
struct drx_cfg_v1310_s {
  struct long_drx_cycle_start_offset_v1310_s_ {
    uint8_t sf60_v1310 = 0;
  };

  // member variables
  bool                                 long_drx_cycle_start_offset_v1310_present = false;
  long_drx_cycle_start_offset_v1310_s_ long_drx_cycle_start_offset_v1310;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DataInactivityTimer-r14 ::= ENUMERATED
struct data_inactivity_timer_r14_opts {
  enum options { s1, s2, s3, s5, s7, s10, s15, s20, s40, s50, s60, s80, s100, s120, s150, s180, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<data_inactivity_timer_r14_opts> data_inactivity_timer_r14_e;

// EPDCCH-Config-r11 ::= SEQUENCE
struct epdcch_cfg_r11_s {
  struct cfg_r11_c_ {
    struct setup_s_ {
      struct sf_pattern_cfg_r11_c_ {
        struct setup_s_ {
          meas_sf_pattern_r10_c sf_pattern_r11;
        };
        typedef setup_e types;

        // choice methods
        sf_pattern_cfg_r11_c_() = default;
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        setup_s_& setup()
        {
          assert_choice_type("setup", type_.to_string(), "subframePatternConfig-r11");
          return c;
        }
        const setup_s_& setup() const
        {
          assert_choice_type("setup", type_.to_string(), "subframePatternConfig-r11");
          return c;
        }
        setup_s_& set_setup()
        {
          set(types::setup);
          return c;
        }

      private:
        types    type_;
        setup_s_ c;
      };

      // member variables
      bool                                 sf_pattern_cfg_r11_present          = false;
      bool                                 start_symbol_r11_present            = false;
      bool                                 set_cfg_to_release_list_r11_present = false;
      bool                                 set_cfg_to_add_mod_list_r11_present = false;
      sf_pattern_cfg_r11_c_                sf_pattern_cfg_r11;
      uint8_t                              start_symbol_r11 = 1;
      epdcch_set_cfg_to_release_list_r11_l set_cfg_to_release_list_r11;
      epdcch_set_cfg_to_add_mod_list_r11_l set_cfg_to_add_mod_list_r11;
    };
    typedef setup_e types;

    // choice methods
    cfg_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "config-r11");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "config-r11");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  cfg_r11_c_ cfg_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityEUTRA ::= SEQUENCE
struct freq_prio_eutra_s {
  uint32_t carrier_freq    = 0;
  uint8_t  cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityEUTRA-r12 ::= SEQUENCE
struct freq_prio_eutra_r12_s {
  uint32_t carrier_freq_r12    = 0;
  uint8_t  cell_resel_prio_r12 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityEUTRA-v1310 ::= SEQUENCE
struct freq_prio_eutra_v1310_s {
  bool                      cell_resel_sub_prio_r13_present = false;
  cell_resel_sub_prio_r13_e cell_resel_sub_prio_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityNR-r15 ::= SEQUENCE
struct freq_prio_nr_r15_s {
  bool                      cell_resel_sub_prio_r15_present = false;
  uint32_t                  carrier_freq_r15                = 0;
  uint8_t                   cell_resel_prio_r15             = 0;
  cell_resel_sub_prio_r13_e cell_resel_sub_prio_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityUTRA-FDD ::= SEQUENCE
struct freq_prio_utra_fdd_s {
  uint16_t carrier_freq    = 0;
  uint8_t  cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityUTRA-TDD ::= SEQUENCE
struct freq_prio_utra_tdd_s {
  uint16_t carrier_freq    = 0;
  uint8_t  cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqsPriorityGERAN ::= SEQUENCE
struct freqs_prio_geran_s {
  carrier_freqs_geran_s carrier_freqs;
  uint8_t               cell_resel_prio = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// NeighCellsToAddModList-r12 ::= SEQUENCE (SIZE (1..8)) OF NeighCellsInfo-r12
using neigh_cells_to_add_mod_list_r12_l = dyn_array<neigh_cells_info_r12_s>;

// NeighCellsToReleaseList-r12 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (0..503)
using neigh_cells_to_release_list_r12_l = bounded_array<uint16_t, 8>;

// PDCCH-CandidateReductions-r13 ::= CHOICE
struct pdcch_candidate_reductions_r13_c {
  struct setup_s_ {
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al1_r13;
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al2_r13;
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al3_r13;
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al4_r13;
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al5_r13;
  };
  typedef setup_e types;

  // choice methods
  pdcch_candidate_reductions_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "PDCCH-CandidateReductions-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "PDCCH-CandidateReductions-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// PDSCH-ConfigDedicated ::= SEQUENCE
struct pdsch_cfg_ded_s {
  struct p_a_opts {
    enum options { db_minus6, db_minus4dot77, db_minus3, db_minus1dot77, db0, db1, db2, db3, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<p_a_opts> p_a_e_;

  // member variables
  p_a_e_ p_a;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicated-v1130 ::= SEQUENCE
struct pdsch_cfg_ded_v1130_s {
  struct qcl_operation_opts {
    enum options { type_a, type_b, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<qcl_operation_opts> qcl_operation_e_;

  // member variables
  bool                                 dmrs_cfg_pdsch_r11_present                 = false;
  bool                                 qcl_operation_present                      = false;
  bool                                 re_map_qcl_cfg_to_release_list_r11_present = false;
  bool                                 re_map_qcl_cfg_to_add_mod_list_r11_present = false;
  dmrs_cfg_r11_c                       dmrs_cfg_pdsch_r11;
  qcl_operation_e_                     qcl_operation;
  re_map_qcl_cfg_to_release_list_r11_l re_map_qcl_cfg_to_release_list_r11;
  re_map_qcl_cfg_to_add_mod_list_r11_l re_map_qcl_cfg_to_add_mod_list_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicated-v1280 ::= SEQUENCE
struct pdsch_cfg_ded_v1280_s {
  struct tbs_idx_alt_r12_opts {
    enum options { a26, a33, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<tbs_idx_alt_r12_opts> tbs_idx_alt_r12_e_;

  // member variables
  bool               tbs_idx_alt_r12_present = false;
  tbs_idx_alt_r12_e_ tbs_idx_alt_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicated-v1310 ::= SEQUENCE
struct pdsch_cfg_ded_v1310_s {
  bool             dmrs_cfg_pdsch_v1310_present = false;
  dmrs_cfg_v1310_s dmrs_cfg_pdsch_v1310;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicated-v1430 ::= SEQUENCE
struct pdsch_cfg_ded_v1430_s {
  struct ce_pdsch_max_bw_r14_opts {
    enum options { bw5, bw20, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ce_pdsch_max_bw_r14_opts> ce_pdsch_max_bw_r14_e_;
  struct ce_sched_enhancement_r14_opts {
    enum options { range1, range2, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ce_sched_enhancement_r14_opts> ce_sched_enhancement_r14_e_;

  // member variables
  bool                        ce_pdsch_max_bw_r14_present        = false;
  bool                        ce_pdsch_ten_processes_r14_present = false;
  bool                        ce_harq_ack_bundling_r14_present   = false;
  bool                        ce_sched_enhancement_r14_present   = false;
  bool                        tbs_idx_alt2_r14_present           = false;
  ce_pdsch_max_bw_r14_e_      ce_pdsch_max_bw_r14;
  ce_sched_enhancement_r14_e_ ce_sched_enhancement_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicated-v1530 ::= SEQUENCE
struct pdsch_cfg_ded_v1530_s {
  struct alt_mcs_table_scaling_cfg_r15_opts {
    enum options { odot5, odot625, odot75, odot875, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<alt_mcs_table_scaling_cfg_r15_opts> alt_mcs_table_scaling_cfg_r15_e_;

  // member variables
  bool                             qcl_operation_v1530_present                   = false;
  bool                             tbs_idx_alt3_r15_present                      = false;
  bool                             ce_cqi_alt_table_cfg_r15_present              = false;
  bool                             ce_pdsch_minus64_qam_cfg_r15_present          = false;
  bool                             ce_pdsch_flex_start_prb_alloc_cfg_r15_present = false;
  bool                             alt_mcs_table_scaling_cfg_r15_present         = false;
  alt_mcs_table_scaling_cfg_r15_e_ alt_mcs_table_scaling_cfg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated ::= SEQUENCE
struct pucch_cfg_ded_s {
  struct ack_nack_repeat_c_ {
    struct setup_s_ {
      struct repeat_factor_opts {
        enum options { n2, n4, n6, spare1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<repeat_factor_opts> repeat_factor_e_;

      // member variables
      repeat_factor_e_ repeat_factor;
      uint16_t         n1_pucch_an_rep = 0;
    };
    typedef setup_e types;

    // choice methods
    ack_nack_repeat_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ackNackRepetition");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ackNackRepetition");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct tdd_ack_nack_feedback_mode_opts {
    enum options { bundling, mux, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<tdd_ack_nack_feedback_mode_opts> tdd_ack_nack_feedback_mode_e_;

  // member variables
  bool                          tdd_ack_nack_feedback_mode_present = false;
  ack_nack_repeat_c_            ack_nack_repeat;
  tdd_ack_nack_feedback_mode_e_ tdd_ack_nack_feedback_mode;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-r13 ::= SEQUENCE
struct pucch_cfg_ded_r13_s {
  struct ack_nack_repeat_r13_c_ {
    struct setup_s_ {
      struct repeat_factor_r13_opts {
        enum options { n2, n4, n6, spare1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<repeat_factor_r13_opts> repeat_factor_r13_e_;

      // member variables
      repeat_factor_r13_e_ repeat_factor_r13;
      uint16_t             n1_pucch_an_rep_r13 = 0;
    };
    typedef setup_e types;

    // choice methods
    ack_nack_repeat_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ackNackRepetition-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ackNackRepetition-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct tdd_ack_nack_feedback_mode_r13_opts {
    enum options { bundling, mux, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<tdd_ack_nack_feedback_mode_r13_opts> tdd_ack_nack_feedback_mode_r13_e_;
  struct pucch_format_r13_c_ {
    struct format3_r13_s_ {
      using n3_pucch_an_list_r13_l_ = bounded_array<uint16_t, 4>;
      struct two_ant_port_activ_pucch_format3_r13_c_ {
        struct setup_s_ {
          using n3_pucch_an_list_p1_r13_l_ = bounded_array<uint16_t, 4>;

          // member variables
          n3_pucch_an_list_p1_r13_l_ n3_pucch_an_list_p1_r13;
        };
        typedef setup_e types;

        // choice methods
        two_ant_port_activ_pucch_format3_r13_c_() = default;
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        setup_s_& setup()
        {
          assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivatedPUCCH-Format3-r13");
          return c;
        }
        const setup_s_& setup() const
        {
          assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivatedPUCCH-Format3-r13");
          return c;
        }
        setup_s_& set_setup()
        {
          set(types::setup);
          return c;
        }

      private:
        types    type_;
        setup_s_ c;
      };

      // member variables
      bool                                    n3_pucch_an_list_r13_present                 = false;
      bool                                    two_ant_port_activ_pucch_format3_r13_present = false;
      n3_pucch_an_list_r13_l_                 n3_pucch_an_list_r13;
      two_ant_port_activ_pucch_format3_r13_c_ two_ant_port_activ_pucch_format3_r13;
    };
    struct ch_sel_r13_s_ {
      struct n1_pucch_an_cs_r13_c_ {
        struct setup_s_ {
          using n1_pucch_an_cs_list_r13_l_    = dyn_array<n1_pucch_an_cs_r10_l>;
          using n1_pucch_an_cs_list_p1_r13_l_ = bounded_array<uint16_t, 4>;

          // member variables
          n1_pucch_an_cs_list_r13_l_    n1_pucch_an_cs_list_r13;
          n1_pucch_an_cs_list_p1_r13_l_ n1_pucch_an_cs_list_p1_r13;
        };
        typedef setup_e types;

        // choice methods
        n1_pucch_an_cs_r13_c_() = default;
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        setup_s_& setup()
        {
          assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-r13");
          return c;
        }
        const setup_s_& setup() const
        {
          assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-r13");
          return c;
        }
        setup_s_& set_setup()
        {
          set(types::setup);
          return c;
        }

      private:
        types    type_;
        setup_s_ c;
      };

      // member variables
      bool                  n1_pucch_an_cs_r13_present = false;
      n1_pucch_an_cs_r13_c_ n1_pucch_an_cs_r13;
    };
    struct format4_r13_s_ {
      using format4_res_cfg_r13_l_           = std::array<format4_res_r13_s, 4>;
      using format4_multi_csi_res_cfg_r13_l_ = dyn_array<format4_res_r13_s>;

      // member variables
      bool                             format4_multi_csi_res_cfg_r13_present = false;
      format4_res_cfg_r13_l_           format4_res_cfg_r13;
      format4_multi_csi_res_cfg_r13_l_ format4_multi_csi_res_cfg_r13;
    };
    struct format5_r13_s_ {
      using format5_res_cfg_r13_l_ = std::array<format5_res_r13_s, 4>;

      // member variables
      bool                   format5_multi_csi_res_cfg_r13_present = false;
      format5_res_cfg_r13_l_ format5_res_cfg_r13;
      format5_res_r13_s      format5_multi_csi_res_cfg_r13;
    };
    struct types_opts {
      enum options { format3_r13, ch_sel_r13, format4_r13, format5_r13, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    pucch_format_r13_c_() = default;
    pucch_format_r13_c_(const pucch_format_r13_c_& other);
    pucch_format_r13_c_& operator=(const pucch_format_r13_c_& other);
    ~pucch_format_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    format3_r13_s_& format3_r13()
    {
      assert_choice_type("format3-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format3_r13_s_>();
    }
    ch_sel_r13_s_& ch_sel_r13()
    {
      assert_choice_type("channelSelection-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<ch_sel_r13_s_>();
    }
    format4_r13_s_& format4_r13()
    {
      assert_choice_type("format4-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format4_r13_s_>();
    }
    format5_r13_s_& format5_r13()
    {
      assert_choice_type("format5-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format5_r13_s_>();
    }
    const format3_r13_s_& format3_r13() const
    {
      assert_choice_type("format3-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format3_r13_s_>();
    }
    const ch_sel_r13_s_& ch_sel_r13() const
    {
      assert_choice_type("channelSelection-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<ch_sel_r13_s_>();
    }
    const format4_r13_s_& format4_r13() const
    {
      assert_choice_type("format4-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format4_r13_s_>();
    }
    const format5_r13_s_& format5_r13() const
    {
      assert_choice_type("format5-r13", type_.to_string(), "pucch-Format-r13");
      return c.get<format5_r13_s_>();
    }
    format3_r13_s_& set_format3_r13()
    {
      set(types::format3_r13);
      return c.get<format3_r13_s_>();
    }
    ch_sel_r13_s_& set_ch_sel_r13()
    {
      set(types::ch_sel_r13);
      return c.get<ch_sel_r13_s_>();
    }
    format4_r13_s_& set_format4_r13()
    {
      set(types::format4_r13);
      return c.get<format4_r13_s_>();
    }
    format5_r13_s_& set_format5_r13()
    {
      set(types::format5_r13);
      return c.get<format5_r13_s_>();
    }

  private:
    types                                                                          type_;
    choice_buffer_t<ch_sel_r13_s_, format3_r13_s_, format4_r13_s_, format5_r13_s_> c;

    void destroy_();
  };
  struct npucch_param_r13_c_ {
    struct setup_s_ {
      uint16_t npucch_id_r13   = 0;
      uint16_t n1_pucch_an_r13 = 0;
    };
    typedef setup_e types;

    // choice methods
    npucch_param_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "nPUCCH-Param-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "nPUCCH-Param-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct nka_pucch_param_r13_c_ {
    struct setup_s_ {
      uint16_t nka_pucch_an_r13 = 0;
    };
    typedef setup_e types;

    // choice methods
    nka_pucch_param_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "nkaPUCCH-Param-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "nkaPUCCH-Param-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct codebooksize_determination_r13_opts {
    enum options { dai, cc, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<codebooksize_determination_r13_opts> codebooksize_determination_r13_e_;
  struct pucch_num_repeat_ce_r13_c_ {
    struct setup_c_ {
      struct mode_a_s_ {
        struct pucch_num_repeat_ce_format1_r13_opts {
          enum options { r1, r2, r4, r8, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<pucch_num_repeat_ce_format1_r13_opts> pucch_num_repeat_ce_format1_r13_e_;
        struct pucch_num_repeat_ce_format2_r13_opts {
          enum options { r1, r2, r4, r8, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<pucch_num_repeat_ce_format2_r13_opts> pucch_num_repeat_ce_format2_r13_e_;

        // member variables
        pucch_num_repeat_ce_format1_r13_e_ pucch_num_repeat_ce_format1_r13;
        pucch_num_repeat_ce_format2_r13_e_ pucch_num_repeat_ce_format2_r13;
      };
      struct mode_b_s_ {
        struct pucch_num_repeat_ce_format1_r13_opts {
          enum options { r4, r8, r16, r32, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<pucch_num_repeat_ce_format1_r13_opts> pucch_num_repeat_ce_format1_r13_e_;
        struct pucch_num_repeat_ce_format2_r13_opts {
          enum options { r4, r8, r16, r32, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<pucch_num_repeat_ce_format2_r13_opts> pucch_num_repeat_ce_format2_r13_e_;

        // member variables
        pucch_num_repeat_ce_format1_r13_e_ pucch_num_repeat_ce_format1_r13;
        pucch_num_repeat_ce_format2_r13_e_ pucch_num_repeat_ce_format2_r13;
      };
      struct types_opts {
        enum options { mode_a, mode_b, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      mode_a_s_& mode_a()
      {
        assert_choice_type("modeA", type_.to_string(), "setup");
        return c.get<mode_a_s_>();
      }
      mode_b_s_& mode_b()
      {
        assert_choice_type("modeB", type_.to_string(), "setup");
        return c.get<mode_b_s_>();
      }
      const mode_a_s_& mode_a() const
      {
        assert_choice_type("modeA", type_.to_string(), "setup");
        return c.get<mode_a_s_>();
      }
      const mode_b_s_& mode_b() const
      {
        assert_choice_type("modeB", type_.to_string(), "setup");
        return c.get<mode_b_s_>();
      }
      mode_a_s_& set_mode_a()
      {
        set(types::mode_a);
        return c.get<mode_a_s_>();
      }
      mode_b_s_& set_mode_b()
      {
        set(types::mode_b);
        return c.get<mode_b_s_>();
      }

    private:
      types                                 type_;
      choice_buffer_t<mode_a_s_, mode_b_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    pucch_num_repeat_ce_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pucch-NumRepetitionCE-r13");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pucch-NumRepetitionCE-r13");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };

  // member variables
  bool                              tdd_ack_nack_feedback_mode_r13_present          = false;
  bool                              pucch_format_r13_present                        = false;
  bool                              two_ant_port_activ_pucch_format1a1b_r13_present = false;
  bool                              simul_pucch_pusch_r13_present                   = false;
  bool                              n1_pucch_an_rep_p1_r13_present                  = false;
  bool                              npucch_param_r13_present                        = false;
  bool                              nka_pucch_param_r13_present                     = false;
  bool                              codebooksize_determination_r13_present          = false;
  bool                              maximum_payload_coderate_r13_present            = false;
  bool                              pucch_num_repeat_ce_r13_present                 = false;
  ack_nack_repeat_r13_c_            ack_nack_repeat_r13;
  tdd_ack_nack_feedback_mode_r13_e_ tdd_ack_nack_feedback_mode_r13;
  pucch_format_r13_c_               pucch_format_r13;
  uint16_t                          n1_pucch_an_rep_p1_r13 = 0;
  npucch_param_r13_c_               npucch_param_r13;
  nka_pucch_param_r13_c_            nka_pucch_param_r13;
  bool                              spatial_bundling_pucch_r13 = false;
  bool                              spatial_bundling_pusch_r13 = false;
  bool                              harq_timing_tdd_r13        = false;
  codebooksize_determination_r13_e_ codebooksize_determination_r13;
  uint8_t                           maximum_payload_coderate_r13 = 0;
  pucch_num_repeat_ce_r13_c_        pucch_num_repeat_ce_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1020 ::= SEQUENCE
struct pucch_cfg_ded_v1020_s {
  struct pucch_format_r10_c_ {
    struct ch_sel_r10_s_ {
      struct n1_pucch_an_cs_r10_c_ {
        struct setup_s_ {
          using n1_pucch_an_cs_list_r10_l_ = dyn_array<n1_pucch_an_cs_r10_l>;

          // member variables
          n1_pucch_an_cs_list_r10_l_ n1_pucch_an_cs_list_r10;
        };
        typedef setup_e types;

        // choice methods
        n1_pucch_an_cs_r10_c_() = default;
        void        set(types::options e = types::nulltype);
        types       type() const { return type_; }
        SRSASN_CODE pack(bit_ref& bref) const;
        SRSASN_CODE unpack(cbit_ref& bref);
        void        to_json(json_writer& j) const;
        // getters
        setup_s_& setup()
        {
          assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-r10");
          return c;
        }
        const setup_s_& setup() const
        {
          assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-r10");
          return c;
        }
        setup_s_& set_setup()
        {
          set(types::setup);
          return c;
        }

      private:
        types    type_;
        setup_s_ c;
      };

      // member variables
      bool                  n1_pucch_an_cs_r10_present = false;
      n1_pucch_an_cs_r10_c_ n1_pucch_an_cs_r10;
    };
    struct types_opts {
      enum options { format3_r10, ch_sel_r10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    pucch_format_r10_c_() = default;
    pucch_format_r10_c_(const pucch_format_r10_c_& other);
    pucch_format_r10_c_& operator=(const pucch_format_r10_c_& other);
    ~pucch_format_r10_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    pucch_format3_conf_r13_s& format3_r10()
    {
      assert_choice_type("format3-r10", type_.to_string(), "pucch-Format-r10");
      return c.get<pucch_format3_conf_r13_s>();
    }
    ch_sel_r10_s_& ch_sel_r10()
    {
      assert_choice_type("channelSelection-r10", type_.to_string(), "pucch-Format-r10");
      return c.get<ch_sel_r10_s_>();
    }
    const pucch_format3_conf_r13_s& format3_r10() const
    {
      assert_choice_type("format3-r10", type_.to_string(), "pucch-Format-r10");
      return c.get<pucch_format3_conf_r13_s>();
    }
    const ch_sel_r10_s_& ch_sel_r10() const
    {
      assert_choice_type("channelSelection-r10", type_.to_string(), "pucch-Format-r10");
      return c.get<ch_sel_r10_s_>();
    }
    pucch_format3_conf_r13_s& set_format3_r10()
    {
      set(types::format3_r10);
      return c.get<pucch_format3_conf_r13_s>();
    }
    ch_sel_r10_s_& set_ch_sel_r10()
    {
      set(types::ch_sel_r10);
      return c.get<ch_sel_r10_s_>();
    }

  private:
    types                                                    type_;
    choice_buffer_t<ch_sel_r10_s_, pucch_format3_conf_r13_s> c;

    void destroy_();
  };

  // member variables
  bool                pucch_format_r10_present                        = false;
  bool                two_ant_port_activ_pucch_format1a1b_r10_present = false;
  bool                simul_pucch_pusch_r10_present                   = false;
  bool                n1_pucch_an_rep_p1_r10_present                  = false;
  pucch_format_r10_c_ pucch_format_r10;
  uint16_t            n1_pucch_an_rep_p1_r10 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1130 ::= SEQUENCE
struct pucch_cfg_ded_v1130_s {
  struct n1_pucch_an_cs_v1130_c_ {
    struct setup_s_ {
      using n1_pucch_an_cs_list_p1_r11_l_ = bounded_array<uint16_t, 4>;

      // member variables
      n1_pucch_an_cs_list_p1_r11_l_ n1_pucch_an_cs_list_p1_r11;
    };
    typedef setup_e types;

    // choice methods
    n1_pucch_an_cs_v1130_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-v1130");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "n1PUCCH-AN-CS-v1130");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct npucch_param_r11_c_ {
    struct setup_s_ {
      uint16_t npucch_id_r11   = 0;
      uint16_t n1_pucch_an_r11 = 0;
    };
    typedef setup_e types;

    // choice methods
    npucch_param_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "nPUCCH-Param-r11");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "nPUCCH-Param-r11");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                    n1_pucch_an_cs_v1130_present = false;
  bool                    npucch_param_r11_present     = false;
  n1_pucch_an_cs_v1130_c_ n1_pucch_an_cs_v1130;
  npucch_param_r11_c_     npucch_param_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1250 ::= SEQUENCE
struct pucch_cfg_ded_v1250_s {
  struct nka_pucch_param_r12_c_ {
    struct setup_s_ {
      uint16_t nka_pucch_an_r12 = 0;
    };
    typedef setup_e types;

    // choice methods
    nka_pucch_param_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "nkaPUCCH-Param-r12");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "nkaPUCCH-Param-r12");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  nka_pucch_param_r12_c_ nka_pucch_param_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1430 ::= SEQUENCE
struct pucch_cfg_ded_v1430_s {
  struct pucch_num_repeat_ce_format1_r14_opts {
    enum options { r64, r128, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<pucch_num_repeat_ce_format1_r14_opts> pucch_num_repeat_ce_format1_r14_e_;

  // member variables
  bool                               pucch_num_repeat_ce_format1_r14_present = false;
  pucch_num_repeat_ce_format1_r14_e_ pucch_num_repeat_ce_format1_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated ::= SEQUENCE
struct pusch_cfg_ded_s {
  uint8_t beta_offset_ack_idx = 0;
  uint8_t beta_offset_ri_idx  = 0;
  uint8_t beta_offset_cqi_idx = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-r13 ::= SEQUENCE
struct pusch_cfg_ded_r13_s {
  struct beta_offset_mc_r13_s_ {
    bool    beta_offset2_ack_idx_mc_r13_present = false;
    uint8_t beta_offset_ack_idx_mc_r13          = 0;
    uint8_t beta_offset2_ack_idx_mc_r13         = 0;
    uint8_t beta_offset_ri_idx_mc_r13           = 0;
    uint8_t beta_offset_cqi_idx_mc_r13          = 0;
  };
  struct pusch_dmrs_r11_c_ {
    struct setup_s_ {
      uint16_t npusch_id_r13    = 0;
      uint16_t ndmrs_csh_id_r13 = 0;
    };
    typedef setup_e types;

    // choice methods
    pusch_dmrs_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pusch-DMRS-r11");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pusch-DMRS-r11");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct uci_on_pusch_c_ {
    struct setup_s_ {
      struct beta_offset_mc_r12_s_ {
        bool    beta_offset2_ack_idx_mc_sf_set2_r13_present = false;
        uint8_t beta_offset_ack_idx_mc_sf_set2_r13          = 0;
        uint8_t beta_offset2_ack_idx_mc_sf_set2_r13         = 0;
        uint8_t beta_offset_ri_idx_mc_sf_set2_r13           = 0;
        uint8_t beta_offset_cqi_idx_mc_sf_set2_r13          = 0;
      };

      // member variables
      bool                  beta_offset2_ack_idx_sf_set2_r13_present = false;
      bool                  beta_offset_mc_r12_present               = false;
      uint8_t               beta_offset_ack_idx_sf_set2_r13          = 0;
      uint8_t               beta_offset2_ack_idx_sf_set2_r13         = 0;
      uint8_t               beta_offset_ri_idx_sf_set2_r13           = 0;
      uint8_t               beta_offset_cqi_idx_sf_set2_r13          = 0;
      beta_offset_mc_r12_s_ beta_offset_mc_r12;
    };
    typedef setup_e types;

    // choice methods
    uci_on_pusch_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "uciOnPUSCH");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "uciOnPUSCH");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                  beta_offset2_ack_idx_r13_present = false;
  bool                  beta_offset_mc_r13_present       = false;
  bool                  group_hop_disabled_r13_present   = false;
  bool                  dmrs_with_occ_activ_r13_present  = false;
  bool                  pusch_dmrs_r11_present           = false;
  bool                  uci_on_pusch_present             = false;
  bool                  pusch_hop_cfg_r13_present        = false;
  uint8_t               beta_offset_ack_idx_r13          = 0;
  uint8_t               beta_offset2_ack_idx_r13         = 0;
  uint8_t               beta_offset_ri_idx_r13           = 0;
  uint8_t               beta_offset_cqi_idx_r13          = 0;
  beta_offset_mc_r13_s_ beta_offset_mc_r13;
  pusch_dmrs_r11_c_     pusch_dmrs_r11;
  uci_on_pusch_c_       uci_on_pusch;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-v1020 ::= SEQUENCE
struct pusch_cfg_ded_v1020_s {
  struct beta_offset_mc_r10_s_ {
    uint8_t beta_offset_ack_idx_mc_r10 = 0;
    uint8_t beta_offset_ri_idx_mc_r10  = 0;
    uint8_t beta_offset_cqi_idx_mc_r10 = 0;
  };

  // member variables
  bool                  beta_offset_mc_r10_present      = false;
  bool                  group_hop_disabled_r10_present  = false;
  bool                  dmrs_with_occ_activ_r10_present = false;
  beta_offset_mc_r10_s_ beta_offset_mc_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-v1130 ::= SEQUENCE
struct pusch_cfg_ded_v1130_s {
  struct pusch_dmrs_r11_c_ {
    struct setup_s_ {
      uint16_t npusch_id_r11    = 0;
      uint16_t ndmrs_csh_id_r11 = 0;
    };
    typedef setup_e types;

    // choice methods
    pusch_dmrs_r11_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pusch-DMRS-r11");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pusch-DMRS-r11");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  pusch_dmrs_r11_c_ pusch_dmrs_r11;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-v1250 ::= SEQUENCE
struct pusch_cfg_ded_v1250_s {
  struct uci_on_pusch_c_ {
    struct setup_s_ {
      struct beta_offset_mc_r12_s_ {
        uint8_t beta_offset_ack_idx_mc_sf_set2_r12 = 0;
        uint8_t beta_offset_ri_idx_mc_sf_set2_r12  = 0;
        uint8_t beta_offset_cqi_idx_mc_sf_set2_r12 = 0;
      };

      // member variables
      bool                  beta_offset_mc_r12_present      = false;
      uint8_t               beta_offset_ack_idx_sf_set2_r12 = 0;
      uint8_t               beta_offset_ri_idx_sf_set2_r12  = 0;
      uint8_t               beta_offset_cqi_idx_sf_set2_r12 = 0;
      beta_offset_mc_r12_s_ beta_offset_mc_r12;
    };
    typedef setup_e types;

    // choice methods
    uci_on_pusch_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "uciOnPUSCH");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "uciOnPUSCH");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  uci_on_pusch_c_ uci_on_pusch;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-v1430 ::= SEQUENCE
struct pusch_cfg_ded_v1430_s {
  bool                   ce_pusch_nb_max_tbs_r14_present = false;
  bool                   ce_pusch_max_bw_r14_present     = false;
  bool                   tdd_pusch_up_pts_r14_present    = false;
  bool                   enable256_qam_r14_present       = false;
  tdd_pusch_up_pts_r14_c tdd_pusch_up_pts_r14;
  bool                   ul_dmrs_ifdma_r14 = false;
  enable256_qam_r14_c    enable256_qam_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicated-v1530 ::= SEQUENCE
struct pusch_cfg_ded_v1530_s {
  struct ce_pusch_flex_start_prb_alloc_cfg_r15_c_ {
    struct setup_s_ {
      bool   offset_ce_mode_b_r15_present = false;
      int8_t offset_ce_mode_b_r15         = -1;
    };
    typedef setup_e types;

    // choice methods
    ce_pusch_flex_start_prb_alloc_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ce-PUSCH-FlexibleStartPRB-AllocConfig-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ce-PUSCH-FlexibleStartPRB-AllocConfig-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct ce_pusch_sub_prb_cfg_r15_c_ {
    struct setup_s_ {
      bool    location_ce_mode_b_r15_present = false;
      uint8_t location_ce_mode_b_r15         = 0;
      uint8_t six_tone_cyclic_shift_r15      = 0;
      uint8_t three_tone_cyclic_shift_r15    = 0;
    };
    typedef setup_e types;

    // choice methods
    ce_pusch_sub_prb_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ce-PUSCH-SubPRB-Config-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ce-PUSCH-SubPRB-Config-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                                     ce_pusch_sub_prb_cfg_r15_present = false;
  ce_pusch_flex_start_prb_alloc_cfg_r15_c_ ce_pusch_flex_start_prb_alloc_cfg_r15;
  ce_pusch_sub_prb_cfg_r15_c_              ce_pusch_sub_prb_cfg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-EnhancementsConfig-r14 ::= CHOICE
struct pusch_enhance_cfg_r14_c {
  struct setup_s_ {
    struct interv_ul_hop_pusch_enh_r14_c_ {
      struct interv_fdd_pusch_enh_r14_opts {
        enum options { int1, int2, int4, int8, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_fdd_pusch_enh_r14_opts> interv_fdd_pusch_enh_r14_e_;
      struct interv_tdd_pusch_enh_r14_opts {
        enum options { int1, int5, int10, int20, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<interv_tdd_pusch_enh_r14_opts> interv_tdd_pusch_enh_r14_e_;
      struct types_opts {
        enum options { interv_fdd_pusch_enh_r14, interv_tdd_pusch_enh_r14, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      interv_ul_hop_pusch_enh_r14_c_() = default;
      interv_ul_hop_pusch_enh_r14_c_(const interv_ul_hop_pusch_enh_r14_c_& other);
      interv_ul_hop_pusch_enh_r14_c_& operator=(const interv_ul_hop_pusch_enh_r14_c_& other);
      ~interv_ul_hop_pusch_enh_r14_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      interv_fdd_pusch_enh_r14_e_& interv_fdd_pusch_enh_r14()
      {
        assert_choice_type("interval-FDD-PUSCH-Enh-r14", type_.to_string(), "interval-ULHoppingPUSCH-Enh-r14");
        return c.get<interv_fdd_pusch_enh_r14_e_>();
      }
      interv_tdd_pusch_enh_r14_e_& interv_tdd_pusch_enh_r14()
      {
        assert_choice_type("interval-TDD-PUSCH-Enh-r14", type_.to_string(), "interval-ULHoppingPUSCH-Enh-r14");
        return c.get<interv_tdd_pusch_enh_r14_e_>();
      }
      const interv_fdd_pusch_enh_r14_e_& interv_fdd_pusch_enh_r14() const
      {
        assert_choice_type("interval-FDD-PUSCH-Enh-r14", type_.to_string(), "interval-ULHoppingPUSCH-Enh-r14");
        return c.get<interv_fdd_pusch_enh_r14_e_>();
      }
      const interv_tdd_pusch_enh_r14_e_& interv_tdd_pusch_enh_r14() const
      {
        assert_choice_type("interval-TDD-PUSCH-Enh-r14", type_.to_string(), "interval-ULHoppingPUSCH-Enh-r14");
        return c.get<interv_tdd_pusch_enh_r14_e_>();
      }
      interv_fdd_pusch_enh_r14_e_& set_interv_fdd_pusch_enh_r14()
      {
        set(types::interv_fdd_pusch_enh_r14);
        return c.get<interv_fdd_pusch_enh_r14_e_>();
      }
      interv_tdd_pusch_enh_r14_e_& set_interv_tdd_pusch_enh_r14()
      {
        set(types::interv_tdd_pusch_enh_r14);
        return c.get<interv_tdd_pusch_enh_r14_e_>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };

    // member variables
    bool                           pusch_hop_offset_pusch_enh_r14_present = false;
    bool                           interv_ul_hop_pusch_enh_r14_present    = false;
    uint8_t                        pusch_hop_offset_pusch_enh_r14         = 1;
    interv_ul_hop_pusch_enh_r14_c_ interv_ul_hop_pusch_enh_r14;
  };
  typedef setup_e types;

  // choice methods
  pusch_enhance_cfg_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "PUSCH-EnhancementsConfig-r14");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "PUSCH-EnhancementsConfig-r14");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// PeriodicBSR-Timer-r12 ::= ENUMERATED
struct periodic_bsr_timer_r12_opts {
  enum options {
    sf5,
    sf10,
    sf16,
    sf20,
    sf32,
    sf40,
    sf64,
    sf80,
    sf128,
    sf160,
    sf320,
    sf640,
    sf1280,
    sf2560,
    infinity,
    spare1,
    nulltype
  } value;
  typedef int16_t number_type;

  std::string to_string() const;
  int16_t     to_number() const;
};
typedef enumerated<periodic_bsr_timer_r12_opts> periodic_bsr_timer_r12_e;

// PhysicalConfigDedicatedSTTI-r15 ::= CHOICE
struct phys_cfg_ded_stti_r15_c {
  struct setup_s_ {
    bool                                 ant_info_ded_stti_r15_present              = false;
    bool                                 ant_info_ul_stti_r15_present               = false;
    bool                                 pucch_cfg_ded_v1530_present                = false;
    bool                                 sched_request_cfg_v1530_present            = false;
    bool                                 ul_pwr_ctrl_ded_stti_r15_present           = false;
    bool                                 cqi_report_cfg_r15_present                 = false;
    bool                                 csi_rs_cfg_r15_present                     = false;
    bool                                 csi_rs_cfg_nzp_to_release_list_r15_present = false;
    bool                                 csi_rs_cfg_nzp_to_add_mod_list_r15_present = false;
    bool                                 csi_rs_cfg_zp_to_release_list_r15_present  = false;
    bool                                 csi_rs_cfg_zp_to_add_mod_list_r11_present  = false;
    bool                                 csi_rs_cfg_zp_ap_list_r15_present          = false;
    bool                                 eimta_main_cfg_r12_present                 = false;
    bool                                 eimta_main_cfg_serv_cell_r15_present       = false;
    bool                                 slot_or_subslot_pdsch_cfg_r15_present      = false;
    bool                                 slot_or_subslot_pusch_cfg_r15_present      = false;
    bool                                 spdcch_cfg_r15_present                     = false;
    bool                                 spucch_cfg_r15_present                     = false;
    bool                                 short_tti_r15_present                      = false;
    ant_info_ded_stti_r15_c              ant_info_ded_stti_r15;
    ant_info_ul_stti_r15_s               ant_info_ul_stti_r15;
    pucch_cfg_ded_v1530_s                pucch_cfg_ded_v1530;
    sched_request_cfg_v1530_c            sched_request_cfg_v1530;
    ul_pwr_ctrl_ded_stti_r15_s           ul_pwr_ctrl_ded_stti_r15;
    cqi_report_cfg_r15_c                 cqi_report_cfg_r15;
    csi_rs_cfg_r15_c                     csi_rs_cfg_r15;
    csi_rs_cfg_nzp_to_release_list_r15_l csi_rs_cfg_nzp_to_release_list_r15;
    csi_rs_cfg_nzp_to_add_mod_list_r15_l csi_rs_cfg_nzp_to_add_mod_list_r15;
    csi_rs_cfg_zp_to_release_list_r11_l  csi_rs_cfg_zp_to_release_list_r15;
    csi_rs_cfg_zp_to_add_mod_list_r11_l  csi_rs_cfg_zp_to_add_mod_list_r11;
    csi_rs_cfg_zp_ap_list_r14_c          csi_rs_cfg_zp_ap_list_r15;
    eimta_main_cfg_r12_c                 eimta_main_cfg_r12;
    eimta_main_cfg_serv_cell_r12_c       eimta_main_cfg_serv_cell_r15;
    bool                                 semi_open_loop_stti_r15 = false;
    slot_or_subslot_pdsch_cfg_r15_c      slot_or_subslot_pdsch_cfg_r15;
    slot_or_subslot_pusch_cfg_r15_c      slot_or_subslot_pusch_cfg_r15;
    spdcch_cfg_r15_c                     spdcch_cfg_r15;
    spucch_cfg_r15_c                     spucch_cfg_r15;
    bool                                 srs_dci7_trigger_cfg_r15  = false;
    bool                                 short_processing_time_r15 = false;
    short_tti_r15_s                      short_tti_r15;
  };
  typedef setup_e types;

  // choice methods
  phys_cfg_ded_stti_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "PhysicalConfigDedicatedSTTI-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "PhysicalConfigDedicatedSTTI-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReject-v1130-IEs ::= SEQUENCE
struct rrc_conn_reject_v1130_ies_s {
  struct depriorit_req_r11_s_ {
    struct depriorit_type_r11_opts {
      enum options { freq, e_utra, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<depriorit_type_r11_opts> depriorit_type_r11_e_;
    struct depriorit_timer_r11_opts {
      enum options { min5, min10, min15, min30, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<depriorit_timer_r11_opts> depriorit_timer_r11_e_;

    // member variables
    depriorit_type_r11_e_  depriorit_type_r11;
    depriorit_timer_r11_e_ depriorit_timer_r11;
  };

  // member variables
  bool                        depriorit_req_r11_present = false;
  bool                        non_crit_ext_present      = false;
  depriorit_req_r11_s_        depriorit_req_r11;
  rrc_conn_reject_v1320_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RetxBSR-Timer-r12 ::= ENUMERATED
struct retx_bsr_timer_r12_opts {
  enum options { sf320, sf640, sf1280, sf2560, sf5120, sf10240, spare2, spare1, nulltype } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<retx_bsr_timer_r12_opts> retx_bsr_timer_r12_e;

// SPS-ConfigDL ::= CHOICE
struct sps_cfg_dl_c {
  struct setup_s_ {
    struct semi_persist_sched_interv_dl_opts {
      enum options {
        sf10,
        sf20,
        sf32,
        sf40,
        sf64,
        sf80,
        sf128,
        sf160,
        sf320,
        sf640,
        spare6,
        spare5,
        spare4,
        spare3,
        spare2,
        spare1,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<semi_persist_sched_interv_dl_opts> semi_persist_sched_interv_dl_e_;
    struct two_ant_port_activ_r10_c_ {
      struct setup_s_ {
        n1_pucch_an_persistent_list_l n1_pucch_an_persistent_list_p1_r10;
      };
      typedef setup_e types;

      // choice methods
      two_ant_port_activ_r10_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivated-r10");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "twoAntennaPortActivated-r10");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                            ext = false;
    semi_persist_sched_interv_dl_e_ semi_persist_sched_interv_dl;
    uint8_t                         nof_conf_sps_processes = 1;
    n1_pucch_an_persistent_list_l   n1_pucch_an_persistent_list;
    // ...
    // group 0
    copy_ptr<two_ant_port_activ_r10_c_> two_ant_port_activ_r10;

    // sequence methods
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
  };
  typedef setup_e types;

  // choice methods
  sps_cfg_dl_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigDL");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SPS-ConfigDL");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SPS-ConfigSL-ToAddModList-r14 ::= SEQUENCE (SIZE (1..8)) OF SPS-ConfigSL-r14
using sps_cfg_sl_to_add_mod_list_r14_l = dyn_array<sps_cfg_sl_r14_s>;

// SPS-ConfigSL-ToReleaseList-r14 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..8)
using sps_cfg_sl_to_release_list_r14_l = bounded_array<uint8_t, 8>;

// SPS-ConfigUL-STTI-ToAddModList-r15 ::= SEQUENCE (SIZE (1..6)) OF SPS-ConfigUL-STTI-r15
using sps_cfg_ul_stti_to_add_mod_list_r15_l = dyn_array<sps_cfg_ul_stti_r15_c>;

// SPS-ConfigUL-STTI-ToReleaseList-r15 ::= SEQUENCE (SIZE (1..6)) OF INTEGER (1..6)
using sps_cfg_ul_stti_to_release_list_r15_l = bounded_array<uint8_t, 6>;

// SPS-ConfigUL-ToAddModList-r14 ::= SEQUENCE (SIZE (1..8)) OF SPS-ConfigUL
using sps_cfg_ul_to_add_mod_list_r14_l = dyn_array<sps_cfg_ul_c>;

// SPS-ConfigUL-ToAddModList-r15 ::= SEQUENCE (SIZE (1..6)) OF SPS-ConfigUL
using sps_cfg_ul_to_add_mod_list_r15_l = dyn_array<sps_cfg_ul_c>;

// SPS-ConfigUL-ToReleaseList-r14 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..8)
using sps_cfg_ul_to_release_list_r14_l = bounded_array<uint8_t, 8>;

// SPS-ConfigUL-ToReleaseList-r15 ::= SEQUENCE (SIZE (1..6)) OF INTEGER (1..6)
using sps_cfg_ul_to_release_list_r15_l = bounded_array<uint8_t, 6>;

// SRB-ToAddMod ::= SEQUENCE
struct srb_to_add_mod_s {
  struct rlc_cfg_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    rlc_cfg_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rlc_cfg_c& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "rlc-Config");
      return c;
    }
    const rlc_cfg_c& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "rlc-Config");
      return c;
    }
    rlc_cfg_c& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types     type_;
    rlc_cfg_c c;
  };
  struct lc_ch_cfg_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    lc_ch_cfg_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    lc_ch_cfg_s& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "logicalChannelConfig");
      return c;
    }
    const lc_ch_cfg_s& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "logicalChannelConfig");
      return c;
    }
    lc_ch_cfg_s& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types       type_;
    lc_ch_cfg_s c;
  };

  // member variables
  bool         ext               = false;
  bool         rlc_cfg_present   = false;
  bool         lc_ch_cfg_present = false;
  uint8_t      srb_id            = 1;
  rlc_cfg_c_   rlc_cfg;
  lc_ch_cfg_c_ lc_ch_cfg;
  // ...
  // group 0
  bool                           pdcp_ver_change_r15_present = false;
  bool                           srb_id_v1530_present        = false;
  copy_ptr<rlc_cfg_v1530_c>      rlc_cfg_v1530;
  copy_ptr<rlc_bearer_cfg_r15_c> rlc_bearer_cfg_dupl_r15;
  uint8_t                        srb_id_v1530 = 4;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SRS-TPC-PDCCH-Config-r14 ::= CHOICE
struct srs_tpc_pdcch_cfg_r14_c {
  struct setup_s_ {
    using srs_cc_set_idxlist_r14_l_ = dyn_array<srs_cc_set_idx_r14_s>;

    // member variables
    bool                      srs_cc_set_idxlist_r14_present = false;
    fixed_bitstring<16>       srs_tpc_rnti_r14;
    uint8_t                   start_bit_of_format3_b_r14 = 0;
    uint8_t                   field_type_format3_b_r14   = 1;
    srs_cc_set_idxlist_r14_l_ srs_cc_set_idxlist_r14;
  };
  typedef setup_e types;

  // choice methods
  srs_tpc_pdcch_cfg_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SRS-TPC-PDCCH-Config-r14");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SRS-TPC-PDCCH-Config-r14");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// STAG-ToAddModList-r11 ::= SEQUENCE (SIZE (1..3)) OF STAG-ToAddMod-r11
using stag_to_add_mod_list_r11_l = dyn_array<stag_to_add_mod_r11_s>;

// STAG-ToReleaseList-r11 ::= SEQUENCE (SIZE (1..3)) OF INTEGER (1..3)
using stag_to_release_list_r11_l = bounded_array<uint8_t, 3>;

// SchedulingRequestConfig ::= CHOICE
struct sched_request_cfg_c {
  struct setup_s_ {
    struct dsr_trans_max_opts {
      enum options { n4, n8, n16, n32, n64, spare3, spare2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<dsr_trans_max_opts> dsr_trans_max_e_;

    // member variables
    uint16_t         sr_pucch_res_idx = 0;
    uint8_t          sr_cfg_idx       = 0;
    dsr_trans_max_e_ dsr_trans_max;
  };
  typedef setup_e types;

  // choice methods
  sched_request_cfg_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfig");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfig");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SchedulingRequestConfig-v1020 ::= SEQUENCE
struct sched_request_cfg_v1020_s {
  bool     sr_pucch_res_idx_p1_r10_present = false;
  uint16_t sr_pucch_res_idx_p1_r10         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SoundingRS-UL-ConfigDedicated ::= CHOICE
struct srs_ul_cfg_ded_c {
  struct setup_s_ {
    struct srs_bw_opts {
      enum options { bw0, bw1, bw2, bw3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_bw_opts> srs_bw_e_;
    struct srs_hop_bw_opts {
      enum options { hbw0, hbw1, hbw2, hbw3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_hop_bw_opts> srs_hop_bw_e_;
    struct cyclic_shift_opts {
      enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cyclic_shift_opts> cyclic_shift_e_;

    // member variables
    srs_bw_e_       srs_bw;
    srs_hop_bw_e_   srs_hop_bw;
    uint8_t         freq_domain_position = 0;
    bool            dur                  = false;
    uint16_t        srs_cfg_idx          = 0;
    uint8_t         tx_comb              = 0;
    cyclic_shift_e_ cyclic_shift;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicated");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicated");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-UL-ConfigDedicated-v1020 ::= SEQUENCE
struct srs_ul_cfg_ded_v1020_s {
  srs_ant_port_e srs_ant_port_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SoundingRS-UL-ConfigDedicated-v1310 ::= CHOICE
struct srs_ul_cfg_ded_v1310_c {
  struct setup_s_ {
    struct cyclic_shift_v1310_opts {
      enum options { cs8, cs9, cs10, cs11, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cyclic_shift_v1310_opts> cyclic_shift_v1310_e_;
    struct tx_comb_num_r13_opts {
      enum options { n2, n4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tx_comb_num_r13_opts> tx_comb_num_r13_e_;

    // member variables
    bool                  tx_comb_v1310_present      = false;
    bool                  cyclic_shift_v1310_present = false;
    bool                  tx_comb_num_r13_present    = false;
    uint8_t               tx_comb_v1310              = 2;
    cyclic_shift_v1310_e_ cyclic_shift_v1310;
    tx_comb_num_r13_e_    tx_comb_num_r13;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_v1310_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicated-v1310");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicated-v1310");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-UL-ConfigDedicatedAperiodic-r10 ::= CHOICE
struct srs_ul_cfg_ded_aperiodic_r10_c {
  struct setup_s_ {
    using srs_cfg_ap_dci_format4_r10_l_ = dyn_array<srs_cfg_ap_r10_s>;
    struct srs_activ_ap_r10_c_ {
      struct setup_s_ {
        bool             ext = false;
        srs_cfg_ap_r10_s srs_cfg_ap_dci_format0_r10;
        srs_cfg_ap_r10_s srs_cfg_ap_dci_format1a2b2c_r10;
        // ...
      };
      typedef setup_e types;

      // choice methods
      srs_activ_ap_r10_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-r10");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-r10");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                          srs_cfg_ap_dci_format4_r10_present = false;
    bool                          srs_activ_ap_r10_present           = false;
    uint8_t                       srs_cfg_idx_ap_r10                 = 0;
    srs_cfg_ap_dci_format4_r10_l_ srs_cfg_ap_dci_format4_r10;
    srs_activ_ap_r10_c_           srs_activ_ap_r10;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_aperiodic_r10_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-r10");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-r10");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-UL-ConfigDedicatedAperiodic-v1310 ::= CHOICE
struct srs_ul_cfg_ded_aperiodic_v1310_c {
  struct setup_s_ {
    using srs_cfg_ap_dci_format4_v1310_l_ = dyn_array<srs_cfg_ap_v1310_s>;
    struct srs_activ_ap_v1310_c_ {
      struct setup_s_ {
        bool               srs_cfg_ap_dci_format0_v1310_present      = false;
        bool               srs_cfg_ap_dci_format1a2b2c_v1310_present = false;
        srs_cfg_ap_v1310_s srs_cfg_ap_dci_format0_v1310;
        srs_cfg_ap_v1310_s srs_cfg_ap_dci_format1a2b2c_v1310;
      };
      typedef setup_e types;

      // choice methods
      srs_activ_ap_v1310_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-v1310");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-v1310");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                            srs_cfg_ap_dci_format4_v1310_present = false;
    bool                            srs_activ_ap_v1310_present           = false;
    srs_cfg_ap_dci_format4_v1310_l_ srs_cfg_ap_dci_format4_v1310;
    srs_activ_ap_v1310_c_           srs_activ_ap_v1310;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_aperiodic_v1310_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-v1310");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-v1310");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-UL-ConfigDedicatedAperiodicUpPTsExt-r13 ::= CHOICE
struct srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c {
  struct setup_s_ {
    struct srs_up_pts_add_r13_opts {
      enum options { sym2, sym4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_up_pts_add_r13_opts> srs_up_pts_add_r13_e_;
    using srs_cfg_ap_dci_format4_r13_l_ = dyn_array<srs_cfg_ap_r13_s>;
    struct srs_activ_ap_r13_c_ {
      struct setup_s_ {
        srs_cfg_ap_r13_s srs_cfg_ap_dci_format0_r13;
        srs_cfg_ap_r13_s srs_cfg_ap_dci_format1a2b2c_r13;
      };
      typedef setup_e types;

      // choice methods
      srs_activ_ap_r13_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-r13");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "srs-ActivateAp-r13");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };

    // member variables
    bool                          srs_cfg_ap_dci_format4_r13_present = false;
    bool                          srs_activ_ap_r13_present           = false;
    srs_up_pts_add_r13_e_         srs_up_pts_add_r13;
    uint8_t                       srs_cfg_idx_ap_r13 = 0;
    srs_cfg_ap_dci_format4_r13_l_ srs_cfg_ap_dci_format4_r13;
    srs_activ_ap_r13_c_           srs_activ_ap_r13;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodicUpPTsExt-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodicUpPTsExt-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-UL-ConfigDedicatedUpPTsExt-r13 ::= CHOICE
struct srs_ul_cfg_ded_up_pts_ext_r13_c {
  struct setup_s_ {
    struct srs_up_pts_add_r13_opts {
      enum options { sym2, sym4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_up_pts_add_r13_opts> srs_up_pts_add_r13_e_;
    struct srs_bw_r13_opts {
      enum options { bw0, bw1, bw2, bw3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_bw_r13_opts> srs_bw_r13_e_;
    struct srs_hop_bw_r13_opts {
      enum options { hbw0, hbw1, hbw2, hbw3, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<srs_hop_bw_r13_opts> srs_hop_bw_r13_e_;
    struct cyclic_shift_r13_opts {
      enum options { cs0, cs1, cs2, cs3, cs4, cs5, cs6, cs7, cs8, cs9, cs10, cs11, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<cyclic_shift_r13_opts> cyclic_shift_r13_e_;
    struct tx_comb_num_r13_opts {
      enum options { n2, n4, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tx_comb_num_r13_opts> tx_comb_num_r13_e_;

    // member variables
    srs_up_pts_add_r13_e_ srs_up_pts_add_r13;
    srs_bw_r13_e_         srs_bw_r13;
    srs_hop_bw_r13_e_     srs_hop_bw_r13;
    uint8_t               freq_domain_position_r13 = 0;
    bool                  dur_r13                  = false;
    uint16_t              srs_cfg_idx_r13          = 0;
    uint8_t               tx_comb_r13              = 0;
    cyclic_shift_r13_e_   cyclic_shift_r13;
    srs_ant_port_e        srs_ant_port_r13;
    tx_comb_num_r13_e_    tx_comb_num_r13;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_up_pts_ext_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedUpPTsExt-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedUpPTsExt-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// UplinkPowerControlDedicated ::= SEQUENCE
struct ul_pwr_ctrl_ded_s {
  struct delta_mcs_enabled_opts {
    enum options { en0, en1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_mcs_enabled_opts> delta_mcs_enabled_e_;

  // member variables
  bool                 filt_coef_present = false;
  int8_t               p0_ue_pusch       = -8;
  delta_mcs_enabled_e_ delta_mcs_enabled;
  bool                 accumulation_enabled = false;
  int8_t               p0_ue_pucch          = -8;
  uint8_t              psrs_offset          = 0;
  filt_coef_e          filt_coef;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicated-v1020 ::= SEQUENCE
struct ul_pwr_ctrl_ded_v1020_s {
  bool                               delta_tx_d_offset_list_pucch_r10_present = false;
  bool                               psrs_offset_ap_r10_present               = false;
  delta_tx_d_offset_list_pucch_r10_s delta_tx_d_offset_list_pucch_r10;
  uint8_t                            psrs_offset_ap_r10 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicated-v1130 ::= SEQUENCE
struct ul_pwr_ctrl_ded_v1130_s {
  bool                                 psrs_offset_v1130_present                  = false;
  bool                                 psrs_offset_ap_v1130_present               = false;
  bool                                 delta_tx_d_offset_list_pucch_v1130_present = false;
  uint8_t                              psrs_offset_v1130                          = 16;
  uint8_t                              psrs_offset_ap_v1130                       = 16;
  delta_tx_d_offset_list_pucch_v1130_s delta_tx_d_offset_list_pucch_v1130;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicated-v1250 ::= SEQUENCE
struct ul_pwr_ctrl_ded_v1250_s {
  struct set2_pwr_ctrl_param_c_ {
    struct setup_s_ {
      fixed_bitstring<10> tpc_sf_set_r12;
      int8_t              p0_nominal_pusch_sf_set2_r12 = -126;
      alpha_r12_e         alpha_sf_set2_r12;
      int8_t              p0_ue_pusch_sf_set2_r12 = -8;
    };
    typedef setup_e types;

    // choice methods
    set2_pwr_ctrl_param_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "set2PowerControlParameter");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "set2PowerControlParameter");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  set2_pwr_ctrl_param_c_ set2_pwr_ctrl_param;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicated-v1530 ::= SEQUENCE
struct ul_pwr_ctrl_ded_v1530_s {
  bool        alpha_ue_r15_present    = false;
  bool        p0_ue_pusch_r15_present = false;
  alpha_r12_e alpha_ue_r15;
  int8_t      p0_ue_pusch_r15 = -16;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandClassPriorityList1XRTT ::= SEQUENCE (SIZE (1..32)) OF BandClassPriority1XRTT
using band_class_prio_list1_xrtt_l = dyn_array<band_class_prio1_xrtt_s>;

// BandClassPriorityListHRPD ::= SEQUENCE (SIZE (1..32)) OF BandClassPriorityHRPD
using band_class_prio_list_hrpd_l = dyn_array<band_class_prio_hrpd_s>;

// CarrierFreqCDMA2000 ::= SEQUENCE
struct carrier_freq_cdma2000_s {
  bandclass_cdma2000_e band_class;
  uint16_t             arfcn = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CarrierFreqListUTRA-TDD-r10 ::= SEQUENCE (SIZE (1..6)) OF INTEGER (0..16383)
using carrier_freq_list_utra_tdd_r10_l = bounded_array<uint16_t, 6>;

// DRB-ToAddModList ::= SEQUENCE (SIZE (1..11)) OF DRB-ToAddMod
using drb_to_add_mod_list_l = dyn_array<drb_to_add_mod_s>;

// DRB-ToAddModList-r15 ::= SEQUENCE (SIZE (1..15)) OF DRB-ToAddMod
using drb_to_add_mod_list_r15_l = dyn_array<drb_to_add_mod_s>;

// DRB-ToReleaseList ::= SEQUENCE (SIZE (1..11)) OF INTEGER (1..32)
using drb_to_release_list_l = bounded_array<uint8_t, 11>;

// DRB-ToReleaseList-r15 ::= SEQUENCE (SIZE (1..15)) OF INTEGER (1..32)
using drb_to_release_list_r15_l = bounded_array<uint8_t, 15>;

// FreqPriorityEUTRA-v9e0 ::= SEQUENCE
struct freq_prio_eutra_v9e0_s {
  bool     carrier_freq_v9e0_present = false;
  uint32_t carrier_freq_v9e0         = 65536;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FreqPriorityListEUTRA ::= SEQUENCE (SIZE (1..8)) OF FreqPriorityEUTRA
using freq_prio_list_eutra_l = dyn_array<freq_prio_eutra_s>;

// FreqPriorityListEUTRA-v1310 ::= SEQUENCE (SIZE (1..8)) OF FreqPriorityEUTRA-v1310
using freq_prio_list_eutra_v1310_l = dyn_array<freq_prio_eutra_v1310_s>;

// FreqPriorityListExtEUTRA-r12 ::= SEQUENCE (SIZE (1..8)) OF FreqPriorityEUTRA-r12
using freq_prio_list_ext_eutra_r12_l = dyn_array<freq_prio_eutra_r12_s>;

// FreqPriorityListExtEUTRA-v1310 ::= SEQUENCE (SIZE (1..8)) OF FreqPriorityEUTRA-v1310
using freq_prio_list_ext_eutra_v1310_l = dyn_array<freq_prio_eutra_v1310_s>;

// FreqPriorityListNR-r15 ::= SEQUENCE (SIZE (1..8)) OF FreqPriorityNR-r15
using freq_prio_list_nr_r15_l = dyn_array<freq_prio_nr_r15_s>;

// FreqPriorityListUTRA-FDD ::= SEQUENCE (SIZE (1..16)) OF FreqPriorityUTRA-FDD
using freq_prio_list_utra_fdd_l = dyn_array<freq_prio_utra_fdd_s>;

// FreqPriorityListUTRA-TDD ::= SEQUENCE (SIZE (1..16)) OF FreqPriorityUTRA-TDD
using freq_prio_list_utra_tdd_l = dyn_array<freq_prio_utra_tdd_s>;

// FreqsPriorityListGERAN ::= SEQUENCE (SIZE (1..16)) OF FreqsPriorityGERAN
using freqs_prio_list_geran_l = dyn_array<freqs_prio_geran_s>;

// MAC-MainConfig ::= SEQUENCE
struct mac_main_cfg_s {
  struct ul_sch_cfg_s_ {
    struct max_harq_tx_opts {
      enum options { n1, n2, n3, n4, n5, n6, n7, n8, n10, n12, n16, n20, n24, n28, spare2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<max_harq_tx_opts> max_harq_tx_e_;

    // member variables
    bool                     max_harq_tx_present        = false;
    bool                     periodic_bsr_timer_present = false;
    max_harq_tx_e_           max_harq_tx;
    periodic_bsr_timer_r12_e periodic_bsr_timer;
    retx_bsr_timer_r12_e     retx_bsr_timer;
    bool                     tti_bundling = false;
  };
  struct phr_cfg_c_ {
    struct setup_s_ {
      struct periodic_phr_timer_opts {
        enum options { sf10, sf20, sf50, sf100, sf200, sf500, sf1000, infinity, nulltype } value;
        typedef int16_t number_type;

        std::string to_string() const;
        int16_t     to_number() const;
      };
      typedef enumerated<periodic_phr_timer_opts> periodic_phr_timer_e_;
      struct prohibit_phr_timer_opts {
        enum options { sf0, sf10, sf20, sf50, sf100, sf200, sf500, sf1000, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<prohibit_phr_timer_opts> prohibit_phr_timer_e_;
      struct dl_pathloss_change_opts {
        enum options { db1, db3, db6, infinity, nulltype } value;
        typedef int8_t number_type;

        std::string to_string() const;
        int8_t      to_number() const;
      };
      typedef enumerated<dl_pathloss_change_opts> dl_pathloss_change_e_;

      // member variables
      periodic_phr_timer_e_ periodic_phr_timer;
      prohibit_phr_timer_e_ prohibit_phr_timer;
      dl_pathloss_change_e_ dl_pathloss_change;
    };
    typedef setup_e types;

    // choice methods
    phr_cfg_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "phr-Config");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "phr-Config");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct mac_main_cfg_v1020_s_ {
    struct scell_deactivation_timer_r10_opts {
      enum options { rf2, rf4, rf8, rf16, rf32, rf64, rf128, spare, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<scell_deactivation_timer_r10_opts> scell_deactivation_timer_r10_e_;

    // member variables
    bool                            scell_deactivation_timer_r10_present = false;
    bool                            extended_bsr_sizes_r10_present       = false;
    bool                            extended_phr_r10_present             = false;
    scell_deactivation_timer_r10_e_ scell_deactivation_timer_r10;
  };
  struct dual_connect_phr_c_ {
    struct setup_s_ {
      struct phr_mode_other_cg_r12_opts {
        enum options { real, virtual_value, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<phr_mode_other_cg_r12_opts> phr_mode_other_cg_r12_e_;

      // member variables
      phr_mode_other_cg_r12_e_ phr_mode_other_cg_r12;
    };
    typedef setup_e types;

    // choice methods
    dual_connect_phr_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "dualConnectivityPHR");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "dualConnectivityPHR");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct lc_ch_sr_cfg_r12_c_ {
    struct setup_s_ {
      struct lc_ch_sr_prohibit_timer_r12_opts {
        enum options { sf20, sf40, sf64, sf128, sf512, sf1024, sf2560, spare1, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<lc_ch_sr_prohibit_timer_r12_opts> lc_ch_sr_prohibit_timer_r12_e_;

      // member variables
      lc_ch_sr_prohibit_timer_r12_e_ lc_ch_sr_prohibit_timer_r12;
    };
    typedef setup_e types;

    // choice methods
    lc_ch_sr_cfg_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "logicalChannelSR-Config-r12");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "logicalChannelSR-Config-r12");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct edrx_cfg_cycle_start_offset_r13_c_ {
    struct setup_c_ {
      struct types_opts {
        enum options { sf5120, sf10240, nulltype } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& sf5120()
      {
        assert_choice_type("sf5120", type_.to_string(), "setup");
        return c.get<uint8_t>();
      }
      uint8_t& sf10240()
      {
        assert_choice_type("sf10240", type_.to_string(), "setup");
        return c.get<uint8_t>();
      }
      const uint8_t& sf5120() const
      {
        assert_choice_type("sf5120", type_.to_string(), "setup");
        return c.get<uint8_t>();
      }
      const uint8_t& sf10240() const
      {
        assert_choice_type("sf10240", type_.to_string(), "setup");
        return c.get<uint8_t>();
      }
      uint8_t& set_sf5120()
      {
        set(types::sf5120);
        return c.get<uint8_t>();
      }
      uint8_t& set_sf10240()
      {
        set(types::sf10240);
        return c.get<uint8_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    edrx_cfg_cycle_start_offset_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "eDRX-Config-CycleStartOffset-r13");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "eDRX-Config-CycleStartOffset-r13");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct drx_cfg_r13_c_ {
    typedef setup_e types;

    // choice methods
    drx_cfg_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    drx_cfg_r13_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "drx-Config-r13");
      return c;
    }
    const drx_cfg_r13_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "drx-Config-r13");
      return c;
    }
    drx_cfg_r13_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types         type_;
    drx_cfg_r13_s c;
  };
  struct skip_ul_tx_r14_c_ {
    struct setup_s_ {
      bool skip_ul_tx_sps_r14_present     = false;
      bool skip_ul_tx_dynamic_r14_present = false;
    };
    typedef setup_e types;

    // choice methods
    skip_ul_tx_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "skipUplinkTx-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "skipUplinkTx-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct data_inactivity_timer_cfg_r14_c_ {
    struct setup_s_ {
      data_inactivity_timer_r14_e data_inactivity_timer_r14;
    };
    typedef setup_e types;

    // choice methods
    data_inactivity_timer_cfg_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "dataInactivityTimerConfig-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "dataInactivityTimerConfig-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct short_tti_and_spt_r15_c_ {
    struct setup_s_ {
      struct periodic_bsr_timer_r15_opts {
        enum options {
          sf1,
          sf5,
          sf10,
          sf16,
          sf20,
          sf32,
          sf40,
          sf64,
          sf80,
          sf128,
          sf160,
          sf320,
          sf640,
          sf1280,
          sf2560,
          infinity,
          nulltype
        } value;
        typedef int16_t number_type;

        std::string to_string() const;
        int16_t     to_number() const;
      };
      typedef enumerated<periodic_bsr_timer_r15_opts> periodic_bsr_timer_r15_e_;
      struct proc_timeline_r15_opts {
        enum options { nplus4set1, nplus6set1, nplus6set2, nplus8set2, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<proc_timeline_r15_opts> proc_timeline_r15_e_;

      // member variables
      bool                      drx_cfg_r15_present            = false;
      bool                      periodic_bsr_timer_r15_present = false;
      bool                      proc_timeline_r15_present      = false;
      bool                      ssr_prohibit_timer_r15_present = false;
      drx_cfg_r15_s             drx_cfg_r15;
      periodic_bsr_timer_r15_e_ periodic_bsr_timer_r15;
      proc_timeline_r15_e_      proc_timeline_r15;
      uint8_t                   ssr_prohibit_timer_r15 = 0;
    };
    typedef setup_e types;

    // choice methods
    short_tti_and_spt_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "shortTTI-AndSPT-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "shortTTI-AndSPT-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct dormant_state_timers_r15_c_ {
    struct setup_s_ {
      struct scell_hibernation_timer_r15_opts {
        enum options { rf2, rf4, rf8, rf16, rf32, rf64, rf128, spare, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<scell_hibernation_timer_r15_opts> scell_hibernation_timer_r15_e_;
      struct dormant_scell_deactivation_timer_r15_opts {
        enum options {
          rf2,
          rf4,
          rf8,
          rf16,
          rf32,
          rf64,
          rf128,
          rf320,
          rf640,
          rf1280,
          rf2560,
          rf5120,
          rf10240,
          spare3,
          spare2,
          spare1,
          nulltype
        } value;
        typedef uint16_t number_type;

        std::string to_string() const;
        uint16_t    to_number() const;
      };
      typedef enumerated<dormant_scell_deactivation_timer_r15_opts> dormant_scell_deactivation_timer_r15_e_;

      // member variables
      bool                                    scell_hibernation_timer_r15_present          = false;
      bool                                    dormant_scell_deactivation_timer_r15_present = false;
      scell_hibernation_timer_r15_e_          scell_hibernation_timer_r15;
      dormant_scell_deactivation_timer_r15_e_ dormant_scell_deactivation_timer_r15;
    };
    typedef setup_e types;

    // choice methods
    dormant_state_timers_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "dormantStateTimers-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "dormantStateTimers-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool               ext                = false;
  bool               ul_sch_cfg_present = false;
  bool               drx_cfg_present    = false;
  bool               phr_cfg_present    = false;
  ul_sch_cfg_s_      ul_sch_cfg;
  drx_cfg_c          drx_cfg;
  time_align_timer_e time_align_timer_ded;
  phr_cfg_c_         phr_cfg;
  // ...
  // group 0
  bool    sr_prohibit_timer_r9_present = false;
  uint8_t sr_prohibit_timer_r9         = 0;
  // group 1
  copy_ptr<mac_main_cfg_v1020_s_> mac_main_cfg_v1020;
  // group 2
  copy_ptr<stag_to_release_list_r11_l> stag_to_release_list_r11;
  copy_ptr<stag_to_add_mod_list_r11_l> stag_to_add_mod_list_r11;
  copy_ptr<drx_cfg_v1130_s>            drx_cfg_v1130;
  // group 3
  bool                          e_harq_pattern_r12_present = false;
  bool                          e_harq_pattern_r12         = false;
  copy_ptr<dual_connect_phr_c_> dual_connect_phr;
  copy_ptr<lc_ch_sr_cfg_r12_c_> lc_ch_sr_cfg_r12;
  // group 4
  bool                                         extended_phr2_r13_present = false;
  copy_ptr<drx_cfg_v1310_s>                    drx_cfg_v1310;
  bool                                         extended_phr2_r13 = false;
  copy_ptr<edrx_cfg_cycle_start_offset_r13_c_> edrx_cfg_cycle_start_offset_r13;
  // group 5
  copy_ptr<drx_cfg_r13_c_> drx_cfg_r13;
  // group 6
  copy_ptr<skip_ul_tx_r14_c_>                skip_ul_tx_r14;
  copy_ptr<data_inactivity_timer_cfg_r14_c_> data_inactivity_timer_cfg_r14;
  // group 7
  bool rai_activation_r14_present = false;
  // group 8
  bool                                  mpdcch_ul_harq_ack_feedback_cfg_r15_present = false;
  copy_ptr<short_tti_and_spt_r15_c_>    short_tti_and_spt_r15;
  bool                                  mpdcch_ul_harq_ack_feedback_cfg_r15 = false;
  copy_ptr<dormant_state_timers_r15_c_> dormant_state_timers_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasSubframePatternPCell-r10 ::= CHOICE
struct meas_sf_pattern_pcell_r10_c {
  typedef setup_e types;

  // choice methods
  meas_sf_pattern_pcell_r10_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  meas_sf_pattern_r10_c& setup()
  {
    assert_choice_type("setup", type_.to_string(), "MeasSubframePatternPCell-r10");
    return c;
  }
  const meas_sf_pattern_r10_c& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "MeasSubframePatternPCell-r10");
    return c;
  }
  meas_sf_pattern_r10_c& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types                 type_;
  meas_sf_pattern_r10_c c;
};

// NAICS-AssistanceInfo-r12 ::= CHOICE
struct naics_assist_info_r12_c {
  struct setup_s_ {
    bool                              neigh_cells_to_release_list_r12_present = false;
    bool                              neigh_cells_to_add_mod_list_r12_present = false;
    bool                              serv_cellp_a_r12_present                = false;
    neigh_cells_to_release_list_r12_l neigh_cells_to_release_list_r12;
    neigh_cells_to_add_mod_list_r12_l neigh_cells_to_add_mod_list_r12;
    p_a_e                             serv_cellp_a_r12;
  };
  typedef setup_e types;

  // choice methods
  naics_assist_info_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "NAICS-AssistanceInfo-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "NAICS-AssistanceInfo-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// NeighCellsCRS-Info-r11 ::= CHOICE
struct neigh_cells_crs_info_r11_c {
  typedef setup_e types;

  // choice methods
  neigh_cells_crs_info_r11_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  crs_assist_info_list_r11_l& setup()
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r11");
    return c;
  }
  const crs_assist_info_list_r11_l& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r11");
    return c;
  }
  crs_assist_info_list_r11_l& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types                      type_;
  crs_assist_info_list_r11_l c;
};

// NeighCellsCRS-Info-r13 ::= CHOICE
struct neigh_cells_crs_info_r13_c {
  typedef setup_e types;

  // choice methods
  neigh_cells_crs_info_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  crs_assist_info_list_r13_l& setup()
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r13");
    return c;
  }
  const crs_assist_info_list_r13_l& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r13");
    return c;
  }
  crs_assist_info_list_r13_l& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types                      type_;
  crs_assist_info_list_r13_l c;
};

// NeighCellsCRS-Info-r15 ::= CHOICE
struct neigh_cells_crs_info_r15_c {
  typedef setup_e types;

  // choice methods
  neigh_cells_crs_info_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  crs_assist_info_list_r15_l& setup()
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r15");
    return c;
  }
  const crs_assist_info_list_r15_l& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "NeighCellsCRS-Info-r15");
    return c;
  }
  crs_assist_info_list_r15_l& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types                      type_;
  crs_assist_info_list_r15_l c;
};

// PhysicalConfigDedicated ::= SEQUENCE
struct phys_cfg_ded_s {
  struct ant_info_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    ant_info_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    ant_info_ded_s& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "antennaInfo");
      return c;
    }
    const ant_info_ded_s& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "antennaInfo");
      return c;
    }
    ant_info_ded_s& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types          type_;
    ant_info_ded_s c;
  };
  struct ant_info_r10_c_ {
    struct types_opts {
      enum options { explicit_value_r10, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    ant_info_r10_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    ant_info_ded_r10_s& explicit_value_r10()
    {
      assert_choice_type("explicitValue-r10", type_.to_string(), "antennaInfo-r10");
      return c;
    }
    const ant_info_ded_r10_s& explicit_value_r10() const
    {
      assert_choice_type("explicitValue-r10", type_.to_string(), "antennaInfo-r10");
      return c;
    }
    ant_info_ded_r10_s& set_explicit_value_r10()
    {
      set(types::explicit_value_r10);
      return c;
    }

  private:
    types              type_;
    ant_info_ded_r10_s c;
  };
  struct add_spec_emission_ca_r10_c_ {
    struct setup_s_ {
      uint8_t add_spec_emission_pcell_r10 = 1;
    };
    typedef setup_e types;

    // choice methods
    add_spec_emission_ca_r10_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "additionalSpectrumEmissionCA-r10");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "additionalSpectrumEmissionCA-r10");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct ce_mode_r13_c_ {
    struct setup_opts {
      enum options { ce_mode_a, ce_mode_b, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<setup_opts> setup_e_;
    typedef setup_e                types;

    // choice methods
    ce_mode_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_e_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "ce-Mode-r13");
      return c;
    }
    const setup_e_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "ce-Mode-r13");
      return c;
    }
    setup_e_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_e_ c;
  };
  struct type_a_srs_tpc_pdcch_group_r14_c_ {
    using setup_l_ = dyn_array<srs_tpc_pdcch_cfg_r14_c>;
    typedef setup_e types;

    // choice methods
    type_a_srs_tpc_pdcch_group_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_l_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "typeA-SRS-TPC-PDCCH-Group-r14");
      return c;
    }
    const setup_l_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "typeA-SRS-TPC-PDCCH-Group-r14");
      return c;
    }
    setup_l_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_l_ c;
  };
  struct must_cfg_r14_c_ {
    struct setup_s_ {
      struct k_max_r14_opts {
        enum options { l1, l3, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<k_max_r14_opts> k_max_r14_e_;
      struct p_a_must_r14_opts {
        enum options { db_minus6, db_minus4dot77, db_minus3, db_minus1dot77, db0, db1, db2, db3, nulltype } value;
        typedef float number_type;

        std::string to_string() const;
        float       to_number() const;
        std::string to_number_string() const;
      };
      typedef enumerated<p_a_must_r14_opts> p_a_must_r14_e_;

      // member variables
      bool            p_a_must_r14_present = false;
      k_max_r14_e_    k_max_r14;
      p_a_must_r14_e_ p_a_must_r14;
    };
    typedef setup_e types;

    // choice methods
    must_cfg_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "must-Config-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "must-Config-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  using srs_ul_periodic_cfg_ded_list_r14_l_            = dyn_array<srs_ul_cfg_ded_c>;
  using srs_ul_periodic_cfg_ded_up_pts_ext_list_r14_l_ = dyn_array<srs_ul_cfg_ded_up_pts_ext_r13_c>;
  using srs_ul_aperiodic_cfg_ded_list_r14_l_           = dyn_array<srs_ul_cfg_ded_aperiodic_r10_c>;
  using srs_ul_cfg_ded_ap_up_pts_ext_list_r14_l_       = dyn_array<srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c>;
  struct semi_static_cfi_cfg_r15_c_ {
    struct setup_c_ {
      struct types_opts {
        enum options { cfi_cfg_r15, cfi_pattern_cfg_r15, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      cfi_cfg_r15_s& cfi_cfg_r15()
      {
        assert_choice_type("cfi-Config-r15", type_.to_string(), "setup");
        return c.get<cfi_cfg_r15_s>();
      }
      cfi_pattern_cfg_r15_s& cfi_pattern_cfg_r15()
      {
        assert_choice_type("cfi-PatternConfig-r15", type_.to_string(), "setup");
        return c.get<cfi_pattern_cfg_r15_s>();
      }
      const cfi_cfg_r15_s& cfi_cfg_r15() const
      {
        assert_choice_type("cfi-Config-r15", type_.to_string(), "setup");
        return c.get<cfi_cfg_r15_s>();
      }
      const cfi_pattern_cfg_r15_s& cfi_pattern_cfg_r15() const
      {
        assert_choice_type("cfi-PatternConfig-r15", type_.to_string(), "setup");
        return c.get<cfi_pattern_cfg_r15_s>();
      }
      cfi_cfg_r15_s& set_cfi_cfg_r15()
      {
        set(types::cfi_cfg_r15);
        return c.get<cfi_cfg_r15_s>();
      }
      cfi_pattern_cfg_r15_s& set_cfi_pattern_cfg_r15()
      {
        set(types::cfi_pattern_cfg_r15);
        return c.get<cfi_pattern_cfg_r15_s>();
      }

    private:
      types                                                 type_;
      choice_buffer_t<cfi_cfg_r15_s, cfi_pattern_cfg_r15_s> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    semi_static_cfi_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "semiStaticCFI-Config-r15");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "semiStaticCFI-Config-r15");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct blind_pdsch_repeat_cfg_r15_c_ {
    struct setup_s_ {
      struct max_num_sf_pdsch_repeats_r15_opts {
        enum options { n4, n6, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<max_num_sf_pdsch_repeats_r15_opts> max_num_sf_pdsch_repeats_r15_e_;
      struct max_num_slot_subslot_pdsch_repeats_r15_opts {
        enum options { n4, n6, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<max_num_slot_subslot_pdsch_repeats_r15_opts> max_num_slot_subslot_pdsch_repeats_r15_e_;
      struct rv_sf_pdsch_repeats_r15_opts {
        enum options { dlrvseq1, dlrvseq2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<rv_sf_pdsch_repeats_r15_opts> rv_sf_pdsch_repeats_r15_e_;
      struct rv_slotsublot_pdsch_repeats_r15_opts {
        enum options { dlrvseq1, dlrvseq2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<rv_slotsublot_pdsch_repeats_r15_opts> rv_slotsublot_pdsch_repeats_r15_e_;
      struct mcs_restrict_sf_pdsch_repeats_r15_opts {
        enum options { n0, n1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<mcs_restrict_sf_pdsch_repeats_r15_opts> mcs_restrict_sf_pdsch_repeats_r15_e_;
      struct mcs_restrict_slot_subslot_pdsch_repeats_r15_opts {
        enum options { n0, n1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<mcs_restrict_slot_subslot_pdsch_repeats_r15_opts>
          mcs_restrict_slot_subslot_pdsch_repeats_r15_e_;

      // member variables
      bool                                           max_num_sf_pdsch_repeats_r15_present                 = false;
      bool                                           max_num_slot_subslot_pdsch_repeats_r15_present       = false;
      bool                                           rv_sf_pdsch_repeats_r15_present                      = false;
      bool                                           rv_slotsublot_pdsch_repeats_r15_present              = false;
      bool                                           nof_processes_sf_pdsch_repeats_r15_present           = false;
      bool                                           nof_processes_slot_subslot_pdsch_repeats_r15_present = false;
      bool                                           mcs_restrict_sf_pdsch_repeats_r15_present            = false;
      bool                                           mcs_restrict_slot_subslot_pdsch_repeats_r15_present  = false;
      bool                                           blind_sf_pdsch_repeats_r15                           = false;
      bool                                           blind_slot_subslot_pdsch_repeats_r15                 = false;
      max_num_sf_pdsch_repeats_r15_e_                max_num_sf_pdsch_repeats_r15;
      max_num_slot_subslot_pdsch_repeats_r15_e_      max_num_slot_subslot_pdsch_repeats_r15;
      rv_sf_pdsch_repeats_r15_e_                     rv_sf_pdsch_repeats_r15;
      rv_slotsublot_pdsch_repeats_r15_e_             rv_slotsublot_pdsch_repeats_r15;
      uint8_t                                        nof_processes_sf_pdsch_repeats_r15           = 1;
      uint8_t                                        nof_processes_slot_subslot_pdsch_repeats_r15 = 1;
      mcs_restrict_sf_pdsch_repeats_r15_e_           mcs_restrict_sf_pdsch_repeats_r15;
      mcs_restrict_slot_subslot_pdsch_repeats_r15_e_ mcs_restrict_slot_subslot_pdsch_repeats_r15;
    };
    typedef setup_e types;

    // choice methods
    blind_pdsch_repeat_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "blindPDSCH-Repetition-Config-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "blindPDSCH-Repetition-Config-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                ext                         = false;
  bool                pdsch_cfg_ded_present       = false;
  bool                pucch_cfg_ded_present       = false;
  bool                pusch_cfg_ded_present       = false;
  bool                ul_pwr_ctrl_ded_present     = false;
  bool                tpc_pdcch_cfg_pucch_present = false;
  bool                tpc_pdcch_cfg_pusch_present = false;
  bool                cqi_report_cfg_present      = false;
  bool                srs_ul_cfg_ded_present      = false;
  bool                ant_info_present            = false;
  bool                sched_request_cfg_present   = false;
  pdsch_cfg_ded_s     pdsch_cfg_ded;
  pucch_cfg_ded_s     pucch_cfg_ded;
  pusch_cfg_ded_s     pusch_cfg_ded;
  ul_pwr_ctrl_ded_s   ul_pwr_ctrl_ded;
  tpc_pdcch_cfg_c     tpc_pdcch_cfg_pucch;
  tpc_pdcch_cfg_c     tpc_pdcch_cfg_pusch;
  cqi_report_cfg_s    cqi_report_cfg;
  srs_ul_cfg_ded_c    srs_ul_cfg_ded;
  ant_info_c_         ant_info;
  sched_request_cfg_c sched_request_cfg;
  // ...
  // group 0
  copy_ptr<cqi_report_cfg_v920_s> cqi_report_cfg_v920;
  copy_ptr<ant_info_ded_v920_s>   ant_info_v920;
  // group 1
  bool                                     cif_presence_r10_present = false;
  copy_ptr<ant_info_r10_c_>                ant_info_r10;
  copy_ptr<ant_info_ul_r10_s>              ant_info_ul_r10;
  bool                                     cif_presence_r10 = false;
  copy_ptr<cqi_report_cfg_r10_s>           cqi_report_cfg_r10;
  copy_ptr<csi_rs_cfg_r10_s>               csi_rs_cfg_r10;
  copy_ptr<pucch_cfg_ded_v1020_s>          pucch_cfg_ded_v1020;
  copy_ptr<pusch_cfg_ded_v1020_s>          pusch_cfg_ded_v1020;
  copy_ptr<sched_request_cfg_v1020_s>      sched_request_cfg_v1020;
  copy_ptr<srs_ul_cfg_ded_v1020_s>         srs_ul_cfg_ded_v1020;
  copy_ptr<srs_ul_cfg_ded_aperiodic_r10_c> srs_ul_cfg_ded_aperiodic_r10;
  copy_ptr<ul_pwr_ctrl_ded_v1020_s>        ul_pwr_ctrl_ded_v1020;
  // group 2
  copy_ptr<add_spec_emission_ca_r10_c_> add_spec_emission_ca_r10;
  // group 3
  copy_ptr<csi_rs_cfg_nzp_to_release_list_r11_l> csi_rs_cfg_nzp_to_release_list_r11;
  copy_ptr<csi_rs_cfg_nzp_to_add_mod_list_r11_l> csi_rs_cfg_nzp_to_add_mod_list_r11;
  copy_ptr<csi_rs_cfg_zp_to_release_list_r11_l>  csi_rs_cfg_zp_to_release_list_r11;
  copy_ptr<csi_rs_cfg_zp_to_add_mod_list_r11_l>  csi_rs_cfg_zp_to_add_mod_list_r11;
  copy_ptr<epdcch_cfg_r11_s>                     epdcch_cfg_r11;
  copy_ptr<pdsch_cfg_ded_v1130_s>                pdsch_cfg_ded_v1130;
  copy_ptr<cqi_report_cfg_v1130_s>               cqi_report_cfg_v1130;
  copy_ptr<pucch_cfg_ded_v1130_s>                pucch_cfg_ded_v1130;
  copy_ptr<pusch_cfg_ded_v1130_s>                pusch_cfg_ded_v1130;
  copy_ptr<ul_pwr_ctrl_ded_v1130_s>              ul_pwr_ctrl_ded_v1130;
  // group 4
  copy_ptr<ant_info_ded_v1250_s>           ant_info_v1250;
  copy_ptr<eimta_main_cfg_r12_c>           eimta_main_cfg_r12;
  copy_ptr<eimta_main_cfg_serv_cell_r12_c> eimta_main_cfg_pcell_r12;
  copy_ptr<pucch_cfg_ded_v1250_s>          pucch_cfg_ded_v1250;
  copy_ptr<cqi_report_cfg_v1250_s>         cqi_report_cfg_pcell_v1250;
  copy_ptr<ul_pwr_ctrl_ded_v1250_s>        ul_pwr_ctrl_ded_v1250;
  copy_ptr<pusch_cfg_ded_v1250_s>          pusch_cfg_ded_v1250;
  copy_ptr<csi_rs_cfg_v1250_s>             csi_rs_cfg_v1250;
  // group 5
  copy_ptr<pdsch_cfg_ded_v1280_s> pdsch_cfg_ded_v1280;
  // group 6
  copy_ptr<pdsch_cfg_ded_v1310_s>                     pdsch_cfg_ded_v1310;
  copy_ptr<pucch_cfg_ded_r13_s>                       pucch_cfg_ded_r13;
  copy_ptr<pusch_cfg_ded_r13_s>                       pusch_cfg_ded_r13;
  copy_ptr<pdcch_candidate_reductions_r13_c>          pdcch_candidate_reductions_r13;
  copy_ptr<cqi_report_cfg_v1310_s>                    cqi_report_cfg_v1310;
  copy_ptr<srs_ul_cfg_ded_v1310_c>                    srs_ul_cfg_ded_v1310;
  copy_ptr<srs_ul_cfg_ded_up_pts_ext_r13_c>           srs_ul_cfg_ded_up_pts_ext_r13;
  copy_ptr<srs_ul_cfg_ded_aperiodic_v1310_c>          srs_ul_cfg_ded_aperiodic_v1310;
  copy_ptr<srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c> srs_ul_cfg_ded_aperiodic_up_pts_ext_r13;
  copy_ptr<csi_rs_cfg_v1310_s>                        csi_rs_cfg_v1310;
  copy_ptr<ce_mode_r13_c_>                            ce_mode_r13;
  copy_ptr<csi_rs_cfg_nzp_to_add_mod_list_ext_r13_l>  csi_rs_cfg_nzp_to_add_mod_list_ext_r13;
  copy_ptr<csi_rs_cfg_nzp_to_release_list_ext_r13_l>  csi_rs_cfg_nzp_to_release_list_ext_r13;
  // group 7
  copy_ptr<cqi_report_cfg_v1320_s> cqi_report_cfg_v1320;
  // group 8
  bool                                                     ce_pdsch_pusch_enhancement_cfg_r14_present = false;
  bool                                                     semi_open_loop_r14_present                 = false;
  copy_ptr<type_a_srs_tpc_pdcch_group_r14_c_>              type_a_srs_tpc_pdcch_group_r14;
  copy_ptr<must_cfg_r14_c_>                                must_cfg_r14;
  copy_ptr<pusch_enhance_cfg_r14_c>                        pusch_enhance_cfg_r14;
  copy_ptr<ant_info_ded_v1430_s>                           ant_info_v1430;
  copy_ptr<pucch_cfg_ded_v1430_s>                          pucch_cfg_ded_v1430;
  copy_ptr<pdsch_cfg_ded_v1430_s>                          pdsch_cfg_ded_v1430;
  copy_ptr<pusch_cfg_ded_v1430_s>                          pusch_cfg_ded_v1430;
  copy_ptr<srs_ul_periodic_cfg_ded_list_r14_l_>            srs_ul_periodic_cfg_ded_list_r14;
  copy_ptr<srs_ul_periodic_cfg_ded_up_pts_ext_list_r14_l_> srs_ul_periodic_cfg_ded_up_pts_ext_list_r14;
  copy_ptr<srs_ul_aperiodic_cfg_ded_list_r14_l_>           srs_ul_aperiodic_cfg_ded_list_r14;
  copy_ptr<srs_ul_cfg_ded_ap_up_pts_ext_list_r14_l_>       srs_ul_cfg_ded_ap_up_pts_ext_list_r14;
  copy_ptr<csi_rs_cfg_v1430_s>                             csi_rs_cfg_v1430;
  copy_ptr<csi_rs_cfg_zp_ap_list_r14_c>                    csi_rs_cfg_zp_ap_list_r14;
  copy_ptr<cqi_report_cfg_v1430_s>                         cqi_report_cfg_v1430;
  bool                                                     semi_open_loop_r14 = false;
  // group 9
  copy_ptr<csi_rs_cfg_v1480_s> csi_rs_cfg_v1480;
  // group 10
  copy_ptr<phys_cfg_ded_stti_r15_c>       phys_cfg_ded_stti_r15;
  copy_ptr<pdsch_cfg_ded_v1530_s>         pdsch_cfg_ded_v1530;
  copy_ptr<pusch_cfg_ded_v1530_s>         pusch_cfg_ded_v1530;
  copy_ptr<cqi_report_cfg_v1530_s>        cqi_report_cfg_v1530;
  copy_ptr<ant_info_ded_v1530_c>          ant_info_v1530;
  copy_ptr<csi_rs_cfg_v1530_s>            csi_rs_cfg_v1530;
  copy_ptr<ul_pwr_ctrl_ded_v1530_s>       ul_pwr_ctrl_ded_v1530;
  copy_ptr<semi_static_cfi_cfg_r15_c_>    semi_static_cfi_cfg_r15;
  copy_ptr<blind_pdsch_repeat_cfg_r15_c_> blind_pdsch_repeat_cfg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RLF-TimersAndConstants-r13 ::= CHOICE
struct rlf_timers_and_consts_r13_c {
  struct setup_s_ {
    struct t301_v1310_opts {
      enum options { ms2500, ms3000, ms3500, ms4000, ms5000, ms6000, ms8000, ms10000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t301_v1310_opts> t301_v1310_e_;
    struct t310_v1330_opts {
      enum options { ms4000, ms6000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t310_v1330_opts> t310_v1330_e_;

    // member variables
    bool          ext = false;
    t301_v1310_e_ t301_v1310;
    // ...
    // group 0
    bool          t310_v1330_present = false;
    t310_v1330_e_ t310_v1330;

    // sequence methods
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
  };
  typedef setup_e types;

  // choice methods
  rlf_timers_and_consts_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstants-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstants-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RLF-TimersAndConstants-r9 ::= CHOICE
struct rlf_timers_and_consts_r9_c {
  struct setup_s_ {
    struct t301_r9_opts {
      enum options { ms100, ms200, ms300, ms400, ms600, ms1000, ms1500, ms2000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t301_r9_opts> t301_r9_e_;
    struct t310_r9_opts {
      enum options { ms0, ms50, ms100, ms200, ms500, ms1000, ms2000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t310_r9_opts> t310_r9_e_;
    struct n310_r9_opts {
      enum options { n1, n2, n3, n4, n6, n8, n10, n20, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<n310_r9_opts> n310_r9_e_;
    struct t311_r9_opts {
      enum options { ms1000, ms3000, ms5000, ms10000, ms15000, ms20000, ms30000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t311_r9_opts> t311_r9_e_;
    struct n311_r9_opts {
      enum options { n1, n2, n3, n4, n5, n6, n8, n10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<n311_r9_opts> n311_r9_e_;

    // member variables
    bool       ext = false;
    t301_r9_e_ t301_r9;
    t310_r9_e_ t310_r9;
    n310_r9_e_ n310_r9;
    t311_r9_e_ t311_r9;
    n311_r9_e_ n311_r9;
    // ...
  };
  typedef setup_e types;

  // choice methods
  rlf_timers_and_consts_r9_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstants-r9");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstants-r9");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReject-v1020-IEs ::= SEQUENCE
struct rrc_conn_reject_v1020_ies_s {
  bool                        extended_wait_time_r10_present = false;
  bool                        non_crit_ext_present           = false;
  uint16_t                    extended_wait_time_r10         = 1;
  rrc_conn_reject_v1130_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPS-Config ::= SEQUENCE
struct sps_cfg_s {
  bool                semi_persist_sched_c_rnti_present = false;
  bool                sps_cfg_dl_present                = false;
  bool                sps_cfg_ul_present                = false;
  fixed_bitstring<16> semi_persist_sched_c_rnti;
  sps_cfg_dl_c        sps_cfg_dl;
  sps_cfg_ul_c        sps_cfg_ul;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPS-Config-v1430 ::= SEQUENCE
struct sps_cfg_v1430_s {
  bool                             ul_sps_v_rnti_r14_present              = false;
  bool                             sl_sps_v_rnti_r14_present              = false;
  bool                             sps_cfg_ul_to_add_mod_list_r14_present = false;
  bool                             sps_cfg_ul_to_release_list_r14_present = false;
  bool                             sps_cfg_sl_to_add_mod_list_r14_present = false;
  bool                             sps_cfg_sl_to_release_list_r14_present = false;
  fixed_bitstring<16>              ul_sps_v_rnti_r14;
  fixed_bitstring<16>              sl_sps_v_rnti_r14;
  sps_cfg_ul_to_add_mod_list_r14_l sps_cfg_ul_to_add_mod_list_r14;
  sps_cfg_ul_to_release_list_r14_l sps_cfg_ul_to_release_list_r14;
  sps_cfg_sl_to_add_mod_list_r14_l sps_cfg_sl_to_add_mod_list_r14;
  sps_cfg_sl_to_release_list_r14_l sps_cfg_sl_to_release_list_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SPS-Config-v1530 ::= SEQUENCE
struct sps_cfg_v1530_s {
  bool                                  semi_persist_sched_c_rnti_r15_present       = false;
  bool                                  sps_cfg_dl_present                          = false;
  bool                                  sps_cfg_ul_stti_to_add_mod_list_r15_present = false;
  bool                                  sps_cfg_ul_stti_to_release_list_r15_present = false;
  bool                                  sps_cfg_ul_to_add_mod_list_r15_present      = false;
  bool                                  sps_cfg_ul_to_release_list_r15_present      = false;
  fixed_bitstring<16>                   semi_persist_sched_c_rnti_r15;
  sps_cfg_dl_c                          sps_cfg_dl;
  sps_cfg_ul_stti_to_add_mod_list_r15_l sps_cfg_ul_stti_to_add_mod_list_r15;
  sps_cfg_ul_stti_to_release_list_r15_l sps_cfg_ul_stti_to_release_list_r15;
  sps_cfg_ul_to_add_mod_list_r15_l      sps_cfg_ul_to_add_mod_list_r15;
  sps_cfg_ul_to_release_list_r15_l      sps_cfg_ul_to_release_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SRB-ToAddModExtList-r15 ::= SEQUENCE (SIZE (1)) OF SRB-ToAddMod
using srb_to_add_mod_ext_list_r15_l = std::array<srb_to_add_mod_s, 1>;

// SRB-ToAddModList ::= SEQUENCE (SIZE (1..2)) OF SRB-ToAddMod
using srb_to_add_mod_list_l = dyn_array<srb_to_add_mod_s>;

// SRB-ToReleaseListDupl-r15 ::= SEQUENCE (SIZE (1..2)) OF INTEGER (1..2)
using srb_to_release_list_dupl_r15_l = bounded_array<uint8_t, 2>;

// IdleModeMobilityControlInfo ::= SEQUENCE
struct idle_mode_mob_ctrl_info_s {
  struct t320_opts {
    enum options { min5, min10, min20, min30, min60, min120, min180, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<t320_opts> t320_e_;

  // member variables
  bool                         ext                                = false;
  bool                         freq_prio_list_eutra_present       = false;
  bool                         freq_prio_list_geran_present       = false;
  bool                         freq_prio_list_utra_fdd_present    = false;
  bool                         freq_prio_list_utra_tdd_present    = false;
  bool                         band_class_prio_list_hrpd_present  = false;
  bool                         band_class_prio_list1_xrtt_present = false;
  bool                         t320_present                       = false;
  freq_prio_list_eutra_l       freq_prio_list_eutra;
  freqs_prio_list_geran_l      freq_prio_list_geran;
  freq_prio_list_utra_fdd_l    freq_prio_list_utra_fdd;
  freq_prio_list_utra_tdd_l    freq_prio_list_utra_tdd;
  band_class_prio_list_hrpd_l  band_class_prio_list_hrpd;
  band_class_prio_list1_xrtt_l band_class_prio_list1_xrtt;
  t320_e_                      t320;
  // ...
  // group 0
  copy_ptr<freq_prio_list_ext_eutra_r12_l> freq_prio_list_ext_eutra_r12;
  // group 1
  copy_ptr<freq_prio_list_eutra_v1310_l>     freq_prio_list_eutra_v1310;
  copy_ptr<freq_prio_list_ext_eutra_v1310_l> freq_prio_list_ext_eutra_v1310;
  // group 2
  copy_ptr<freq_prio_list_nr_r15_l> freq_prio_list_nr_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// IdleModeMobilityControlInfo-v9e0 ::= SEQUENCE
struct idle_mode_mob_ctrl_info_v9e0_s {
  using freq_prio_list_eutra_v9e0_l_ = dyn_array<freq_prio_eutra_v9e0_s>;

  // member variables
  freq_prio_list_eutra_v9e0_l_ freq_prio_list_eutra_v9e0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReestablishment-v8a0-IEs ::= SEQUENCE
struct rrc_conn_reest_v8a0_ies_s {
  bool          late_non_crit_ext_present = false;
  bool          non_crit_ext_present      = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReestablishmentReject-v8a0-IEs ::= SEQUENCE
struct rrc_conn_reest_reject_v8a0_ies_s {
  bool          late_non_crit_ext_present = false;
  bool          non_crit_ext_present      = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReject-v8a0-IEs ::= SEQUENCE
struct rrc_conn_reject_v8a0_ies_s {
  bool                        late_non_crit_ext_present = false;
  bool                        non_crit_ext_present      = false;
  dyn_octstring               late_non_crit_ext;
  rrc_conn_reject_v1020_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionSetup-v8a0-IEs ::= SEQUENCE
struct rrc_conn_setup_v8a0_ies_s {
  bool          late_non_crit_ext_present = false;
  bool          non_crit_ext_present      = false;
  dyn_octstring late_non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicated ::= SEQUENCE
struct rr_cfg_ded_s {
  struct mac_main_cfg_c_ {
    struct types_opts {
      enum options { explicit_value, default_value, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    mac_main_cfg_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    mac_main_cfg_s& explicit_value()
    {
      assert_choice_type("explicitValue", type_.to_string(), "mac-MainConfig");
      return c;
    }
    const mac_main_cfg_s& explicit_value() const
    {
      assert_choice_type("explicitValue", type_.to_string(), "mac-MainConfig");
      return c;
    }
    mac_main_cfg_s& set_explicit_value()
    {
      set(types::explicit_value);
      return c;
    }

  private:
    types          type_;
    mac_main_cfg_s c;
  };
  struct crs_intf_mitig_cfg_r15_c_ {
    struct setup_c_ {
      struct crs_intf_mitig_num_prbs_r15_opts {
        enum options { n6, n24, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<crs_intf_mitig_num_prbs_r15_opts> crs_intf_mitig_num_prbs_r15_e_;
      struct types_opts {
        enum options { crs_intf_mitig_enabled_minus15, crs_intf_mitig_num_prbs_r15, nulltype } value;
        typedef int8_t number_type;

        std::string to_string() const;
        int8_t      to_number() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      crs_intf_mitig_num_prbs_r15_e_& crs_intf_mitig_num_prbs_r15()
      {
        assert_choice_type("crs-IntfMitigNumPRBs-r15", type_.to_string(), "setup");
        return c;
      }
      const crs_intf_mitig_num_prbs_r15_e_& crs_intf_mitig_num_prbs_r15() const
      {
        assert_choice_type("crs-IntfMitigNumPRBs-r15", type_.to_string(), "setup");
        return c;
      }
      crs_intf_mitig_num_prbs_r15_e_& set_crs_intf_mitig_num_prbs_r15()
      {
        set(types::crs_intf_mitig_num_prbs_r15);
        return c;
      }

    private:
      types                          type_;
      crs_intf_mitig_num_prbs_r15_e_ c;
    };
    typedef setup_e types;

    // choice methods
    crs_intf_mitig_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "crs-IntfMitigConfig-r15");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "crs-IntfMitigConfig-r15");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };

  // member variables
  bool                  ext                         = false;
  bool                  srb_to_add_mod_list_present = false;
  bool                  drb_to_add_mod_list_present = false;
  bool                  drb_to_release_list_present = false;
  bool                  mac_main_cfg_present        = false;
  bool                  sps_cfg_present             = false;
  bool                  phys_cfg_ded_present        = false;
  srb_to_add_mod_list_l srb_to_add_mod_list;
  drb_to_add_mod_list_l drb_to_add_mod_list;
  drb_to_release_list_l drb_to_release_list;
  mac_main_cfg_c_       mac_main_cfg;
  sps_cfg_s             sps_cfg;
  phys_cfg_ded_s        phys_cfg_ded;
  // ...
  // group 0
  copy_ptr<rlf_timers_and_consts_r9_c> rlf_timers_and_consts_r9;
  // group 1
  copy_ptr<meas_sf_pattern_pcell_r10_c> meas_sf_pattern_pcell_r10;
  // group 2
  copy_ptr<neigh_cells_crs_info_r11_c> neigh_cells_crs_info_r11;
  // group 3
  copy_ptr<naics_assist_info_r12_c> naics_info_r12;
  // group 4
  copy_ptr<neigh_cells_crs_info_r13_c>  neigh_cells_crs_info_r13;
  copy_ptr<rlf_timers_and_consts_r13_c> rlf_timers_and_consts_r13;
  // group 5
  copy_ptr<sps_cfg_v1430_s> sps_cfg_v1430;
  // group 6
  bool                                     srb_to_release_ext_list_r15_present = false;
  copy_ptr<srb_to_add_mod_ext_list_r15_l>  srb_to_add_mod_ext_list_r15;
  uint8_t                                  srb_to_release_ext_list_r15 = 4;
  copy_ptr<sps_cfg_v1530_s>                sps_cfg_v1530;
  copy_ptr<crs_intf_mitig_cfg_r15_c_>      crs_intf_mitig_cfg_r15;
  copy_ptr<neigh_cells_crs_info_r15_c>     neigh_cells_crs_info_r15;
  copy_ptr<drb_to_add_mod_list_r15_l>      drb_to_add_mod_list_r15;
  copy_ptr<drb_to_release_list_r15_l>      drb_to_release_list_r15;
  copy_ptr<srb_to_release_list_dupl_r15_l> srb_to_release_list_dupl_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RedirectedCarrierInfo-r15-IEs ::= CHOICE
struct redirected_carrier_info_r15_ies_c {
  struct types_opts {
    enum options {
      eutra_r15,
      geran_r15,
      utra_fdd_r15,
      cdma2000_hrpd_r15,
      cdma2000_minus1x_rtt_r15,
      utra_tdd_r15,
      nulltype
    } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  redirected_carrier_info_r15_ies_c() = default;
  redirected_carrier_info_r15_ies_c(const redirected_carrier_info_r15_ies_c& other);
  redirected_carrier_info_r15_ies_c& operator=(const redirected_carrier_info_r15_ies_c& other);
  ~redirected_carrier_info_r15_ies_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  uint32_t& eutra_r15()
  {
    assert_choice_type("eutra-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<uint32_t>();
  }
  carrier_freqs_geran_s& geran_r15()
  {
    assert_choice_type("geran-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freqs_geran_s>();
  }
  uint16_t& utra_fdd_r15()
  {
    assert_choice_type("utra-FDD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<uint16_t>();
  }
  carrier_freq_cdma2000_s& cdma2000_hrpd_r15()
  {
    assert_choice_type("cdma2000-HRPD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_cdma2000_s>();
  }
  carrier_freq_cdma2000_s& cdma2000_minus1x_rtt_r15()
  {
    assert_choice_type("cdma2000-1xRTT-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_cdma2000_s>();
  }
  carrier_freq_list_utra_tdd_r10_l& utra_tdd_r15()
  {
    assert_choice_type("utra-TDD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_list_utra_tdd_r10_l>();
  }
  const uint32_t& eutra_r15() const
  {
    assert_choice_type("eutra-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<uint32_t>();
  }
  const carrier_freqs_geran_s& geran_r15() const
  {
    assert_choice_type("geran-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freqs_geran_s>();
  }
  const uint16_t& utra_fdd_r15() const
  {
    assert_choice_type("utra-FDD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<uint16_t>();
  }
  const carrier_freq_cdma2000_s& cdma2000_hrpd_r15() const
  {
    assert_choice_type("cdma2000-HRPD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_cdma2000_s>();
  }
  const carrier_freq_cdma2000_s& cdma2000_minus1x_rtt_r15() const
  {
    assert_choice_type("cdma2000-1xRTT-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_cdma2000_s>();
  }
  const carrier_freq_list_utra_tdd_r10_l& utra_tdd_r15() const
  {
    assert_choice_type("utra-TDD-r15", type_.to_string(), "RedirectedCarrierInfo-r15-IEs");
    return c.get<carrier_freq_list_utra_tdd_r10_l>();
  }
  uint32_t& set_eutra_r15()
  {
    set(types::eutra_r15);
    return c.get<uint32_t>();
  }
  carrier_freqs_geran_s& set_geran_r15()
  {
    set(types::geran_r15);
    return c.get<carrier_freqs_geran_s>();
  }
  uint16_t& set_utra_fdd_r15()
  {
    set(types::utra_fdd_r15);
    return c.get<uint16_t>();
  }
  carrier_freq_cdma2000_s& set_cdma2000_hrpd_r15()
  {
    set(types::cdma2000_hrpd_r15);
    return c.get<carrier_freq_cdma2000_s>();
  }
  carrier_freq_cdma2000_s& set_cdma2000_minus1x_rtt_r15()
  {
    set(types::cdma2000_minus1x_rtt_r15);
    return c.get<carrier_freq_cdma2000_s>();
  }
  carrier_freq_list_utra_tdd_r10_l& set_utra_tdd_r15()
  {
    set(types::utra_tdd_r15);
    return c.get<carrier_freq_list_utra_tdd_r10_l>();
  }

private:
  types                                                                                             type_;
  choice_buffer_t<carrier_freq_cdma2000_s, carrier_freq_list_utra_tdd_r10_l, carrier_freqs_geran_s> c;

  void destroy_();
};

// RRCConnectionReestablishment-r8-IEs ::= SEQUENCE
struct rrc_conn_reest_r8_ies_s {
  bool                      non_crit_ext_present = false;
  rr_cfg_ded_s              rr_cfg_ded;
  uint8_t                   next_hop_chaining_count = 0;
  rrc_conn_reest_v8a0_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReestablishmentReject-r8-IEs ::= SEQUENCE
struct rrc_conn_reest_reject_r8_ies_s {
  bool                             non_crit_ext_present = false;
  rrc_conn_reest_reject_v8a0_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReject-r8-IEs ::= SEQUENCE
struct rrc_conn_reject_r8_ies_s {
  bool                       non_crit_ext_present = false;
  uint8_t                    wait_time            = 1;
  rrc_conn_reject_v8a0_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionSetup-r8-IEs ::= SEQUENCE
struct rrc_conn_setup_r8_ies_s {
  bool                      non_crit_ext_present = false;
  rr_cfg_ded_s              rr_cfg_ded;
  rrc_conn_setup_v8a0_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCEarlyDataComplete-r15-IEs ::= SEQUENCE
struct rrc_early_data_complete_r15_ies_s {
  bool                              ded_info_nas_r15_present                = false;
  bool                              extended_wait_time_r15_present          = false;
  bool                              idle_mode_mob_ctrl_info_r15_present     = false;
  bool                              idle_mode_mob_ctrl_info_ext_r15_present = false;
  bool                              redirected_carrier_info_r15_present     = false;
  bool                              non_crit_ext_present                    = false;
  dyn_octstring                     ded_info_nas_r15;
  uint16_t                          extended_wait_time_r15 = 1;
  idle_mode_mob_ctrl_info_s         idle_mode_mob_ctrl_info_r15;
  idle_mode_mob_ctrl_info_v9e0_s    idle_mode_mob_ctrl_info_ext_r15;
  redirected_carrier_info_r15_ies_c redirected_carrier_info_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

struct c1_or_crit_ext_opts {
  enum options { c1, crit_exts_future, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<c1_or_crit_ext_opts> c1_or_crit_ext_e;

// RRCConnectionReestablishment ::= SEQUENCE
struct rrc_conn_reest_s {
  struct crit_exts_c_ {
    struct c1_c_ {
      struct types_opts {
        enum options { rrc_conn_reest_r8, spare7, spare6, spare5, spare4, spare3, spare2, spare1, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      c1_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      rrc_conn_reest_r8_ies_s& rrc_conn_reest_r8()
      {
        assert_choice_type("rrcConnectionReestablishment-r8", type_.to_string(), "c1");
        return c;
      }
      const rrc_conn_reest_r8_ies_s& rrc_conn_reest_r8() const
      {
        assert_choice_type("rrcConnectionReestablishment-r8", type_.to_string(), "c1");
        return c;
      }
      rrc_conn_reest_r8_ies_s& set_rrc_conn_reest_r8()
      {
        set(types::rrc_conn_reest_r8);
        return c;
      }

    private:
      types                   type_;
      rrc_conn_reest_r8_ies_s c;
    };
    typedef c1_or_crit_ext_e types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    c1_c_& c1()
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    const c1_c_& c1() const
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    c1_c_& set_c1()
    {
      set(types::c1);
      return c.get<c1_c_>();
    }

  private:
    types                  type_;
    choice_buffer_t<c1_c_> c;

    void destroy_();
  };

  // member variables
  uint8_t      rrc_transaction_id = 0;
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReestablishmentReject ::= SEQUENCE
struct rrc_conn_reest_reject_s {
  struct crit_exts_c_ {
    struct types_opts {
      enum options { rrc_conn_reest_reject_r8, crit_exts_future, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rrc_conn_reest_reject_r8_ies_s& rrc_conn_reest_reject_r8()
    {
      assert_choice_type("rrcConnectionReestablishmentReject-r8", type_.to_string(), "criticalExtensions");
      return c.get<rrc_conn_reest_reject_r8_ies_s>();
    }
    const rrc_conn_reest_reject_r8_ies_s& rrc_conn_reest_reject_r8() const
    {
      assert_choice_type("rrcConnectionReestablishmentReject-r8", type_.to_string(), "criticalExtensions");
      return c.get<rrc_conn_reest_reject_r8_ies_s>();
    }
    rrc_conn_reest_reject_r8_ies_s& set_rrc_conn_reest_reject_r8()
    {
      set(types::rrc_conn_reest_reject_r8);
      return c.get<rrc_conn_reest_reject_r8_ies_s>();
    }

  private:
    types                                           type_;
    choice_buffer_t<rrc_conn_reest_reject_r8_ies_s> c;

    void destroy_();
  };

  // member variables
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReject ::= SEQUENCE
struct rrc_conn_reject_s {
  struct crit_exts_c_ {
    struct c1_c_ {
      struct types_opts {
        enum options { rrc_conn_reject_r8, spare3, spare2, spare1, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      c1_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      rrc_conn_reject_r8_ies_s& rrc_conn_reject_r8()
      {
        assert_choice_type("rrcConnectionReject-r8", type_.to_string(), "c1");
        return c;
      }
      const rrc_conn_reject_r8_ies_s& rrc_conn_reject_r8() const
      {
        assert_choice_type("rrcConnectionReject-r8", type_.to_string(), "c1");
        return c;
      }
      rrc_conn_reject_r8_ies_s& set_rrc_conn_reject_r8()
      {
        set(types::rrc_conn_reject_r8);
        return c;
      }

    private:
      types                    type_;
      rrc_conn_reject_r8_ies_s c;
    };
    typedef c1_or_crit_ext_e types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    c1_c_& c1()
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    const c1_c_& c1() const
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    c1_c_& set_c1()
    {
      set(types::c1);
      return c.get<c1_c_>();
    }

  private:
    types                  type_;
    choice_buffer_t<c1_c_> c;

    void destroy_();
  };

  // member variables
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionSetup ::= SEQUENCE
struct rrc_conn_setup_s {
  struct crit_exts_c_ {
    struct c1_c_ {
      struct types_opts {
        enum options { rrc_conn_setup_r8, spare7, spare6, spare5, spare4, spare3, spare2, spare1, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      c1_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      rrc_conn_setup_r8_ies_s& rrc_conn_setup_r8()
      {
        assert_choice_type("rrcConnectionSetup-r8", type_.to_string(), "c1");
        return c;
      }
      const rrc_conn_setup_r8_ies_s& rrc_conn_setup_r8() const
      {
        assert_choice_type("rrcConnectionSetup-r8", type_.to_string(), "c1");
        return c;
      }
      rrc_conn_setup_r8_ies_s& set_rrc_conn_setup_r8()
      {
        set(types::rrc_conn_setup_r8);
        return c;
      }

    private:
      types                   type_;
      rrc_conn_setup_r8_ies_s c;
    };
    typedef c1_or_crit_ext_e types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    c1_c_& c1()
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    const c1_c_& c1() const
    {
      assert_choice_type("c1", type_.to_string(), "criticalExtensions");
      return c.get<c1_c_>();
    }
    c1_c_& set_c1()
    {
      set(types::c1);
      return c.get<c1_c_>();
    }

  private:
    types                  type_;
    choice_buffer_t<c1_c_> c;

    void destroy_();
  };

  // member variables
  uint8_t      rrc_transaction_id = 0;
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCEarlyDataComplete-r15 ::= SEQUENCE
struct rrc_early_data_complete_r15_s {
  struct crit_exts_c_ {
    struct types_opts {
      enum options { rrc_early_data_complete_r15, crit_exts_future, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    crit_exts_c_() = default;
    crit_exts_c_(const crit_exts_c_& other);
    crit_exts_c_& operator=(const crit_exts_c_& other);
    ~crit_exts_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rrc_early_data_complete_r15_ies_s& rrc_early_data_complete_r15()
    {
      assert_choice_type("rrcEarlyDataComplete-r15", type_.to_string(), "criticalExtensions");
      return c.get<rrc_early_data_complete_r15_ies_s>();
    }
    const rrc_early_data_complete_r15_ies_s& rrc_early_data_complete_r15() const
    {
      assert_choice_type("rrcEarlyDataComplete-r15", type_.to_string(), "criticalExtensions");
      return c.get<rrc_early_data_complete_r15_ies_s>();
    }
    rrc_early_data_complete_r15_ies_s& set_rrc_early_data_complete_r15()
    {
      set(types::rrc_early_data_complete_r15);
      return c.get<rrc_early_data_complete_r15_ies_s>();
    }

  private:
    types                                              type_;
    choice_buffer_t<rrc_early_data_complete_r15_ies_s> c;

    void destroy_();
  };

  // member variables
  crit_exts_c_ crit_exts;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DL-CCCH-MessageType ::= CHOICE
struct dl_ccch_msg_type_c {
  struct c1_c_ {
    struct types_opts {
      enum options { rrc_conn_reest, rrc_conn_reest_reject, rrc_conn_reject, rrc_conn_setup, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    c1_c_() = default;
    c1_c_(const c1_c_& other);
    c1_c_& operator=(const c1_c_& other);
    ~c1_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    rrc_conn_reest_s& rrc_conn_reest()
    {
      assert_choice_type("rrcConnectionReestablishment", type_.to_string(), "c1");
      return c.get<rrc_conn_reest_s>();
    }
    rrc_conn_reest_reject_s& rrc_conn_reest_reject()
    {
      assert_choice_type("rrcConnectionReestablishmentReject", type_.to_string(), "c1");
      return c.get<rrc_conn_reest_reject_s>();
    }
    rrc_conn_reject_s& rrc_conn_reject()
    {
      assert_choice_type("rrcConnectionReject", type_.to_string(), "c1");
      return c.get<rrc_conn_reject_s>();
    }
    rrc_conn_setup_s& rrc_conn_setup()
    {
      assert_choice_type("rrcConnectionSetup", type_.to_string(), "c1");
      return c.get<rrc_conn_setup_s>();
    }
    const rrc_conn_reest_s& rrc_conn_reest() const
    {
      assert_choice_type("rrcConnectionReestablishment", type_.to_string(), "c1");
      return c.get<rrc_conn_reest_s>();
    }
    const rrc_conn_reest_reject_s& rrc_conn_reest_reject() const
    {
      assert_choice_type("rrcConnectionReestablishmentReject", type_.to_string(), "c1");
      return c.get<rrc_conn_reest_reject_s>();
    }
    const rrc_conn_reject_s& rrc_conn_reject() const
    {
      assert_choice_type("rrcConnectionReject", type_.to_string(), "c1");
      return c.get<rrc_conn_reject_s>();
    }
    const rrc_conn_setup_s& rrc_conn_setup() const
    {
      assert_choice_type("rrcConnectionSetup", type_.to_string(), "c1");
      return c.get<rrc_conn_setup_s>();
    }
    rrc_conn_reest_s& set_rrc_conn_reest()
    {
      set(types::rrc_conn_reest);
      return c.get<rrc_conn_reest_s>();
    }
    rrc_conn_reest_reject_s& set_rrc_conn_reest_reject()
    {
      set(types::rrc_conn_reest_reject);
      return c.get<rrc_conn_reest_reject_s>();
    }
    rrc_conn_reject_s& set_rrc_conn_reject()
    {
      set(types::rrc_conn_reject);
      return c.get<rrc_conn_reject_s>();
    }
    rrc_conn_setup_s& set_rrc_conn_setup()
    {
      set(types::rrc_conn_setup);
      return c.get<rrc_conn_setup_s>();
    }

  private:
    types                                                                                           type_;
    choice_buffer_t<rrc_conn_reest_reject_s, rrc_conn_reest_s, rrc_conn_reject_s, rrc_conn_setup_s> c;

    void destroy_();
  };
  struct msg_class_ext_c_ {
    struct c2_c_ {
      struct types_opts {
        enum options { rrc_early_data_complete_r15, spare3, spare2, spare1, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      c2_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      rrc_early_data_complete_r15_s& rrc_early_data_complete_r15()
      {
        assert_choice_type("rrcEarlyDataComplete-r15", type_.to_string(), "c2");
        return c;
      }
      const rrc_early_data_complete_r15_s& rrc_early_data_complete_r15() const
      {
        assert_choice_type("rrcEarlyDataComplete-r15", type_.to_string(), "c2");
        return c;
      }
      rrc_early_data_complete_r15_s& set_rrc_early_data_complete_r15()
      {
        set(types::rrc_early_data_complete_r15);
        return c;
      }

    private:
      types                         type_;
      rrc_early_data_complete_r15_s c;
    };
    struct types_opts {
      enum options { c2, msg_class_ext_future_r15, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    msg_class_ext_c_() = default;
    msg_class_ext_c_(const msg_class_ext_c_& other);
    msg_class_ext_c_& operator=(const msg_class_ext_c_& other);
    ~msg_class_ext_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    c2_c_& c2()
    {
      assert_choice_type("c2", type_.to_string(), "messageClassExtension");
      return c.get<c2_c_>();
    }
    const c2_c_& c2() const
    {
      assert_choice_type("c2", type_.to_string(), "messageClassExtension");
      return c.get<c2_c_>();
    }
    c2_c_& set_c2()
    {
      set(types::c2);
      return c.get<c2_c_>();
    }

  private:
    types                  type_;
    choice_buffer_t<c2_c_> c;

    void destroy_();
  };
  struct types_opts {
    enum options { c1, msg_class_ext, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  dl_ccch_msg_type_c() = default;
  dl_ccch_msg_type_c(const dl_ccch_msg_type_c& other);
  dl_ccch_msg_type_c& operator=(const dl_ccch_msg_type_c& other);
  ~dl_ccch_msg_type_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  c1_c_& c1()
  {
    assert_choice_type("c1", type_.to_string(), "DL-CCCH-MessageType");
    return c.get<c1_c_>();
  }
  msg_class_ext_c_& msg_class_ext()
  {
    assert_choice_type("messageClassExtension", type_.to_string(), "DL-CCCH-MessageType");
    return c.get<msg_class_ext_c_>();
  }
  const c1_c_& c1() const
  {
    assert_choice_type("c1", type_.to_string(), "DL-CCCH-MessageType");
    return c.get<c1_c_>();
  }
  const msg_class_ext_c_& msg_class_ext() const
  {
    assert_choice_type("messageClassExtension", type_.to_string(), "DL-CCCH-MessageType");
    return c.get<msg_class_ext_c_>();
  }
  c1_c_& set_c1()
  {
    set(types::c1);
    return c.get<c1_c_>();
  }
  msg_class_ext_c_& set_msg_class_ext()
  {
    set(types::msg_class_ext);
    return c.get<msg_class_ext_c_>();
  }

private:
  types                                    type_;
  choice_buffer_t<c1_c_, msg_class_ext_c_> c;

  void destroy_();
};

// DL-CCCH-Message ::= SEQUENCE
struct dl_ccch_msg_s {
  dl_ccch_msg_type_c msg;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDCCH-CandidateReductionValue-r14 ::= ENUMERATED
struct pdcch_candidate_reduction_value_r14_opts {
  enum options { n0, n50, n100, n150, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<pdcch_candidate_reduction_value_r14_opts> pdcch_candidate_reduction_value_r14_e;

// PDCCH-CandidateReductionsLAA-UL-r14 ::= CHOICE
struct pdcch_candidate_reductions_laa_ul_r14_c {
  struct setup_s_ {
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al1_r14;
    pdcch_candidate_reduction_value_r13_e pdcch_candidate_reduction_al2_r14;
    pdcch_candidate_reduction_value_r14_e pdcch_candidate_reduction_al3_r14;
    pdcch_candidate_reduction_value_r14_e pdcch_candidate_reduction_al4_r14;
    pdcch_candidate_reduction_value_r14_e pdcch_candidate_reduction_al5_r14;
  };
  typedef setup_e types;

  // choice methods
  pdcch_candidate_reductions_laa_ul_r14_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "PDCCH-CandidateReductionsLAA-UL-r14");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "PDCCH-CandidateReductionsLAA-UL-r14");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// AUL-Config-r15 ::= CHOICE
struct aul_cfg_r15_c {
  struct setup_s_ {
    struct tx_mode_ul_aul_r15_opts {
      enum options { tm1, tm2, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<tx_mode_ul_aul_r15_opts> tx_mode_ul_aul_r15_e_;
    struct aul_start_partial_bw_inside_mcot_r15_opts {
      enum options { o34, o43, o52, o61, oos1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<aul_start_partial_bw_inside_mcot_r15_opts> aul_start_partial_bw_inside_mcot_r15_e_;
    struct aul_start_partial_bw_outside_mcot_r15_opts {
      enum options { o16, o25, o34, o43, o52, o61, oos1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<aul_start_partial_bw_outside_mcot_r15_opts> aul_start_partial_bw_outside_mcot_r15_e_;
    struct aul_retx_timer_r15_opts {
      enum options {
        psf4,
        psf5,
        psf6,
        psf8,
        psf10,
        psf12,
        psf20,
        psf28,
        psf37,
        psf44,
        psf68,
        psf84,
        psf100,
        psf116,
        psf132,
        psf164,
        psf324,
        nulltype
      } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<aul_retx_timer_r15_opts> aul_retx_timer_r15_e_;
    struct contention_win_size_timer_r15_opts {
      enum options { n0, n5, n10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<contention_win_size_timer_r15_opts> contention_win_size_timer_r15_e_;

    // member variables
    fixed_bitstring<16>                      aul_crnti_r15;
    fixed_bitstring<40>                      aul_sfs_r15;
    uint8_t                                  aul_harq_processes_r15 = 1;
    tx_mode_ul_aul_r15_e_                    tx_mode_ul_aul_r15;
    fixed_bitstring<5>                       aul_start_full_bw_inside_mcot_r15;
    fixed_bitstring<7>                       aul_start_full_bw_outside_mcot_r15;
    aul_start_partial_bw_inside_mcot_r15_e_  aul_start_partial_bw_inside_mcot_r15;
    aul_start_partial_bw_outside_mcot_r15_e_ aul_start_partial_bw_outside_mcot_r15;
    aul_retx_timer_r15_e_                    aul_retx_timer_r15;
    uint8_t                                  ending_symbol_aul_r15     = 12;
    uint8_t                                  sf_offset_cot_sharing_r15 = 2;
    contention_win_size_timer_r15_e_         contention_win_size_timer_r15;
  };
  typedef setup_e types;

  // choice methods
  aul_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "AUL-Config-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "AUL-Config-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportPeriodicSCell-r15 ::= CHOICE
struct cqi_report_periodic_scell_r15_c {
  struct setup_s_ {
    struct csi_sf_pattern_dormant_r15_c_ {
      struct setup_s_ {
        meas_sf_pattern_r10_c csi_meas_sf_set1_r15;
        meas_sf_pattern_r10_c csi_meas_sf_set2_r15;
      };
      typedef setup_e types;

      // choice methods
      csi_sf_pattern_dormant_r15_c_() = default;
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      setup_s_& setup()
      {
        assert_choice_type("setup", type_.to_string(), "csi-SubframePatternDormant-r15");
        return c;
      }
      const setup_s_& setup() const
      {
        assert_choice_type("setup", type_.to_string(), "csi-SubframePatternDormant-r15");
        return c;
      }
      setup_s_& set_setup()
      {
        set(types::setup);
        return c;
      }

    private:
      types    type_;
      setup_s_ c;
    };
    struct cqi_format_ind_dormant_r15_c_ {
      struct wideband_cqi_r15_s_ {
        struct csi_report_mode_r15_opts {
          enum options { submode1, submode2, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<csi_report_mode_r15_opts> csi_report_mode_r15_e_;

        // member variables
        bool                   csi_report_mode_r15_present = false;
        csi_report_mode_r15_e_ csi_report_mode_r15;
      };
      struct subband_cqi_r15_s_ {
        struct periodicity_factor_r15_opts {
          enum options { n2, n4, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<periodicity_factor_r15_opts> periodicity_factor_r15_e_;

        // member variables
        uint8_t                   k_r15 = 1;
        periodicity_factor_r15_e_ periodicity_factor_r15;
      };
      struct types_opts {
        enum options { wideband_cqi_r15, subband_cqi_r15, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      cqi_format_ind_dormant_r15_c_() = default;
      cqi_format_ind_dormant_r15_c_(const cqi_format_ind_dormant_r15_c_& other);
      cqi_format_ind_dormant_r15_c_& operator=(const cqi_format_ind_dormant_r15_c_& other);
      ~cqi_format_ind_dormant_r15_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      wideband_cqi_r15_s_& wideband_cqi_r15()
      {
        assert_choice_type("widebandCQI-r15", type_.to_string(), "cqi-FormatIndicatorDormant-r15");
        return c.get<wideband_cqi_r15_s_>();
      }
      subband_cqi_r15_s_& subband_cqi_r15()
      {
        assert_choice_type("subbandCQI-r15", type_.to_string(), "cqi-FormatIndicatorDormant-r15");
        return c.get<subband_cqi_r15_s_>();
      }
      const wideband_cqi_r15_s_& wideband_cqi_r15() const
      {
        assert_choice_type("widebandCQI-r15", type_.to_string(), "cqi-FormatIndicatorDormant-r15");
        return c.get<wideband_cqi_r15_s_>();
      }
      const subband_cqi_r15_s_& subband_cqi_r15() const
      {
        assert_choice_type("subbandCQI-r15", type_.to_string(), "cqi-FormatIndicatorDormant-r15");
        return c.get<subband_cqi_r15_s_>();
      }
      wideband_cqi_r15_s_& set_wideband_cqi_r15()
      {
        set(types::wideband_cqi_r15);
        return c.get<wideband_cqi_r15_s_>();
      }
      subband_cqi_r15_s_& set_subband_cqi_r15()
      {
        set(types::subband_cqi_r15);
        return c.get<subband_cqi_r15_s_>();
      }

    private:
      types                                                    type_;
      choice_buffer_t<subband_cqi_r15_s_, wideband_cqi_r15_s_> c;

      void destroy_();
    };

    // member variables
    bool                          ri_cfg_idx_dormant_r15_present     = false;
    bool                          csi_sf_pattern_dormant_r15_present = false;
    bool                          cqi_format_ind_dormant_r15_present = false;
    uint16_t                      cqi_pmi_cfg_idx_dormant_r15        = 0;
    uint16_t                      ri_cfg_idx_dormant_r15             = 0;
    csi_sf_pattern_dormant_r15_c_ csi_sf_pattern_dormant_r15;
    cqi_format_ind_dormant_r15_c_ cqi_format_ind_dormant_r15;
  };
  typedef setup_e types;

  // choice methods
  cqi_report_periodic_scell_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodicSCell-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ReportPeriodicSCell-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CrossCarrierSchedulingConfigLAA-UL-r14 ::= SEQUENCE
struct cross_carrier_sched_cfg_laa_ul_r14_s {
  uint8_t sched_cell_id_r14     = 0;
  uint8_t cif_in_sched_cell_r14 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// LBT-Config-r14 ::= CHOICE
struct lbt_cfg_r14_c {
  struct types_opts {
    enum options { max_energy_detection_thres_r14, energy_detection_thres_offset_r14, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  lbt_cfg_r14_c() = default;
  lbt_cfg_r14_c(const lbt_cfg_r14_c& other);
  lbt_cfg_r14_c& operator=(const lbt_cfg_r14_c& other);
  ~lbt_cfg_r14_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  int8_t& max_energy_detection_thres_r14()
  {
    assert_choice_type("maxEnergyDetectionThreshold-r14", type_.to_string(), "LBT-Config-r14");
    return c.get<int8_t>();
  }
  int8_t& energy_detection_thres_offset_r14()
  {
    assert_choice_type("energyDetectionThresholdOffset-r14", type_.to_string(), "LBT-Config-r14");
    return c.get<int8_t>();
  }
  const int8_t& max_energy_detection_thres_r14() const
  {
    assert_choice_type("maxEnergyDetectionThreshold-r14", type_.to_string(), "LBT-Config-r14");
    return c.get<int8_t>();
  }
  const int8_t& energy_detection_thres_offset_r14() const
  {
    assert_choice_type("energyDetectionThresholdOffset-r14", type_.to_string(), "LBT-Config-r14");
    return c.get<int8_t>();
  }
  int8_t& set_max_energy_detection_thres_r14()
  {
    set(types::max_energy_detection_thres_r14);
    return c.get<int8_t>();
  }
  int8_t& set_energy_detection_thres_offset_r14()
  {
    set(types::energy_detection_thres_offset_r14);
    return c.get<int8_t>();
  }

private:
  types               type_;
  pod_choice_buffer_t c;

  void destroy_();
};

// PDCCH-ConfigLAA-r14 ::= SEQUENCE
struct pdcch_cfg_laa_r14_s {
  struct max_nof_sched_sfs_format0_b_r14_opts {
    enum options { sf2, sf3, sf4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_nof_sched_sfs_format0_b_r14_opts> max_nof_sched_sfs_format0_b_r14_e_;
  struct max_nof_sched_sfs_format4_b_r14_opts {
    enum options { sf2, sf3, sf4, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_nof_sched_sfs_format4_b_r14_opts> max_nof_sched_sfs_format4_b_r14_e_;

  // member variables
  bool                                    max_nof_sched_sfs_format0_b_r14_present          = false;
  bool                                    max_nof_sched_sfs_format4_b_r14_present          = false;
  bool                                    skip_monitoring_dci_format0_a_r14_present        = false;
  bool                                    skip_monitoring_dci_format4_a_r14_present        = false;
  bool                                    pdcch_candidate_reductions_format0_a_r14_present = false;
  bool                                    pdcch_candidate_reductions_format4_a_r14_present = false;
  bool                                    pdcch_candidate_reductions_format0_b_r14_present = false;
  bool                                    pdcch_candidate_reductions_format4_b_r14_present = false;
  max_nof_sched_sfs_format0_b_r14_e_      max_nof_sched_sfs_format0_b_r14;
  max_nof_sched_sfs_format4_b_r14_e_      max_nof_sched_sfs_format4_b_r14;
  pdcch_candidate_reductions_r13_c        pdcch_candidate_reductions_format0_a_r14;
  pdcch_candidate_reductions_laa_ul_r14_c pdcch_candidate_reductions_format4_a_r14;
  pdcch_candidate_reductions_laa_ul_r14_c pdcch_candidate_reductions_format0_b_r14;
  pdcch_candidate_reductions_laa_ul_r14_c pdcch_candidate_reductions_format4_b_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ModeConfigLAA-r15 ::= SEQUENCE
struct pusch_mode_cfg_laa_r15_s {
  bool laa_pusch_mode1 = false;
  bool laa_pusch_mode2 = false;
  bool laa_pusch_mode3 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SoundingRS-UL-ConfigDedicatedAperiodic-v1430 ::= CHOICE
struct srs_ul_cfg_ded_aperiodic_v1430_c {
  struct setup_s_ {
    bool    srs_sf_ind_r14_present = false;
    uint8_t srs_sf_ind_r14         = 1;
  };
  typedef setup_e types;

  // choice methods
  srs_ul_cfg_ded_aperiodic_v1430_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-v1430");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SoundingRS-UL-ConfigDedicatedAperiodic-v1430");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CQI-ReportConfigSCell-r10 ::= SEQUENCE
struct cqi_report_cfg_scell_r10_s {
  bool                        cqi_report_mode_aperiodic_r10_present = false;
  bool                        cqi_report_periodic_scell_r10_present = false;
  bool                        pmi_ri_report_r10_present             = false;
  cqi_report_mode_aperiodic_e cqi_report_mode_aperiodic_r10;
  int8_t                      nom_pdsch_rs_epre_offset_r10 = -1;
  cqi_report_periodic_r10_c   cqi_report_periodic_scell_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ReportConfigSCell-r15 ::= SEQUENCE
struct cqi_report_cfg_scell_r15_s {
  struct alt_cqi_table_minus1024_qam_r15_opts {
    enum options { all_sfs, csi_sf_set1, csi_sf_set2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<alt_cqi_table_minus1024_qam_r15_opts> alt_cqi_table_minus1024_qam_r15_e_;

  // member variables
  bool                               cqi_report_periodic_scell_r15_present   = false;
  bool                               alt_cqi_table_minus1024_qam_r15_present = false;
  cqi_report_periodic_scell_r15_c    cqi_report_periodic_scell_r15;
  alt_cqi_table_minus1024_qam_r15_e_ alt_cqi_table_minus1024_qam_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CQI-ShortConfigSCell-r15 ::= CHOICE
struct cqi_short_cfg_scell_r15_c {
  struct setup_s_ {
    struct cqi_format_ind_short_r15_c_ {
      struct wideband_cqi_short_r15_s_ {
        struct csi_report_mode_short_r15_opts {
          enum options { submode1, submode2, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<csi_report_mode_short_r15_opts> csi_report_mode_short_r15_e_;

        // member variables
        bool                         csi_report_mode_short_r15_present = false;
        csi_report_mode_short_r15_e_ csi_report_mode_short_r15;
      };
      struct subband_cqi_short_r15_s_ {
        struct periodicity_factor_r15_opts {
          enum options { n2, n4, nulltype } value;
          typedef uint8_t number_type;

          std::string to_string() const;
          uint8_t     to_number() const;
        };
        typedef enumerated<periodicity_factor_r15_opts> periodicity_factor_r15_e_;

        // member variables
        uint8_t                   k_r15 = 1;
        periodicity_factor_r15_e_ periodicity_factor_r15;
      };
      struct types_opts {
        enum options { wideband_cqi_short_r15, subband_cqi_short_r15, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      cqi_format_ind_short_r15_c_() = default;
      cqi_format_ind_short_r15_c_(const cqi_format_ind_short_r15_c_& other);
      cqi_format_ind_short_r15_c_& operator=(const cqi_format_ind_short_r15_c_& other);
      ~cqi_format_ind_short_r15_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      wideband_cqi_short_r15_s_& wideband_cqi_short_r15()
      {
        assert_choice_type("widebandCQI-Short-r15", type_.to_string(), "cqi-FormatIndicatorShort-r15");
        return c.get<wideband_cqi_short_r15_s_>();
      }
      subband_cqi_short_r15_s_& subband_cqi_short_r15()
      {
        assert_choice_type("subbandCQI-Short-r15", type_.to_string(), "cqi-FormatIndicatorShort-r15");
        return c.get<subband_cqi_short_r15_s_>();
      }
      const wideband_cqi_short_r15_s_& wideband_cqi_short_r15() const
      {
        assert_choice_type("widebandCQI-Short-r15", type_.to_string(), "cqi-FormatIndicatorShort-r15");
        return c.get<wideband_cqi_short_r15_s_>();
      }
      const subband_cqi_short_r15_s_& subband_cqi_short_r15() const
      {
        assert_choice_type("subbandCQI-Short-r15", type_.to_string(), "cqi-FormatIndicatorShort-r15");
        return c.get<subband_cqi_short_r15_s_>();
      }
      wideband_cqi_short_r15_s_& set_wideband_cqi_short_r15()
      {
        set(types::wideband_cqi_short_r15);
        return c.get<wideband_cqi_short_r15_s_>();
      }
      subband_cqi_short_r15_s_& set_subband_cqi_short_r15()
      {
        set(types::subband_cqi_short_r15);
        return c.get<subband_cqi_short_r15_s_>();
      }

    private:
      types                                                                type_;
      choice_buffer_t<subband_cqi_short_r15_s_, wideband_cqi_short_r15_s_> c;

      void destroy_();
    };

    // member variables
    bool                        ri_cfg_idx_short_r15_present     = false;
    bool                        cqi_format_ind_short_r15_present = false;
    uint16_t                    cqi_pmi_cfg_idx_short_r15        = 0;
    uint16_t                    ri_cfg_idx_short_r15             = 0;
    cqi_format_ind_short_r15_c_ cqi_format_ind_short_r15;
  };
  typedef setup_e types;

  // choice methods
  cqi_short_cfg_scell_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ShortConfigSCell-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "CQI-ShortConfigSCell-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// CrossCarrierSchedulingConfig-r10 ::= SEQUENCE
struct cross_carrier_sched_cfg_r10_s {
  struct sched_cell_info_r10_c_ {
    struct own_r10_s_ {
      bool cif_presence_r10 = false;
    };
    struct other_r10_s_ {
      uint8_t sched_cell_id_r10 = 0;
      uint8_t pdsch_start_r10   = 1;
    };
    struct types_opts {
      enum options { own_r10, other_r10, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sched_cell_info_r10_c_() = default;
    sched_cell_info_r10_c_(const sched_cell_info_r10_c_& other);
    sched_cell_info_r10_c_& operator=(const sched_cell_info_r10_c_& other);
    ~sched_cell_info_r10_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    own_r10_s_& own_r10()
    {
      assert_choice_type("own-r10", type_.to_string(), "schedulingCellInfo-r10");
      return c.get<own_r10_s_>();
    }
    other_r10_s_& other_r10()
    {
      assert_choice_type("other-r10", type_.to_string(), "schedulingCellInfo-r10");
      return c.get<other_r10_s_>();
    }
    const own_r10_s_& own_r10() const
    {
      assert_choice_type("own-r10", type_.to_string(), "schedulingCellInfo-r10");
      return c.get<own_r10_s_>();
    }
    const other_r10_s_& other_r10() const
    {
      assert_choice_type("other-r10", type_.to_string(), "schedulingCellInfo-r10");
      return c.get<other_r10_s_>();
    }
    own_r10_s_& set_own_r10()
    {
      set(types::own_r10);
      return c.get<own_r10_s_>();
    }
    other_r10_s_& set_other_r10()
    {
      set(types::other_r10);
      return c.get<other_r10_s_>();
    }

  private:
    types                                     type_;
    choice_buffer_t<other_r10_s_, own_r10_s_> c;

    void destroy_();
  };

  // member variables
  sched_cell_info_r10_c_ sched_cell_info_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CrossCarrierSchedulingConfig-r13 ::= SEQUENCE
struct cross_carrier_sched_cfg_r13_s {
  struct sched_cell_info_r13_c_ {
    struct own_r13_s_ {
      bool cif_presence_r13 = false;
    };
    struct other_r13_s_ {
      uint8_t sched_cell_id_r13     = 0;
      uint8_t pdsch_start_r13       = 1;
      uint8_t cif_in_sched_cell_r13 = 1;
    };
    struct types_opts {
      enum options { own_r13, other_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    sched_cell_info_r13_c_() = default;
    sched_cell_info_r13_c_(const sched_cell_info_r13_c_& other);
    sched_cell_info_r13_c_& operator=(const sched_cell_info_r13_c_& other);
    ~sched_cell_info_r13_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    own_r13_s_& own_r13()
    {
      assert_choice_type("own-r13", type_.to_string(), "schedulingCellInfo-r13");
      return c.get<own_r13_s_>();
    }
    other_r13_s_& other_r13()
    {
      assert_choice_type("other-r13", type_.to_string(), "schedulingCellInfo-r13");
      return c.get<other_r13_s_>();
    }
    const own_r13_s_& own_r13() const
    {
      assert_choice_type("own-r13", type_.to_string(), "schedulingCellInfo-r13");
      return c.get<own_r13_s_>();
    }
    const other_r13_s_& other_r13() const
    {
      assert_choice_type("other-r13", type_.to_string(), "schedulingCellInfo-r13");
      return c.get<other_r13_s_>();
    }
    own_r13_s_& set_own_r13()
    {
      set(types::own_r13);
      return c.get<own_r13_s_>();
    }
    other_r13_s_& set_other_r13()
    {
      set(types::other_r13);
      return c.get<other_r13_s_>();
    }

  private:
    types                                     type_;
    choice_buffer_t<other_r13_s_, own_r13_s_> c;

    void destroy_();
  };

  // member variables
  sched_cell_info_r13_c_ sched_cell_info_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DeltaFList-SPUCCH-r15 ::= CHOICE
struct delta_flist_spucch_r15_c {
  struct setup_s_ {
    struct delta_f_slot_spucch_format1_r15_opts {
      enum options {
        delta_f_minus1,
        delta_f0,
        delta_f1,
        delta_f2,
        delta_f3,
        delta_f4,
        delta_f5,
        delta_f6,
        nulltype
      } value;
      typedef int8_t number_type;

      std::string to_string() const;
      int8_t      to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_format1_r15_opts> delta_f_slot_spucch_format1_r15_e_;
    struct delta_f_slot_spucch_format1a_r15_opts {
      enum options { delta_f1, delta_f2, delta_f3, delta_f4, delta_f5, delta_f6, delta_f7, delta_f8, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_format1a_r15_opts> delta_f_slot_spucch_format1a_r15_e_;
    struct delta_f_slot_spucch_format1b_r15_opts {
      enum options { delta_f3, delta_f4, delta_f5, delta_f6, delta_f7, delta_f8, delta_f9, delta_f10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_format1b_r15_opts> delta_f_slot_spucch_format1b_r15_e_;
    struct delta_f_slot_spucch_format3_r15_opts {
      enum options { delta_f4, delta_f5, delta_f6, delta_f7, delta_f8, delta_f9, delta_f10, delta_f11, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_format3_r15_opts> delta_f_slot_spucch_format3_r15_e_;
    struct delta_f_slot_spucch_rm_format4_r15_opts {
      enum options {
        delta_f13,
        delta_f14,
        delta_f15,
        delta_f16,
        delta_f17,
        delta_f18,
        delta_f19,
        delta_f20,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_rm_format4_r15_opts> delta_f_slot_spucch_rm_format4_r15_e_;
    struct delta_f_slot_spucch_tbcc_format4_r15_opts {
      enum options {
        delta_f10,
        delta_f11,
        delta_f12,
        delta_f13,
        delta_f14,
        delta_f15,
        delta_f16,
        delta_f17,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_slot_spucch_tbcc_format4_r15_opts> delta_f_slot_spucch_tbcc_format4_r15_e_;
    struct delta_f_subslot_spucch_format1and1a_r15_opts {
      enum options {
        delta_f5,
        delta_f6,
        delta_f7,
        delta_f8,
        delta_f9,
        delta_f10,
        delta_f11,
        delta_f12,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_subslot_spucch_format1and1a_r15_opts> delta_f_subslot_spucch_format1and1a_r15_e_;
    struct delta_f_subslot_spucch_format1b_r15_opts {
      enum options {
        delta_f6,
        delta_f7,
        delta_f8,
        delta_f9,
        delta_f10,
        delta_f11,
        delta_f12,
        delta_f13,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_subslot_spucch_format1b_r15_opts> delta_f_subslot_spucch_format1b_r15_e_;
    struct delta_f_subslot_spucch_rm_format4_r15_opts {
      enum options {
        delta_f15,
        delta_f16,
        delta_f17,
        delta_f18,
        delta_f19,
        delta_f20,
        delta_f21,
        delta_f22,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_subslot_spucch_rm_format4_r15_opts> delta_f_subslot_spucch_rm_format4_r15_e_;
    struct delta_f_subslot_spucch_tbcc_format4_r15_opts {
      enum options {
        delta_f10,
        delta_f11,
        delta_f12,
        delta_f13,
        delta_f14,
        delta_f15,
        delta_f16,
        delta_f17,
        nulltype
      } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<delta_f_subslot_spucch_tbcc_format4_r15_opts> delta_f_subslot_spucch_tbcc_format4_r15_e_;

    // member variables
    bool                                       ext                                             = false;
    bool                                       delta_f_slot_spucch_format1_r15_present         = false;
    bool                                       delta_f_slot_spucch_format1a_r15_present        = false;
    bool                                       delta_f_slot_spucch_format1b_r15_present        = false;
    bool                                       delta_f_slot_spucch_format3_r15_present         = false;
    bool                                       delta_f_slot_spucch_rm_format4_r15_present      = false;
    bool                                       delta_f_slot_spucch_tbcc_format4_r15_present    = false;
    bool                                       delta_f_subslot_spucch_format1and1a_r15_present = false;
    bool                                       delta_f_subslot_spucch_format1b_r15_present     = false;
    bool                                       delta_f_subslot_spucch_rm_format4_r15_present   = false;
    bool                                       delta_f_subslot_spucch_tbcc_format4_r15_present = false;
    delta_f_slot_spucch_format1_r15_e_         delta_f_slot_spucch_format1_r15;
    delta_f_slot_spucch_format1a_r15_e_        delta_f_slot_spucch_format1a_r15;
    delta_f_slot_spucch_format1b_r15_e_        delta_f_slot_spucch_format1b_r15;
    delta_f_slot_spucch_format3_r15_e_         delta_f_slot_spucch_format3_r15;
    delta_f_slot_spucch_rm_format4_r15_e_      delta_f_slot_spucch_rm_format4_r15;
    delta_f_slot_spucch_tbcc_format4_r15_e_    delta_f_slot_spucch_tbcc_format4_r15;
    delta_f_subslot_spucch_format1and1a_r15_e_ delta_f_subslot_spucch_format1and1a_r15;
    delta_f_subslot_spucch_format1b_r15_e_     delta_f_subslot_spucch_format1b_r15;
    delta_f_subslot_spucch_rm_format4_r15_e_   delta_f_subslot_spucch_rm_format4_r15;
    delta_f_subslot_spucch_tbcc_format4_r15_e_ delta_f_subslot_spucch_tbcc_format4_r15;
    // ...
  };
  typedef setup_e types;

  // choice methods
  delta_flist_spucch_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "DeltaFList-SPUCCH-r15");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "DeltaFList-SPUCCH-r15");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// LAA-SCellConfiguration-r13 ::= SEQUENCE
struct laa_scell_cfg_r13_s {
  struct sf_start_position_r13_opts {
    enum options { s0, s07, nulltype } value;
    typedef float number_type;

    std::string to_string() const;
    float       to_number() const;
    std::string to_number_string() const;
  };
  typedef enumerated<sf_start_position_r13_opts> sf_start_position_r13_e_;

  // member variables
  sf_start_position_r13_e_ sf_start_position_r13;
  fixed_bitstring<8>       laa_scell_sf_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// LAA-SCellConfiguration-v1430 ::= SEQUENCE
struct laa_scell_cfg_v1430_s {
  struct cross_carrier_sched_cfg_ul_r14_c_ {
    struct setup_s_ {
      cross_carrier_sched_cfg_laa_ul_r14_s cross_carrier_sched_cfg_laa_ul_r14;
    };
    typedef setup_e types;

    // choice methods
    cross_carrier_sched_cfg_ul_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "crossCarrierSchedulingConfig-UL-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "crossCarrierSchedulingConfig-UL-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                              cross_carrier_sched_cfg_ul_r14_present      = false;
  bool                              lbt_cfg_r14_present                         = false;
  bool                              pdcch_cfg_laa_r14_present                   = false;
  bool                              absence_of_any_other_technology_r14_present = false;
  bool                              srs_ul_cfg_ded_aperiodic_v1430_present      = false;
  cross_carrier_sched_cfg_ul_r14_c_ cross_carrier_sched_cfg_ul_r14;
  lbt_cfg_r14_c                     lbt_cfg_r14;
  pdcch_cfg_laa_r14_s               pdcch_cfg_laa_r14;
  srs_ul_cfg_ded_aperiodic_v1430_c  srs_ul_cfg_ded_aperiodic_v1430;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// LAA-SCellConfiguration-v1530 ::= SEQUENCE
struct laa_scell_cfg_v1530_s {
  bool                     aul_cfg_r15_present            = false;
  bool                     pusch_mode_cfg_laa_r15_present = false;
  aul_cfg_r15_c            aul_cfg_r15;
  pusch_mode_cfg_laa_r15_s pusch_mode_cfg_laa_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDCCH-ConfigSCell-r13 ::= SEQUENCE
struct pdcch_cfg_scell_r13_s {
  bool skip_monitoring_dci_format0_minus1_a_r13_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PDSCH-ConfigDedicatedSCell-v1430 ::= SEQUENCE
struct pdsch_cfg_ded_scell_v1430_s {
  bool tbs_idx_alt2_r14_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUCCH-ConfigDedicated-v1370 ::= SEQUENCE
struct pucch_cfg_ded_v1370_s {
  struct pucch_format_v1370_c_ {
    typedef setup_e types;

    // choice methods
    pucch_format_v1370_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    pucch_format3_conf_r13_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pucch-Format-v1370");
      return c;
    }
    const pucch_format3_conf_r13_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pucch-Format-v1370");
      return c;
    }
    pucch_format3_conf_r13_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types                    type_;
    pucch_format3_conf_r13_s c;
  };

  // member variables
  pucch_format_v1370_c_ pucch_format_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicatedSCell-r10 ::= SEQUENCE
struct pusch_cfg_ded_scell_r10_s {
  bool group_hop_disabled_r10_present  = false;
  bool dmrs_with_occ_activ_r10_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicatedSCell-v1430 ::= SEQUENCE
struct pusch_cfg_ded_scell_v1430_s {
  bool                enable256_qam_r14_present = false;
  enable256_qam_r14_c enable256_qam_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PUSCH-ConfigDedicatedScell-v1530 ::= SEQUENCE
struct pusch_cfg_ded_scell_v1530_s {
  struct uci_on_pusch_r15_c_ {
    struct setup_s_ {
      uint8_t beta_offset_aul_r15 = 0;
    };
    typedef setup_e types;

    // choice methods
    uci_on_pusch_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "uci-OnPUSCH-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "uci-OnPUSCH-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  uci_on_pusch_r15_c_ uci_on_pusch_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SchedulingRequestConfigSCell-r13 ::= CHOICE
struct sched_request_cfg_scell_r13_c {
  struct setup_s_ {
    struct dsr_trans_max_r13_opts {
      enum options { n4, n8, n16, n32, n64, spare3, spare2, spare1, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<dsr_trans_max_r13_opts> dsr_trans_max_r13_e_;

    // member variables
    bool                 sr_pucch_res_idx_p1_r13_present = false;
    uint16_t             sr_pucch_res_idx_r13            = 0;
    uint16_t             sr_pucch_res_idx_p1_r13         = 0;
    uint8_t              sr_cfg_idx_r13                  = 0;
    dsr_trans_max_r13_e_ dsr_trans_max_r13;
  };
  typedef setup_e types;

  // choice methods
  sched_request_cfg_scell_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfigSCell-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SchedulingRequestConfigSCell-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SoundingRS-AperiodicSet-r14 ::= SEQUENCE
struct srs_aperiodic_set_r14_s {
  using srs_cc_set_idx_list_r14_l_ = dyn_array<srs_cc_set_idx_r14_s>;

  // member variables
  bool                           srs_cc_set_idx_list_r14_present = false;
  srs_cc_set_idx_list_r14_l_     srs_cc_set_idx_list_r14;
  srs_ul_cfg_ded_aperiodic_r10_c srs_ul_cfg_ded_aperiodic_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SoundingRS-AperiodicSetUpPTsExt-r14 ::= SEQUENCE
struct srs_aperiodic_set_up_pts_ext_r14_s {
  using srs_cc_set_idx_list_r14_l_ = dyn_array<srs_cc_set_idx_r14_s>;

  // member variables
  bool                                      srs_cc_set_idx_list_r14_present = false;
  srs_cc_set_idx_list_r14_l_                srs_cc_set_idx_list_r14;
  srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c srs_ul_cfg_ded_aperiodic_up_pts_ext_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// TPC-PDCCH-ConfigSCell-r13 ::= CHOICE
struct tpc_pdcch_cfg_scell_r13_c {
  struct setup_s_ {
    tpc_idx_c tpc_idx_pucch_scell_r13;
  };
  typedef setup_e types;

  // choice methods
  tpc_pdcch_cfg_scell_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "TPC-PDCCH-ConfigSCell-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "TPC-PDCCH-ConfigSCell-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// UplinkPUSCH-LessPowerControlDedicated-v1430 ::= SEQUENCE
struct ul_pusch_less_pwr_ctrl_ded_v1430_s {
  bool   p0_ue_periodic_srs_r14_present  = false;
  bool   p0_ue_aperiodic_srs_r14_present = false;
  int8_t p0_ue_periodic_srs_r14          = -8;
  int8_t p0_ue_aperiodic_srs_r14         = -8;
  bool   accumulation_enabled_r14        = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicatedSCell-r10 ::= SEQUENCE
struct ul_pwr_ctrl_ded_scell_r10_s {
  struct delta_mcs_enabled_r10_opts {
    enum options { en0, en1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_mcs_enabled_r10_opts> delta_mcs_enabled_r10_e_;
  struct pathloss_ref_linking_r10_opts {
    enum options { pcell, scell, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<pathloss_ref_linking_r10_opts> pathloss_ref_linking_r10_e_;

  // member variables
  bool                        psrs_offset_ap_r10_present = false;
  bool                        filt_coef_r10_present      = false;
  int8_t                      p0_ue_pusch_r10            = -8;
  delta_mcs_enabled_r10_e_    delta_mcs_enabled_r10;
  bool                        accumulation_enabled_r10 = false;
  uint8_t                     psrs_offset_r10          = 0;
  uint8_t                     psrs_offset_ap_r10       = 0;
  filt_coef_e                 filt_coef_r10;
  pathloss_ref_linking_r10_e_ pathloss_ref_linking_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlDedicatedSCell-v1310 ::= SEQUENCE
struct ul_pwr_ctrl_ded_scell_v1310_s {
  bool                               delta_tx_d_offset_list_pucch_r10_present = false;
  int8_t                             p0_ue_pucch                              = -8;
  delta_tx_d_offset_list_pucch_r10_s delta_tx_d_offset_list_pucch_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoCommon ::= SEQUENCE
struct ant_info_common_s {
  struct ant_ports_count_opts {
    enum options { an1, an2, an4, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<ant_ports_count_opts> ant_ports_count_e_;

  // member variables
  ant_ports_count_e_ ant_ports_count;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// HighSpeedConfigSCell-r14 ::= SEQUENCE
struct high_speed_cfg_scell_r14_s {
  bool high_speed_enhanced_demod_flag_r14_present = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MAC-MainConfigSCell-r11 ::= SEQUENCE
struct mac_main_cfg_scell_r11_s {
  bool    ext                 = false;
  bool    stag_id_r11_present = false;
  uint8_t stag_id_r11         = 1;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-Config ::= SEQUENCE
struct prach_cfg_s {
  bool             prach_cfg_info_present = false;
  uint16_t         root_seq_idx           = 0;
  prach_cfg_info_s prach_cfg_info;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PRACH-ConfigSCell-r10 ::= SEQUENCE
struct prach_cfg_scell_r10_s {
  uint8_t prach_cfg_idx_r10 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysicalConfigDedicatedSCell-r10 ::= SEQUENCE
struct phys_cfg_ded_scell_r10_s {
  struct non_ul_cfg_r10_s_ {
    bool                          ant_info_r10_present                = false;
    bool                          cross_carrier_sched_cfg_r10_present = false;
    bool                          csi_rs_cfg_r10_present              = false;
    bool                          pdsch_cfg_ded_r10_present           = false;
    ant_info_ded_r10_s            ant_info_r10;
    cross_carrier_sched_cfg_r10_s cross_carrier_sched_cfg_r10;
    csi_rs_cfg_r10_s              csi_rs_cfg_r10;
    pdsch_cfg_ded_s               pdsch_cfg_ded_r10;
  };
  struct ul_cfg_r10_s_ {
    bool                           ant_info_ul_r10_present              = false;
    bool                           pusch_cfg_ded_scell_r10_present      = false;
    bool                           ul_pwr_ctrl_ded_scell_r10_present    = false;
    bool                           cqi_report_cfg_scell_r10_present     = false;
    bool                           srs_ul_cfg_ded_r10_present           = false;
    bool                           srs_ul_cfg_ded_v1020_present         = false;
    bool                           srs_ul_cfg_ded_aperiodic_r10_present = false;
    ant_info_ul_r10_s              ant_info_ul_r10;
    pusch_cfg_ded_scell_r10_s      pusch_cfg_ded_scell_r10;
    ul_pwr_ctrl_ded_scell_r10_s    ul_pwr_ctrl_ded_scell_r10;
    cqi_report_cfg_scell_r10_s     cqi_report_cfg_scell_r10;
    srs_ul_cfg_ded_c               srs_ul_cfg_ded_r10;
    srs_ul_cfg_ded_v1020_s         srs_ul_cfg_ded_v1020;
    srs_ul_cfg_ded_aperiodic_r10_c srs_ul_cfg_ded_aperiodic_r10;
  };
  struct pucch_scell_c_ {
    struct setup_s_ {
      bool                          pucch_cfg_ded_r13_present             = false;
      bool                          sched_request_cfg_r13_present         = false;
      bool                          tpc_pdcch_cfg_pucch_scell_r13_present = false;
      bool                          pusch_cfg_ded_r13_present             = false;
      bool                          ul_pwr_ctrl_ded_r13_present           = false;
      pucch_cfg_ded_r13_s           pucch_cfg_ded_r13;
      sched_request_cfg_scell_r13_c sched_request_cfg_r13;
      tpc_pdcch_cfg_scell_r13_c     tpc_pdcch_cfg_pucch_scell_r13;
      pusch_cfg_ded_r13_s           pusch_cfg_ded_r13;
      ul_pwr_ctrl_ded_scell_v1310_s ul_pwr_ctrl_ded_r13;
    };
    typedef setup_e types;

    // choice methods
    pucch_scell_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pucch-SCell");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pucch-SCell");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  using srs_ul_periodic_cfg_ded_list_r14_l_            = dyn_array<srs_ul_cfg_ded_c>;
  using srs_ul_periodic_cfg_ded_up_pts_ext_list_r14_l_ = dyn_array<srs_ul_cfg_ded_up_pts_ext_r13_c>;
  using srs_ul_aperiodic_cfg_ded_list_r14_l_           = dyn_array<srs_aperiodic_set_r14_s>;
  using srs_ul_cfg_ded_ap_up_pts_ext_list_r14_l_       = dyn_array<srs_aperiodic_set_up_pts_ext_r14_s>;
  struct must_cfg_r14_c_ {
    struct setup_s_ {
      struct k_max_r14_opts {
        enum options { l1, l3, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<k_max_r14_opts> k_max_r14_e_;
      struct p_a_must_r14_opts {
        enum options { db_minus6, db_minus4dot77, db_minus3, db_minus1dot77, db0, db1, db2, db3, nulltype } value;
        typedef float number_type;

        std::string to_string() const;
        float       to_number() const;
        std::string to_number_string() const;
      };
      typedef enumerated<p_a_must_r14_opts> p_a_must_r14_e_;

      // member variables
      bool            p_a_must_r14_present = false;
      k_max_r14_e_    k_max_r14;
      p_a_must_r14_e_ p_a_must_r14;
    };
    typedef setup_e types;

    // choice methods
    must_cfg_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "must-Config-r14");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "must-Config-r14");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct semi_static_cfi_cfg_r15_c_ {
    struct setup_c_ {
      struct types_opts {
        enum options { cfi_cfg_r15, cfi_pattern_cfg_r15, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      cfi_cfg_r15_s& cfi_cfg_r15()
      {
        assert_choice_type("cfi-Config-r15", type_.to_string(), "setup");
        return c.get<cfi_cfg_r15_s>();
      }
      cfi_pattern_cfg_r15_s& cfi_pattern_cfg_r15()
      {
        assert_choice_type("cfi-PatternConfig-r15", type_.to_string(), "setup");
        return c.get<cfi_pattern_cfg_r15_s>();
      }
      const cfi_cfg_r15_s& cfi_cfg_r15() const
      {
        assert_choice_type("cfi-Config-r15", type_.to_string(), "setup");
        return c.get<cfi_cfg_r15_s>();
      }
      const cfi_pattern_cfg_r15_s& cfi_pattern_cfg_r15() const
      {
        assert_choice_type("cfi-PatternConfig-r15", type_.to_string(), "setup");
        return c.get<cfi_pattern_cfg_r15_s>();
      }
      cfi_cfg_r15_s& set_cfi_cfg_r15()
      {
        set(types::cfi_cfg_r15);
        return c.get<cfi_cfg_r15_s>();
      }
      cfi_pattern_cfg_r15_s& set_cfi_pattern_cfg_r15()
      {
        set(types::cfi_pattern_cfg_r15);
        return c.get<cfi_pattern_cfg_r15_s>();
      }

    private:
      types                                                 type_;
      choice_buffer_t<cfi_cfg_r15_s, cfi_pattern_cfg_r15_s> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    semi_static_cfi_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "semiStaticCFI-Config-r15");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "semiStaticCFI-Config-r15");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct blind_pdsch_repeat_cfg_r15_c_ {
    struct setup_s_ {
      struct max_num_sf_pdsch_repeats_r15_opts {
        enum options { n4, n6, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<max_num_sf_pdsch_repeats_r15_opts> max_num_sf_pdsch_repeats_r15_e_;
      struct max_num_slot_subslot_pdsch_repeats_r15_opts {
        enum options { n4, n6, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<max_num_slot_subslot_pdsch_repeats_r15_opts> max_num_slot_subslot_pdsch_repeats_r15_e_;
      struct rv_sf_pdsch_repeats_r15_opts {
        enum options { dlrvseq1, dlrvseq2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<rv_sf_pdsch_repeats_r15_opts> rv_sf_pdsch_repeats_r15_e_;
      struct rv_slotsublot_pdsch_repeats_r15_opts {
        enum options { dlrvseq1, dlrvseq2, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<rv_slotsublot_pdsch_repeats_r15_opts> rv_slotsublot_pdsch_repeats_r15_e_;
      struct mcs_restrict_sf_pdsch_repeats_r15_opts {
        enum options { n0, n1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<mcs_restrict_sf_pdsch_repeats_r15_opts> mcs_restrict_sf_pdsch_repeats_r15_e_;
      struct mcs_restrict_slot_subslot_pdsch_repeats_r15_opts {
        enum options { n0, n1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<mcs_restrict_slot_subslot_pdsch_repeats_r15_opts>
          mcs_restrict_slot_subslot_pdsch_repeats_r15_e_;

      // member variables
      bool                                           max_num_sf_pdsch_repeats_r15_present                 = false;
      bool                                           max_num_slot_subslot_pdsch_repeats_r15_present       = false;
      bool                                           rv_sf_pdsch_repeats_r15_present                      = false;
      bool                                           rv_slotsublot_pdsch_repeats_r15_present              = false;
      bool                                           nof_processes_sf_pdsch_repeats_r15_present           = false;
      bool                                           nof_processes_slot_subslot_pdsch_repeats_r15_present = false;
      bool                                           mcs_restrict_sf_pdsch_repeats_r15_present            = false;
      bool                                           mcs_restrict_slot_subslot_pdsch_repeats_r15_present  = false;
      bool                                           blind_sf_pdsch_repeats_r15                           = false;
      bool                                           blind_slot_subslot_pdsch_repeats_r15                 = false;
      max_num_sf_pdsch_repeats_r15_e_                max_num_sf_pdsch_repeats_r15;
      max_num_slot_subslot_pdsch_repeats_r15_e_      max_num_slot_subslot_pdsch_repeats_r15;
      rv_sf_pdsch_repeats_r15_e_                     rv_sf_pdsch_repeats_r15;
      rv_slotsublot_pdsch_repeats_r15_e_             rv_slotsublot_pdsch_repeats_r15;
      uint8_t                                        nof_processes_sf_pdsch_repeats_r15           = 1;
      uint8_t                                        nof_processes_slot_subslot_pdsch_repeats_r15 = 1;
      mcs_restrict_sf_pdsch_repeats_r15_e_           mcs_restrict_sf_pdsch_repeats_r15;
      mcs_restrict_slot_subslot_pdsch_repeats_r15_e_ mcs_restrict_slot_subslot_pdsch_repeats_r15;
    };
    typedef setup_e types;

    // choice methods
    blind_pdsch_repeat_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "blindPDSCH-Repetition-Config-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "blindPDSCH-Repetition-Config-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool              ext                    = false;
  bool              non_ul_cfg_r10_present = false;
  bool              ul_cfg_r10_present     = false;
  non_ul_cfg_r10_s_ non_ul_cfg_r10;
  ul_cfg_r10_s_     ul_cfg_r10;
  // ...
  // group 0
  copy_ptr<csi_rs_cfg_nzp_to_release_list_r11_l> csi_rs_cfg_nzp_to_release_list_r11;
  copy_ptr<csi_rs_cfg_nzp_to_add_mod_list_r11_l> csi_rs_cfg_nzp_to_add_mod_list_r11;
  copy_ptr<csi_rs_cfg_zp_to_release_list_r11_l>  csi_rs_cfg_zp_to_release_list_r11;
  copy_ptr<csi_rs_cfg_zp_to_add_mod_list_r11_l>  csi_rs_cfg_zp_to_add_mod_list_r11;
  copy_ptr<epdcch_cfg_r11_s>                     epdcch_cfg_r11;
  copy_ptr<pdsch_cfg_ded_v1130_s>                pdsch_cfg_ded_v1130;
  copy_ptr<cqi_report_cfg_v1130_s>               cqi_report_cfg_v1130;
  copy_ptr<pusch_cfg_ded_v1130_s>                pusch_cfg_ded_v1130;
  copy_ptr<ul_pwr_ctrl_ded_v1130_s>              ul_pwr_ctrl_ded_scell_v1130;
  // group 1
  copy_ptr<ant_info_ded_v1250_s>           ant_info_v1250;
  copy_ptr<eimta_main_cfg_serv_cell_r12_c> eimta_main_cfg_scell_r12;
  copy_ptr<cqi_report_cfg_v1250_s>         cqi_report_cfg_scell_v1250;
  copy_ptr<ul_pwr_ctrl_ded_v1250_s>        ul_pwr_ctrl_ded_scell_v1250;
  copy_ptr<csi_rs_cfg_v1250_s>             csi_rs_cfg_v1250;
  // group 2
  copy_ptr<pdsch_cfg_ded_v1280_s> pdsch_cfg_ded_v1280;
  // group 3
  bool                                                pucch_cell_r13_present = false;
  copy_ptr<pucch_scell_c_>                            pucch_scell;
  copy_ptr<cross_carrier_sched_cfg_r13_s>             cross_carrier_sched_cfg_r13;
  copy_ptr<pdcch_cfg_scell_r13_s>                     pdcch_cfg_scell_r13;
  copy_ptr<cqi_report_cfg_v1310_s>                    cqi_report_cfg_v1310;
  copy_ptr<pdsch_cfg_ded_v1310_s>                     pdsch_cfg_ded_v1310;
  copy_ptr<srs_ul_cfg_ded_v1310_c>                    srs_ul_cfg_ded_v1310;
  copy_ptr<srs_ul_cfg_ded_up_pts_ext_r13_c>           srs_ul_cfg_ded_up_pts_ext_r13;
  copy_ptr<srs_ul_cfg_ded_aperiodic_v1310_c>          srs_ul_cfg_ded_aperiodic_v1310;
  copy_ptr<srs_ul_cfg_ded_aperiodic_up_pts_ext_r13_c> srs_ul_cfg_ded_aperiodic_up_pts_ext_r13;
  copy_ptr<csi_rs_cfg_v1310_s>                        csi_rs_cfg_v1310;
  copy_ptr<laa_scell_cfg_r13_s>                       laa_scell_cfg_r13;
  copy_ptr<csi_rs_cfg_nzp_to_add_mod_list_ext_r13_l>  csi_rs_cfg_nzp_to_add_mod_list_ext_r13;
  copy_ptr<csi_rs_cfg_nzp_to_release_list_ext_r13_l>  csi_rs_cfg_nzp_to_release_list_ext_r13;
  // group 4
  copy_ptr<cqi_report_cfg_v1320_s> cqi_report_cfg_v1320;
  // group 5
  bool                                                     semi_open_loop_r14_present = false;
  copy_ptr<laa_scell_cfg_v1430_s>                          laa_scell_cfg_v1430;
  copy_ptr<srs_tpc_pdcch_cfg_r14_c>                        type_b_srs_tpc_pdcch_cfg_r14;
  copy_ptr<ul_pusch_less_pwr_ctrl_ded_v1430_s>             ul_pusch_less_pwr_ctrl_ded_v1430;
  copy_ptr<srs_ul_periodic_cfg_ded_list_r14_l_>            srs_ul_periodic_cfg_ded_list_r14;
  copy_ptr<srs_ul_periodic_cfg_ded_up_pts_ext_list_r14_l_> srs_ul_periodic_cfg_ded_up_pts_ext_list_r14;
  copy_ptr<srs_ul_aperiodic_cfg_ded_list_r14_l_>           srs_ul_aperiodic_cfg_ded_list_r14;
  copy_ptr<srs_ul_cfg_ded_ap_up_pts_ext_list_r14_l_>       srs_ul_cfg_ded_ap_up_pts_ext_list_r14;
  copy_ptr<must_cfg_r14_c_>                                must_cfg_r14;
  copy_ptr<pusch_cfg_ded_scell_v1430_s>                    pusch_cfg_ded_v1430;
  copy_ptr<csi_rs_cfg_v1430_s>                             csi_rs_cfg_v1430;
  copy_ptr<csi_rs_cfg_zp_ap_list_r14_c>                    csi_rs_cfg_zp_ap_list_r14;
  copy_ptr<cqi_report_cfg_v1430_s>                         cqi_report_cfg_v1430;
  bool                                                     semi_open_loop_r14 = false;
  copy_ptr<pdsch_cfg_ded_scell_v1430_s>                    pdsch_cfg_ded_scell_v1430;
  // group 6
  copy_ptr<csi_rs_cfg_v1480_s> csi_rs_cfg_v1480;
  // group 7
  copy_ptr<phys_cfg_ded_stti_r15_c>       phys_cfg_ded_stti_r15;
  copy_ptr<pdsch_cfg_ded_v1530_s>         pdsch_cfg_ded_v1530;
  copy_ptr<cqi_report_cfg_v1530_s>        cqi_report_cfg_v1530;
  copy_ptr<cqi_report_cfg_scell_r15_s>    cqi_report_cfg_scell_r15;
  copy_ptr<cqi_short_cfg_scell_r15_c>     cqi_short_cfg_scell_r15;
  copy_ptr<csi_rs_cfg_v1530_s>            csi_rs_cfg_v1530;
  copy_ptr<ul_pwr_ctrl_ded_v1530_s>       ul_pwr_ctrl_ded_scell_v1530;
  copy_ptr<laa_scell_cfg_v1530_s>         laa_scell_cfg_v1530;
  copy_ptr<pusch_cfg_ded_scell_v1530_s>   pusch_cfg_ded_v1530;
  copy_ptr<semi_static_cfi_cfg_r15_c_>    semi_static_cfi_cfg_r15;
  copy_ptr<blind_pdsch_repeat_cfg_r15_c_> blind_pdsch_repeat_cfg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysicalConfigDedicatedSCell-v1370 ::= SEQUENCE
struct phys_cfg_ded_scell_v1370_s {
  struct pucch_scell_v1370_c_ {
    struct setup_s_ {
      bool                  pucch_cfg_ded_v1370_present = false;
      pucch_cfg_ded_v1370_s pucch_cfg_ded_v1370;
    };
    typedef setup_e types;

    // choice methods
    pucch_scell_v1370_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "pucch-SCell-v1370");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "pucch-SCell-v1370");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  pucch_scell_v1370_c_ pucch_scell_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-ConfigCommonSCell-r11 ::= SEQUENCE
struct rach_cfg_common_scell_r11_s {
  struct ra_supervision_info_r11_s_ {
    preamb_trans_max_e preamb_trans_max_r11;
  };

  // member variables
  bool                       ext = false;
  pwr_ramp_params_s          pwr_ramp_params_r11;
  ra_supervision_info_r11_s_ ra_supervision_info_r11;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommon-v1530 ::= SEQUENCE
struct ul_pwr_ctrl_common_v1530_s {
  delta_flist_spucch_r15_c delta_flist_spucch_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommonPUSCH-LessCell-v1430 ::= SEQUENCE
struct ul_pwr_ctrl_common_pusch_less_cell_v1430_s {
  bool        p0_nominal_periodic_srs_r14_present  = false;
  bool        p0_nominal_aperiodic_srs_r14_present = false;
  bool        alpha_srs_r14_present                = false;
  int8_t      p0_nominal_periodic_srs_r14          = -126;
  int8_t      p0_nominal_aperiodic_srs_r14         = -126;
  alpha_r12_e alpha_srs_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommonSCell-r10 ::= SEQUENCE
struct ul_pwr_ctrl_common_scell_r10_s {
  int8_t      p0_nominal_pusch_r10 = -126;
  alpha_r12_e alpha_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommonSCell-v1130 ::= SEQUENCE
struct ul_pwr_ctrl_common_scell_v1130_s {
  int8_t delta_preamb_msg3_r11 = -1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommonSCell-v1310 ::= SEQUENCE
struct ul_pwr_ctrl_common_scell_v1310_s {
  struct delta_f_pucch_format3_r12_opts {
    enum options {
      delta_f_minus1,
      delta_f0,
      delta_f1,
      delta_f2,
      delta_f3,
      delta_f4,
      delta_f5,
      delta_f6,
      nulltype
    } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format3_r12_opts> delta_f_pucch_format3_r12_e_;
  struct delta_f_pucch_format1b_cs_r12_opts {
    enum options { delta_f1, delta_f2, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format1b_cs_r12_opts> delta_f_pucch_format1b_cs_r12_e_;
  struct delta_f_pucch_format4_r13_opts {
    enum options {
      delta_f16,
      delta_f15,
      delta_f14,
      delta_f13,
      delta_f12,
      delta_f11,
      delta_f10,
      spare1,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format4_r13_opts> delta_f_pucch_format4_r13_e_;
  struct delta_f_pucch_format5_minus13_opts {
    enum options { delta_f13, delta_f12, delta_f11, delta_f10, delta_f9, delta_f8, delta_f7, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format5_minus13_opts> delta_f_pucch_format5_minus13_e_;

  // member variables
  bool                             delta_f_pucch_format3_r12_present     = false;
  bool                             delta_f_pucch_format1b_cs_r12_present = false;
  bool                             delta_f_pucch_format4_r13_present     = false;
  bool                             delta_f_pucch_format5_minus13_present = false;
  int8_t                           p0_nominal_pucch                      = -127;
  delta_flist_pucch_s              delta_flist_pucch;
  delta_f_pucch_format3_r12_e_     delta_f_pucch_format3_r12;
  delta_f_pucch_format1b_cs_r12_e_ delta_f_pucch_format1b_cs_r12;
  delta_f_pucch_format4_r13_e_     delta_f_pucch_format4_r13;
  delta_f_pucch_format5_minus13_e_ delta_f_pucch_format5_minus13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AntennaInfoDedicated-v10i0 ::= SEQUENCE
struct ant_info_ded_v10i0_s {
  struct max_layers_mimo_r10_opts {
    enum options { two_layers, four_layers, eight_layers, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<max_layers_mimo_r10_opts> max_layers_mimo_r10_e_;

  // member variables
  bool                   max_layers_mimo_r10_present = false;
  max_layers_mimo_r10_e_ max_layers_mimo_r10;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonSCell-r10 ::= SEQUENCE
struct rr_cfg_common_scell_r10_s {
  struct non_ul_cfg_r10_s_ {
    struct dl_bw_r10_opts {
      enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<dl_bw_r10_opts> dl_bw_r10_e_;

    // member variables
    bool                mbsfn_sf_cfg_list_r10_present = false;
    bool                tdd_cfg_r10_present           = false;
    dl_bw_r10_e_        dl_bw_r10;
    ant_info_common_s   ant_info_common_r10;
    mbsfn_sf_cfg_list_l mbsfn_sf_cfg_list_r10;
    phich_cfg_s         phich_cfg_r10;
    pdsch_cfg_common_s  pdsch_cfg_common_r10;
    tdd_cfg_s           tdd_cfg_r10;
  };
  struct ul_cfg_r10_s_ {
    struct ul_freq_info_r10_s_ {
      struct ul_bw_r10_opts {
        enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<ul_bw_r10_opts> ul_bw_r10_e_;

      // member variables
      bool         ul_carrier_freq_r10_present = false;
      bool         ul_bw_r10_present           = false;
      uint32_t     ul_carrier_freq_r10         = 0;
      ul_bw_r10_e_ ul_bw_r10;
      uint8_t      add_spec_emission_scell_r10 = 1;
    };

    // member variables
    bool                           p_max_r10_present           = false;
    bool                           prach_cfg_scell_r10_present = false;
    ul_freq_info_r10_s_            ul_freq_info_r10;
    int8_t                         p_max_r10 = -30;
    ul_pwr_ctrl_common_scell_r10_s ul_pwr_ctrl_common_scell_r10;
    srs_ul_cfg_common_c            srs_ul_cfg_common_r10;
    ul_cp_len_e                    ul_cp_len_r10;
    prach_cfg_scell_r10_s          prach_cfg_scell_r10;
    pusch_cfg_common_s             pusch_cfg_common_r10;
  };
  struct ul_cfg_r14_s_ {
    struct ul_freq_info_r14_s_ {
      struct ul_bw_r14_opts {
        enum options { n6, n15, n25, n50, n75, n100, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<ul_bw_r14_opts> ul_bw_r14_e_;

      // member variables
      bool         ul_carrier_freq_r14_present = false;
      bool         ul_bw_r14_present           = false;
      uint32_t     ul_carrier_freq_r14         = 0;
      ul_bw_r14_e_ ul_bw_r14;
      uint8_t      add_spec_emission_scell_r14 = 1;
    };

    // member variables
    bool                                       p_max_r14_present                                = false;
    bool                                       prach_cfg_scell_r14_present                      = false;
    bool                                       ul_pwr_ctrl_common_pusch_less_cell_v1430_present = false;
    ul_freq_info_r14_s_                        ul_freq_info_r14;
    int8_t                                     p_max_r14 = -30;
    srs_ul_cfg_common_c                        srs_ul_cfg_common_r14;
    ul_cp_len_e                                ul_cp_len_r14;
    prach_cfg_scell_r10_s                      prach_cfg_scell_r14;
    ul_pwr_ctrl_common_pusch_less_cell_v1430_s ul_pwr_ctrl_common_pusch_less_cell_v1430;
  };
  struct harq_ref_cfg_r14_opts {
    enum options { sa2, sa4, sa5, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<harq_ref_cfg_r14_opts> harq_ref_cfg_r14_e_;

  // member variables
  bool              ext                = false;
  bool              ul_cfg_r10_present = false;
  non_ul_cfg_r10_s_ non_ul_cfg_r10;
  ul_cfg_r10_s_     ul_cfg_r10;
  // ...
  // group 0
  bool     ul_carrier_freq_v1090_present = false;
  uint32_t ul_carrier_freq_v1090         = 65536;
  // group 1
  copy_ptr<rach_cfg_common_scell_r11_s>      rach_cfg_common_scell_r11;
  copy_ptr<prach_cfg_s>                      prach_cfg_scell_r11;
  copy_ptr<tdd_cfg_v1130_s>                  tdd_cfg_v1130;
  copy_ptr<ul_pwr_ctrl_common_scell_v1130_s> ul_pwr_ctrl_common_scell_v1130;
  // group 2
  copy_ptr<pusch_cfg_common_v1270_s> pusch_cfg_common_v1270;
  // group 3
  copy_ptr<pucch_cfg_common_s>               pucch_cfg_common_r13;
  copy_ptr<ul_pwr_ctrl_common_scell_v1310_s> ul_pwr_ctrl_common_scell_v1310;
  // group 4
  bool                                 harq_ref_cfg_r14_present    = false;
  bool                                 srs_flex_timing_r14_present = false;
  copy_ptr<high_speed_cfg_scell_r14_s> high_speed_cfg_scell_r14;
  copy_ptr<prach_cfg_v1430_s>          prach_cfg_v1430;
  copy_ptr<ul_cfg_r14_s_>              ul_cfg_r14;
  harq_ref_cfg_r14_e_                  harq_ref_cfg_r14;
  // group 5
  copy_ptr<mbsfn_sf_cfg_list_v1430_l> mbsfn_sf_cfg_list_v1430;
  // group 6
  copy_ptr<ul_pwr_ctrl_common_v1530_s> ul_pwr_ctrl_common_scell_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicatedSCell-r10 ::= SEQUENCE
struct rr_cfg_ded_scell_r10_s {
  bool                     ext                            = false;
  bool                     phys_cfg_ded_scell_r10_present = false;
  phys_cfg_ded_scell_r10_s phys_cfg_ded_scell_r10;
  // ...
  // group 0
  copy_ptr<mac_main_cfg_scell_r11_s> mac_main_cfg_scell_r11;
  // group 1
  copy_ptr<naics_assist_info_r12_c> naics_info_r12;
  // group 2
  copy_ptr<neigh_cells_crs_info_r13_c> neigh_cells_crs_info_scell_r13;
  // group 3
  copy_ptr<phys_cfg_ded_scell_v1370_s> phys_cfg_ded_scell_v1370;
  // group 4
  bool                                 crs_intf_mitig_enabled_r15_present = false;
  bool                                 crs_intf_mitig_enabled_r15         = false;
  copy_ptr<neigh_cells_crs_info_r15_c> neigh_cells_crs_info_r15;
  copy_ptr<sps_cfg_v1530_s>            sps_cfg_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddModExt-r13 ::= SEQUENCE
struct scell_to_add_mod_ext_r13_s {
  struct cell_identif_r13_s_ {
    uint16_t pci_r13             = 0;
    uint32_t dl_carrier_freq_r13 = 0;
  };

  // member variables
  bool                      cell_identif_r13_present        = false;
  bool                      rr_cfg_common_scell_r13_present = false;
  bool                      rr_cfg_ded_scell_r13_present    = false;
  bool                      ant_info_ded_scell_r13_present  = false;
  uint8_t                   scell_idx_r13                   = 1;
  cell_identif_r13_s_       cell_identif_r13;
  rr_cfg_common_scell_r10_s rr_cfg_common_scell_r13;
  rr_cfg_ded_scell_r10_s    rr_cfg_ded_scell_r13;
  ant_info_ded_v10i0_s      ant_info_ded_scell_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CipheringAlgorithm-r12 ::= ENUMERATED
struct ciphering_algorithm_r12_opts {
  enum options { eea0, eea1, eea2, eea3_v1130, spare4, spare3, spare2, spare1, /*...*/ nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<ciphering_algorithm_r12_opts, true> ciphering_algorithm_r12_e;

// SCellConfigCommon-r15 ::= SEQUENCE
struct scell_cfg_common_r15_s {
  bool                      rr_cfg_common_scell_r15_present = false;
  bool                      rr_cfg_ded_scell_r15_present    = false;
  bool                      ant_info_ded_scell_r15_present  = false;
  rr_cfg_common_scell_r10_s rr_cfg_common_scell_r15;
  rr_cfg_ded_scell_r10_s    rr_cfg_ded_scell_r15;
  ant_info_ded_v10i0_s      ant_info_ded_scell_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddModListExt-r13 ::= SEQUENCE (SIZE (1..31)) OF SCellToAddModExt-r13
using scell_to_add_mod_list_ext_r13_l = dyn_array<scell_to_add_mod_ext_r13_s>;

// SCellToReleaseListExt-r13 ::= SEQUENCE (SIZE (1..31)) OF INTEGER (1..31)
using scell_to_release_list_ext_r13_l = bounded_array<uint8_t, 31>;

// SL-DiscTxPoolToAddMod-r12 ::= SEQUENCE
struct sl_disc_tx_pool_to_add_mod_r12_s {
  uint8_t                pool_id_r12 = 1;
  sl_disc_res_pool_r12_s pool_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TF-IndexPair-r12b ::= SEQUENCE
struct sl_tf_idx_pair_r12b_s {
  bool    disc_sf_idx_r12b_present  = false;
  bool    disc_prb_idx_r12b_present = false;
  uint8_t disc_sf_idx_r12b          = 0;
  uint8_t disc_prb_idx_r12b         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellGroupToAddMod-r15 ::= SEQUENCE
struct scell_group_to_add_mod_r15_s {
  bool                            scell_cfg_common_r15_present      = false;
  bool                            scell_to_release_list_r15_present = false;
  bool                            scell_to_add_mod_list_r15_present = false;
  uint8_t                         scell_group_idx_r15               = 1;
  scell_cfg_common_r15_s          scell_cfg_common_r15;
  scell_to_release_list_ext_r13_l scell_to_release_list_r15;
  scell_to_add_mod_list_ext_r13_l scell_to_add_mod_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscTxPoolToAddModList-r12 ::= SEQUENCE (SIZE (1..4)) OF SL-DiscTxPoolToAddMod-r12
using sl_disc_tx_pool_to_add_mod_list_r12_l = dyn_array<sl_disc_tx_pool_to_add_mod_r12_s>;

// SL-HoppingConfigDisc-r12 ::= SEQUENCE
struct sl_hop_cfg_disc_r12_s {
  struct c_r12_opts {
    enum options { n1, n5, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<c_r12_opts> c_r12_e_;

  // member variables
  uint8_t  a_r12 = 1;
  uint8_t  b_r12 = 1;
  c_r12_e_ c_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TF-IndexPairList-r12b ::= SEQUENCE (SIZE (1..64)) OF SL-TF-IndexPair-r12b
using sl_tf_idx_pair_list_r12b_l = dyn_array<sl_tf_idx_pair_r12b_s>;

// SL-TxPoolToReleaseList-r12 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..4)
using sl_tx_pool_to_release_list_r12_l = bounded_array<uint8_t, 4>;

// SecurityAlgorithmConfig ::= SEQUENCE
struct security_algorithm_cfg_s {
  struct integrity_prot_algorithm_opts {
    enum options { eia0_v920, eia1, eia2, eia3_v1130, spare4, spare3, spare2, spare1, /*...*/ nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<integrity_prot_algorithm_opts, true> integrity_prot_algorithm_e_;

  // member variables
  ciphering_algorithm_r12_e   ciphering_algorithm;
  integrity_prot_algorithm_e_ integrity_prot_algorithm;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRB-ToAddModSCG-r12 ::= SEQUENCE
struct drb_to_add_mod_scg_r12_s {
  struct drb_type_r12_c_ {
    struct scg_r12_s_ {
      bool       eps_bearer_id_r12_present = false;
      bool       pdcp_cfg_r12_present      = false;
      uint8_t    eps_bearer_id_r12         = 0;
      pdcp_cfg_s pdcp_cfg_r12;
    };
    struct types_opts {
      enum options { split_r12, scg_r12, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    drb_type_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    scg_r12_s_& scg_r12()
    {
      assert_choice_type("scg-r12", type_.to_string(), "drb-Type-r12");
      return c;
    }
    const scg_r12_s_& scg_r12() const
    {
      assert_choice_type("scg-r12", type_.to_string(), "drb-Type-r12");
      return c;
    }
    scg_r12_s_& set_scg_r12()
    {
      set(types::scg_r12);
      return c;
    }

  private:
    types      type_;
    scg_r12_s_ c;
  };

  // member variables
  bool            ext                       = false;
  bool            drb_type_r12_present      = false;
  bool            rlc_cfg_scg_r12_present   = false;
  bool            rlc_cfg_v1250_present     = false;
  bool            lc_ch_id_scg_r12_present  = false;
  bool            lc_ch_cfg_scg_r12_present = false;
  uint8_t         drb_id_r12                = 1;
  drb_type_r12_c_ drb_type_r12;
  rlc_cfg_c       rlc_cfg_scg_r12;
  rlc_cfg_v1250_s rlc_cfg_v1250;
  uint8_t         lc_ch_id_scg_r12 = 3;
  lc_ch_cfg_s     lc_ch_cfg_scg_r12;
  // ...
  // group 0
  copy_ptr<rlc_cfg_v1430_c> rlc_cfg_v1430;
  // group 1
  bool                           lc_ch_id_scg_r15_present = false;
  uint8_t                        lc_ch_id_scg_r15         = 32;
  copy_ptr<rlc_cfg_v1530_c>      rlc_cfg_v1530;
  copy_ptr<rlc_bearer_cfg_r15_c> rlc_bearer_cfg_dupl_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// IKE-Identity-r13 ::= SEQUENCE
struct ike_id_r13_s {
  dyn_octstring id_i_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// IP-Address-r13 ::= CHOICE
struct ip_address_r13_c {
  struct types_opts {
    enum options { ipv4_r13, ipv6_r13, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  ip_address_r13_c() = default;
  ip_address_r13_c(const ip_address_r13_c& other);
  ip_address_r13_c& operator=(const ip_address_r13_c& other);
  ~ip_address_r13_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  fixed_bitstring<32>& ipv4_r13()
  {
    assert_choice_type("ipv4-r13", type_.to_string(), "IP-Address-r13");
    return c.get<fixed_bitstring<32> >();
  }
  fixed_bitstring<128>& ipv6_r13()
  {
    assert_choice_type("ipv6-r13", type_.to_string(), "IP-Address-r13");
    return c.get<fixed_bitstring<128> >();
  }
  const fixed_bitstring<32>& ipv4_r13() const
  {
    assert_choice_type("ipv4-r13", type_.to_string(), "IP-Address-r13");
    return c.get<fixed_bitstring<32> >();
  }
  const fixed_bitstring<128>& ipv6_r13() const
  {
    assert_choice_type("ipv6-r13", type_.to_string(), "IP-Address-r13");
    return c.get<fixed_bitstring<128> >();
  }
  fixed_bitstring<32>& set_ipv4_r13()
  {
    set(types::ipv4_r13);
    return c.get<fixed_bitstring<32> >();
  }
  fixed_bitstring<128>& set_ipv6_r13()
  {
    set(types::ipv6_r13);
    return c.get<fixed_bitstring<128> >();
  }

private:
  types                                  type_;
  choice_buffer_t<fixed_bitstring<128> > c;

  void destroy_();
};

// PhysicalConfigDedicated-v1370 ::= SEQUENCE
struct phys_cfg_ded_v1370_s {
  bool                  pucch_cfg_ded_v1370_present = false;
  pucch_cfg_ded_v1370_s pucch_cfg_ded_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RAN-AreaConfig-r15 ::= SEQUENCE
struct ran_area_cfg_r15_s {
  using ran_area_code_list_r15_l_ = bounded_array<uint16_t, 32>;

  // member variables
  bool                      ran_area_code_list_r15_present = false;
  fixed_bitstring<24>       tac_minus5_gc_r15;
  ran_area_code_list_r15_l_ ran_area_code_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonSCell-v10l0 ::= SEQUENCE
struct rr_cfg_common_scell_v10l0_s {
  struct ul_cfg_v10l0_s_ {
    uint16_t add_spec_emission_scell_v10l0 = 33;
  };

  // member variables
  ul_cfg_v10l0_s_ ul_cfg_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonSCell-v1440 ::= SEQUENCE
struct rr_cfg_common_scell_v1440_s {
  struct ul_cfg_v1440_s_ {
    struct ul_freq_info_v1440_s_ {
      uint16_t add_spec_emission_scell_v1440 = 33;
    };

    // member variables
    ul_freq_info_v1440_s_ ul_freq_info_v1440;
  };

  // member variables
  ul_cfg_v1440_s_ ul_cfg_v1440;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellGroupToAddModList-r15 ::= SEQUENCE (SIZE (1..4)) OF SCellGroupToAddMod-r15
using scell_group_to_add_mod_list_r15_l = dyn_array<scell_group_to_add_mod_r15_s>;

// SCellGroupToReleaseList-r15 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..4)
using scell_group_to_release_list_r15_l = bounded_array<uint8_t, 4>;

// SL-DiscTxConfigScheduled-r13 ::= SEQUENCE
struct sl_disc_tx_cfg_sched_r13_s {
  bool                       ext                          = false;
  bool                       disc_tx_cfg_r13_present      = false;
  bool                       disc_tf_idx_list_r13_present = false;
  bool                       disc_hop_cfg_r13_present     = false;
  sl_disc_res_pool_r12_s     disc_tx_cfg_r13;
  sl_tf_idx_pair_list_r12b_l disc_tf_idx_list_r13;
  sl_hop_cfg_disc_r12_s      disc_hop_cfg_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscTxPoolDedicated-r13 ::= SEQUENCE
struct sl_disc_tx_pool_ded_r13_s {
  bool                                  pool_to_release_list_r13_present = false;
  bool                                  pool_to_add_mod_list_r13_present = false;
  sl_tx_pool_to_release_list_r12_l      pool_to_release_list_r13;
  sl_disc_tx_pool_to_add_mod_list_r12_l pool_to_add_mod_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-ReliabilityList-r15 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..8)
using sl_reliability_list_r15_l = bounded_array<uint8_t, 8>;

// SL-TxPoolToAddMod-r14 ::= SEQUENCE
struct sl_tx_pool_to_add_mod_r14_s {
  uint8_t                    pool_id_r14 = 1;
  sl_comm_res_pool_v2x_r14_s pool_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SecurityConfigHO-v1530 ::= SEQUENCE
struct security_cfg_ho_v1530_s {
  struct handov_type_v1530_c_ {
    struct intra5_gc_r15_s_ {
      bool                     security_algorithm_cfg_r15_present = false;
      bool                     nas_container_r15_present          = false;
      security_algorithm_cfg_s security_algorithm_cfg_r15;
      bool                     key_change_ind_r15          = false;
      uint8_t                  next_hop_chaining_count_r15 = 0;
      dyn_octstring            nas_container_r15;
    };
    struct ngc_to_epc_r15_s_ {
      security_algorithm_cfg_s security_algorithm_cfg_r15;
      uint8_t                  next_hop_chaining_count_r15 = 0;
    };
    struct epc_to_ngc_r15_s_ {
      security_algorithm_cfg_s security_algorithm_cfg_r15;
      dyn_octstring            nas_container_r15;
    };
    struct types_opts {
      enum options { intra5_gc_r15, ngc_to_epc_r15, epc_to_ngc_r15, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    handov_type_v1530_c_() = default;
    handov_type_v1530_c_(const handov_type_v1530_c_& other);
    handov_type_v1530_c_& operator=(const handov_type_v1530_c_& other);
    ~handov_type_v1530_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    intra5_gc_r15_s_& intra5_gc_r15()
    {
      assert_choice_type("intra5GC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<intra5_gc_r15_s_>();
    }
    ngc_to_epc_r15_s_& ngc_to_epc_r15()
    {
      assert_choice_type("ngc-ToEPC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<ngc_to_epc_r15_s_>();
    }
    epc_to_ngc_r15_s_& epc_to_ngc_r15()
    {
      assert_choice_type("epc-ToNGC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<epc_to_ngc_r15_s_>();
    }
    const intra5_gc_r15_s_& intra5_gc_r15() const
    {
      assert_choice_type("intra5GC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<intra5_gc_r15_s_>();
    }
    const ngc_to_epc_r15_s_& ngc_to_epc_r15() const
    {
      assert_choice_type("ngc-ToEPC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<ngc_to_epc_r15_s_>();
    }
    const epc_to_ngc_r15_s_& epc_to_ngc_r15() const
    {
      assert_choice_type("epc-ToNGC-r15", type_.to_string(), "handoverType-v1530");
      return c.get<epc_to_ngc_r15_s_>();
    }
    intra5_gc_r15_s_& set_intra5_gc_r15()
    {
      set(types::intra5_gc_r15);
      return c.get<intra5_gc_r15_s_>();
    }
    ngc_to_epc_r15_s_& set_ngc_to_epc_r15()
    {
      set(types::ngc_to_epc_r15);
      return c.get<ngc_to_epc_r15_s_>();
    }
    epc_to_ngc_r15_s_& set_epc_to_ngc_r15()
    {
      set(types::epc_to_ngc_r15);
      return c.get<epc_to_ngc_r15_s_>();
    }

  private:
    types                                                                   type_;
    choice_buffer_t<epc_to_ngc_r15_s_, intra5_gc_r15_s_, ngc_to_epc_r15_s_> c;

    void destroy_();
  };

  // member variables
  bool                 ext = false;
  handov_type_v1530_c_ handov_type_v1530;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommon-v1310 ::= SEQUENCE
struct ul_pwr_ctrl_common_v1310_s {
  struct delta_f_pucch_format4_r13_opts {
    enum options {
      delta_f16,
      delta_f15,
      delta_f14,
      delta_f13,
      delta_f12,
      delta_f11,
      delta_f10,
      spare1,
      nulltype
    } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format4_r13_opts> delta_f_pucch_format4_r13_e_;
  struct delta_f_pucch_format5_minus13_opts {
    enum options { delta_f13, delta_f12, delta_f11, delta_f10, delta_f9, delta_f8, delta_f7, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format5_minus13_opts> delta_f_pucch_format5_minus13_e_;

  // member variables
  bool                             delta_f_pucch_format4_r13_present     = false;
  bool                             delta_f_pucch_format5_minus13_present = false;
  delta_f_pucch_format4_r13_e_     delta_f_pucch_format4_r13;
  delta_f_pucch_format5_minus13_e_ delta_f_pucch_format5_minus13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UplinkPowerControlCommonPSCell-r12 ::= SEQUENCE
struct ul_pwr_ctrl_common_ps_cell_r12_s {
  struct delta_f_pucch_format3_r12_opts {
    enum options {
      delta_f_minus1,
      delta_f0,
      delta_f1,
      delta_f2,
      delta_f3,
      delta_f4,
      delta_f5,
      delta_f6,
      nulltype
    } value;
    typedef int8_t number_type;

    std::string to_string() const;
    int8_t      to_number() const;
  };
  typedef enumerated<delta_f_pucch_format3_r12_opts> delta_f_pucch_format3_r12_e_;
  struct delta_f_pucch_format1b_cs_r12_opts {
    enum options { delta_f1, delta_f2, spare2, spare1, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<delta_f_pucch_format1b_cs_r12_opts> delta_f_pucch_format1b_cs_r12_e_;

  // member variables
  delta_f_pucch_format3_r12_e_     delta_f_pucch_format3_r12;
  delta_f_pucch_format1b_cs_r12_e_ delta_f_pucch_format1b_cs_r12;
  int8_t                           p0_nominal_pucch_r12 = -127;
  delta_flist_pucch_s              delta_flist_pucch_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-Id-List-r13 ::= SEQUENCE (SIZE (1..32)) OF WLAN-Identifiers-r12
using wlan_id_list_r13_l = dyn_array<wlan_ids_r12_s>;

// WLAN-SuspendConfig-r14 ::= SEQUENCE
struct wlan_suspend_cfg_r14_s {
  bool wlan_suspend_resume_allowed_r14_present         = false;
  bool wlan_suspend_triggers_status_report_r14_present = false;
  bool wlan_suspend_resume_allowed_r14                 = false;
  bool wlan_suspend_triggers_status_report_r14         = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// DRB-ToAddModListSCG-r12 ::= SEQUENCE (SIZE (1..11)) OF DRB-ToAddModSCG-r12
using drb_to_add_mod_list_scg_r12_l = dyn_array<drb_to_add_mod_scg_r12_s>;

// DRB-ToAddModListSCG-r15 ::= SEQUENCE (SIZE (1..15)) OF DRB-ToAddModSCG-r12
using drb_to_add_mod_list_scg_r15_l = dyn_array<drb_to_add_mod_scg_r12_s>;

// LogicalChGroupInfoList-r13 ::= SEQUENCE (SIZE (1..4)) OF SL-PriorityList-r13
using lc_ch_group_info_list_r13_l = dyn_array<sl_prio_list_r13_l>;

// LogicalChGroupInfoList-v1530 ::= SEQUENCE (SIZE (1..4)) OF SL-ReliabilityList-r15
using lc_ch_group_info_list_v1530_l = dyn_array<sl_reliability_list_r15_l>;

// MAC-MainConfigSL-r12 ::= SEQUENCE
struct mac_main_cfg_sl_r12_s {
  bool                     periodic_bsr_timer_sl_present = false;
  periodic_bsr_timer_r12_e periodic_bsr_timer_sl;
  retx_bsr_timer_r12_e     retx_bsr_timer_sl;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-RAN-AreaCell-r15 ::= SEQUENCE
struct plmn_ran_area_cell_r15_s {
  using ran_area_cells_r15_l_ = bounded_array<fixed_bitstring<28>, 32>;

  // member variables
  bool                  plmn_id_r15_present = false;
  plmn_id_s             plmn_id_r15;
  ran_area_cells_r15_l_ ran_area_cells_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-RAN-AreaConfig-r15 ::= SEQUENCE
struct plmn_ran_area_cfg_r15_s {
  using ran_area_r15_l_ = dyn_array<ran_area_cfg_r15_s>;

  // member variables
  bool            plmn_id_r15_present = false;
  plmn_id_s       plmn_id_r15;
  ran_area_r15_l_ ran_area_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-ConfigDedicated ::= SEQUENCE
struct rach_cfg_ded_s {
  uint8_t ra_preamb_idx     = 0;
  uint8_t ra_prach_mask_idx = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RACH-Skip-r14 ::= SEQUENCE
struct rach_skip_r14_s {
  struct target_ta_r14_c_ {
    struct types_opts {
      enum options { ta0_r14, mcg_ptag_r14, scg_ptag_r14, mcg_stag_r14, scg_stag_r14, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    target_ta_r14_c_() = default;
    target_ta_r14_c_(const target_ta_r14_c_& other);
    target_ta_r14_c_& operator=(const target_ta_r14_c_& other);
    ~target_ta_r14_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    uint8_t& mcg_stag_r14()
    {
      assert_choice_type("mcg-STAG-r14", type_.to_string(), "targetTA-r14");
      return c.get<uint8_t>();
    }
    uint8_t& scg_stag_r14()
    {
      assert_choice_type("scg-STAG-r14", type_.to_string(), "targetTA-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& mcg_stag_r14() const
    {
      assert_choice_type("mcg-STAG-r14", type_.to_string(), "targetTA-r14");
      return c.get<uint8_t>();
    }
    const uint8_t& scg_stag_r14() const
    {
      assert_choice_type("scg-STAG-r14", type_.to_string(), "targetTA-r14");
      return c.get<uint8_t>();
    }
    uint8_t& set_mcg_stag_r14()
    {
      set(types::mcg_stag_r14);
      return c.get<uint8_t>();
    }
    uint8_t& set_scg_stag_r14()
    {
      set(types::scg_stag_r14);
      return c.get<uint8_t>();
    }

  private:
    types               type_;
    pod_choice_buffer_t c;

    void destroy_();
  };
  struct ul_cfg_info_r14_s_ {
    struct ul_sched_interv_r14_opts {
      enum options { sf2, sf5, sf10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<ul_sched_interv_r14_opts> ul_sched_interv_r14_e_;

    // member variables
    uint8_t                nof_conf_ul_processes_r14 = 1;
    ul_sched_interv_r14_e_ ul_sched_interv_r14;
    uint8_t                ul_start_sf_r14 = 0;
    fixed_bitstring<16>    ul_grant_r14;
  };

  // member variables
  bool               ul_cfg_info_r14_present = false;
  target_ta_r14_c_   target_ta_r14;
  ul_cfg_info_r14_s_ ul_cfg_info_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RLF-TimersAndConstantsSCG-r12 ::= CHOICE
struct rlf_timers_and_consts_scg_r12_c {
  struct setup_s_ {
    struct t313_r12_opts {
      enum options { ms0, ms50, ms100, ms200, ms500, ms1000, ms2000, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<t313_r12_opts> t313_r12_e_;
    struct n313_r12_opts {
      enum options { n1, n2, n3, n4, n6, n8, n10, n20, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<n313_r12_opts> n313_r12_e_;
    struct n314_r12_opts {
      enum options { n1, n2, n3, n4, n5, n6, n8, n10, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<n314_r12_opts> n314_r12_e_;

    // member variables
    bool        ext = false;
    t313_r12_e_ t313_r12;
    n313_r12_e_ n313_r12;
    n314_r12_e_ n314_r12;
    // ...
  };
  typedef setup_e types;

  // choice methods
  rlf_timers_and_consts_scg_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstantsSCG-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RLF-TimersAndConstantsSCG-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReconfiguration-v1530-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1530_ies_s {
  using ded_info_nas_list_r15_l_ = bounded_array<dyn_octstring, 15>;

  // member variables
  bool                              security_cfg_ho_v1530_present           = false;
  bool                              scell_group_to_release_list_r15_present = false;
  bool                              scell_group_to_add_mod_list_r15_present = false;
  bool                              ded_info_nas_list_r15_present           = false;
  bool                              p_max_ue_fr1_r15_present                = false;
  bool                              smtc_r15_present                        = false;
  bool                              non_crit_ext_present                    = false;
  security_cfg_ho_v1530_s           security_cfg_ho_v1530;
  scell_group_to_release_list_r15_l scell_group_to_release_list_r15;
  scell_group_to_add_mod_list_r15_l scell_group_to_add_mod_list_r15;
  ded_info_nas_list_r15_l_          ded_info_nas_list_r15;
  int8_t                            p_max_ue_fr1_r15 = -30;
  mtc_ssb_nr_r15_s                  smtc_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonPSCell-r12 ::= SEQUENCE
struct rr_cfg_common_ps_cell_r12_s {
  bool                             ext = false;
  rr_cfg_common_scell_r10_s        basic_fields_r12;
  pucch_cfg_common_s               pucch_cfg_common_r12;
  rach_cfg_common_s                rach_cfg_common_r12;
  ul_pwr_ctrl_common_ps_cell_r12_s ul_pwr_ctrl_common_ps_cell_r12;
  // ...
  // group 0
  copy_ptr<ul_pwr_ctrl_common_v1310_s> ul_pwr_ctrl_common_ps_cell_v1310;
  // group 1
  copy_ptr<ul_pwr_ctrl_common_v1530_s> ul_pwr_ctrl_common_ps_cell_v1530;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonPSCell-v12f0 ::= SEQUENCE
struct rr_cfg_common_ps_cell_v12f0_s {
  rr_cfg_common_scell_v10l0_s basic_fields_v12f0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigCommonPSCell-v1440 ::= SEQUENCE
struct rr_cfg_common_ps_cell_v1440_s {
  rr_cfg_common_scell_v1440_s basic_fields_v1440;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicatedPSCell-r12 ::= SEQUENCE
struct rr_cfg_ded_ps_cell_r12_s {
  bool                    ext                              = false;
  bool                    phys_cfg_ded_ps_cell_r12_present = false;
  bool                    sps_cfg_r12_present              = false;
  bool                    naics_info_r12_present           = false;
  phys_cfg_ded_s          phys_cfg_ded_ps_cell_r12;
  sps_cfg_s               sps_cfg_r12;
  naics_assist_info_r12_c naics_info_r12;
  // ...
  // group 0
  copy_ptr<neigh_cells_crs_info_r13_c> neigh_cells_crs_info_ps_cell_r13;
  // group 1
  copy_ptr<sps_cfg_v1430_s> sps_cfg_v1430;
  // group 2
  bool                                 crs_intf_mitig_enabled_r15_present = false;
  copy_ptr<sps_cfg_v1530_s>            sps_cfg_v1530;
  bool                                 crs_intf_mitig_enabled_r15 = false;
  copy_ptr<neigh_cells_crs_info_r15_c> neigh_cells_crs_info_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicatedPSCell-v1370 ::= SEQUENCE
struct rr_cfg_ded_ps_cell_v1370_s {
  bool                 phys_cfg_ded_ps_cell_v1370_present = false;
  phys_cfg_ded_v1370_s phys_cfg_ded_ps_cell_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddMod-r10 ::= SEQUENCE
struct scell_to_add_mod_r10_s {
  struct cell_identif_r10_s_ {
    uint16_t pci_r10             = 0;
    uint32_t dl_carrier_freq_r10 = 0;
  };
  struct scell_state_r15_opts {
    enum options { activ, dormant, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<scell_state_r15_opts> scell_state_r15_e_;

  // member variables
  bool                      ext                             = false;
  bool                      cell_identif_r10_present        = false;
  bool                      rr_cfg_common_scell_r10_present = false;
  bool                      rr_cfg_ded_scell_r10_present    = false;
  uint8_t                   scell_idx_r10                   = 1;
  cell_identif_r10_s_       cell_identif_r10;
  rr_cfg_common_scell_r10_s rr_cfg_common_scell_r10;
  rr_cfg_ded_scell_r10_s    rr_cfg_ded_scell_r10;
  // ...
  // group 0
  bool     dl_carrier_freq_v1090_present = false;
  uint32_t dl_carrier_freq_v1090         = 65536;
  // group 1
  copy_ptr<ant_info_ded_v10i0_s> ant_info_ded_scell_v10i0;
  // group 2
  bool    srs_switch_from_serv_cell_idx_r14_present = false;
  uint8_t srs_switch_from_serv_cell_idx_r14         = 0;
  // group 3
  bool               scell_state_r15_present = false;
  scell_state_r15_e_ scell_state_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddMod-v10l0 ::= SEQUENCE
struct scell_to_add_mod_v10l0_s {
  bool                        rr_cfg_common_scell_v10l0_present = false;
  rr_cfg_common_scell_v10l0_s rr_cfg_common_scell_v10l0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddModExt-v1370 ::= SEQUENCE
struct scell_to_add_mod_ext_v1370_s {
  bool                        rr_cfg_common_scell_v1370_present = false;
  rr_cfg_common_scell_v10l0_s rr_cfg_common_scell_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddModExt-v1430 ::= SEQUENCE
struct scell_to_add_mod_ext_v1430_s {
  struct scell_state_r15_opts {
    enum options { activ, dormant, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<scell_state_r15_opts> scell_state_r15_e_;

  // member variables
  bool    ext                                       = false;
  bool    srs_switch_from_serv_cell_idx_r14_present = false;
  uint8_t srs_switch_from_serv_cell_idx_r14         = 0;
  // ...
  // group 0
  bool               scell_state_r15_present = false;
  scell_state_r15_e_ scell_state_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscTxRefCarrierDedicated-r13 ::= CHOICE
struct sl_disc_tx_ref_carrier_ded_r13_c {
  struct types_opts {
    enum options { pcell, scell, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  sl_disc_tx_ref_carrier_ded_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  uint8_t& scell()
  {
    assert_choice_type("sCell", type_.to_string(), "SL-DiscTxRefCarrierDedicated-r13");
    return c;
  }
  const uint8_t& scell() const
  {
    assert_choice_type("sCell", type_.to_string(), "SL-DiscTxRefCarrierDedicated-r13");
    return c;
  }
  uint8_t& set_scell()
  {
    set(types::scell);
    return c;
  }

private:
  types   type_;
  uint8_t c;
};

// SL-DiscTxResource-r13 ::= CHOICE
struct sl_disc_tx_res_r13_c {
  struct setup_c_ {
    struct types_opts {
      enum options { sched_r13, ue_sel_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    setup_c_() = default;
    setup_c_(const setup_c_& other);
    setup_c_& operator=(const setup_c_& other);
    ~setup_c_() { destroy_(); }
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sl_disc_tx_cfg_sched_r13_s& sched_r13()
    {
      assert_choice_type("scheduled-r13", type_.to_string(), "setup");
      return c.get<sl_disc_tx_cfg_sched_r13_s>();
    }
    sl_disc_tx_pool_ded_r13_s& ue_sel_r13()
    {
      assert_choice_type("ue-Selected-r13", type_.to_string(), "setup");
      return c.get<sl_disc_tx_pool_ded_r13_s>();
    }
    const sl_disc_tx_cfg_sched_r13_s& sched_r13() const
    {
      assert_choice_type("scheduled-r13", type_.to_string(), "setup");
      return c.get<sl_disc_tx_cfg_sched_r13_s>();
    }
    const sl_disc_tx_pool_ded_r13_s& ue_sel_r13() const
    {
      assert_choice_type("ue-Selected-r13", type_.to_string(), "setup");
      return c.get<sl_disc_tx_pool_ded_r13_s>();
    }
    sl_disc_tx_cfg_sched_r13_s& set_sched_r13()
    {
      set(types::sched_r13);
      return c.get<sl_disc_tx_cfg_sched_r13_s>();
    }
    sl_disc_tx_pool_ded_r13_s& set_ue_sel_r13()
    {
      set(types::ue_sel_r13);
      return c.get<sl_disc_tx_pool_ded_r13_s>();
    }

  private:
    types                                                                  type_;
    choice_buffer_t<sl_disc_tx_cfg_sched_r13_s, sl_disc_tx_pool_ded_r13_s> c;

    void destroy_();
  };
  typedef setup_e types;

  // choice methods
  sl_disc_tx_res_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_c_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SL-DiscTxResource-r13");
    return c;
  }
  const setup_c_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SL-DiscTxResource-r13");
    return c;
  }
  setup_c_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_c_ c;
};

// SL-GapPattern-r13 ::= SEQUENCE
struct sl_gap_pattern_r13_s {
  struct gap_period_r13_opts {
    enum options {
      sf40,
      sf60,
      sf70,
      sf80,
      sf120,
      sf140,
      sf160,
      sf240,
      sf280,
      sf320,
      sf640,
      sf1280,
      sf2560,
      sf5120,
      sf10240,
      nulltype
    } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<gap_period_r13_opts> gap_period_r13_e_;

  // member variables
  bool                        ext = false;
  gap_period_r13_e_           gap_period_r13;
  sl_offset_ind_r12_c         gap_offset_r12;
  bounded_bitstring<1, 10240> gap_sf_bitmap_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TxPoolToAddModListV2X-r14 ::= SEQUENCE (SIZE (1..8)) OF SL-TxPoolToAddMod-r14
using sl_tx_pool_to_add_mod_list_v2x_r14_l = dyn_array<sl_tx_pool_to_add_mod_r14_s>;

// SL-TxPoolToReleaseListV2X-r14 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (1..8)
using sl_tx_pool_to_release_list_v2x_r14_l = bounded_array<uint8_t, 8>;

// SubframeAssignment-r15 ::= ENUMERATED
struct sf_assign_r15_opts {
  enum options { sa0, sa1, sa2, sa3, sa4, sa5, sa6, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<sf_assign_r15_opts> sf_assign_r15_e;

// TunnelConfigLWIP-r13 ::= SEQUENCE
struct tunnel_cfg_lwip_r13_s {
  bool             ext = false;
  ip_address_r13_c ip_address_r13;
  ike_id_r13_s     ike_id_r13;
  // ...
  // group 0
  bool     lwip_counter_r13_present = false;
  uint32_t lwip_counter_r13         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-MobilityConfig-r13 ::= SEQUENCE
struct wlan_mob_cfg_r13_s {
  struct assoc_timer_r13_opts {
    enum options { s10, s30, s60, s120, s240, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<assoc_timer_r13_opts> assoc_timer_r13_e_;

  // member variables
  bool               ext                                  = false;
  bool               wlan_to_release_list_r13_present     = false;
  bool               wlan_to_add_list_r13_present         = false;
  bool               assoc_timer_r13_present              = false;
  bool               success_report_requested_r13_present = false;
  wlan_id_list_r13_l wlan_to_release_list_r13;
  wlan_id_list_r13_l wlan_to_add_list_r13;
  assoc_timer_r13_e_ assoc_timer_r13;
  // ...
  // group 0
  copy_ptr<wlan_suspend_cfg_r14_s> wlan_suspend_cfg_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CA-BandwidthClass-r10 ::= ENUMERATED
struct ca_bw_class_r10_opts {
  enum options { a, b, c, d, e, f, /*...*/ nulltype } value;

  std::string to_string() const;
};
typedef enumerated<ca_bw_class_r10_opts, true> ca_bw_class_r10_e;

// LWA-Config-r13 ::= SEQUENCE
struct lwa_cfg_r13_s {
  bool               ext                        = false;
  bool               lwa_mob_cfg_r13_present    = false;
  bool               lwa_wt_counter_r13_present = false;
  wlan_mob_cfg_r13_s lwa_mob_cfg_r13;
  uint32_t           lwa_wt_counter_r13 = 0;
  // ...
  // group 0
  bool               wt_mac_address_r14_present = false;
  fixed_octstring<6> wt_mac_address_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// LWIP-Config-r13 ::= SEQUENCE
struct lwip_cfg_r13_s {
  bool                  ext                         = false;
  bool                  lwip_mob_cfg_r13_present    = false;
  bool                  tunnel_cfg_lwip_r13_present = false;
  wlan_mob_cfg_r13_s    lwip_mob_cfg_r13;
  tunnel_cfg_lwip_r13_s tunnel_cfg_lwip_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MobilityControlInfoSCG-r12 ::= SEQUENCE
struct mob_ctrl_info_scg_r12_s {
  struct t307_r12_opts {
    enum options { ms50, ms100, ms150, ms200, ms500, ms1000, ms2000, spare1, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<t307_r12_opts> t307_r12_e_;

  // member variables
  bool                      ext                                 = false;
  bool                      ue_id_scg_r12_present               = false;
  bool                      rach_cfg_ded_r12_present            = false;
  bool                      ciphering_algorithm_scg_r12_present = false;
  t307_r12_e_               t307_r12;
  fixed_bitstring<16>       ue_id_scg_r12;
  rach_cfg_ded_s            rach_cfg_ded_r12;
  ciphering_algorithm_r12_e ciphering_algorithm_scg_r12;
  // ...
  // group 0
  bool                      make_before_break_scg_r14_present = false;
  copy_ptr<rach_skip_r14_s> rach_skip_scg_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PLMN-RAN-AreaCellList-r15 ::= SEQUENCE (SIZE (1..8)) OF PLMN-RAN-AreaCell-r15
using plmn_ran_area_cell_list_r15_l = dyn_array<plmn_ran_area_cell_r15_s>;

// PLMN-RAN-AreaConfigList-r15 ::= SEQUENCE (SIZE (1..8)) OF PLMN-RAN-AreaConfig-r15
using plmn_ran_area_cfg_list_r15_l = dyn_array<plmn_ran_area_cfg_r15_s>;

// PSCellToAddMod-r12 ::= SEQUENCE
struct ps_cell_to_add_mod_r12_s {
  struct cell_identif_r12_s_ {
    uint16_t pci_r12             = 0;
    uint32_t dl_carrier_freq_r12 = 0;
  };

  // member variables
  bool                        ext                               = false;
  bool                        cell_identif_r12_present          = false;
  bool                        rr_cfg_common_ps_cell_r12_present = false;
  bool                        rr_cfg_ded_ps_cell_r12_present    = false;
  uint8_t                     scell_idx_r12                     = 1;
  cell_identif_r12_s_         cell_identif_r12;
  rr_cfg_common_ps_cell_r12_s rr_cfg_common_ps_cell_r12;
  rr_cfg_ded_ps_cell_r12_s    rr_cfg_ded_ps_cell_r12;
  // ...
  // group 0
  copy_ptr<ant_info_ded_v10i0_s> ant_info_ded_ps_cell_v1280;
  // group 1
  bool    scell_idx_r13_present = false;
  uint8_t scell_idx_r13         = 1;
  // group 2
  copy_ptr<rr_cfg_ded_ps_cell_v1370_s> rr_cfg_ded_ps_cell_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PSCellToAddMod-v12f0 ::= SEQUENCE
struct ps_cell_to_add_mod_v12f0_s {
  bool                          rr_cfg_common_ps_cell_r12_present = false;
  rr_cfg_common_ps_cell_v12f0_s rr_cfg_common_ps_cell_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PSCellToAddMod-v1440 ::= SEQUENCE
struct ps_cell_to_add_mod_v1440_s {
  bool                          rr_cfg_common_ps_cell_r14_present = false;
  rr_cfg_common_ps_cell_v1440_s rr_cfg_common_ps_cell_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RCLWI-Config-r13 ::= SEQUENCE
struct rclwi_cfg_r13_s {
  struct cmd_c_ {
    struct steer_to_wlan_r13_s_ {
      wlan_id_list_r12_l mob_cfg_r13;
    };
    struct types_opts {
      enum options { steer_to_wlan_r13, steer_to_lte_r13, nulltype } value;

      std::string to_string() const;
    };
    typedef enumerated<types_opts> types;

    // choice methods
    cmd_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    steer_to_wlan_r13_s_& steer_to_wlan_r13()
    {
      assert_choice_type("steerToWLAN-r13", type_.to_string(), "command");
      return c;
    }
    const steer_to_wlan_r13_s_& steer_to_wlan_r13() const
    {
      assert_choice_type("steerToWLAN-r13", type_.to_string(), "command");
      return c;
    }
    steer_to_wlan_r13_s_& set_steer_to_wlan_r13()
    {
      set(types::steer_to_wlan_r13);
      return c;
    }

  private:
    types                type_;
    steer_to_wlan_r13_s_ c;
  };

  // member variables
  bool   ext = false;
  cmd_c_ cmd;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReconfiguration-v1510-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1510_ies_s {
  struct nr_cfg_r15_c_ {
    struct setup_s_ {
      bool          nr_secondary_cell_group_cfg_r15_present = false;
      bool          p_max_eutra_r15_present                 = false;
      bool          endc_release_and_add_r15                = false;
      dyn_octstring nr_secondary_cell_group_cfg_r15;
      int8_t        p_max_eutra_r15 = -30;
    };
    typedef setup_e types;

    // choice methods
    nr_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "nr-Config-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "nr-Config-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct tdm_pattern_cfg_r15_c_ {
    struct setup_s_ {
      sf_assign_r15_e sf_assign_r15;
      uint8_t         harq_offset_r15 = 0;
    };
    typedef setup_e types;

    // choice methods
    tdm_pattern_cfg_r15_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "tdm-PatternConfig-r15");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "tdm-PatternConfig-r15");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                       nr_cfg_r15_present               = false;
  bool                       sk_counter_r15_present           = false;
  bool                       nr_radio_bearer_cfg1_r15_present = false;
  bool                       nr_radio_bearer_cfg2_r15_present = false;
  bool                       tdm_pattern_cfg_r15_present      = false;
  bool                       non_crit_ext_present             = false;
  nr_cfg_r15_c_              nr_cfg_r15;
  uint32_t                   sk_counter_r15 = 0;
  dyn_octstring              nr_radio_bearer_cfg1_r15;
  dyn_octstring              nr_radio_bearer_cfg2_r15;
  tdm_pattern_cfg_r15_c_     tdm_pattern_cfg_r15;
  rrc_conn_recfg_v1530_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicatedSCG-r12 ::= SEQUENCE
struct rr_cfg_ded_scg_r12_s {
  bool                            ext                                   = false;
  bool                            drb_to_add_mod_list_scg_r12_present   = false;
  bool                            mac_main_cfg_scg_r12_present          = false;
  bool                            rlf_timers_and_consts_scg_r12_present = false;
  drb_to_add_mod_list_scg_r12_l   drb_to_add_mod_list_scg_r12;
  mac_main_cfg_s                  mac_main_cfg_scg_r12;
  rlf_timers_and_consts_scg_r12_c rlf_timers_and_consts_scg_r12;
  // ...
  // group 0
  copy_ptr<drb_to_add_mod_list_scg_r15_l> drb_to_add_mod_list_scg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCellToAddModList-r10 ::= SEQUENCE (SIZE (1..4)) OF SCellToAddMod-r10
using scell_to_add_mod_list_r10_l = dyn_array<scell_to_add_mod_r10_s>;

// SCellToAddModList-v10l0 ::= SEQUENCE (SIZE (1..4)) OF SCellToAddMod-v10l0
using scell_to_add_mod_list_v10l0_l = dyn_array<scell_to_add_mod_v10l0_s>;

// SCellToAddModListExt-v1370 ::= SEQUENCE (SIZE (1..31)) OF SCellToAddModExt-v1370
using scell_to_add_mod_list_ext_v1370_l = dyn_array<scell_to_add_mod_ext_v1370_s>;

// SCellToAddModListExt-v1430 ::= SEQUENCE (SIZE (1..31)) OF SCellToAddModExt-v1430
using scell_to_add_mod_list_ext_v1430_l = dyn_array<scell_to_add_mod_ext_v1430_s>;

// SCellToReleaseList-r10 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (1..7)
using scell_to_release_list_r10_l = bounded_array<uint8_t, 4>;

// SL-CommTxPoolToAddMod-r12 ::= SEQUENCE
struct sl_comm_tx_pool_to_add_mod_r12_s {
  uint8_t                pool_id_r12 = 1;
  sl_comm_res_pool_r12_s pool_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CommTxPoolToAddModExt-r13 ::= SEQUENCE
struct sl_comm_tx_pool_to_add_mod_ext_r13_s {
  uint8_t                pool_id_v1310 = 5;
  sl_comm_res_pool_r12_s pool_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscTxResourceInfoPerFreq-r13 ::= SEQUENCE
struct sl_disc_tx_res_info_per_freq_r13_s {
  bool                             ext                                 = false;
  bool                             disc_tx_res_r13_present             = false;
  bool                             disc_tx_res_ps_r13_present          = false;
  bool                             disc_tx_ref_carrier_ded_r13_present = false;
  bool                             disc_cell_sel_info_r13_present      = false;
  uint32_t                         disc_tx_carrier_freq_r13            = 0;
  sl_disc_tx_res_r13_c             disc_tx_res_r13;
  sl_disc_tx_res_r13_c             disc_tx_res_ps_r13;
  sl_disc_tx_ref_carrier_ded_r13_c disc_tx_ref_carrier_ded_r13;
  cell_sel_info_nfreq_r13_s        disc_cell_sel_info_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-GapPatternList-r13 ::= SEQUENCE (SIZE (1..8)) OF SL-GapPattern-r13
using sl_gap_pattern_list_r13_l = dyn_array<sl_gap_pattern_r13_s>;

// SL-TF-IndexPair-r12 ::= SEQUENCE
struct sl_tf_idx_pair_r12_s {
  bool    disc_sf_idx_r12_present  = false;
  bool    disc_prb_idx_r12_present = false;
  uint8_t disc_sf_idx_r12          = 1;
  uint8_t disc_prb_idx_r12         = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-V2X-ConfigDedicated-r14 ::= SEQUENCE
struct sl_v2x_cfg_ded_r14_s {
  struct comm_tx_res_r14_c_ {
    struct setup_c_ {
      struct sched_r14_s_ {
        bool                        v2x_sched_pool_r14_present = false;
        bool                        mcs_r14_present            = false;
        fixed_bitstring<16>         sl_v_rnti_r14;
        mac_main_cfg_sl_r12_s       mac_main_cfg_r14;
        sl_comm_res_pool_v2x_r14_s  v2x_sched_pool_r14;
        uint8_t                     mcs_r14 = 0;
        lc_ch_group_info_list_r13_l lc_ch_group_info_list_r14;
      };
      struct ue_sel_r14_s_ {
        struct v2x_comm_tx_pool_normal_ded_r14_s_ {
          bool                                 pool_to_release_list_r14_present         = false;
          bool                                 pool_to_add_mod_list_r14_present         = false;
          bool                                 v2x_comm_tx_pool_sensing_cfg_r14_present = false;
          sl_tx_pool_to_release_list_v2x_r14_l pool_to_release_list_r14;
          sl_tx_pool_to_add_mod_list_v2x_r14_l pool_to_add_mod_list_r14;
          sl_comm_tx_pool_sensing_cfg_r14_s    v2x_comm_tx_pool_sensing_cfg_r14;
        };

        // member variables
        v2x_comm_tx_pool_normal_ded_r14_s_ v2x_comm_tx_pool_normal_ded_r14;
      };
      struct types_opts {
        enum options { sched_r14, ue_sel_r14, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sched_r14_s_& sched_r14()
      {
        assert_choice_type("scheduled-r14", type_.to_string(), "setup");
        return c.get<sched_r14_s_>();
      }
      ue_sel_r14_s_& ue_sel_r14()
      {
        assert_choice_type("ue-Selected-r14", type_.to_string(), "setup");
        return c.get<ue_sel_r14_s_>();
      }
      const sched_r14_s_& sched_r14() const
      {
        assert_choice_type("scheduled-r14", type_.to_string(), "setup");
        return c.get<sched_r14_s_>();
      }
      const ue_sel_r14_s_& ue_sel_r14() const
      {
        assert_choice_type("ue-Selected-r14", type_.to_string(), "setup");
        return c.get<ue_sel_r14_s_>();
      }
      sched_r14_s_& set_sched_r14()
      {
        set(types::sched_r14);
        return c.get<sched_r14_s_>();
      }
      ue_sel_r14_s_& set_ue_sel_r14()
      {
        set(types::ue_sel_r14);
        return c.get<ue_sel_r14_s_>();
      }

    private:
      types                                        type_;
      choice_buffer_t<sched_r14_s_, ue_sel_r14_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    comm_tx_res_r14_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-r14");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-r14");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct comm_tx_res_v1530_c_ {
    struct setup_c_ {
      struct sched_v1530_s_ {
        bool                          lc_ch_group_info_list_v1530_present = false;
        bool                          mcs_r15_present                     = false;
        lc_ch_group_info_list_v1530_l lc_ch_group_info_list_v1530;
        uint8_t                       mcs_r15 = 0;
      };
      struct ue_sel_v1530_s_ {
        bool                           v2x_freq_sel_cfg_list_r15_present = false;
        sl_v2x_freq_sel_cfg_list_r15_l v2x_freq_sel_cfg_list_r15;
      };
      struct types_opts {
        enum options { sched_v1530, ue_sel_v1530, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sched_v1530_s_& sched_v1530()
      {
        assert_choice_type("scheduled-v1530", type_.to_string(), "setup");
        return c.get<sched_v1530_s_>();
      }
      ue_sel_v1530_s_& ue_sel_v1530()
      {
        assert_choice_type("ue-Selected-v1530", type_.to_string(), "setup");
        return c.get<ue_sel_v1530_s_>();
      }
      const sched_v1530_s_& sched_v1530() const
      {
        assert_choice_type("scheduled-v1530", type_.to_string(), "setup");
        return c.get<sched_v1530_s_>();
      }
      const ue_sel_v1530_s_& ue_sel_v1530() const
      {
        assert_choice_type("ue-Selected-v1530", type_.to_string(), "setup");
        return c.get<ue_sel_v1530_s_>();
      }
      sched_v1530_s_& set_sched_v1530()
      {
        set(types::sched_v1530);
        return c.get<sched_v1530_s_>();
      }
      ue_sel_v1530_s_& set_ue_sel_v1530()
      {
        set(types::ue_sel_v1530);
        return c.get<ue_sel_v1530_s_>();
      }

    private:
      types                                            type_;
      choice_buffer_t<sched_v1530_s_, ue_sel_v1530_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    comm_tx_res_v1530_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-v1530");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-v1530");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };

  // member variables
  bool                              ext                                    = false;
  bool                              comm_tx_res_r14_present                = false;
  bool                              v2x_inter_freq_info_list_r14_present   = false;
  bool                              thres_sl_tx_prioritization_r14_present = false;
  bool                              type_tx_sync_r14_present               = false;
  bool                              cbr_ded_tx_cfg_list_r14_present        = false;
  comm_tx_res_r14_c_                comm_tx_res_r14;
  sl_inter_freq_info_list_v2x_r14_l v2x_inter_freq_info_list_r14;
  uint8_t                           thres_sl_tx_prioritization_r14 = 1;
  sl_type_tx_sync_r14_e             type_tx_sync_r14;
  sl_cbr_common_tx_cfg_list_r14_s   cbr_ded_tx_cfg_list_r14;
  // ...
  // group 0
  bool                                   slss_tx_multi_freq_r15_present = false;
  copy_ptr<comm_tx_res_v1530_c_>         comm_tx_res_v1530;
  copy_ptr<sl_v2x_packet_dupl_cfg_r15_s> v2x_packet_dupl_cfg_r15;
  copy_ptr<sl_v2x_sync_freq_list_r15_l>  sync_freq_list_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BandIndication-r14 ::= SEQUENCE
struct band_ind_r14_s {
  bool              ca_bw_class_ul_r14_present = false;
  uint16_t          band_eutra_r14             = 1;
  ca_bw_class_r10_e ca_bw_class_dl_r14;
  ca_bw_class_r10_e ca_bw_class_ul_r14;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// LWA-Configuration-r13 ::= CHOICE
struct lwa_cfg_r13_c {
  struct setup_s_ {
    lwa_cfg_r13_s lwa_cfg_r13;
  };
  typedef setup_e types;

  // choice methods
  lwa_cfg_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "LWA-Configuration-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "LWA-Configuration-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// LWIP-Configuration-r13 ::= CHOICE
struct lwip_cfg_r13_c {
  struct setup_s_ {
    lwip_cfg_r13_s lwip_cfg_r13;
  };
  typedef setup_e types;

  // choice methods
  lwip_cfg_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "LWIP-Configuration-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "LWIP-Configuration-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// MeasCSI-RS-Config-r12 ::= SEQUENCE
struct meas_csi_rs_cfg_r12_s {
  bool             ext                = false;
  uint8_t          meas_csi_rs_id_r12 = 1;
  uint16_t         pci_r12            = 0;
  uint16_t         scrambling_id_r12  = 0;
  uint8_t          res_cfg_r12        = 0;
  uint8_t          sf_offset_r12      = 0;
  q_offset_range_e csi_rs_individual_offset_r12;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysCellIdRangeUTRA-FDD-r9 ::= SEQUENCE
struct pci_range_utra_fdd_r9_s {
  bool     range_r9_present = false;
  uint16_t start_r9         = 0;
  uint16_t range_r9         = 2;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PowerCoordinationInfo-r12 ::= SEQUENCE
struct pwr_coordination_info_r12_s {
  uint8_t p_menb_r12        = 1;
  uint8_t p_senb_r12        = 1;
  uint8_t pwr_ctrl_mode_r12 = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RAN-NotificationAreaInfo-r15 ::= CHOICE
struct ran_notif_area_info_r15_c {
  struct types_opts {
    enum options { cell_list_r15, ran_area_cfg_list_r15, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<types_opts> types;

  // choice methods
  ran_notif_area_info_r15_c() = default;
  ran_notif_area_info_r15_c(const ran_notif_area_info_r15_c& other);
  ran_notif_area_info_r15_c& operator=(const ran_notif_area_info_r15_c& other);
  ~ran_notif_area_info_r15_c() { destroy_(); }
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  plmn_ran_area_cell_list_r15_l& cell_list_r15()
  {
    assert_choice_type("cellList-r15", type_.to_string(), "RAN-NotificationAreaInfo-r15");
    return c.get<plmn_ran_area_cell_list_r15_l>();
  }
  plmn_ran_area_cfg_list_r15_l& ran_area_cfg_list_r15()
  {
    assert_choice_type("ran-AreaConfigList-r15", type_.to_string(), "RAN-NotificationAreaInfo-r15");
    return c.get<plmn_ran_area_cfg_list_r15_l>();
  }
  const plmn_ran_area_cell_list_r15_l& cell_list_r15() const
  {
    assert_choice_type("cellList-r15", type_.to_string(), "RAN-NotificationAreaInfo-r15");
    return c.get<plmn_ran_area_cell_list_r15_l>();
  }
  const plmn_ran_area_cfg_list_r15_l& ran_area_cfg_list_r15() const
  {
    assert_choice_type("ran-AreaConfigList-r15", type_.to_string(), "RAN-NotificationAreaInfo-r15");
    return c.get<plmn_ran_area_cfg_list_r15_l>();
  }
  plmn_ran_area_cell_list_r15_l& set_cell_list_r15()
  {
    set(types::cell_list_r15);
    return c.get<plmn_ran_area_cell_list_r15_l>();
  }
  plmn_ran_area_cfg_list_r15_l& set_ran_area_cfg_list_r15()
  {
    set(types::ran_area_cfg_list_r15);
    return c.get<plmn_ran_area_cfg_list_r15_l>();
  }

private:
  types                                                                        type_;
  choice_buffer_t<plmn_ran_area_cell_list_r15_l, plmn_ran_area_cfg_list_r15_l> c;

  void destroy_();
};

// RCLWI-Configuration-r13 ::= CHOICE
struct rclwi_cfg_r13_c {
  struct setup_s_ {
    rclwi_cfg_r13_s rclwi_cfg_r13;
  };
  typedef setup_e types;

  // choice methods
  rclwi_cfg_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RCLWI-Configuration-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RCLWI-Configuration-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReconfiguration-v1430-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1430_ies_s {
  bool                              sl_v2x_cfg_ded_r14_present              = false;
  bool                              scell_to_add_mod_list_ext_v1430_present = false;
  bool                              per_cc_gap_ind_request_r14_present      = false;
  bool                              sib_type2_ded_r14_present               = false;
  bool                              non_crit_ext_present                    = false;
  sl_v2x_cfg_ded_r14_s              sl_v2x_cfg_ded_r14;
  scell_to_add_mod_list_ext_v1430_l scell_to_add_mod_list_ext_v1430;
  dyn_octstring                     sib_type2_ded_r14;
  rrc_conn_recfg_v1510_ies_s        non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RadioResourceConfigDedicated-v1370 ::= SEQUENCE
struct rr_cfg_ded_v1370_s {
  bool                 phys_cfg_ded_v1370_present = false;
  phys_cfg_ded_v1370_s phys_cfg_ded_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCG-ConfigPartSCG-r12 ::= SEQUENCE
struct scg_cfg_part_scg_r12_s {
  bool                        ext                                   = false;
  bool                        rr_cfg_ded_scg_r12_present            = false;
  bool                        scell_to_release_list_scg_r12_present = false;
  bool                        pscell_to_add_mod_r12_present         = false;
  bool                        scell_to_add_mod_list_scg_r12_present = false;
  bool                        mob_ctrl_info_scg_r12_present         = false;
  rr_cfg_ded_scg_r12_s        rr_cfg_ded_scg_r12;
  scell_to_release_list_r10_l scell_to_release_list_scg_r12;
  ps_cell_to_add_mod_r12_s    pscell_to_add_mod_r12;
  scell_to_add_mod_list_r10_l scell_to_add_mod_list_scg_r12;
  mob_ctrl_info_scg_r12_s     mob_ctrl_info_scg_r12;
  // ...
  // group 0
  copy_ptr<scell_to_release_list_ext_r13_l> scell_to_release_list_scg_ext_r13;
  copy_ptr<scell_to_add_mod_list_ext_r13_l> scell_to_add_mod_list_scg_ext_r13;
  // group 1
  copy_ptr<scell_to_add_mod_list_ext_v1370_l> scell_to_add_mod_list_scg_ext_v1370;
  // group 2
  copy_ptr<ps_cell_to_add_mod_v1440_s> pscell_to_add_mod_v1440;
  // group 3
  copy_ptr<scell_group_to_release_list_r15_l> scell_group_to_release_list_scg_r15;
  copy_ptr<scell_group_to_add_mod_list_r15_l> scell_group_to_add_mod_list_scg_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCG-ConfigPartSCG-v12f0 ::= SEQUENCE
struct scg_cfg_part_scg_v12f0_s {
  bool                          pscell_to_add_mod_v12f0_present         = false;
  bool                          scell_to_add_mod_list_scg_v12f0_present = false;
  ps_cell_to_add_mod_v12f0_s    pscell_to_add_mod_v12f0;
  scell_to_add_mod_list_v10l0_l scell_to_add_mod_list_scg_v12f0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-CommTxPoolToAddModList-r12 ::= SEQUENCE (SIZE (1..4)) OF SL-CommTxPoolToAddMod-r12
using sl_comm_tx_pool_to_add_mod_list_r12_l = dyn_array<sl_comm_tx_pool_to_add_mod_r12_s>;

// SL-CommTxPoolToAddModListExt-r13 ::= SEQUENCE (SIZE (1..4)) OF SL-CommTxPoolToAddModExt-r13
using sl_comm_tx_pool_to_add_mod_list_ext_r13_l = dyn_array<sl_comm_tx_pool_to_add_mod_ext_r13_s>;

// SL-DiscSysInfoToReportFreqList-r13 ::= SEQUENCE (SIZE (1..8)) OF INTEGER (0..262143)
using sl_disc_sys_info_to_report_freq_list_r13_l = bounded_array<uint32_t, 8>;

// SL-DiscTxInfoInterFreqListAdd-r13 ::= SEQUENCE
struct sl_disc_tx_info_inter_freq_list_add_r13_s {
  using disc_tx_freq_to_add_mod_list_r13_l_ = dyn_array<sl_disc_tx_res_info_per_freq_r13_s>;
  using disc_tx_freq_to_release_list_r13_l_ = bounded_array<uint32_t, 8>;

  // member variables
  bool                                ext                                      = false;
  bool                                disc_tx_freq_to_add_mod_list_r13_present = false;
  bool                                disc_tx_freq_to_release_list_r13_present = false;
  disc_tx_freq_to_add_mod_list_r13_l_ disc_tx_freq_to_add_mod_list_r13;
  disc_tx_freq_to_release_list_r13_l_ disc_tx_freq_to_release_list_r13;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-GapConfig-r13 ::= SEQUENCE
struct sl_gap_cfg_r13_s {
  sl_gap_pattern_list_r13_l gap_pattern_list_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-TF-IndexPairList-r12 ::= SEQUENCE (SIZE (1..64)) OF SL-TF-IndexPair-r12
using sl_tf_idx_pair_list_r12_l = dyn_array<sl_tf_idx_pair_r12_s>;

// SL-TxPoolToReleaseListExt-r13 ::= SEQUENCE (SIZE (1..4)) OF INTEGER (5..8)
using sl_tx_pool_to_release_list_ext_r13_l = bounded_array<uint8_t, 4>;

// AltTTT-CellsToAddMod-r12 ::= SEQUENCE
struct alt_ttt_cells_to_add_mod_r12_s {
  uint8_t     cell_idx_r12 = 1;
  pci_range_s pci_range_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// BT-NameList-r15 ::= SEQUENCE (SIZE (1..4)) OF OCTET STRING (SIZE (1..248))
using bt_name_list_r15_l = bounded_array<dyn_octstring, 4>;

// BandCombination-r14 ::= SEQUENCE (SIZE (1..64)) OF BandIndication-r14
using band_combination_r14_l = dyn_array<band_ind_r14_s>;

// BlackCellsToAddMod ::= SEQUENCE
struct black_cells_to_add_mod_s {
  uint8_t     cell_idx = 1;
  pci_range_s pci_range;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellsToAddMod ::= SEQUENCE
struct cells_to_add_mod_s {
  uint8_t          cell_idx = 1;
  uint16_t         pci      = 0;
  q_offset_range_e cell_individual_offset;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellsToAddModCDMA2000 ::= SEQUENCE
struct cells_to_add_mod_cdma2000_s {
  uint8_t  cell_idx = 1;
  uint16_t pci      = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellsToAddModNR-r15 ::= SEQUENCE
struct cells_to_add_mod_nr_r15_s {
  uint8_t  cell_idx_r15 = 1;
  uint16_t pci_r15      = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellsToAddModUTRA-FDD ::= SEQUENCE
struct cells_to_add_mod_utra_fdd_s {
  uint8_t  cell_idx = 1;
  uint16_t pci      = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellsToAddModUTRA-TDD ::= SEQUENCE
struct cells_to_add_mod_utra_tdd_s {
  uint8_t cell_idx = 1;
  uint8_t pci      = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// FlightPathInfoReportConfig-r15 ::= SEQUENCE
struct flight_path_info_report_cfg_r15_s {
  bool    include_time_stamp_r15_present = false;
  uint8_t max_way_point_num_r15          = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasCSI-RS-ToAddModList-r12 ::= SEQUENCE (SIZE (1..96)) OF MeasCSI-RS-Config-r12
using meas_csi_rs_to_add_mod_list_r12_l = dyn_array<meas_csi_rs_cfg_r12_s>;

// MeasCSI-RS-ToRemoveList-r12 ::= SEQUENCE (SIZE (1..96)) OF INTEGER (1..96)
using meas_csi_rs_to_rem_list_r12_l = dyn_array<uint8_t>;

// MeasIdleConfigDedicated-r15 ::= SEQUENCE
struct meas_idle_cfg_ded_r15_s {
  struct meas_idle_dur_r15_opts {
    enum options { sec10, sec30, sec60, sec120, sec180, sec240, sec300, spare, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<meas_idle_dur_r15_opts> meas_idle_dur_r15_e_;

  // member variables
  bool                     ext                                      = false;
  bool                     meas_idle_carrier_list_eutra_r15_present = false;
  eutra_carrier_list_r15_l meas_idle_carrier_list_eutra_r15;
  meas_idle_dur_r15_e_     meas_idle_dur_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasSubframeCellList-r10 ::= SEQUENCE (SIZE (1..32)) OF PhysCellIdRange
using meas_sf_cell_list_r10_l = dyn_array<pci_range_s>;

// PhysCellIdRangeUTRA-FDDList-r9 ::= SEQUENCE (SIZE (1..4)) OF PhysCellIdRangeUTRA-FDD-r9
using pci_range_utra_fdd_list_r9_l = dyn_array<pci_range_utra_fdd_r9_s>;

// RRC-InactiveConfig-r15 ::= SEQUENCE
struct rrc_inactive_cfg_r15_s {
  struct ran_paging_cycle_r15_opts {
    enum options { rf32, rf64, rf128, rf256, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<ran_paging_cycle_r15_opts> ran_paging_cycle_r15_e_;
  struct periodic_rnau_timer_r15_opts {
    enum options { min5, min10, min20, min30, min60, min120, min360, min720, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<periodic_rnau_timer_r15_opts> periodic_rnau_timer_r15_e_;

  // member variables
  bool                       ran_paging_cycle_r15_present        = false;
  bool                       ran_notif_area_info_r15_present     = false;
  bool                       periodic_rnau_timer_r15_present     = false;
  bool                       next_hop_chaining_count_r15_present = false;
  bool                       non_crit_ext_present                = false;
  fixed_bitstring<40>        full_i_rnti_r15;
  fixed_bitstring<24>        short_i_rnti_r15;
  ran_paging_cycle_r15_e_    ran_paging_cycle_r15;
  ran_notif_area_info_r15_c  ran_notif_area_info_r15;
  periodic_rnau_timer_r15_e_ periodic_rnau_timer_r15;
  uint8_t                    next_hop_chaining_count_r15 = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReconfiguration-v1310-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1310_ies_s {
  bool                            scell_to_release_list_ext_r13_present = false;
  bool                            scell_to_add_mod_list_ext_r13_present = false;
  bool                            lwa_cfg_r13_present                   = false;
  bool                            lwip_cfg_r13_present                  = false;
  bool                            rclwi_cfg_r13_present                 = false;
  bool                            non_crit_ext_present                  = false;
  scell_to_release_list_ext_r13_l scell_to_release_list_ext_r13;
  scell_to_add_mod_list_ext_r13_l scell_to_add_mod_list_ext_r13;
  lwa_cfg_r13_c                   lwa_cfg_r13;
  lwip_cfg_r13_c                  lwip_cfg_r13;
  rclwi_cfg_r13_c                 rclwi_cfg_r13;
  rrc_conn_recfg_v1430_ies_s      non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReconfiguration-v1370-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1370_ies_s {
  bool                              rr_cfg_ded_v1370_present                = false;
  bool                              scell_to_add_mod_list_ext_v1370_present = false;
  bool                              non_crit_ext_present                    = false;
  rr_cfg_ded_v1370_s                rr_cfg_ded_v1370;
  scell_to_add_mod_list_ext_v1370_l scell_to_add_mod_list_ext_v1370;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SCG-Configuration-r12 ::= CHOICE
struct scg_cfg_r12_c {
  struct setup_s_ {
    struct scg_cfg_part_mcg_r12_s_ {
      bool                        ext                               = false;
      bool                        scg_counter_r12_present           = false;
      bool                        pwr_coordination_info_r12_present = false;
      uint32_t                    scg_counter_r12                   = 0;
      pwr_coordination_info_r12_s pwr_coordination_info_r12;
      // ...
    };

    // member variables
    bool                    scg_cfg_part_mcg_r12_present = false;
    bool                    scg_cfg_part_scg_r12_present = false;
    scg_cfg_part_mcg_r12_s_ scg_cfg_part_mcg_r12;
    scg_cfg_part_scg_r12_s  scg_cfg_part_scg_r12;
  };
  typedef setup_e types;

  // choice methods
  scg_cfg_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SCG-Configuration-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SCG-Configuration-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SCG-Configuration-v12f0 ::= CHOICE
struct scg_cfg_v12f0_c {
  struct setup_s_ {
    bool                     scg_cfg_part_scg_v12f0_present = false;
    scg_cfg_part_scg_v12f0_s scg_cfg_part_scg_v12f0;
  };
  typedef setup_e types;

  // choice methods
  scg_cfg_v12f0_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "SCG-Configuration-v12f0");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "SCG-Configuration-v12f0");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// SL-CommConfig-r12 ::= SEQUENCE
struct sl_comm_cfg_r12_s {
  struct comm_tx_res_r12_c_ {
    struct setup_c_ {
      struct sched_r12_s_ {
        bool                   mcs_r12_present = false;
        fixed_bitstring<16>    sl_rnti_r12;
        mac_main_cfg_sl_r12_s  mac_main_cfg_r12;
        sl_comm_res_pool_r12_s sc_comm_tx_cfg_r12;
        uint8_t                mcs_r12 = 0;
      };
      struct ue_sel_r12_s_ {
        struct comm_tx_pool_normal_ded_r12_s_ {
          bool                                  pool_to_release_list_r12_present = false;
          bool                                  pool_to_add_mod_list_r12_present = false;
          sl_tx_pool_to_release_list_r12_l      pool_to_release_list_r12;
          sl_comm_tx_pool_to_add_mod_list_r12_l pool_to_add_mod_list_r12;
        };

        // member variables
        comm_tx_pool_normal_ded_r12_s_ comm_tx_pool_normal_ded_r12;
      };
      struct types_opts {
        enum options { sched_r12, ue_sel_r12, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sched_r12_s_& sched_r12()
      {
        assert_choice_type("scheduled-r12", type_.to_string(), "setup");
        return c.get<sched_r12_s_>();
      }
      ue_sel_r12_s_& ue_sel_r12()
      {
        assert_choice_type("ue-Selected-r12", type_.to_string(), "setup");
        return c.get<ue_sel_r12_s_>();
      }
      const sched_r12_s_& sched_r12() const
      {
        assert_choice_type("scheduled-r12", type_.to_string(), "setup");
        return c.get<sched_r12_s_>();
      }
      const ue_sel_r12_s_& ue_sel_r12() const
      {
        assert_choice_type("ue-Selected-r12", type_.to_string(), "setup");
        return c.get<ue_sel_r12_s_>();
      }
      sched_r12_s_& set_sched_r12()
      {
        set(types::sched_r12);
        return c.get<sched_r12_s_>();
      }
      ue_sel_r12_s_& set_ue_sel_r12()
      {
        set(types::ue_sel_r12);
        return c.get<ue_sel_r12_s_>();
      }

    private:
      types                                        type_;
      choice_buffer_t<sched_r12_s_, ue_sel_r12_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    comm_tx_res_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-r12");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-r12");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct comm_tx_res_v1310_c_ {
    struct setup_c_ {
      struct sched_v1310_s_ {
        lc_ch_group_info_list_r13_l lc_ch_group_info_list_r13;
        bool                        multiple_tx_r13 = false;
      };
      struct ue_sel_v1310_s_ {
        struct comm_tx_pool_normal_ded_ext_r13_s_ {
          bool                                      pool_to_release_list_ext_r13_present = false;
          bool                                      pool_to_add_mod_list_ext_r13_present = false;
          sl_tx_pool_to_release_list_ext_r13_l      pool_to_release_list_ext_r13;
          sl_comm_tx_pool_to_add_mod_list_ext_r13_l pool_to_add_mod_list_ext_r13;
        };

        // member variables
        comm_tx_pool_normal_ded_ext_r13_s_ comm_tx_pool_normal_ded_ext_r13;
      };
      struct types_opts {
        enum options { sched_v1310, ue_sel_v1310, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sched_v1310_s_& sched_v1310()
      {
        assert_choice_type("scheduled-v1310", type_.to_string(), "setup");
        return c.get<sched_v1310_s_>();
      }
      ue_sel_v1310_s_& ue_sel_v1310()
      {
        assert_choice_type("ue-Selected-v1310", type_.to_string(), "setup");
        return c.get<ue_sel_v1310_s_>();
      }
      const sched_v1310_s_& sched_v1310() const
      {
        assert_choice_type("scheduled-v1310", type_.to_string(), "setup");
        return c.get<sched_v1310_s_>();
      }
      const ue_sel_v1310_s_& ue_sel_v1310() const
      {
        assert_choice_type("ue-Selected-v1310", type_.to_string(), "setup");
        return c.get<ue_sel_v1310_s_>();
      }
      sched_v1310_s_& set_sched_v1310()
      {
        set(types::sched_v1310);
        return c.get<sched_v1310_s_>();
      }
      ue_sel_v1310_s_& set_ue_sel_v1310()
      {
        set(types::ue_sel_v1310);
        return c.get<ue_sel_v1310_s_>();
      }

    private:
      types                                            type_;
      choice_buffer_t<sched_v1310_s_, ue_sel_v1310_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    comm_tx_res_v1310_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-v1310");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "commTxResources-v1310");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };

  // member variables
  bool               ext                     = false;
  bool               comm_tx_res_r12_present = false;
  comm_tx_res_r12_c_ comm_tx_res_r12;
  // ...
  // group 0
  bool                           comm_tx_allow_relay_ded_r13_present = false;
  copy_ptr<comm_tx_res_v1310_c_> comm_tx_res_v1310;
  bool                           comm_tx_allow_relay_ded_r13 = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-DiscConfig-r12 ::= SEQUENCE
struct sl_disc_cfg_r12_s {
  struct disc_tx_res_r12_c_ {
    struct setup_c_ {
      struct sched_r12_s_ {
        bool                      disc_tx_cfg_r12_present      = false;
        bool                      disc_tf_idx_list_r12_present = false;
        bool                      disc_hop_cfg_r12_present     = false;
        sl_disc_res_pool_r12_s    disc_tx_cfg_r12;
        sl_tf_idx_pair_list_r12_l disc_tf_idx_list_r12;
        sl_hop_cfg_disc_r12_s     disc_hop_cfg_r12;
      };
      struct ue_sel_r12_s_ {
        struct disc_tx_pool_ded_r12_s_ {
          bool                                  pool_to_release_list_r12_present = false;
          bool                                  pool_to_add_mod_list_r12_present = false;
          sl_tx_pool_to_release_list_r12_l      pool_to_release_list_r12;
          sl_disc_tx_pool_to_add_mod_list_r12_l pool_to_add_mod_list_r12;
        };

        // member variables
        bool                    disc_tx_pool_ded_r12_present = false;
        disc_tx_pool_ded_r12_s_ disc_tx_pool_ded_r12;
      };
      struct types_opts {
        enum options { sched_r12, ue_sel_r12, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sched_r12_s_& sched_r12()
      {
        assert_choice_type("scheduled-r12", type_.to_string(), "setup");
        return c.get<sched_r12_s_>();
      }
      ue_sel_r12_s_& ue_sel_r12()
      {
        assert_choice_type("ue-Selected-r12", type_.to_string(), "setup");
        return c.get<ue_sel_r12_s_>();
      }
      const sched_r12_s_& sched_r12() const
      {
        assert_choice_type("scheduled-r12", type_.to_string(), "setup");
        return c.get<sched_r12_s_>();
      }
      const ue_sel_r12_s_& ue_sel_r12() const
      {
        assert_choice_type("ue-Selected-r12", type_.to_string(), "setup");
        return c.get<ue_sel_r12_s_>();
      }
      sched_r12_s_& set_sched_r12()
      {
        set(types::sched_r12);
        return c.get<sched_r12_s_>();
      }
      ue_sel_r12_s_& set_ue_sel_r12()
      {
        set(types::ue_sel_r12);
        return c.get<ue_sel_r12_s_>();
      }

    private:
      types                                        type_;
      choice_buffer_t<sched_r12_s_, ue_sel_r12_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    disc_tx_res_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discTxResources-r12");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discTxResources-r12");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct disc_tf_idx_list_v1260_c_ {
    struct setup_s_ {
      sl_tf_idx_pair_list_r12b_l disc_tf_idx_list_r12b;
    };
    typedef setup_e types;

    // choice methods
    disc_tf_idx_list_v1260_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discTF-IndexList-v1260");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discTF-IndexList-v1260");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct disc_tx_res_ps_r13_c_ {
    struct setup_c_ {
      struct ue_sel_r13_s_ {
        sl_disc_tx_pool_ded_r13_s disc_tx_pool_ps_ded_r13;
      };
      struct types_opts {
        enum options { sched_r13, ue_sel_r13, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      setup_c_() = default;
      setup_c_(const setup_c_& other);
      setup_c_& operator=(const setup_c_& other);
      ~setup_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      sl_disc_tx_cfg_sched_r13_s& sched_r13()
      {
        assert_choice_type("scheduled-r13", type_.to_string(), "setup");
        return c.get<sl_disc_tx_cfg_sched_r13_s>();
      }
      ue_sel_r13_s_& ue_sel_r13()
      {
        assert_choice_type("ue-Selected-r13", type_.to_string(), "setup");
        return c.get<ue_sel_r13_s_>();
      }
      const sl_disc_tx_cfg_sched_r13_s& sched_r13() const
      {
        assert_choice_type("scheduled-r13", type_.to_string(), "setup");
        return c.get<sl_disc_tx_cfg_sched_r13_s>();
      }
      const ue_sel_r13_s_& ue_sel_r13() const
      {
        assert_choice_type("ue-Selected-r13", type_.to_string(), "setup");
        return c.get<ue_sel_r13_s_>();
      }
      sl_disc_tx_cfg_sched_r13_s& set_sched_r13()
      {
        set(types::sched_r13);
        return c.get<sl_disc_tx_cfg_sched_r13_s>();
      }
      ue_sel_r13_s_& set_ue_sel_r13()
      {
        set(types::ue_sel_r13);
        return c.get<ue_sel_r13_s_>();
      }

    private:
      types                                                      type_;
      choice_buffer_t<sl_disc_tx_cfg_sched_r13_s, ue_sel_r13_s_> c;

      void destroy_();
    };
    typedef setup_e types;

    // choice methods
    disc_tx_res_ps_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_c_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discTxResourcesPS-r13");
      return c;
    }
    const setup_c_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discTxResourcesPS-r13");
      return c;
    }
    setup_c_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_c_ c;
  };
  struct disc_tx_inter_freq_info_r13_c_ {
    struct setup_s_ {
      bool                                      disc_tx_carrier_freq_r13_present             = false;
      bool                                      disc_tx_ref_carrier_ded_r13_present          = false;
      bool                                      disc_tx_info_inter_freq_list_add_r13_present = false;
      uint32_t                                  disc_tx_carrier_freq_r13                     = 0;
      sl_disc_tx_ref_carrier_ded_r13_c          disc_tx_ref_carrier_ded_r13;
      sl_disc_tx_info_inter_freq_list_add_r13_s disc_tx_info_inter_freq_list_add_r13;
    };
    typedef setup_e types;

    // choice methods
    disc_tx_inter_freq_info_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discTxInterFreqInfo-r13");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discTxInterFreqInfo-r13");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };
  struct disc_rx_gap_cfg_r13_c_ {
    typedef setup_e types;

    // choice methods
    disc_rx_gap_cfg_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sl_gap_cfg_r13_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discRxGapConfig-r13");
      return c;
    }
    const sl_gap_cfg_r13_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discRxGapConfig-r13");
      return c;
    }
    sl_gap_cfg_r13_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types            type_;
    sl_gap_cfg_r13_s c;
  };
  struct disc_tx_gap_cfg_r13_c_ {
    typedef setup_e types;

    // choice methods
    disc_tx_gap_cfg_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sl_gap_cfg_r13_s& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discTxGapConfig-r13");
      return c;
    }
    const sl_gap_cfg_r13_s& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discTxGapConfig-r13");
      return c;
    }
    sl_gap_cfg_r13_s& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types            type_;
    sl_gap_cfg_r13_s c;
  };
  struct disc_sys_info_to_report_cfg_r13_c_ {
    typedef setup_e types;

    // choice methods
    disc_sys_info_to_report_cfg_r13_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    sl_disc_sys_info_to_report_freq_list_r13_l& setup()
    {
      assert_choice_type("setup", type_.to_string(), "discSysInfoToReportConfig-r13");
      return c;
    }
    const sl_disc_sys_info_to_report_freq_list_r13_l& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "discSysInfoToReportConfig-r13");
      return c;
    }
    sl_disc_sys_info_to_report_freq_list_r13_l& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types                                      type_;
    sl_disc_sys_info_to_report_freq_list_r13_l c;
  };

  // member variables
  bool               ext                     = false;
  bool               disc_tx_res_r12_present = false;
  disc_tx_res_r12_c_ disc_tx_res_r12;
  // ...
  // group 0
  copy_ptr<disc_tf_idx_list_v1260_c_> disc_tf_idx_list_v1260;
  // group 1
  bool                                         gap_requests_allowed_ded_r13_present = false;
  copy_ptr<disc_tx_res_ps_r13_c_>              disc_tx_res_ps_r13;
  copy_ptr<disc_tx_inter_freq_info_r13_c_>     disc_tx_inter_freq_info_r13;
  bool                                         gap_requests_allowed_ded_r13 = false;
  copy_ptr<disc_rx_gap_cfg_r13_c_>             disc_rx_gap_cfg_r13;
  copy_ptr<disc_tx_gap_cfg_r13_c_>             disc_tx_gap_cfg_r13;
  copy_ptr<disc_sys_info_to_report_cfg_r13_c_> disc_sys_info_to_report_cfg_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// SL-SyncTxControl-r12 ::= SEQUENCE
struct sl_sync_tx_ctrl_r12_s {
  struct network_ctrl_sync_tx_r12_opts {
    enum options { on, off, nulltype } value;

    std::string to_string() const;
  };
  typedef enumerated<network_ctrl_sync_tx_r12_opts> network_ctrl_sync_tx_r12_e_;

  // member variables
  bool                        network_ctrl_sync_tx_r12_present = false;
  network_ctrl_sync_tx_r12_e_ network_ctrl_sync_tx_r12;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// UECapabilityEnquiry-v1530-IEs ::= SEQUENCE
struct ue_cap_enquiry_v1530_ies_s {
  bool request_stti_spt_cap_r15_present = false;
  bool eutra_nr_only_r15_present        = false;
  bool non_crit_ext_present             = false;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// WLAN-ChannelList-r13 ::= SEQUENCE (SIZE (1..16)) OF INTEGER (0..255)
using wlan_ch_list_r13_l = bounded_array<uint16_t, 16>;

// WLAN-NameList-r15 ::= SEQUENCE (SIZE (1..4)) OF OCTET STRING (SIZE (1..32))
using wlan_name_list_r15_l = bounded_array<dyn_octstring, 4>;

// WhiteCellsToAddMod-r13 ::= SEQUENCE
struct white_cells_to_add_mod_r13_s {
  uint8_t     cell_idx_r13 = 1;
  pci_range_s pci_range_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// AltTTT-CellsToAddModList-r12 ::= SEQUENCE (SIZE (1..32)) OF AltTTT-CellsToAddMod-r12
using alt_ttt_cells_to_add_mod_list_r12_l = dyn_array<alt_ttt_cells_to_add_mod_r12_s>;

// BT-NameListConfig-r15 ::= CHOICE
struct bt_name_list_cfg_r15_c {
  typedef setup_e types;

  // choice methods
  bt_name_list_cfg_r15_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  bt_name_list_r15_l& setup()
  {
    assert_choice_type("setup", type_.to_string(), "BT-NameListConfig-r15");
    return c;
  }
  const bt_name_list_r15_l& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "BT-NameListConfig-r15");
    return c;
  }
  bt_name_list_r15_l& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types              type_;
  bt_name_list_r15_l c;
};

// BandCombinationList-r14 ::= SEQUENCE (SIZE (1..384)) OF BandCombination-r14
using band_combination_list_r14_l = dyn_array<band_combination_r14_l>;

// BlackCellsToAddModList ::= SEQUENCE (SIZE (1..32)) OF BlackCellsToAddMod
using black_cells_to_add_mod_list_l = dyn_array<black_cells_to_add_mod_s>;

// CDMA2000-Type ::= ENUMERATED
struct cdma2000_type_opts {
  enum options { type1_xrtt, type_hrpd, nulltype } value;
  typedef uint8_t number_type;

  std::string to_string() const;
  uint8_t     to_number() const;
};
typedef enumerated<cdma2000_type_opts> cdma2000_type_e;

// CSG-AllowedReportingCells-r9 ::= SEQUENCE
struct csg_allowed_report_cells_r9_s {
  bool                         pci_range_utra_fdd_list_r9_present = false;
  pci_range_utra_fdd_list_r9_l pci_range_utra_fdd_list_r9;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CandidateServingFreqListNR-r15 ::= SEQUENCE (SIZE (1..32)) OF INTEGER (0..3279165)
using candidate_serving_freq_list_nr_r15_l = bounded_array<uint32_t, 32>;

// CarrierFreqGERAN ::= SEQUENCE
struct carrier_freq_geran_s {
  uint16_t         arfcn = 0;
  band_ind_geran_e band_ind;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// CellIndexList ::= SEQUENCE (SIZE (1..32)) OF INTEGER (1..32)
using cell_idx_list_l = bounded_array<uint8_t, 32>;

// CellsToAddModList ::= SEQUENCE (SIZE (1..32)) OF CellsToAddMod
using cells_to_add_mod_list_l = dyn_array<cells_to_add_mod_s>;

// CellsToAddModListCDMA2000 ::= SEQUENCE (SIZE (1..32)) OF CellsToAddModCDMA2000
using cells_to_add_mod_list_cdma2000_l = dyn_array<cells_to_add_mod_cdma2000_s>;

// CellsToAddModListNR-r15 ::= SEQUENCE (SIZE (1..32)) OF CellsToAddModNR-r15
using cells_to_add_mod_list_nr_r15_l = dyn_array<cells_to_add_mod_nr_r15_s>;

// CellsToAddModListUTRA-FDD ::= SEQUENCE (SIZE (1..32)) OF CellsToAddModUTRA-FDD
using cells_to_add_mod_list_utra_fdd_l = dyn_array<cells_to_add_mod_utra_fdd_s>;

// CellsToAddModListUTRA-TDD ::= SEQUENCE (SIZE (1..32)) OF CellsToAddModUTRA-TDD
using cells_to_add_mod_list_utra_tdd_l = dyn_array<cells_to_add_mod_utra_tdd_s>;

// MeasCycleSCell-r10 ::= ENUMERATED
struct meas_cycle_scell_r10_opts {
  enum options { sf160, sf256, sf320, sf512, sf640, sf1024, sf1280, spare1, nulltype } value;
  typedef uint16_t number_type;

  std::string to_string() const;
  uint16_t    to_number() const;
};
typedef enumerated<meas_cycle_scell_r10_opts> meas_cycle_scell_r10_e;

// MeasDS-Config-r12 ::= CHOICE
struct meas_ds_cfg_r12_c {
  struct setup_s_ {
    struct dmtc_period_offset_r12_c_ {
      struct types_opts {
        enum options { ms40_r12, ms80_r12, ms160_r12, /*...*/ nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<types_opts, true> types;

      // choice methods
      dmtc_period_offset_r12_c_() = default;
      dmtc_period_offset_r12_c_(const dmtc_period_offset_r12_c_& other);
      dmtc_period_offset_r12_c_& operator=(const dmtc_period_offset_r12_c_& other);
      ~dmtc_period_offset_r12_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& ms40_r12()
      {
        assert_choice_type("ms40-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      uint8_t& ms80_r12()
      {
        assert_choice_type("ms80-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      uint8_t& ms160_r12()
      {
        assert_choice_type("ms160-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      const uint8_t& ms40_r12() const
      {
        assert_choice_type("ms40-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      const uint8_t& ms80_r12() const
      {
        assert_choice_type("ms80-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      const uint8_t& ms160_r12() const
      {
        assert_choice_type("ms160-r12", type_.to_string(), "dmtc-PeriodOffset-r12");
        return c.get<uint8_t>();
      }
      uint8_t& set_ms40_r12()
      {
        set(types::ms40_r12);
        return c.get<uint8_t>();
      }
      uint8_t& set_ms80_r12()
      {
        set(types::ms80_r12);
        return c.get<uint8_t>();
      }
      uint8_t& set_ms160_r12()
      {
        set(types::ms160_r12);
        return c.get<uint8_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };
    struct ds_occasion_dur_r12_c_ {
      struct types_opts {
        enum options { dur_fdd_r12, dur_tdd_r12, nulltype } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts> types;

      // choice methods
      ds_occasion_dur_r12_c_() = default;
      ds_occasion_dur_r12_c_(const ds_occasion_dur_r12_c_& other);
      ds_occasion_dur_r12_c_& operator=(const ds_occasion_dur_r12_c_& other);
      ~ds_occasion_dur_r12_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& dur_fdd_r12()
      {
        assert_choice_type("durationFDD-r12", type_.to_string(), "ds-OccasionDuration-r12");
        return c.get<uint8_t>();
      }
      uint8_t& dur_tdd_r12()
      {
        assert_choice_type("durationTDD-r12", type_.to_string(), "ds-OccasionDuration-r12");
        return c.get<uint8_t>();
      }
      const uint8_t& dur_fdd_r12() const
      {
        assert_choice_type("durationFDD-r12", type_.to_string(), "ds-OccasionDuration-r12");
        return c.get<uint8_t>();
      }
      const uint8_t& dur_tdd_r12() const
      {
        assert_choice_type("durationTDD-r12", type_.to_string(), "ds-OccasionDuration-r12");
        return c.get<uint8_t>();
      }
      uint8_t& set_dur_fdd_r12()
      {
        set(types::dur_fdd_r12);
        return c.get<uint8_t>();
      }
      uint8_t& set_dur_tdd_r12()
      {
        set(types::dur_tdd_r12);
        return c.get<uint8_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };

    // member variables
    bool                              ext                                     = false;
    bool                              meas_csi_rs_to_rem_list_r12_present     = false;
    bool                              meas_csi_rs_to_add_mod_list_r12_present = false;
    dmtc_period_offset_r12_c_         dmtc_period_offset_r12;
    ds_occasion_dur_r12_c_            ds_occasion_dur_r12;
    meas_csi_rs_to_rem_list_r12_l     meas_csi_rs_to_rem_list_r12;
    meas_csi_rs_to_add_mod_list_r12_l meas_csi_rs_to_add_mod_list_r12;
    // ...
  };
  typedef setup_e types;

  // choice methods
  meas_ds_cfg_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "MeasDS-Config-r12");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "MeasDS-Config-r12");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// MeasGapConfig ::= CHOICE
struct meas_gap_cfg_c {
  struct setup_s_ {
    struct gap_offset_c_ {
      struct types_opts {
        enum options {
          gp0,
          gp1,
          // ...
          gp2_r14,
          gp3_r14,
          gp_ncsg0_r14,
          gp_ncsg1_r14,
          gp_ncsg2_r14,
          gp_ncsg3_r14,
          gp_non_uniform1_r14,
          gp_non_uniform2_r14,
          gp_non_uniform3_r14,
          gp_non_uniform4_r14,
          gp4_r15,
          gp5_r15,
          gp6_r15,
          gp7_r15,
          gp8_r15,
          gp9_r15,
          gp10_r15,
          gp11_r15,
          nulltype
        } value;

        std::string to_string() const;
      };
      typedef enumerated<types_opts, true, 18> types;

      // choice methods
      gap_offset_c_() = default;
      gap_offset_c_(const gap_offset_c_& other);
      gap_offset_c_& operator=(const gap_offset_c_& other);
      ~gap_offset_c_() { destroy_(); }
      void        set(types::options e = types::nulltype);
      types       type() const { return type_; }
      SRSASN_CODE pack(bit_ref& bref) const;
      SRSASN_CODE unpack(cbit_ref& bref);
      void        to_json(json_writer& j) const;
      // getters
      uint8_t& gp0()
      {
        assert_choice_type("gp0", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp1()
      {
        assert_choice_type("gp1", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp2_r14()
      {
        assert_choice_type("gp2-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp3_r14()
      {
        assert_choice_type("gp3-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp_ncsg0_r14()
      {
        assert_choice_type("gp-ncsg0-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp_ncsg1_r14()
      {
        assert_choice_type("gp-ncsg1-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp_ncsg2_r14()
      {
        assert_choice_type("gp-ncsg2-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp_ncsg3_r14()
      {
        assert_choice_type("gp-ncsg3-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint16_t& gp_non_uniform1_r14()
      {
        assert_choice_type("gp-nonUniform1-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      uint16_t& gp_non_uniform2_r14()
      {
        assert_choice_type("gp-nonUniform2-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      uint16_t& gp_non_uniform3_r14()
      {
        assert_choice_type("gp-nonUniform3-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      uint16_t& gp_non_uniform4_r14()
      {
        assert_choice_type("gp-nonUniform4-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      uint8_t& gp4_r15()
      {
        assert_choice_type("gp4-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp5_r15()
      {
        assert_choice_type("gp5-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp6_r15()
      {
        assert_choice_type("gp6-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp7_r15()
      {
        assert_choice_type("gp7-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp8_r15()
      {
        assert_choice_type("gp8-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp9_r15()
      {
        assert_choice_type("gp9-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp10_r15()
      {
        assert_choice_type("gp10-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& gp11_r15()
      {
        assert_choice_type("gp11-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp0() const
      {
        assert_choice_type("gp0", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp1() const
      {
        assert_choice_type("gp1", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp2_r14() const
      {
        assert_choice_type("gp2-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp3_r14() const
      {
        assert_choice_type("gp3-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp_ncsg0_r14() const
      {
        assert_choice_type("gp-ncsg0-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp_ncsg1_r14() const
      {
        assert_choice_type("gp-ncsg1-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp_ncsg2_r14() const
      {
        assert_choice_type("gp-ncsg2-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp_ncsg3_r14() const
      {
        assert_choice_type("gp-ncsg3-r14", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint16_t& gp_non_uniform1_r14() const
      {
        assert_choice_type("gp-nonUniform1-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& gp_non_uniform2_r14() const
      {
        assert_choice_type("gp-nonUniform2-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& gp_non_uniform3_r14() const
      {
        assert_choice_type("gp-nonUniform3-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      const uint16_t& gp_non_uniform4_r14() const
      {
        assert_choice_type("gp-nonUniform4-r14", type_.to_string(), "gapOffset");
        return c.get<uint16_t>();
      }
      const uint8_t& gp4_r15() const
      {
        assert_choice_type("gp4-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp5_r15() const
      {
        assert_choice_type("gp5-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp6_r15() const
      {
        assert_choice_type("gp6-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp7_r15() const
      {
        assert_choice_type("gp7-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp8_r15() const
      {
        assert_choice_type("gp8-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp9_r15() const
      {
        assert_choice_type("gp9-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp10_r15() const
      {
        assert_choice_type("gp10-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      const uint8_t& gp11_r15() const
      {
        assert_choice_type("gp11-r15", type_.to_string(), "gapOffset");
        return c.get<uint8_t>();
      }
      uint8_t& set_gp0()
      {
        set(types::gp0);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp1()
      {
        set(types::gp1);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp2_r14()
      {
        set(types::gp2_r14);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp3_r14()
      {
        set(types::gp3_r14);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp_ncsg0_r14()
      {
        set(types::gp_ncsg0_r14);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp_ncsg1_r14()
      {
        set(types::gp_ncsg1_r14);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp_ncsg2_r14()
      {
        set(types::gp_ncsg2_r14);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp_ncsg3_r14()
      {
        set(types::gp_ncsg3_r14);
        return c.get<uint8_t>();
      }
      uint16_t& set_gp_non_uniform1_r14()
      {
        set(types::gp_non_uniform1_r14);
        return c.get<uint16_t>();
      }
      uint16_t& set_gp_non_uniform2_r14()
      {
        set(types::gp_non_uniform2_r14);
        return c.get<uint16_t>();
      }
      uint16_t& set_gp_non_uniform3_r14()
      {
        set(types::gp_non_uniform3_r14);
        return c.get<uint16_t>();
      }
      uint16_t& set_gp_non_uniform4_r14()
      {
        set(types::gp_non_uniform4_r14);
        return c.get<uint16_t>();
      }
      uint8_t& set_gp4_r15()
      {
        set(types::gp4_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp5_r15()
      {
        set(types::gp5_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp6_r15()
      {
        set(types::gp6_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp7_r15()
      {
        set(types::gp7_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp8_r15()
      {
        set(types::gp8_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp9_r15()
      {
        set(types::gp9_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp10_r15()
      {
        set(types::gp10_r15);
        return c.get<uint8_t>();
      }
      uint8_t& set_gp11_r15()
      {
        set(types::gp11_r15);
        return c.get<uint8_t>();
      }

    private:
      types               type_;
      pod_choice_buffer_t c;

      void destroy_();
    };

    // member variables
    gap_offset_c_ gap_offset;
  };
  typedef setup_e types;

  // choice methods
  meas_gap_cfg_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "MeasGapConfig");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "MeasGapConfig");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// MeasRSSI-ReportConfig-r13 ::= SEQUENCE
struct meas_rssi_report_cfg_r13_s {
  bool    ch_occupancy_thres_r13_present = false;
  uint8_t ch_occupancy_thres_r13         = 0;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasSensing-Config-r15 ::= SEQUENCE
struct meas_sensing_cfg_r15_s {
  struct sensing_periodicity_r15_opts {
    enum options { ms20, ms50, ms100, ms200, ms300, ms400, ms500, ms600, ms700, ms800, ms900, ms1000, nulltype } value;
    typedef uint16_t number_type;

    std::string to_string() const;
    uint16_t    to_number() const;
  };
  typedef enumerated<sensing_periodicity_r15_opts> sensing_periodicity_r15_e_;

  // member variables
  uint8_t                    sensing_subch_num_r15 = 1;
  sensing_periodicity_r15_e_ sensing_periodicity_r15;
  uint8_t                    sensing_resel_counter_r15 = 5;
  uint8_t                    sensing_prio_r15          = 1;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// MeasSubframePatternConfigNeigh-r10 ::= CHOICE
struct meas_sf_pattern_cfg_neigh_r10_c {
  struct setup_s_ {
    bool                    meas_sf_cell_list_r10_present = false;
    meas_sf_pattern_r10_c   meas_sf_pattern_neigh_r10;
    meas_sf_cell_list_r10_l meas_sf_cell_list_r10;
  };
  typedef setup_e types;

  // choice methods
  meas_sf_pattern_cfg_neigh_r10_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "MeasSubframePatternConfigNeigh-r10");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "MeasSubframePatternConfigNeigh-r10");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// MobilityControlInfo-v10l0 ::= SEQUENCE
struct mob_ctrl_info_v10l0_s {
  bool     add_spec_emission_v10l0_present = false;
  uint16_t add_spec_emission_v10l0         = 33;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// PhysCellIdGERAN ::= SEQUENCE
struct pci_geran_s {
  fixed_bitstring<3> network_colour_code;
  fixed_bitstring<3> base_station_colour_code;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// QuantityConfigRS-NR-r15 ::= SEQUENCE
struct quant_cfg_rs_nr_r15_s {
  bool        filt_coeff_rsrp_r15_present = false;
  bool        filt_coeff_rsrq_r15_present = false;
  bool        filt_coef_sinr_r13_present  = false;
  filt_coef_e filt_coeff_rsrp_r15;
  filt_coef_e filt_coeff_rsrq_r15;
  filt_coef_e filt_coef_sinr_r13;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RMTC-Config-r13 ::= CHOICE
struct rmtc_cfg_r13_c {
  struct setup_s_ {
    struct rmtc_period_r13_opts {
      enum options { ms40, ms80, ms160, ms320, ms640, nulltype } value;
      typedef uint16_t number_type;

      std::string to_string() const;
      uint16_t    to_number() const;
    };
    typedef enumerated<rmtc_period_r13_opts> rmtc_period_r13_e_;
    struct meas_dur_r13_opts {
      enum options { sym1, sym14, sym28, sym42, sym70, nulltype } value;
      typedef uint8_t number_type;

      std::string to_string() const;
      uint8_t     to_number() const;
    };
    typedef enumerated<meas_dur_r13_opts> meas_dur_r13_e_;

    // member variables
    bool               ext                        = false;
    bool               rmtc_sf_offset_r13_present = false;
    rmtc_period_r13_e_ rmtc_period_r13;
    uint16_t           rmtc_sf_offset_r13 = 0;
    meas_dur_r13_e_    meas_dur_r13;
    // ...
  };
  typedef setup_e types;

  // choice methods
  rmtc_cfg_r13_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
  // getters
  setup_s_& setup()
  {
    assert_choice_type("setup", type_.to_string(), "RMTC-Config-r13");
    return c;
  }
  const setup_s_& setup() const
  {
    assert_choice_type("setup", type_.to_string(), "RMTC-Config-r13");
    return c;
  }
  setup_s_& set_setup()
  {
    set(types::setup);
    return c;
  }

private:
  types    type_;
  setup_s_ c;
};

// RRCConnectionReconfiguration-v1250-IEs ::= SEQUENCE
struct rrc_conn_recfg_v1250_ies_s {
  struct wlan_offload_info_r12_c_ {
    struct setup_s_ {
      struct t350_r12_opts {
        enum options { min5, min10, min20, min30, min60, min120, min180, spare1, nulltype } value;
        typedef uint8_t number_type;

        std::string to_string() const;
        uint8_t     to_number() const;
      };
      typedef enumerated<t350_r12_opts> t350_r12_e_;

      // member variables
      bool                   t350_r12_present = false;
      wlan_offload_cfg_r12_s wlan_offload_cfg_ded_r12;
      t350_r12_e_            t350_r12;
    };
    typedef setup_e types;

    // choice methods
    wlan_offload_info_r12_c_() = default;
    void        set(types::options e = types::nulltype);
    types       type() const { return type_; }
    SRSASN_CODE pack(bit_ref& bref) const;
    SRSASN_CODE unpack(cbit_ref& bref);
    void        to_json(json_writer& j) const;
    // getters
    setup_s_& setup()
    {
      assert_choice_type("setup", type_.to_string(), "wlan-OffloadInfo-r12");
      return c;
    }
    const setup_s_& setup() const
    {
      assert_choice_type("setup", type_.to_string(), "wlan-OffloadInfo-r12");
      return c;
    }
    setup_s_& set_setup()
    {
      set(types::setup);
      return c;
    }

  private:
    types    type_;
    setup_s_ c;
  };

  // member variables
  bool                       wlan_offload_info_r12_present = false;
  bool                       scg_cfg_r12_present           = false;
  bool                       sl_sync_tx_ctrl_r12_present   = false;
  bool                       sl_disc_cfg_r12_present       = false;
  bool                       sl_comm_cfg_r12_present       = false;
  bool                       non_crit_ext_present          = false;
  wlan_offload_info_r12_c_   wlan_offload_info_r12;
  scg_cfg_r12_c              scg_cfg_r12;
  sl_sync_tx_ctrl_r12_s      sl_sync_tx_ctrl_r12;
  sl_disc_cfg_r12_s          sl_disc_cfg_r12;
  sl_comm_cfg_r12_s          sl_comm_cfg_r12;
  rrc_conn_recfg_v1310_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionReconfiguration-v12f0-IEs ::= SEQUENCE
struct rrc_conn_recfg_v12f0_ies_s {
  bool                       scg_cfg_v12f0_present     = false;
  bool                       late_non_crit_ext_present = false;
  bool                       non_crit_ext_present      = false;
  scg_cfg_v12f0_c            scg_cfg_v12f0;
  dyn_octstring              late_non_crit_ext;
  rrc_conn_recfg_v1370_ies_s non_crit_ext;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RRCConnectionRelease-v1530-IEs ::= SEQUENCE
struct rrc_conn_release_v1530_ies_s {
  struct cn_type_r15_opts {
    enum options { epc, fivegc, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<cn_type_r15_opts> cn_type_r15_e_;

  // member variables
  bool                    drb_continue_rohc_r15_present       = false;
  bool                    next_hop_chaining_count_r15_present = false;
  bool                    meas_idle_cfg_r15_present           = false;
  bool                    rrc_inactive_cfg_r15_present        = false;
  bool                    cn_type_r15_present                 = false;
  bool                    non_crit_ext_present                = false;
  uint8_t                 next_hop_chaining_count_r15         = 0;
  meas_idle_cfg_ded_r15_s meas_idle_cfg_r15;
  rrc_inactive_cfg_r15_s  rrc_inactive_cfg_r15;
  cn_type_r15_e_          cn_type_r15;

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RS-ConfigSSB-NR-r15 ::= SEQUENCE
struct rs_cfg_ssb_nr_r15_s {
  struct subcarrier_spacing_ssb_r15_opts {
    enum options { khz15, khz30, khz120, khz240, nulltype } value;
    typedef uint8_t number_type;

    std::string to_string() const;
    uint8_t     to_number() const;
  };
  typedef enumerated<subcarrier_spacing_ssb_r15_opts> subcarrier_spacing_ssb_r15_e_;

  // member variables
  bool                          ext = false;
  mtc_ssb_nr_r15_s              meas_timing_cfg_r15;
  subcarrier_spacing_ssb_r15_e_ subcarrier_spacing_ssb_r15;
  // ...

  // sequence methods
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  void        to_json(json_writer& j) const;
};

// RSRQ-RangeConfig-r12 ::= CHOICE
struct rsrq_range_cfg_r12_c {
  typedef setup_e types;

  // choice methods
  rsrq_range_cfg_r12_c() = default;
  void        set(types::options e = types::nulltype);
  types       type() const { return type_; }
  SRSASN_CODE pack(bit_ref& bref) const;
  SRSASN_CODE unpack(cbit_ref& bref);
  vo